/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Paragon CIM Protocol Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"
#include "cim.h"
#include "events.h"
#include "paragon_cim.h"

/* fix to CIM implementation bug */
#define WRITE_CIM_EEPROM_WORKAROUND            0
#define READ_CIM_EEPROM_WORKAROUND             1

#define CARRIAGE_RETURN                        0x0d

static char          display_rxpkt = 1;
static int           pkt_buffer = FPD_CIM_BUFFER;
static FPD_cmdprog_t pcmds[] = {
  /* link,           cmd,            len, valid, ram, chksum, dest */
    {   0, PCMD_AUTO_SKEW,             1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_PHSYNC_PVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_PHSYNC_NVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_NHSYNC_PVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_NHSYNC_NVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_CONFIGURATION,     5,     1,   1,      1, FPD_CIM_BUFFER},
    {   0, PCMD_PS2KBD_DATA1,          2,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_PS2KBD_DATA2,          3,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_SUNKBD_DATA1,          2,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_SUNKBD_DATA2,          3,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_SUNKBD_DATA3,          4,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_PS2MOUSE_CMD1,         2,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_PS2MOUSE_CMD2,         3,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_EEPROM_REPORT,     4,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_FOR_POWER_CIM,         0,     1,   0,      1, FPD_CIM_BUFFER},
    {   0, PCMD_FOR_POWER_STRIP,       0,     1,   0,      1, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_DEV_TYPE,  27,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_CONT_SEND,  1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_ERASE_ERR,  1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_PROG_ERR,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_BLKCHK_ERR, 1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_ACK,        1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_NACK,       1,     1,   1,      0, FPD_CIM_BUFFER},
};

static int invalidate_all_cmds( int fd, int link_if )
{
    FPD_cmdinv_t cmd;
    int rc;

    cmd.link_if = link_if;
    rc = ioctl(fd, FPD_INVALIDATE_ALL_CMDS, &cmd);

    return rc;
}

unsigned char compute_paragon_chksum( unsigned char *pbuf, int len )
{
    unsigned char chksum = 0, i;

    for( i = 0; i < len; i++ ) {
        chksum -= pbuf[i];
    }

    return chksum;
}

static int configure_lookup_table( int fd, int link_if )
{
    int i, rc;
    int n_cmds = sizeof(pcmds)/sizeof(pcmds[0]);

    /* invalidate all commands in the lookup table */
    if( invalidate_all_cmds(fd, link_if) != 0 ) {
        fprintf(stderr, "ERROR: Failed to invalidate all cmds!\n");
        return -1;
    }

    /* write commands */
    for( i = 0; i < n_cmds; i++ ) {
        pcmds[i].link_if = link_if;
        rc = ioctl(fd, FPD_WRITE_ENTRY_CMD_TABLE, &pcmds[i]);
        if( rc != 0 ) {
            fprintf(stderr, "ERROR: FPD_WRITE_ENTRY_CMD_TABLE ioctl failed (%s)\n", strerror(errno));
            if( invalidate_all_cmds(fd, link_if) != 0 ) {
                fprintf(stderr, "ERROR: Failed to invalidate all cmds!\n");
                return -1;
            }
            return -2;
        }
    }

    return 0;
}

void dump_data( unsigned char *bufaddr, int pktlen )
{
    int i;

    for( i = 0; i < pktlen; i++ ) {
        printf("%02x ", bufaddr[i]);
    }
    printf("\n");

    return;
}

static int get_echo_rsp( int fd, unsigned int link_if, unsigned char *pdata )
{
    FPD_echorsp_t cmd;
    int rc;

    cmd.link_if = link_if;
    rc = ioctl(fd, FPD_GET_ECHO_RSP, &cmd);
    if( rc == 0 ) {
        *pdata = (unsigned char)cmd.echo_rsp;
    }
    else {
        fprintf(stderr, "ERROR: FPD_GET_ECHO_RSP ioctl failed (%s)\n", strerror(errno));
    }

    return rc;
}

int paragon_cim_send_pkt( int fd, unsigned int link_if, FPD_data_t *pdata )
{
    int rc;

    rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
    if( rc == 0 ) {
        if( pdata->actual_len != pdata->requested_len ) {
            fprintf(stderr, "FAILED.\n");
            fprintf(stderr, "Actual bytes written %d\n", pdata->actual_len);
            return -1;
        }
    }
    else {
        fprintf(stderr, "FAILED.\n");
        if(errno) {
            fprintf(stderr, "FPD_SEND_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
        }
        return -2;
    }

    return 0;
}

int paragon_cim_receive_pkt( int fd, unsigned int link_if, FPD_data_t *pdata )
{
    int rc;

    rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
    if( rc != 0 ) {
        fprintf(stderr, "Reading %d bytes of data from %s buffer....FAILED\n",
                pdata->requested_len, pdata->type?"Priority":"CIM");
        if(errno) {
            fprintf(stderr, "FPD_RECEIVE_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
        }
    }

    return rc;
}

int paragon_cim_init( int fd, int link_if, int connected )
{
    line_connection_t old_conn, *pconn = NULL;

    if( connected ) {
        /* retrieve current line connection to be used for re-connection */
        if( cim_get_line_connection_info(link_if, &pconn) == 0 ) {
            old_conn = *pconn;
        }

        /* required to disconnect link before re-configuring */
        if( cim_disconnect_link(fd, link_if) != 0 ) {
            fprintf(stderr, "\nERROR: Failed to disconnect Link %d\n", link_if);
            return -1;
        }
    }

    if( configure_lookup_table(fd, link_if) != 0 ) {
        return -2;
    }

    if( connected ) {
        /* re-connect link */
        pconn = &old_conn;
        if( cim_switch_link(fd, link_if, pconn->target_port, pconn->protocol, pconn->tx_parity, pconn->rx_parity) != 0 ) {
            printf("\nERROR: Failed to re-connect Link %d\n", link_if);
            return -3;
        }
    }

    return 0;
}

int paragon_cim_send_echo( int fd, int link_if, unsigned char *pdev_id )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    int correct_rsp = 0;
    unsigned char device_id;

    pdata->link_if = link_if;
    pdata->requested_len = 1;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_ECHO_REQUEST;

    printf("Sending ECHO REQUEST......................");

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.\n");

        /* wait for reply */
        while( !correct_rsp ) {
            if( cim_poll_once(fd, DEFAULT_POLL_TIMEOUT, pevt) != 0 ) {
                return -1;
            }

            memset(&process, 0, sizeof(rx_notify_t));
            process.link_if = link_if;
            handle_events(fd, pevt, &process);

            if( process.rx_cim ) {
                /* retrieve incoming data */
                pdata->link_if = link_if;
                pdata->requested_len = 256;
                pdata->type = pkt_buffer;
                pdata->buf = protocol_buffer;
                paragon_cim_receive_pkt(fd, link_if, pdata);

                if( display_rxpkt ) {
                    if( pdata->actual_len > 0 ) {
                        printf("RxPKT: ");
                        dump_data(pdata->buf, pdata->actual_len);
                    }
                }
            }
            else if( process.rx_priority ) {
                fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
            }

            if( process.echo_rsp ) {
                printf("Detecting CIM Type........................");
                correct_rsp = 1;
                /* get echo response */
                if( get_echo_rsp(fd, link_if, &device_id) == 0 ) {
                    if( pdev_id ) {
                        *pdev_id = device_id;
                    }

                    switch( device_id ) {
                        case DEVICE_ID_POWER_CIM_OLD:
                            printf("old Power CIM\n");
                            break;
                        case DEVICE_ID_POWER_CIM_NEW:
                            printf("KX2.0 Power CIM\n");
                            break;
                        case DEVICE_ID_PC_CIM:
                            printf("PC CIM\n");
                            break;
                        case DEVICE_ID_P2CIM_PS2DUAL:
                            printf("P2CIM-PS2DUAL\n");
                            break;
                        case DEVICE_ID_SUN_CIM:
                            printf("SUN CIM\n");
                            break;
                        default:
                            printf("Unknown device [%02x]\n", device_id);
                            return -2;
                    }
                }
            }
        }
    }

    return 0;
}

int paragon_cim_send_kbms_init( int fd, int link_if )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    int correct_rsp = 0;

    pdata->link_if = link_if;
    pdata->requested_len = 1;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_KBMS_INIT;

    printf("\nSending KB/MS INIT....");

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.\n");

        /* wait for reply */
        while( !correct_rsp ) {
            if( cim_poll_once(fd, DEFAULT_POLL_TIMEOUT, pevt) != 0 ) {
                return -1;
            }

            memset(&process, 0, sizeof(rx_notify_t));
            process.link_if = link_if;
            handle_events(fd, pevt, &process);

            if( process.rx_cim ) {
                do {
                    /* retrieve incoming data */
                    pdata->link_if = link_if;
                    pdata->requested_len = 256;
                    pdata->type = pkt_buffer;
                    pdata->buf = protocol_buffer;
                    paragon_cim_receive_pkt(fd, link_if, pdata);

                    if( pdata->actual_len > 0 ) {
                        printf("RxPKT: ");
                        dump_data(pdata->buf, pdata->actual_len);

                        /* don't know what to expect yet so we'll wait for timeout */
                    }
                } while( pdata->actual_len > 0 );
            }
            else if( process.rx_priority ) {
                fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
            }
        }
    }

    return 0;
}

int paragon_cim_send_beep( int fd, int link_if, unsigned char beep_time )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 2;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_BEEP;
    pdata->buf[1] = beep_time;

    printf("\nSending BEEP....");

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.\n");
    }

    /* no reply expected */

    return 0;
}

int paragon_cim_send_eeprom_read( int fd, int link_if, char offset, pcim_eeprom_report_t *preport )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    int correct_rsp = 0;

    pdata->link_if = link_if;
    pdata->requested_len = 2;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_READ_CIM_EEPROM;
    pdata->buf[1] = offset;

    printf("Sending READ_CIM_EEPROM offset %02x.........", offset);

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.\n");

        /* wait for reply */
        while( !correct_rsp ) {
            if( cim_poll_once(fd, DEFAULT_POLL_TIMEOUT, pevt) != 0 ) {
                return -1;
            }

            memset(&process, 0, sizeof(rx_notify_t));
            process.link_if = link_if;
            handle_events(fd, pevt, &process);

            if( process.rx_cim ) {
                do {
                    /* retrieve incoming data */
                    pdata->link_if = link_if;
                    pdata->requested_len = 256;
                    pdata->type = pkt_buffer;
                    pdata->buf = protocol_buffer;
                    paragon_cim_receive_pkt(fd, link_if, pdata);

                    if( pdata->actual_len > 0 ) {
                        if( display_rxpkt ) {
                            printf("RxPKT: ");
                            dump_data(pdata->buf, pdata->actual_len);
                        }

                        if( pdata->buf[0] == PCMD_CIM_EEPROM_REPORT ) {
                            if( pdata->buf[1] == offset ) {
                                correct_rsp = 1;
                                if( preport ) {
#if READ_CIM_EEPROM_WORKAROUND
                                    preport->cmd = pdata->buf[0];
                                    preport->addr = pdata->buf[1];
                                    preport->datalo = pdata->buf[3];
                                    preport->datahi = pdata->buf[2];
#else
                                    memcpy(preport, pdata->buf, sizeof(pcim_eeprom_report_t));
#endif
                                }
                            }
                            else {
                                printf("Wrong offset: 0x%02x (exp 0x%02x)\n", pdata->buf[1], offset);
                            }
                        }
                    }
                } while( pdata->actual_len > 0 );
            }
            else if( process.rx_priority ) {
                fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
            }
        }
    }

    return 0;
}

int paragon_cim_send_eeprom_write( int fd, int link_if, char offset, char datalo, char datahi )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    pcim_eeprom_report_t eeprom;

    pdata->link_if = link_if;
    pdata->requested_len = 4;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_WRITE_CIM_EEPROM;
    pdata->buf[1] = offset;
#if WRITE_CIM_EEPROM_WORKAROUND
    pdata->buf[2] = datahi;
    pdata->buf[3] = datalo;
#else
    pdata->buf[2] = datalo;
    pdata->buf[3] = datahi;
#endif

    printf("\nSending WRITE_CIM_EEPROM offset %02x....", offset);

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.");
    }

    /* only way to verify is to read back the EEPROM */
    if( paragon_cim_send_eeprom_read(fd, link_if, offset, &eeprom) == 0 ) {
        printf("Verifying EEPROM offset %02x............", offset);
        if( eeprom.datalo == datalo && eeprom.datahi == datahi ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED\n");
            printf("Offset %02x: Expected: %02x    Got: %02x\n",
                   offset, datalo, eeprom.datalo);
            printf("Offset %02x: Expected: %02x    Got: %02x\n",
                   offset+1, datahi, eeprom.datahi);
        }
    }
    else {
        printf("FAILED\n");
        return -1;
    }

    return 0;
}

int paragon_cim_get_serial_number( int fd, int link_if )
{
    unsigned char sernum[PCIM_EEPROM_SERIAL_NUMBER_SIZE];
    pcim_eeprom_report_t eeprom;
    int cim_type;
    int offset;
    int size, revised_size;
    int i;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_SERIAL_NUMBER;
        size = PCIM_EEPROM_SERIAL_NUMBER_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_SERIAL_NUMBER;
        size = DCIM_EEPROM_SERIAL_NUMBER_SIZE;
    }
    else {
        return -1;
    }

    revised_size = size >> 1;
    if( size & 0x1 ) {
        ++revised_size;
    }
    for( i = 0; i < revised_size; i++ ) {
        if( paragon_cim_send_eeprom_read(fd, link_if, offset+i, &eeprom) == 0 ) {
            sernum[2*i] = eeprom.datalo;
            sernum[2*i+1] = eeprom.datahi;
        }
        else {
            return -2;
        }
    }

    printf("Serial Number: ");
    for( i = 0; i < size; i++ ) {
        printf("%02x ", sernum[i]);
    }
    printf("\n");

    return 0;
}

int paragon_cim_get_timestamp( int fd, int link_if )
{
    unsigned short timestamp;
    pcim_eeprom_report_t eeprom;
    int cim_type;
    int offset;
    int size;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_TIMESTAMP;
        size = PCIM_EEPROM_TIMESTAMP_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_TIMESTAMP;
        size = DCIM_EEPROM_TIMESTAMP_SIZE;
    }
    else {
        return -1;
    }

    if( paragon_cim_send_eeprom_read(fd, link_if, offset, &eeprom) == 0 ) {
        timestamp = eeprom.datalo | (eeprom.datahi << 8);
    }
    else {
        return -2;
    }

    printf("Timestamp: %d (0x%04x)\n", timestamp, timestamp);

    return 0;
}

int paragon_cim_get_channel_name( int fd, int link_if )
{
    char name[DCIM_EEPROM_CHANNEL_NAME_SIZE];
    pcim_eeprom_report_t eeprom;
    int cim_type;
    int offset;
    int size, revised_size;
    int i;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_CHANNEL_NAME;
        size = PCIM_EEPROM_CHANNEL_NAME_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_CHANNEL_NAME;
        size = DCIM_EEPROM_CHANNEL_NAME_SIZE;
    }
    else {
        return -1;
    }

    revised_size = size >> 1;
    if( size & 0x1 ) {
        ++revised_size;
    }
    for( i = 0; i < revised_size; i++ ) {
        if( paragon_cim_send_eeprom_read(fd, link_if, offset+i, &eeprom) == 0 ) {
            name[2*i] = eeprom.datalo;
            name[2*i+1] = eeprom.datahi;
        }
        else {
            return -2;
        }
    }

    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_NAME_EXTENSION;
        size = PCIM_EEPROM_NAME_EXTENSION_SIZE;

        revised_size = size >> 1;
        if( size & 0x1 ) {
            ++revised_size;
        }
        for( i = 0; i < revised_size; i++ ) {
            if( paragon_cim_send_eeprom_read(fd, link_if, offset+i, &eeprom) == 0 ) {
                name[2*i+PCIM_EEPROM_CHANNEL_NAME_SIZE] = eeprom.datalo;
                name[2*i+PCIM_EEPROM_CHANNEL_NAME_SIZE+1] = eeprom.datahi;
            }
            else {
                return -3;
            }
        }

        size = PCIM_EEPROM_CHANNEL_NAME_SIZE + PCIM_EEPROM_NAME_EXTENSION_SIZE;
    }

    printf("Channel Name: %s [ ", name);
    for( i = 0; i < size; i++ ) {
        printf("%02x ", name[i]);
    }
    printf("]\n");

    return 0;
}

int paragon_cim_get_device_type( int fd, int link_if, char *ptype )
{
    char name[PCIM_EEPROM_DEVICE_TYPE_SIZE];
    pcim_eeprom_report_t eeprom;
    int cim_type;
    int offset;
    int size, revised_size;
    int i;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_DEVICE_TYPE;
        size = PCIM_EEPROM_DEVICE_TYPE_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_DEVICE_TYPE;
        size = DCIM_EEPROM_DEVICE_TYPE_SIZE;
    }
    else {
        return -1;
    }

    revised_size = size >> 1;
    if( size & 0x1 ) {
        ++revised_size;
    }
    for( i = 0; i < revised_size; i++ ) {
        if( paragon_cim_send_eeprom_read(fd, link_if, offset+i, &eeprom) == 0 ) {
            name[2*i] = eeprom.datalo;
            /* only 3 bytes are valid */
            if( i == 0 ) {
                name[2*i+1] = eeprom.datahi;
            }
        }
        else {
            return -2;
        }
    }

    if( ptype != NULL ) {
        strncpy(ptype, name, PCIM_EEPROM_DEVICE_TYPE_SIZE);
    }

    printf("Device Type: %s [ ", name);
    for( i = 0; i < size; i++ ) {
        printf("%02x ", name[i]);
    }
    printf("]\n");

    return 0;
}

int paragon_cim_get_fw_version( int fd, int link_if )
{
    char version[PCIM_EEPROM_FW_VERSION_SIZE];
    pcim_eeprom_report_t eeprom;
    int cim_type;
    int offset;
    int size, revised_size;
    int i;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_FW_VERSION;
        size = PCIM_EEPROM_FW_VERSION_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_FW_VERSION;
        size = DCIM_EEPROM_FW_VERSION_SIZE;
    }
    else {
        return -1;
    }

    revised_size = size >> 1;
    if( size & 0x1 ) {
        ++revised_size;
    }
    for( i = 0; i < revised_size; i++ ) {
        if( paragon_cim_send_eeprom_read(fd, link_if, offset+i, &eeprom) == 0 ) {
            version[2*i] = eeprom.datalo;
            version[2*i+1] = eeprom.datahi;
        }
        else {
            return -2;
        }
    }

    printf("Firmware Version: ");
    for( i = 0; i < size; i++ ) {
        printf("%02x ", version[i]);
    }
    printf("\n");

    return 0;
}

int paragon_cim_get_hw_version( int fd, int link_if )
{
    char version[PCIM_EEPROM_HW_VERSION_SIZE];
    pcim_eeprom_report_t eeprom;
    int cim_type;
    int offset;
    int size, revised_size;
    int i;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_HW_VERSION;
        size = PCIM_EEPROM_HW_VERSION_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_HW_VERSION;
        size = DCIM_EEPROM_HW_VERSION_SIZE;
    }
    else {
        return -1;
    }

    revised_size = size >> 1;
    if( size & 0x1 ) {
        ++revised_size;
    }
    for( i = 0; i < revised_size; i++ ) {
        if( paragon_cim_send_eeprom_read(fd, link_if, offset+i, &eeprom) == 0 ) {
            version[2*i] = eeprom.datalo;
            version[2*i+1] = eeprom.datahi;
        }
        else {
            return -2;
        }
    }

    printf("Hardware Version: ");
    for( i = 0; i < size; i++ ) {
        printf("%02x ", version[i]);
    }
    printf("\n");

    return 0;
}

int paragon_cim_set_timestamp( int fd, int link_if )
{
    unsigned short timestamp;
    int cim_type;
    int offset;
    unsigned char lval, hval;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_TIMESTAMP;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_TIMESTAMP;
    }
    else {
        return -1;
    }

    /* ask user for new timestamp */
    printf("\nEnter Timestamp (in decimal): ");
    scanf("%hd", &timestamp);
    lval = timestamp & 0xff;
    hval = (timestamp & 0xff00) >> 8;
    
    if( paragon_cim_send_eeprom_write(fd, link_if, offset, lval, hval) == 0 ) {
        return -2;
    }

    return 0;
}

int paragon_cim_set_channel_name( int fd, int link_if )
{
    char name[128];
    int cim_type;
    int offset;
    int size, revised_size;
    int i;

    /* determine CIM type */
    cim_type = cim_get_type(link_if);
    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_CHANNEL_NAME;
        size = PCIM_EEPROM_CHANNEL_NAME_SIZE;
    }
    else if( cim_type == DCIM ) {
        offset = DCIM_EEPROM_CHANNEL_NAME;
        size = DCIM_EEPROM_CHANNEL_NAME_SIZE;
    }
    else {
        return -1;
    }

    memset(name, 0, 128);
    fgets(name, 128, stdin);
    printf("\nEnter Server Name [max 20 chars]: ");
    if( fgets(name, 128, stdin) != name ) {
        printf("Oh Crap!\n");
        return -1;
    }

    revised_size = size >> 1;
    if( size & 0x1 ) {
        ++revised_size;
    }
    for( i = 0; i < revised_size; i++ ) {
        if( paragon_cim_send_eeprom_write(fd, link_if, offset+i, name[2*i], name[2*i+1]) != 0 ) {
            return -2;
        }
    }

    if( cim_type == PCIM ) {
        offset = PCIM_EEPROM_NAME_EXTENSION;
        size = PCIM_EEPROM_NAME_EXTENSION_SIZE;

        revised_size = size >> 1;
        if( size & 0x1 ) {
            ++revised_size;
        }
        for( i = 0; i < revised_size; i++ ) {
            if( paragon_cim_send_eeprom_write(fd, link_if, offset+i, name[2*i+PCIM_EEPROM_CHANNEL_NAME_SIZE], name[2*i+1+PCIM_EEPROM_CHANNEL_NAME_SIZE]) != 0 ) {
                return -2;
            }
        }
    }

    printf("Channel Name: %s [ ", name);
    for( i = 0; i < size; i++ ) {
        printf("%02x ", name[i]);
    }
    printf("]\n");

    return 0;
}

void paragon_cim_set_display_rxpkt( char val)
{
    display_rxpkt = val;
    return;
}

int paragon_cim_send_power_cmd( int fd, int link_if, char id, powercim_msg_t *pmsg )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    int correct_rsp = 0;
    int i;

    pdata->link_if = link_if;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;
    display_rxpkt = 1;

    switch( id ) {
        case POWER_REQUEST_INFO:
            pdata->requested_len = 4;
            pdata->buf[0] = PCMD_FOR_POWER_CIM;
            pdata->buf[1] = 2;
            pdata->buf[2] = id;
            pdata->buf[3] = compute_paragon_chksum(pdata->buf, 3);
            printf("Sending Power CIM Request Info.........");
            break;
        case POWER_ENTER_BSNMP:
            pdata->requested_len = 4;
            pdata->buf[0] = PCMD_FOR_POWER_CIM;
            pdata->buf[1] = 2;
            pdata->buf[2] = id;
            pdata->buf[3] = compute_paragon_chksum(pdata->buf, 3);
            printf("Sending Power CIM Enter BSNMP..........");
            break;
        case POWER_READ_EEPROM:
            pdata->requested_len = 7;
            pdata->buf[0] = PCMD_FOR_POWER_CIM;
            pdata->buf[1] = 5;
            pdata->buf[2] = id;
            pdata->buf[3] = (pmsg->eeprom_rd.addr & 0xff00) >> 8;
            pdata->buf[4] = (pmsg->eeprom_rd.addr & 0xff);
            pdata->buf[5] = pmsg->eeprom_rd.len;
            pdata->buf[6] = compute_paragon_chksum(pdata->buf, 6);
            printf("Sending Power CIM Read EEPROM..........");
            break;
        case POWER_WRITE_EEPROM:
            pdata->requested_len = 12;
            pdata->buf[0] = PCMD_FOR_POWER_CIM;
            pdata->buf[1] = 0x0a;
            pdata->buf[2] = id;
            pdata->buf[3] = (pmsg->eeprom_wr.addr & 0xff00) >> 8;
            pdata->buf[4] = (pmsg->eeprom_wr.addr & 0xff);
            pdata->buf[5] = pmsg->eeprom_wr.len;
            for( i = 0; i < pmsg->eeprom_wr.len; i++ ) {
                pdata->buf[6+i] = pmsg->eeprom_wr.data[i];
            }
            pdata->buf[6+pmsg->eeprom_wr.len] = compute_paragon_chksum(pdata->buf, 6+pmsg->eeprom_wr.len);
            printf("Sending Power CIM Write EEPROM.........");
            break;
        default:
            return -1;
    }

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.\n");

        /* wait for reply */
        while( !correct_rsp ) {
            if( cim_poll_once(fd, DEFAULT_POLL_TIMEOUT, pevt) != 0 ) {
                return -1;
            }

            memset(&process, 0, sizeof(rx_notify_t));
            process.link_if = link_if;
            handle_events(fd, pevt, &process);

            if( process.rx_cim ) {
                do {
                    /* retrieve incoming data */
                    pdata->link_if = link_if;
                    pdata->requested_len = 256;
                    pdata->type = pkt_buffer;
                    pdata->buf = protocol_buffer;
                    paragon_cim_receive_pkt(fd, link_if, pdata);

                    if( pdata->actual_len > 0 ) {
                        if( display_rxpkt ) {
                            printf("RxPKT: ");
                            dump_data(pdata->buf, pdata->actual_len);
                        }

                        if( pdata->buf[0] == PCMD_FOR_POWER_CIM ) {
                            if( pdata->buf[2] == (id | 0x80) ) {
                                correct_rsp = 1;
                            }
                        }
                    }
                } while( pdata->actual_len > 0 );
            }
            else if( process.rx_priority ) {
                fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
            }
        }
    }

    return 0;
}

int paragon_cim_send_power_strip_cmd( int fd, int link_if, char id, char *pbuf )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    int correct_rsp = 0;
    int len;
    int i;

    pdata->link_if = link_if;
    pdata->type = pkt_buffer;
    pdata->buf = protocol_buffer;
    display_rxpkt = 1;

    pdata->buf[0] = PCMD_FOR_POWER_STRIP;

    switch( id ) {
        case SNMPON:
            pdata->requested_len = 11;
            pdata->buf[0] = PCMD_FOR_POWER_STRIP;
            pdata->buf[1] = 9;
            pdata->buf[2] = POWER_STRIP_CMD_DATA;
            pdata->buf[3] = 'S';
            pdata->buf[4] = 'N';
            pdata->buf[5] = 'M';
            pdata->buf[6] = 'P';
            pdata->buf[7] = 'O';
            pdata->buf[8] = 'N';
            pdata->buf[9] = CARRIAGE_RETURN;
            pdata->buf[10] = compute_paragon_chksum(pdata->buf, 10);
            printf("Sending Power Strip SNMPON.............");
            break;
        case SNMPON_OUTLET:
            pdata->buf[0] = PCMD_FOR_POWER_STRIP;
            pdata->buf[2] = POWER_STRIP_CMD_DATA;
            pdata->buf[3] = 'S';
            pdata->buf[4] = 'N';
            pdata->buf[5] = 'M';
            pdata->buf[6] = 'P';
            pdata->buf[7] = 'O';
            pdata->buf[8] = 'N';
            pdata->buf[9] = ' ';
            pdata->buf[10] = pbuf[0];
            if( strlen(pbuf) == 3 ) {
                pdata->buf[1] = 12;
                pdata->buf[11] = pbuf[1];
                pdata->buf[12] = CARRIAGE_RETURN;
                pdata->buf[13] = compute_paragon_chksum(pdata->buf, 13);
                pdata->requested_len = 14;
            }
            else {
                pdata->buf[1] = 11;
                pdata->buf[11] = CARRIAGE_RETURN;
                pdata->buf[12] = compute_paragon_chksum(pdata->buf, 12);
                pdata->requested_len = 13;
            }
            printf("Sending Power Strip SNMPON.............");
            break;
        case SPOFF:
            pdata->requested_len = 10;
            pdata->buf[0] = PCMD_FOR_POWER_STRIP;
            pdata->buf[1] = 8;
            pdata->buf[2] = POWER_STRIP_CMD_DATA;
            pdata->buf[3] = 'S';
            pdata->buf[4] = 'P';
            pdata->buf[5] = 'O';
            pdata->buf[6] = 'F';
            pdata->buf[7] = 'F';
            pdata->buf[8] = CARRIAGE_RETURN;
            pdata->buf[9] = compute_paragon_chksum(pdata->buf, 9);
            printf("Sending Power Strip SPOFF..............");
            break;
        case SPOFF_OUTLET:
            pdata->buf[0] = PCMD_FOR_POWER_STRIP;
            pdata->buf[2] = POWER_STRIP_CMD_DATA;
            pdata->buf[3] = 'S';
            pdata->buf[4] = 'P';
            pdata->buf[5] = 'O';
            pdata->buf[6] = 'F';
            pdata->buf[7] = 'F';
            pdata->buf[8] = ' ';
            pdata->buf[9] = pbuf[0];
            if( strlen(pbuf) == 3 ) {
                pdata->buf[1] = 11;
                pdata->buf[10] = pbuf[1];
                pdata->buf[11] = CARRIAGE_RETURN;
                pdata->buf[12] = compute_paragon_chksum(pdata->buf, 12);
                pdata->requested_len = 13;
            }
            else {
                pdata->buf[1] = 10;
                pdata->buf[10] = CARRIAGE_RETURN;
                pdata->buf[11] = compute_paragon_chksum(pdata->buf, 11);
                pdata->requested_len = 12;
            }
            printf("Sending Power Strip SPOFF..............");
            break;
        case SNMPS:
            pdata->requested_len = 11;
            pdata->buf[0] = PCMD_FOR_POWER_STRIP;
            pdata->buf[1] = 9;
            pdata->buf[2] = POWER_STRIP_CMD_DATA;
            pdata->buf[3] = 'S';
            pdata->buf[4] = 'N';
            pdata->buf[5] = 'M';
            pdata->buf[6] = 'P';
            pdata->buf[7] = 'S';
            pdata->buf[8] = ' ';
            pdata->buf[9] = CARRIAGE_RETURN;
            pdata->buf[10] = compute_paragon_chksum(pdata->buf, 10);
            printf("Sending Power Strip SNMPS..............");
            break;
        case USER_DEFINED:
            len = strlen(pbuf) - 1; /* minus NL feed */
            pdata->buf[0] = PCMD_FOR_POWER_STRIP;
            pdata->buf[1] = len + 3;
            pdata->buf[2] = POWER_STRIP_CMD_DATA;
            for( i = 0; i < len; i++ ) {
                pdata->buf[3+i] = pbuf[i];
            }
            pdata->buf[3+len] = CARRIAGE_RETURN;
            pdata->buf[4+len] = compute_paragon_chksum(pdata->buf, 4+len);
            pdata->requested_len = pdata->buf[1] + 2;
            printf("Sending User-defined BSNMP command.....");
            break;
        default:
            return -1;
    }

#if 0
    {
        printf("Tx Packet: ");
        for( i = 0; i < pdata->requested_len; i++ ) {
            printf("%02x ", pdata->buf[i]);
        }
        printf("\n");
    }
#endif

    if( paragon_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        printf("passed.\n");

        /* wait for reply */
        while( !correct_rsp ) {
            if( cim_poll_once(fd, DEFAULT_POLL_TIMEOUT, pevt) != 0 ) {
                return -1;
            }

            memset(&process, 0, sizeof(rx_notify_t));
            process.link_if = link_if;
            handle_events(fd, pevt, &process);

            if( process.rx_cim ) {
                do {
                    /* retrieve incoming data */
                    pdata->link_if = link_if;
                    pdata->requested_len = 256;
                    pdata->type = pkt_buffer;
                    pdata->buf = protocol_buffer;
                    paragon_cim_receive_pkt(fd, link_if, pdata);

                    if( pdata->actual_len > 0 ) {
                        if( display_rxpkt ) {
                            printf("RxPKT: ");
                            dump_data(pdata->buf, pdata->actual_len);
                        }

                        if( pdata->buf[0] == PCMD_FOR_POWER_STRIP ) {
                            if( pdata->buf[2] == (POWER_STRIP_CMD_DATA | 0x80) ) {
                                correct_rsp = 1;
                            }
                        }
                    }
                } while( pdata->actual_len > 0 );
            }
            else if( process.rx_priority ) {
                fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
            }
        }
    }

    return 0;
}
