/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Virtual Media CIM Protocol Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"
#include "cim.h"
#include "events.h"
#include "vm_cim.h"


/* request: category, cmd, payload length
 * result : category, cmd, payload length
 */
static int vm_buffer = FPD_CIM_BUFFER;
static enum version_component version_component = CIM;
static enum version_type version_type = FW;
static unsigned char version_component_index = 0;
static unsigned char cc_ref = 0;
static unsigned char tx_timeout = 5;
static unsigned short eeprom_rd_offset = 0;
static unsigned short eeprom_rd_cnt = 1;
static unsigned short eeprom_wr_offset = 0;
static unsigned short eeprom_wr_cnt = 1;
static unsigned char *eeprom_wr_buf;
static enum kbd_model keybd_model = STD_101_KEY;
static enum kbd_language keybd_lang = US_ENGLISH;
static enum mouse_position mouse_mode = ABSOLUTE;
static enum led_settings led_setting = LED_TOGGLE;
static enum test_mode test_mode = TEST_EXIT;
static unsigned char test_option_cnt = 0;
static unsigned char test_option[256];
static char display_printout = 1;
static unsigned char fpga_reg = 0;
static unsigned char fpga_data = 0;
static unsigned char usb_device_config = 0;
static unsigned char mass_storage_id = 0;
static unsigned char stall_settings = 0;


static unsigned char vm_cim_validate_header( vm_pkt_hdr_t *phdr )
{
    /* check for Start-Of-Packet and End-Of-Header */
    if( phdr->sop != VM_SOP ) {
        return RESULT_INVALID_PKT_SOP;
    }

    if( phdr->eoh != VM_EOH ) {
        return RESULT_INVALID_PKT_EOH;
    }

    if( (phdr->command & PKT_SRC_MASK) != PKT_SRC_CIM ) {
        return RESULT_INVALID_PKT_SOURCE;
    }

    return RESULT_NO_ERROR;
}

static int vm_cim_create_header( vm_pkt_hdr_t *phdr, enum vm_category category, unsigned char cmd, unsigned short len )
{
    static unsigned char seqnum = 0;

    if( phdr ) {
        phdr->sop = VM_SOP;
        phdr->protocol_version = 1;
        phdr->sequence_number = seqnum++;
        phdr->category = category;
        phdr->command = cmd;
        phdr->payload_lenlo = len & 0xff;
        phdr->payload_lenhi = (len & 0xff00) >> 8;
        phdr->eoh = VM_EOH;

        return 0;
    }
    else {
        return -1;
    }
}

static int vm_cim_get_payload_len(enum vm_category category, unsigned char cmd)
{
    int payload_len = -1;

    switch( category ) {
        case INITIALIZATION:
            switch( cmd ) {
                case RESET:
                    payload_len = sizeof(cim_reset_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case INIT:
                    payload_len = sizeof(initialization_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case INFORMATION:
            switch( cmd ) {
                case VERSION:
                    payload_len = sizeof(version_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CIM_HEARTBEAT:
                    payload_len = sizeof(cim_status_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case LINK_STATS:
                    payload_len = sizeof(link_stats_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case SWITCH_HEARTBEAT:
                    payload_len = sizeof(switch_heartbeat_rsp_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CIM_FPGA_ERROR:
                    payload_len = sizeof(cim_fpga_error_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CIM_FPGA_READ:
                    payload_len = sizeof(cim_fpga_read_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case CONTROL:
            switch( cmd ) {
                case CIM_SET_COLOR_CAL_VOLTAGE:
                    payload_len = sizeof(cim_set_cc_voltage_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CIM_SET_TX_TIMEOUT_VALUE:
                    payload_len = sizeof(cim_set_tx_timeout_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CIM_SET_LED:
                    payload_len = sizeof(cim_set_led_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CIM_SET_TEST_MODE:
                    payload_len = 2 + test_option_cnt;
                    break;
                case CIM_FPGA_WRITE:
                    payload_len = sizeof(cim_fpga_write_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case MEMORY:
            switch( cmd ) {
                case CIM_EEPROM_WRITE:
                    payload_len = 4 + eeprom_wr_cnt;
                    break;
                case CIM_EEPROM_READ:
                    payload_len = sizeof(cim_eeprom_read_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case KEYBOARD:
            switch( cmd ) {
                case KBD_MODEL_SET:
                    payload_len = sizeof(kbd_model_set_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case KBD_LANGUAGE_SET:
                    payload_len = sizeof(kbd_language_set_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case TARGET_KBD_LED_SETTINGS:
                    payload_len = sizeof(kbd_led_setting_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case KBD_DATA:
                    payload_len = sizeof(kbd_data_pkt_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case MOUSE:
            switch( cmd ) {
                case MOUSE_MODE_SET:
                    payload_len = sizeof(mouse_mode_set_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case MOUSE_DATA:
                    payload_len = sizeof(mouse_data_pkt_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case USB:
            switch( cmd ) {
                case USB_CIM_DISCONNECT:
                    payload_len = sizeof(usb_cim_disc_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case USB_CIM_CONNECT:
                    payload_len = sizeof(usb_cim_conn_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case VM_USB_DEVICE_CONFIG:
                    payload_len = sizeof(usb_dev_cfg_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case STALL_BULK_EP:
                    payload_len = sizeof(stall_bulkep_req_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case MASS_STORAGE_RESET:
                    payload_len = sizeof(mass_storage_reset_rsp_t) - sizeof(vm_pkt_hdr_t);
                    break;
                case CLR_FEATURE_HALT:
                    payload_len = sizeof(clear_feat_halt_rsp_t) - sizeof(vm_pkt_hdr_t);
                    break;
                default:
                    break;
            }
            break;
        case SERIAL:
        case POWER_STRIP:
        case SWITCH_TO_SWITCH:
        case Z_CHAINS:
        case UPDATE:
        default:
            break;
    }

    return payload_len;
}

static int vm_cim_create_payload(unsigned char *pbuf, enum vm_category category, unsigned char cmd)
{
    int i;

    switch( category ) {
        case INFORMATION:
            switch( cmd ) {
                case VERSION:
                {
                    version_req_t *preq = (version_req_t *)pbuf;
                    preq->component = version_component;
                    preq->version_type = version_type;
                    preq->component_index = version_component_index;
                    break;
                }
                case CIM_FPGA_READ:
                {
                    cim_fpga_read_req_t *preq = (cim_fpga_read_req_t *)pbuf;
                    preq->reg = fpga_reg;
                    break;
                }
                default:
                    break;
            }
            break;
        case CONTROL:
            switch( cmd ) {
                case CIM_SET_COLOR_CAL_VOLTAGE:
                {
                    cim_set_cc_voltage_req_t *preq = (cim_set_cc_voltage_req_t *)pbuf;
                    preq->voltage_ref = cc_ref;
                    break;
                }
                case CIM_SET_TX_TIMEOUT_VALUE:
                {
                    cim_set_tx_timeout_req_t *preq = (cim_set_tx_timeout_req_t *)pbuf;
                    preq->timeout = tx_timeout;
                    break;
                }
                case CIM_SET_LED:
                {
                    cim_set_led_req_t *preq = (cim_set_led_req_t *)pbuf;
                    preq->cmd = led_setting;
                    break;
                }
                case CIM_SET_TEST_MODE:
                {
                    cim_set_test_mode_req_t *preq = (cim_set_test_mode_req_t *)pbuf;
                    preq->test_mode = test_mode;
                    preq->option_cnt = test_option_cnt;
                    for( i = 0; i < test_option_cnt; i++ ) {
                        preq->options[i] = test_option[i];
                    }
                    break;
                }
                case CIM_FPGA_WRITE:
                {
                    cim_fpga_write_req_t *preq = (cim_fpga_write_req_t *)pbuf;
                    preq->reg = fpga_reg;
                    preq->value = fpga_data;
                    break;
                }
                default:
                    break;
            }
            break;
        case MEMORY:
            switch( cmd ) {
                case CIM_EEPROM_WRITE:
                {
                    cim_eeprom_write_req_t *preq = (cim_eeprom_write_req_t *)pbuf;
                    preq->offset_lo = eeprom_wr_offset & 0xff;
                    preq->offset_hi = (eeprom_wr_offset & 0xff00) >> 8;
                    preq->len_lo = eeprom_wr_cnt & 0xff;
                    preq->len_hi = (eeprom_wr_cnt & 0xff00) >> 8;
                    for( i = 0; i < eeprom_wr_cnt; i++ ) {
                        preq->buf[i] = eeprom_wr_buf[i];
                    }
                    break;
                }
                case CIM_EEPROM_READ:
                {
                    cim_eeprom_read_req_t *preq = (cim_eeprom_read_req_t *)pbuf;
                    preq->offset_lo = eeprom_rd_offset & 0xff;
                    preq->offset_hi = (eeprom_rd_offset & 0xff00) >> 8;
                    preq->len_lo = eeprom_rd_cnt & 0xff;
                    preq->len_hi = (eeprom_rd_cnt & 0xff00) >> 8;
                    break;
                }
                default:
                    break;
            }
            break;
        case KEYBOARD:
            switch( cmd ) {
                case KBD_MODEL_SET:
                {
                    kbd_model_set_req_t *preq = (kbd_model_set_req_t *)pbuf;
                    preq->kbd_model = keybd_model;
                    break;
                }
                case KBD_LANGUAGE_SET:
                {
                    kbd_language_set_req_t *preq = (kbd_language_set_req_t *)pbuf;
                    preq->language = keybd_lang;
                    break;
                }
                case KBD_DATA:
                {
                    break;
                }
                default:
                    break;
            }
            break;
        case MOUSE:
            switch( cmd ) {
                case MOUSE_MODE_SET:
                {
                    mouse_mode_set_req_t *preq = (mouse_mode_set_req_t *)pbuf;
                    preq->mouse_mode = mouse_mode;
                    break;
                }
                case MOUSE_DATA:
                {
                    break;
                }
                default:
                    break;
            }
            break;
        case USB:
            switch( cmd ) {
                case VM_USB_DEVICE_CONFIG:
                {
                    usb_dev_cfg_req_t *preq = (usb_dev_cfg_req_t *)pbuf;
                    preq->settings = usb_device_config;
                    break;
                }
                case STALL_BULK_EP:
                {
                    stall_bulkep_req_t *preq = (stall_bulkep_req_t *)pbuf;
                    preq->usbdevID = mass_storage_id;
                    preq->settings = stall_settings;
                    break;
                }
                case MASS_STORAGE_RESET:
                {
                    mass_storage_reset_rsp_t *prsp = (mass_storage_reset_rsp_t *)pbuf;
                    prsp->result = RESULT_NO_ERROR;
                    break;
                }
                case CLR_FEATURE_HALT:
                {
                    clear_feat_halt_rsp_t *prsp = (clear_feat_halt_rsp_t *)pbuf;
                    prsp->result = RESULT_NO_ERROR;
                    break;
                }
                default:
                    break;
            }
            break;
        default:
            return -1;
    }

    return 0;
}

static void vm_cim_print_cmd(enum vm_category category, unsigned char cmd)
{
    printf("\nSending ");

    switch( category ) {
        case INITIALIZATION:
            switch( cmd ) {
                case RESET:
                    printf("CIM Reset...................");
                    break;
                case INIT:
                    printf("CIM Initialization..........");
                    break;
                default:
                    break;
            }
            break;
        case INFORMATION:
            switch( cmd ) {
                case VERSION:
                    printf("Version.....................");
                    break;
                case CIM_HEARTBEAT:
                    printf("CIM Heartbeat...............");
                    break;
                case SWITCH_HEARTBEAT:
                    printf("Switch Heartbeat............");
                    break;
                case LINK_STATS:
                    printf("Link Statistics.............");
                    break;
                case CIM_FPGA_ERROR:
                    printf("CIM FPGA Errors.............");
                    break;
                case CIM_FPGA_READ:
                    printf("CIM FPGA Read...............");
                    break;
                default:
                    break;
            }
            break;
        case CONTROL:
            switch( cmd ) {
                case CIM_SET_COLOR_CAL_VOLTAGE:
                    printf("CIM Set Color Cal Voltage...");
                    break;
                case CIM_SET_TX_TIMEOUT_VALUE:
                    printf("CIM Set TX Timeout..........");
                    break;
                case CIM_SET_LED:
                    printf("CIM Set LED.................");
                    break;
                case CIM_SET_TEST_MODE:
                    printf("CIM Set Test Mode...........");
                    break;
                case CIM_FPGA_WRITE:
                    printf("CIM FPGA Write..............");
                    break;
                default:
                    break;
            }
            break;
        case MEMORY:
            switch( cmd ) {
                case CIM_EEPROM_WRITE:
                    printf("CIM Write EEPROM............");
                    break;
                case CIM_EEPROM_READ:
                    printf("CIM Read EEPROM.............");
                    break;
                default:
                    break;
            }
            break;
        case KEYBOARD:
            switch( cmd ) {
                case KBD_MODEL_SET:
                    printf("Keybd Model Set.............");
                    break;
                case KBD_LANGUAGE_SET:
                    printf("Keybd Language Set..........");
                    break;
                case TARGET_KBD_LED_SETTINGS:
                    printf("Target Keybd LED Settings...");
                    break;
                case KBD_DATA:
                    printf("Keybd Data..................");
                    break;
                default:
                    break;
            }
            break;
        case MOUSE:
            switch( cmd ) {
                case MOUSE_MODE_SET:
                    printf("Mouse Mode Set..............");
                    break;
                case MOUSE_DATA:
                    printf("Mouse Data..................");
                    break;
                default:
                    break;
            }
            break;
        case USB:
            switch( cmd ) {
                case USB_CIM_DISCONNECT:
                    printf("USB CIM Disconnect..........");
                    break;
                case USB_CIM_CONNECT:
                    printf("USB CIM Connect.............");
                    break;
                case VM_USB_DEVICE_CONFIG:
                    printf("USB Device Configuration....");
                    break;
                case STALL_BULK_EP:
                    printf("Stall USB Bulk Endpoint.....");
                    break;
                case MASS_STORAGE_RESET:
                    printf("Mass Storage Reset Rsp......");
                    break;
                case CLR_FEATURE_HALT:
                    printf("Clear Feature Halt Rsp......");
                    break;
                default:
                    break;
            }
            break;
        case UPDATE:
        case SERIAL:
        case POWER_STRIP:
        case SWITCH_TO_SWITCH:
        case Z_CHAINS:
        default:
            break;
    }

    return;
}

void vm_parse_request_result( unsigned char code )
{
    if( code != RESULT_NO_ERROR ) {
        printf("\nERROR: (0x%02x) ", code);
    }

    switch( code ) {
        case RESULT_NO_ERROR:
            printf("\nResult is good\n");
            break;
        case RESULT_CIM_RESET_FAILED:
            printf("CIM Reset failed\n");
            break;
        case RESULT_WRONG_CIM_SETTINGS:
            printf("Wrong CIM settings\n");
            break;
        case RESULT_WRONG_KBD_SETTINGS:
            printf("Wrong keyboard settings\n");
            break;
        case RESULT_WRONG_VERSION_INFO:
            printf("Wrong version info\n");
            break;
        case RESULT_INVALID_VERSION_COMPONENT:
            printf("Invalid version component\n");
            break;
        case RESULT_INVALID_VERSION_TYPE:
            printf("Invalid version type\n");
            break;
        case RESULT_INVALID_COMPONENT_INDEX:
            printf("Invalid component index\n");
            break;
        case RESULT_HEARTBEAT_ERROR:
            printf("Heartbeat error\n");
            break;
        case RESULT_CIM_SET_LED_FAILED:
            printf("CIM set LED failed\n");
            break;
        case RESULT_CIM_FPGA_READ_FAILED:
            printf("CIM FPGA read failed\n");
            break;
        case RESULT_CIM_FPGA_WRITE_FAILED:
            printf("CIM FPGA write failed\n");
            break;
        case RESULT_COLORCAL_CNTL_FAILED:
            printf("Color calibration control failed\n");
            break;
        case RESULT_CIM_TX_TIMEOUT_FAILED:
            printf("CIM TX timeout failed\n");
            break;
        case RESULT_NVRAM_WRITE_ERROR:
            printf("NVRAM write error\n");
            break;
        case RESULT_NVRAM_REQ_LEN_OVER_LIMIT:
            printf("NVRAM requested length is over the limit\n");
            break;
        case RESULT_NVRAM_REQ_OFFSET_OVER_LIMIT:
            printf("NVRAM requested offset is over the limit\n");
            break;
        case RESULT_NVRAM_READ_ERROR:
            printf("NVRAM read error\n");
            break;
        case RESULT_KBD_MODEL_SETTING_ERROR:
            printf("Keyboard model setting error\n");
            break;
        case RESULT_INVALID_KBD_MODEL:
            printf("Invalid keyboard model\n");
            break;
        case RESULT_KBD_LANGUAGE_SETTING_ERROR:
            printf("Keyboard language setting error\n");
            break;
        case RESULT_INVALID_KBD_LANGUAGE:
            printf("Invalid Keyboard language\n");
            break;
        case RESULT_INVALID_KBD_LED_SETTING:
            printf("Invalid Keyboard LED setting\n");
            break;
        case RESULT_MOUSE_MODE_ERROR:
            printf("Mouse mode error\n");
            break;
        case RESULT_INVALID_MOUSE_MODE:
            printf("Invalid mouse mode\n");
            break;
        case RESULT_USB_CIM_DISC_ERROR:
            printf("USB CIM disconnect error\n");
            break;
        case RESULT_USB_CIM_CONN_ERROR:
            printf("USB CIM connect error\n");
            break;
        case RESULT_USB_MS_ENUM_ERROR:
            printf("USB CIM device configuration error\n");
            break;
        case RESULT_USB_STALL_BULKEP_ERROR:
            printf("USB Stall Bulk endpoint error\n");
            break;
        case RESULT_USB_MS_RESET_ERROR:
            printf("Mass Storage reset error\n");
            break;
        case RESULT_USB_MS_CLEAR_FEAT_HALT_ERROR:
            printf("Mass Storage Clear Feature Halt error\n");
            break;
        case RESULT_SERIAL_PARAM_INIT_ERROR:
            printf("Serial parameter init error\n");
            break;
        case RESULT_UNSUPPORTED_SERIAL_BIT_RATE:
            printf("Unsupported serial bit rate\n");
            break;
        case RESULT_INVALID_PKT_SOP:
            printf("Invalid packet Star-of-Packet\n");
            break;
        case RESULT_INVALID_PKT_EOH:
            printf("Invalid packet End-of-Header\n");
            break;
        case RESULT_INVALID_PKT_SOURCE:
            printf("Invalid packet source\n");
            break;
        case RESULT_INVALID_PKT_CATEGORY:
            printf("Invalid packet category\n");
            break;
        case RESULT_INVALID_PKT_COMMAND:
            printf("Invalid packet command\n");
            break;
        case RESULT_INVALID_TEST:
            printf("Invalid test\n");
            break;
        case RESULT_INVALID_TEST_OPTION_CNT:
            printf("Invalid test option count\n");
            break;
        case RESULT_INVALID_TEST_OPTION:
            printf("Invalid test option\n");
            break;
        case RESULT_NOT_READY_TO_UPDATE_CODE:
            printf("CIM not ready to update code\n");
            break;
        case RESULT_CODE_BLK_NOT_SAVED:
            printf("CIM update code block is not saved\n");
            break;
        case RESULT_CODE_CHKSUM_ERROR:
            printf("CIM update code checksum error\n");
            break;
        default:
            printf("Unknown Error\n");
    }

    return;
}

static void Process_Initialization_Pkt( initialization_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case RESET:
        {
            /*
             * CIM Ready Indication
             */
            cim_ready_indication_t *rsp = &pkt->ready_ind;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nCIM is ready for communication\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case INIT:
        {
            /*
             * Initialization Request
             */
            initialization_rsp_t *rsp = &pkt->init_rsp;
            char cim_is_usb = 0;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nNumber of CIM settings  : %d\n", rsp->cim_setting_cnt);
                printf("  CIM Type              : ");
                switch( rsp->cim_type ) {
                    case CIM_TYPE_POWER:
                        printf("Power");
                        break;
                    case CIM_TYPE_SERIAL:
                        printf("Serial");
                        break;
                    case CIM_TYPE_PS2:
                        printf("PS/2");
                        break;
                    case CIM_TYPE_SUN:
                        printf("SUN");
                        break;
                    case CIM_TYPE_SUN_USB:
                        printf("USB SUN");
                        cim_is_usb = 1;
                        break;
                    case CIM_TYPE_USB_FULL_SPEED:
                        printf("USB Full Speed");
                        cim_is_usb = 1;
                        break;
                    case CIM_TYPE_USB_FULL_AND_HIGH_SPEED:
                        printf("USB Full and High Speed");
                        cim_is_usb = 1;
                        break;
                    case CIM_TYPE_SWITCH:
                        printf("Switch");
                        break;
                    case CIM_TYPE_ZCIM:
                        printf("Z-CIM");
                        break;
                    default:
                        printf("Unknown");
                        break;
                }
                printf("\n");
                printf("  Endpoints Supported   : %d\n", rsp->ep_cnt);
                if( cim_is_usb ) {
                    printf("  Target USB Speed      : ");
                    switch( rsp->cim_setting_1 & 0x0F ) {
                        case USB_SPEED_NA:
                            printf("N/A");
                            break;
                        case USB_SPEED_FULL:
                            printf("Full Speed");
                            break;
                        case USB_SPEED_HIGH:
                            printf("High Speed");
                            break;
                        default:
                            printf("Unknown");
                            break;
                    }
                    printf("\n");
                }
                printf("  Supported Mode        : ");
                switch( rsp->cim_setting_2 ) {
                    case SUPPORTED_MODE_NONE:
                        printf("None");
                        break;
                    case SUPPORTED_MODE_VM:
                        printf("Virtual Media");
                        break;
                    case SUPPORTED_MODE_AUDIO:
                        printf("Audio");
                        break;
                    case SUPPORTED_MODE_VM_AND_AUDIO:
                        printf("Virtual Media and Audio");
                        break;
                    default:
                        printf("Unknown");
                        break;
                }
                printf("\n");
                printf("Number of Keybd settings: %d\n", rsp->kbd_setting_cnt);
                printf("  Keybd LED status      : \n");
                printf("    Num Lock            : %s\n",
                       (rsp->kbd_setting_1 & KBD_LED_NUM_LOCK)?"ON":"off");
                printf("    Caps Lock           : %s\n",
                       (rsp->kbd_setting_1 & KBD_LED_CAPS_LOCK)?"ON":"off");
                printf("    Scroll Lock         : %s\n",
                       (rsp->kbd_setting_1 & KBD_LED_SCROLL_LOCK)?"ON":"off");
                printf("    Compose Lock        : %s\n",
                       (rsp->kbd_setting_1 & KBD_LED_COMPOSE_LOCK)?"ON":"off");
                printf("    Kana Lock           : %s\n",
                       (rsp->kbd_setting_1 & KBD_LED_KANA_LOCK)?"ON":"off");
                printf("  Keybd Setting 2       : %x\n", rsp->kbd_setting_2);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        default:
            break;
    }

    return;
}

static void Process_Information_Pkt( information_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case VERSION:
        {
            /*
             * Version Response
             */
            version_rsp_t *rsp = &pkt->version_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nVersion : %c.%c.%c.%c\n",
                       rsp->version[3], rsp->version[2],
                       rsp->version[1], rsp->version[0]);
                printf("Length  : %d\n",
                       (rsp->len[0] | (rsp->len[1] << 8) |
                        (rsp->len[2] << 16) | (rsp->len[3] << 24)));
                printf("Checksum: 0x%02x\n", rsp->checksum);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_HEARTBEAT:
        {
            /*
             * CIM Status Response
             */
            cim_status_rsp_t *rsp = &pkt->cim_status_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nReceived CIM heartbeat\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case SWITCH_HEARTBEAT:
        {
            /*
             * Switch Heartbeat Request
             */
            printf("\nCIM Heartbeat Request\n");
            break;
        }
        case LINK_STATS:
        {
            /*
             * Link Statistics Response
             */
            link_stats_rsp_t *rsp = &pkt->link_stats_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nLink Statistics\n");
                printf("---------------\n");
                printf("Invalid packet count  : %d\n", rsp->rx_inv_pkt_cnt);
                printf("CRC error count       : %d\n", rsp->rx_crc_err_cnt);
                printf("Protocol error count  : %d\n", rsp->rx_prot_err_cnt);
                printf("Packet drop count     : %d\n", rsp->rx_buf_full_cnt);
                printf("Noise count           : %d\n", rsp->noise_cnt);
                printf("Protocol NACK count   : %d\n", rsp->tx_prot_err_cnt);
                printf("Buffer full NACK count: %d\n", rsp->tx_buf_full_cnt);
                printf("CRC NACK count        : %d\n", rsp->tx_crc_cnt);
                printf("Maximum retry         : %d\n", rsp->max_retry);
                printf("RX packet count       : %d\n", (rsp->rx_pkt_cnt_hi << 8) | rsp->rx_pkt_cnt_lo);
                printf("TX packet count       : %d\n", (rsp->tx_pkt_cnt_hi << 8) | rsp->tx_pkt_cnt_lo);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_FPGA_ERROR:
        {
            /*
             * CIM FPGA Error Response
             */
            cim_fpga_error_rsp_t *rsp = &pkt->cim_fpga_error_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nFPGA Errors\n");
                printf("-----------\n");
                printf("RX Priority read from an empty buffer: %d\n", rsp->rx_pri_rd_empty_buf);
                printf("TX Priority buffer sync error        : %d\n", rsp->tx_pri_sync_err);
                printf("TX Priority full buffer with no EOP  : %d\n", rsp->tx_pri_no_eop);
                printf("TX Priority write to a full buffer   : %d\n", rsp->tx_pri_wr_full_buf);
                printf("RX CIM read from an empty buffer     : %d\n", rsp->rx_cim_rd_empty_buf);
                printf("TX CIM buffer sync error             : %d\n", rsp->tx_cim_sync_err);
                printf("TX CIM full buffer with no EOP       : %d\n", rsp->tx_cim_no_eop);
                printf("TX CIM write to a full buffer        : %d\n", rsp->tx_cim_wr_full_buf);
                printf("TX Host Data1 buffer sync error      : %d\n", rsp->tx_host1_sync_err);
                printf("TX Host Data1 full buffer with no EOP: %d\n", rsp->tx_host1_no_eop);
                printf("TX Host Data1 write to a full buffer : %d\n", rsp->tx_host1_wr_full_buf);
                printf("Reserved                             : %d\n", rsp->reserved);
                printf("Priority buffer TX timeout           : %d\n", rsp->tx_pri_timeout);
                printf("CIM buffer TX timeout                : %d\n", rsp->tx_cim_timeout);
                printf("Host buffer 1 TX timeout             : %d\n", rsp->tx_host_timeout1);
                printf("Host buffer 0 TX timeout             : %d\n", rsp->tx_host_timeout0);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_FPGA_READ:
        {
            /*
             * CIM FPGA Read Response
             */
            cim_fpga_read_rsp_t *rsp = &pkt->cim_fpga_read_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nFPGA Register: 0x%02x\n", rsp->reg);
                printf("Data Content : 0x%02x\n", rsp->value);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        default:
            break;
    }

    return;
}

static void Process_Control_Pkt( control_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case CIM_SET_COLOR_CAL_VOLTAGE:
        {
            /*
             * CIM Set Color Calibration Voltage Response
             */
            cim_set_cc_voltage_rsp_t *rsp = &pkt->cc_voltage_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nSet CIM color calibration voltage to %d\n",
                       rsp->voltage_ref);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_SET_TX_TIMEOUT_VALUE:
        {
            /*
             * CIM Set Transmit Timeout Value Response
             */
            cim_set_tx_timeout_rsp_t *rsp = &pkt->tx_timeout_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nSet CIM transmit timeout successfully\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_SET_LED:
        {
            /*
             * CIM Set LED on/off
             */
            cim_set_led_rsp_t *rsp = &pkt->led_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nConfigured CIM LED successfully\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_SET_TEST_MODE:
        {
            /*
             * CIM Set Test Mode
             */
            cim_set_test_mode_rsp_t *rsp = &pkt->test_mode_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nConfigured CIM Test Mode successfully\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_FPGA_WRITE:
        {
            /*
             * CIM FPGA Write Request
             */
            cim_fpga_write_rsp_t *rsp = &pkt->cim_fpga_write_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nWrote 0x%02x to CIM FPGA register 0x%02x\n",
                       rsp->value, rsp->reg);
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        default:
            break;
    }

    return;
}

static void Process_Memory_Pkt( memory_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case CIM_EEPROM_WRITE:
        {
            /*
             * CIM EEPROM Write Response
             */
            cim_eeprom_write_rsp_t *rsp = &pkt->eeprom_write_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nCIM EEPROM write succeeded\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CIM_EEPROM_READ:
        {
            /*
             * CIM EEPROM Read Response
             */
            cim_eeprom_read_rsp_t *rsp = &pkt->eeprom_read_rsp;
            int length = pkt->hdr.payload_lenlo + (pkt->hdr.payload_lenhi << 8);
            int i;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nEEPROM Contents\n");
                printf("---------------\n");
                for( i = 0; i < length; i++ ) {
                    printf("%02x ", rsp->buf[i]);
                }
                printf("\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        default:
            break;
    }

    return;
}

static void Process_Keyboard_Pkt( keyboard_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case KBD_MODEL_SET:
        {
            /*
             * Keyboard Model Set Response
             */
            kbd_model_set_rsp_t *rsp = &pkt->kbd_model_set_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nSucceeded setting Keyboard Model to ");
                switch( rsp->kbd_model ) {
                    case STD_101_KEY:
                        printf("Standard 101 Key");
                        break;
                    case PINNACLE_DEKO:
                        printf("Pinnacle Deko Fast Action");
                        break;
                    case BELLA_DV:
                        printf("Bella DV");
                        break;
                    default:
                        printf("Unknown");
                        break;
                }
                printf("\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case KBD_LANGUAGE_SET:
        {
            /*
             * Keyboard Language Set Response
             */
            kbd_language_set_rsp_t *rsp = &pkt->kbd_language_set_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nSucceeded setting Keyboard Language to ");
                switch( rsp->language ) {
                    case US_ENGLISH:
                        printf("US English");
                        break;
                    case BRITISH_ENGLISH:
                        printf("British English");
                        break;
                    case FRENCH:
                        printf("French");
                        break;
                    case GERMAN:
                        printf("German");
                        break;
                    case ITALIAN:
                        printf("Italian");
                        break;
                    case SPANISH:
                        printf("Spanish");
                        break;
                    case NORWAY:
                        printf("Norway");
                        break;
                    case JAPANESE:
                        printf("Japanese");
                        break;
                    case KOREAN:
                        printf("Korean");
                        break;
                    case CHINESE_TRADITIONAL:
                        printf("Chinese Traditional");
                        break;
                    case CHINESE_SIMPLIFIED:
                        printf("Chinese Simplified");
                        break;
                    default:
                        printf("Unknown");
                        break;
                }
                printf("\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case TARGET_KBD_LED_SETTINGS:
        {
            /*
             * Target Keyboard LED Settings Response
             */
            kbd_led_setting_rsp_t *rsp = &pkt->kbd_led_setting_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nKeybd LED status\n");
                printf("----------------\n");
                printf("Num Lock    : %s\n",
                       (rsp->led_setting & KBD_LED_NUM_LOCK)?"ON":"off");
                printf("Caps Lock   : %s\n",
                       (rsp->led_setting & KBD_LED_CAPS_LOCK)?"ON":"off");
                printf("Scroll Lock : %s\n",
                       (rsp->led_setting & KBD_LED_SCROLL_LOCK)?"ON":"off");
                printf("Compose Lock: %s\n",
                       (rsp->led_setting & KBD_LED_COMPOSE_LOCK)?"ON":"off");
                printf("Kana Lock   : %s\n",
                       (rsp->led_setting & KBD_LED_KANA_LOCK)?"ON":"off");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case TARGET_KBD_LED_SETTINGS_CHG:
        {
            /*
             * Target Keyboard LED Settings Change
             */
            kbd_led_setting_chg_t *rsp = &pkt->kbd_led_setting_chg;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nKeybd LED status\n");
                printf("----------------\n");
                printf("Num Lock    : %s\n",
                       (rsp->led_setting & KBD_LED_NUM_LOCK)?"ON":"off");
                printf("Caps Lock   : %s\n",
                       (rsp->led_setting & KBD_LED_CAPS_LOCK)?"ON":"off");
                printf("Scroll Lock : %s\n",
                       (rsp->led_setting & KBD_LED_SCROLL_LOCK)?"ON":"off");
                printf("Compose Lock: %s\n",
                       (rsp->led_setting & KBD_LED_COMPOSE_LOCK)?"ON":"off");
                printf("Kana Lock   : %s\n",
                       (rsp->led_setting & KBD_LED_KANA_LOCK)?"ON":"off");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        default:
            break;
    }

    return;
}

static void Process_Mouse_Pkt( mouse_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case MOUSE_MODE_SET:
        {
            /*
             * Mouse Mode Set Request
             */
            mouse_mode_set_rsp_t *rsp = &pkt->mouse_mode_set_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nSucceeded setting Mouse mode.\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        default:
            break;
    }

    return;
}

static void Process_USB_Pkt( usb_control_pkt_t *pkt )
{
    switch( pkt->hdr.command & ~PKT_SRC_MASK ) {
        case USB_CIM_DISCONNECT:
        {
            /*
             * USB CIM Disconnect Request
             */
            usb_cim_disc_rsp_t *rsp = &pkt->usb_cim_disc_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nCIM has been disconnected from USB Host\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case USB_CIM_CONNECT:
        {
            /*
             * USB CIM Connect Request
             */
            usb_cim_conn_rsp_t *rsp = &pkt->usb_cim_conn_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nCIM has been reconnected from USB Host\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case MASS_STORAGE_RESET:
        {
            /*
             * USB CIM Mass Storage Reset
             */
            mass_storage_reset_req_t *req = &pkt->mass_storage_reset_req;

            printf("\nUSB Host requested for Mass Storage %d Reset\n", req->usbdevID);
            break;
        }
        case STALL_BULK_EP:
        {
            /*
             * Stall Bulk Endpoint Request
             */
            stall_bulkep_rsp_t *rsp = &pkt->stall_bulkep_rsp;

            if( rsp->result == RESULT_NO_ERROR ) {
                printf("\nStall/Unstall Bulk Endpoint request fulfilled\n");
            }
            else {
                vm_parse_request_result(rsp->result);
            }
            break;
        }
        case CLR_FEATURE_HALT:
        {
            /*
             * Clear Feature HALT USB Standard Device Response
             */
            clear_feat_halt_req_t *req = &pkt->clear_feat_halt_req;

            printf("\nUSB Host requested for Clear Feature Halt for MS %d Bulk-%s\n",
                   req->usbdevID,
                   (req->settings & BULK_ENDPOINT_IN)?"IN":"OUT");
            break;
        }
        default:
            break;
    }

    return;
}

static void vm_parse_response( unsigned char *pkt )
{
    vm_pkt_hdr_t *phdr = (vm_pkt_hdr_t *)pkt;

    /* process according to category */
    switch( phdr->category ) {
        case INITIALIZATION:
            Process_Initialization_Pkt((initialization_pkt_t *)pkt);
            break;
        case INFORMATION:
            Process_Information_Pkt((information_pkt_t *)pkt);
            break;
        case CONTROL:
            Process_Control_Pkt((control_pkt_t *)pkt);
            break;
        case MEMORY:
            Process_Memory_Pkt((memory_pkt_t *)pkt);
            break;
        case KEYBOARD:
            Process_Keyboard_Pkt((keyboard_pkt_t *)pkt);
            break;
        case MOUSE:
            Process_Mouse_Pkt((mouse_pkt_t *)pkt);
            break;
        case USB:
            Process_USB_Pkt((usb_control_pkt_t *)pkt);
            break;
        case UPDATE:
            break;
        case SERIAL:
        case POWER_STRIP:
        case SWITCH_TO_SWITCH:
        case Z_CHAINS:
        default:
            break;
    }

    return;
}

int vm_cim_send_pkt( int fd, unsigned int link_if, FPD_data_t *pdata )
{
    int rc;

    rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
    if( rc == 0 ) {
        if( pdata->actual_len != pdata->requested_len ) {
            fprintf(stderr, "FAILED.\n");
            fprintf(stderr, "Actual bytes written %d\n", pdata->actual_len);
            return -1;
        }
    }
    else {
        fprintf(stderr, "FAILED.\n");
        if(errno) {
            fprintf(stderr, "FPD_SEND_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
        }
        return -2;
    }

    return 0;
}

int vm_cim_receive_pkt( int fd, unsigned int link_if, FPD_data_t *pdata )
{
    int rc;

    rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
    if( rc != 0 ) {
        fprintf(stderr, "Reading %d bytes of data from %s buffer....FAILED\n",
                pdata->requested_len, pdata->type?"Priority":"CIM");
        if(errno) {
            fprintf(stderr, "FPD_RECEIVE_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
        }
    }

    return rc;
}

static void dump_data( unsigned char *bufaddr, int pktlen )
{
    int i;

    for( i = 0; i < pktlen; i++ ) {
        printf("%02x ", bufaddr[i]);
    }
    printf("\n");

    return;
}

int vm_cim_send_protocol( int fd, int link_if, enum vm_category category, unsigned char cmd )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    vm_pkt_hdr_t *phdr;
    rx_notify_t process;
    unsigned char vm_hdr_len = sizeof(vm_pkt_hdr_t);
    int paylen;
    int correct_rsp = 0;
    unsigned short payload_len;
    unsigned char result;

    pdata->link_if = link_if;
    pdata->requested_len = vm_hdr_len;
    pdata->type = vm_buffer;
    pdata->buf = protocol_buffer;
    phdr = (vm_pkt_hdr_t *)pdata->buf;

    /* create packet */
    paylen = vm_cim_get_payload_len(category, cmd);
    if( paylen < 0 ) {
        return -1;
    }

    vm_cim_create_header(phdr, category, cmd, paylen);
    if( paylen > 0 ) {
        /* create payload */
        vm_cim_create_payload(pdata->buf, category, cmd);
    }
    pdata->requested_len = vm_hdr_len + paylen;

    if( display_printout ) {
        vm_cim_print_cmd(category, cmd);
    }

    if( vm_cim_send_pkt(fd, link_if, pdata) == 0 ) {
        if( display_printout ) {
            printf("passed.\n");
            fflush(stdout);
        }

        /* wait for reply */
        while( !correct_rsp ) {
            if( display_printout ) {
                printf("Waiting for response................");
                fflush(stdout);
            }

            if( cim_poll_once(fd, DEFAULT_POLL_TIMEOUT, pevt) != 0 ) {
                return -1;
            }

            memset(&process, 0, sizeof(rx_notify_t));
            process.link_if = link_if;
            handle_events(fd, pevt, &process);

            if((( vm_buffer == FPD_CIM_BUFFER ) && process.rx_cim ) ||
               (( vm_buffer == FPD_PRIORITY_BUFFER ) && process.rx_priority ))
            {
                if( display_printout ) {
                    printf("acquired.\n");
                    fflush(stdout);
                }

                do {
                    /* retrieve incoming data */
                    pdata->link_if = link_if;
                    pdata->requested_len = vm_hdr_len;
                    pdata->type = vm_buffer;
                    pdata->buf = protocol_buffer;
                    vm_cim_receive_pkt(fd, link_if, pdata);

                    if( pdata->actual_len > 0 ) {
                        if( display_printout ) {
                            printf("RxPKT: ");
                            dump_data(pdata->buf, pdata->actual_len);
                        }

                        /* verify packet header */
                        result = vm_cim_validate_header(phdr);
                        if( result == RESULT_NO_ERROR ) {
                            if(( phdr->category == category ) &&
                               ( phdr->command == (cmd | PKT_SRC_CIM) )) {
                                correct_rsp = 1;
                            }

                            /* get payload length */
                            payload_len = phdr->payload_lenlo | (phdr->payload_lenhi << 8);

                            if( payload_len > 0 ) {
                                /* get payload */
                                pdata->link_if = link_if;
                                pdata->requested_len = payload_len;
                                pdata->type = vm_buffer;
                                pdata->buf = &protocol_buffer[vm_hdr_len];
                                vm_cim_receive_pkt(fd, link_if, pdata);
                                if( pdata->actual_len > 0 ) {
                                    if( display_printout ) {
                                        printf("RxPKT: ");
                                        dump_data(pdata->buf, pdata->actual_len);
                                    }
                                }
                            }

                            /* parse the packet */
                            vm_parse_response(protocol_buffer);
                        }
                        else {
                            printf("ERROR: Received invalid packet header (0x%02x)\n", result);
                        }
                    }
                } while( pdata->actual_len > 0 );
            }
            else if((( vm_buffer == FPD_CIM_BUFFER ) && process.rx_priority ) ||
                    (( vm_buffer == FPD_PRIORITY_BUFFER ) && process.rx_cim ))
            {
                fprintf(stderr, "ERROR: Pkt rcvd from different buffer than expected\n");
            }
        }
    }

    return 0;
}

int vm_cim_setup_buffer( int type )
{
    switch( type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            vm_buffer = type;
            break;
        default:
            return -1;
    }
    
    return 0;
}

void vm_cim_set_display( char val )
{
    display_printout = val;
    return;
}

int vm_cim_setup_version( enum version_component component, enum version_type type, unsigned char index )
{
    version_component = component;
    version_type = type;
    version_component_index = index;

    return 0;
}

int vm_cim_setup_ccv( unsigned char ref )
{
    if( ref == 0 || ref == 1 ) {
        cc_ref = ref;
        return 0;
    }

    return -1;
}

int vm_cim_setup_txtimeout( unsigned char timeout )
{
    if( timeout > 0 && timeout < 6 ) {
        tx_timeout = timeout;
        return 0;
    }

    return -1;
}

int vm_cim_setup_write_eeprom( unsigned short offset, unsigned short cnt, unsigned char *pbuf )
{
    if( cnt > 128 )
        return -1;

    eeprom_wr_offset = offset;
    eeprom_wr_cnt    = cnt;
    eeprom_wr_buf    = pbuf;

    return 0;
}

int vm_cim_setup_read_eeprom( unsigned short offset, unsigned short cnt )
{
    if( cnt > 128 )
        return -1;

    eeprom_rd_offset = offset;
    eeprom_rd_cnt    = cnt;

    return 0;
}

int vm_cim_setup_kbd_model( enum kbd_model model )
{
    switch( model ) {
        case STD_101_KEY:
        case PINNACLE_DEKO:
        case BELLA_DV:
            break;
        default:
            return -1;
    }

    keybd_model = model;

    return 0;
}

int vm_cim_setup_kbd_language( enum kbd_language lang )
{
    switch( lang ) {
        case US_ENGLISH:
        case BRITISH_ENGLISH:
        case FRENCH:
        case GERMAN:
        case ITALIAN:
        case SPANISH:
        case NORWAY:
        case JAPANESE:
        case KOREAN:
        case CHINESE_TRADITIONAL:
        case CHINESE_SIMPLIFIED:
            break;
        default:
            return -1;
    }

    keybd_lang = lang;

    return 0;
}

int vm_cim_setup_mouse_mode( enum mouse_position position )
{
    switch( position ) {
        case ABSOLUTE:
            mouse_mode &= ~0x01;
            break;
        case RELATIVE:
            mouse_mode |= 0x01;
            break;
        default:
            return -1;
    }    

    return 0;
}

int vm_cim_setup_led( enum led_settings setting )
{
    switch( setting ) {
        case LED_OFF:
        case LED_ON:
        case LED_TOGGLE:
            led_setting = setting;
            break;
        default:
            return -1;
    }

    return 0;
}

int vm_cim_setup_test_mode( enum test_mode test, unsigned char opt_cnt, unsigned char *opt )
{
    int i;

    switch( test ) {
        case TEST_EXIT:
        case TEST_LOOPBACK:
            test_mode = test;
            test_option_cnt = opt_cnt;
            if( opt_cnt ) {
                for( i = 0; i < opt_cnt; i++ ) {
                    test_option[i] = opt[i];
                }
            }
            break;
        default:
            return -1;
    }

    return 0;
}

int vm_cim_setup_fpga_read( unsigned char addr )
{
    fpga_reg = addr;
    return 0;
}

int vm_cim_setup_fpga_write( unsigned char addr, unsigned char data )
{
    fpga_reg = addr;
    fpga_data = data;
    return 0;
}

int vm_cim_setup_stall_bulkep( unsigned char id, unsigned char setting )
{
    mass_storage_id = id;
    stall_settings = setting;
    return 0;
}
