/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Diagnostics Menu
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>

#include "fpd_ioctl.h"
#include "fpd_menu.h"
#include "fpd-diagnostics.h"
#include "legacy_cim.h"

static int fpd_legacy_cim_menu( int fd );
static int fpd_pdata_menu( int fd );


static void display_fpd_menu( void )
{
    int i = 0;

    printf("\nFPGA Protocol Diagnostics:");
    printf("\n==========================");
    printf("\n\t %d: Exit Diagnostics", i++);
    printf("\n\t %d: Display Info", i++);
    printf("\n\t %d: Display PCI Configuration Space", i++);
    printf("\n\t %d: Read FPGA Registers", i++);
    printf("\n\t %d: Write FPGA Registers", i++);
    printf("\n\t %d: Select Master/Slave", i++);
    printf("\n\t %d: Select Link and Line Interface", i++);
    printf("\n\t %d: Reset Link Interface", i++);
    printf("\n\t %d: Manually Write Packet to CIM buffer", i++);
    printf("\n\t %d: Manually Read Packet from CIM buffer", i++);
    printf("\n\t%d: KX2.0 Protocol and DMA Data Tests", i++);
    printf("\n\t%d: Dump Link Statistics", i++);
    printf("\n\t%d: Check Incoming Events", i++);
    printf("\n\t%d: Display Link Errors", i++);
    printf("\n\t%d: Display PCI Errors", i++);
    printf("\n\t%d: Set DMA Burst Size", i++);
    printf("\n\t%d: Display DMA Burst Size", i++);
    printf("\n\t%d: Legacy CIM Tests", i++);
    printf("\n\t%d: Set Cable Type", i++);
    printf("\n\t%d: Test KX2.0 VM Port", i++);
    printf("\n\t%d: Test Receiver VM Port", i++);
    printf("\n\n");

    return;
}

int fpd_menu( int fd )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    char answer, done;

    if( fpd_init_diag(fd) < 0 ) {
        printf("Failed to initialize Diagnostics!\n");
        return -1;
    }

    while( running ) {
        display_fpd_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case FPD_MENU_EXIT:
                running = 0;
                break;

            case FPD_MENU_DISPLAY_INFO:
                printf("\n");
                fpd_display_info(fd);
                break;

            case FPD_MENU_DISPLAY_PCI_CFG:
                printf("\n");
                fpd_display_pci_cfg(fd);
                break;

            case FPD_MENU_READ_REG:
                fpd_read_reg(fd);
                break;

            case FPD_MENU_WRITE_REG:
                fpd_write_reg(fd);
                break;

            case FPD_MENU_SELECT_MASTER_SLAVE:
                fpd_select_masterslave(fd);
                break;

            case FPD_MENU_SELECT_LINK_AND_LINE_IF:
                fpd_get_switch_info(fd);
                fpd_display_current_setup();

                printf("\n");
                if(( max_link_if > 1 ) ||
                   (( max_link_if == 1 ) && bgnd_link_if_present )) {
                    fflush(stdin);
                    do {
                        printf("\rDo you want to change Link IF   (y/n)                  : ");
                        answer = getchar();
                        switch( answer ) {
                            case 'y':
                            case 'Y':
                            case 'n':
                            case 'N':
                                done = 1;
                                break;
                            default:
                                done = 0;
                                break;
                        }
                    } while( !done );

                    /* CIM connection */
                    if((answer == 'y') || (answer == 'Y')) {
                        fpd_select_linkif();
                    }
                }

                do {
                    printf("\rDo you want to change the setup (y/n)                  : ");
                    answer = getchar();
                    switch( answer ) {
                        case 'y':
                        case 'Y':
                        case 'n':
                        case 'N':
                            done = 1;
                            break;
                        default:
                            done = 0;
                            break;
                    }
                } while( !done );

                if((answer == 'y') || (answer == 'Y')) {
                    fpd_select_linkif_and_line(fd, fpd_get_current_link_if());
                }
                break;

            case FPD_MENU_RESET_LINKIF:
                fpd_reset_linkif(fd, fpd_get_current_link_if());
                break;

            case FPD_MENU_WRITE_BUFFER:
                fpd_write_buffer(fd);
                break;

            case FPD_MENU_READ_BUFFER:
                fpd_read_buffer(fd);
                break;

            case FPD_MENU_PROTOCOL_AND_DMA_DATA_TESTS:
                if( fpd_get_link_if_protocol() == FPD_PROTOCOL_VM ) {
                    fpd_pdata_menu(fd);
                }
                else {
                    fprintf(stderr, "\nRequires VM Protocol!\n");
                }
                break;

            case FPD_MENU_GET_LINK_STATS:
                fpd_get_stats(fd);
                break;

            case FPD_MENU_CHECK_EVENTS:
                fpd_check_events(fd);
                break;

            case FPD_MENU_GET_LINK_ERRORS:
                fpd_get_link_errors(fd);
                break;

            case FPD_MENU_GET_PCI_ERRORS:
                fpd_get_pci_errors(fd);
                break;

            case FPD_MENU_SET_DMA_BURST_SIZE:
                fpd_set_dma_burst_size(fd);
                break;

            case FPD_MENU_GET_DMA_BURST_SIZE:
                fpd_get_dma_burst_size(fd);
                break;

            case FPD_MENU_LEGACY_CIM_TESTS:
                if( fpd_get_link_if_protocol() == FPD_PROTOCOL_PARAGON ) {
                    legacy_cim_init(fd, fpd_get_current_link_if(), 1);
                    fpd_legacy_cim_menu(fd);
                }
                else {
                    fprintf(stderr, "\nRequires Paragon Protocol!\n");
                }
                break;

            case FPD_MENU_SET_CABLE_TYPE:
                fpd_set_cable_type(fd);
                break;

            case FPD_MENU_TEST_KXGEN2_VM_PORT:
                fpd_test_kxgen2_vm_port(fd);
                break;

            case FPD_MENU_TEST_RECEIVER_VM_PORT:
                fpd_test_receiver_vm_port(fd);
                break;

            default:
                break;
        }
    }

    printf("\n");
    fpd_cleanup_diag();

    return result;
}

static void display_fpd_pdata_menu( unsigned int btype )
{
    int i = 0;
    unsigned char type[16];

    memset(type, 0, 16);
    switch( btype ) {
        case FPD_CIM_BUFFER:
            strncpy(type, "CIM", 3);
            break;
        case FPD_PRIORITY_BUFFER:
            strncpy(type, "Priority", 9);
            break;
        case FPD_HOST_BUFFER:
            strncpy(type, "Host", 4);
            break;
        default:
            strncpy(type, "UNKNOWN", 8);
            break;
    }

    printf("\nProtocol Data Tests:");
    printf("\n====================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Display Current Buffer Type, Packet Size and Pattern", i++);
    printf("\n\t %d: Change Buffer Type", i++);
    printf("\n\t %d: Change Packet Size and Pattern", i++);
    printf("\n\t %d: Send Packet to %s buffer", i++, type);
    printf("\n\t %d: Receive Packet from %s buffer", i++, type);
    printf("\n\t %d: Send Packet with CRC error thru %s buffer", i++, type);
    printf("\n\t %d: Receive Packet with CRC error thru %s buffer", i++, type);
    printf("\n\t %d: Send Packet with Protocol error thru %s buffer", i++, type);
    printf("\n\t %d: Receive Packet with Protocol error thru %s buffer", i++, type);
    printf("\n\t%d: Send Invalid Packet thru %s buffer", i++, type);
    printf("\n\t%d: Receive Invalid Packet thru %s buffer", i++, type);
    printf("\n\t%d: Testing %s Buffer Full (tx-side)", i++, type);
    printf("\n\t%d: Initiate testing %s Buffer Full (rx-side)", i++, type);
    printf("\n\t%d: Complete testing %s Buffer Full (rx-side)", i++, type);
    printf("\n\t%d: Continuously Send Packet to %s buffer", i++, type);
    printf("\n\t%d: Continuously Receive Packet from %s buffer", i++, type);
    printf("\n\n");

    return;
}

static int fpd_pdata_menu( int fd )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned int btype;

    btype = fpd_get_buffer_type();

    while( running ) {
        display_fpd_pdata_menu(btype);

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case FPD_BUF_MENU_EXIT:
                running = 0;
                break;

            case FPD_BUF_MENU_DISPLAY_PKT_INFO:
                fpd_display_packet_info(fd);
                break;

            case FPD_BUF_MENU_CHANGE_BUFFER_TYPE:
                fpd_change_buffer_type(fd);
                btype = fpd_get_buffer_type();
                break;

            case FPD_BUF_MENU_CHANGE_PKT:
                fpd_change_packet(fd);
                break;

            case FPD_BUF_MENU_SEND_PKT:
                fpd_send_pkt(fd);
                break;

            case FPD_BUF_MENU_RECEIVE_PKT:
                fpd_receive_pkt(fd);
                break;

            case FPD_BUF_MENU_SEND_PKT_WITH_CRC_ERROR:
                fpd_run_diag(fd, FPD_DIAG_TEST_CRC_ERROR, SEND, 1);
                break;

            case FPD_BUF_MENU_RECEIVE_PKT_WITH_CRC_ERROR:
                fpd_run_diag(fd, FPD_DIAG_TEST_CRC_ERROR, RECEIVE, 1);
                break;

            case FPD_BUF_MENU_SEND_PKT_WITH_PROTOCOL_ERROR:
                fpd_run_diag(fd, FPD_DIAG_TEST_PROTOCOL_ERROR, SEND, 1);
                break;

            case FPD_BUF_MENU_RECEIVE_PKT_WITH_PROTOCOL_ERROR:
                fpd_run_diag(fd, FPD_DIAG_TEST_PROTOCOL_ERROR, RECEIVE, 1);
                break;

            case FPD_BUF_MENU_SEND_INVALID_PKT:
                fpd_run_diag(fd, FPD_DIAG_TEST_INVALID_PKT, SEND, 1);
                break;

            case FPD_BUF_MENU_RECEIVE_INVALID_PKT:
                fpd_run_diag(fd, FPD_DIAG_TEST_INVALID_PKT, RECEIVE, 1);
                break;

            case FPD_BUF_MENU_TEST_BUFFER_FULL_TXSIDE:
                fpd_run_diag(fd, FPD_DIAG_TEST_BUFFER_FULL, SEND, 1);
                break;

            case FPD_BUF_MENU_TEST_BUFFER_FULL_RXSIDE_ON:
                fpd_run_diag(fd, FPD_DIAG_TEST_BUFFER_FULL, RECEIVE, 1);
                break;

            case FPD_BUF_MENU_TEST_BUFFER_FULL_RXSIDE_OFF:
                fpd_run_diag(fd, FPD_DIAG_TEST_BUFFER_FULL, RECEIVE, 0);
                break;

            case FPD_BUF_MENU_CONTINUOUSLY_SEND_PKTS:
                fpd_send_pkts_continuously(fd);
                break;

            case FPD_BUF_MENU_CONTINUOUSLY_RECEIVE_PKTS:
                fpd_receive_pkts_continuously(fd);
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_fpd_legacy_cim_menu( void )
{
    int i = 0;

    printf("\nLegacy CIM Tests:");
    printf("\n=================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Display Current Setup", i++);
    printf("\n\t %d: Display Valid Commands", i++);
    printf("\n\t %d: Change Lookup Table Setup", i++);
    printf("\n\t %d: Change Packet Size and Pattern", i++);
    printf("\n\t %d: Send Packet", i++);
    printf("\n\t %d: Receive Packet", i++);
    printf("\n\t %d: Continuously Send Packet", i++);
    printf("\n\t %d: Continuously Receive Packet", i++);
    printf("\n\n");

    return;
}

static int fpd_legacy_cim_menu( int fd )
{
    char running = 1;
    int result = 1;
    int diag_opt;

    while( running ) {
        display_fpd_legacy_cim_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case FPD_LEGACY_CIM_MENU_EXIT:
                running = 0;
                break;

            case FPD_LEGACY_CIM_MENU_DISPLAY_SETUP:
                legacy_cim_display_setup();
                break;

            case FPD_LEGACY_CIM_MENU_DISPLAY_CMDS:
                legacy_cim_display_valid_cmds(fd, fpd_get_current_link_if());
                break;

            case FPD_LEGACY_CIM_MENU_CHANGE_SETUP:
                legacy_cim_change_setup(fd, fpd_get_current_link_if());
                break;

            case FPD_LEGACY_CIM_MENU_CHANGE_PKT:
                legacy_cim_change_pkt();
                break;

            case FPD_LEGACY_CIM_MENU_SEND_PKT:
                legacy_cim_send_pkt(fd, fpd_get_current_link_if());
                break;

            case FPD_LEGACY_CIM_MENU_RECEIVE_PKT:
                legacy_cim_receive_pkt(fd, fpd_get_current_link_if());
                break;

            case FPD_LEGACY_CIM_MENU_CONTINUOUSLY_SEND_PKTS:
                legacy_cim_send_pkts_continuously(fd, fpd_get_current_link_if());
                break;

            case FPD_LEGACY_CIM_MENU_CONTINUOUSLY_RECEIVE_PKTS:
                legacy_cim_receive_pkts_continuously(fd, fpd_get_current_link_if());
                break;

            default:
                break;
        }
    }

    return result;
}
