/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Application to test the events of the FPGA Protocol Driver.
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <poll.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include "fpd_ioctl.h"

#define FPD_EVENTS_REV     "$Revision: 1.2 $"

#define MAX_DESCRIPTORS 1

/* timeouts in seconds */
#define POLL_TIMEOUT                (60 * 1000)
#define DEFAULT_PACKET_SIZE         32

/* Device ID - answer to ECHO Request */
#define DEVICE_ID_POWER_CIM_OLD                        0x30
#define DEVICE_ID_POWER_CIM_NEW                        0x31
#define DEVICE_ID_PC_CIM                               0x80
#define DEVICE_ID_P2CIM_PS2DUAL                        0x82
#define DEVICE_ID_SUN_CIM                              0x84

static int g_verbose = 0;
static int packet_size = DEFAULT_PACKET_SIZE;
static FPD_msg_t gmsg;
static void *host_buffer;
static int host_buffer_mmap_len = 0;
static int max_link_if = 0;
static int max_line_cnt = 0;

void usage( void );
void fpdlog( char * fmt, ... );
void msg( char *fmt, ... );
void err( char * fmt, ... );
static void poll_events( int fd, int timeout );
static void poll_line_status_change( int fd );
static void display_line_status( FPD_linestat_t *pinfo );
static void handle_events( int fd, FPD_event_t *pevt );


static int init_host_module( int fd )
{
    host_buffer = (void *) mmap(NULL,
                                host_buffer_mmap_len,
                                PROT_READ | PROT_WRITE,
                                MAP_SHARED,
                                fd,
                                0 );
    fprintf(stderr, "host_buffer = %p\n", host_buffer);
    if( host_buffer == (void *) -1 ) {
        return -1;
    }

    return 0;
}

static void cleanup_host_module( void )
{
    munmap(host_buffer, host_buffer_mmap_len);
    return;
}

int main( int argc, char * const argv[] )
{
    unsigned int timeout = POLL_TIMEOUT;
    int ch;
    int fd0 = -1;
    int rc;
    char dev[128] = "/dev/fpd";
    char cim_detect = 0;

    while( (ch = getopt(argc, argv, "t:p:chv")) != EOF ) {
        switch( ch ) {
            case 't':
                timeout = atoi(optarg);
                break;

            case 'p':
                packet_size = atoi(optarg);
                break;

            case 'c':
                cim_detect = 1;
                break;

            case 'v':
                g_verbose = 1;
                break;

            case 'h':
            default:
                usage();
                return 1;
        }
    }

    if( optind == argc-1 ) {
        strcpy(dev, argv[optind++]);
        msg( "using %s as device node ...\n", dev );
    }

    if( (fd0 = open(dev, O_RDWR)) < 0 ) {
        err( "Cannot open %s : %s\n", dev, strerror(errno) );
        return -1;
    }
    fprintf(stderr, "File Descriptor %d (%s)\n", fd0, dev);

    /* determine host buffer information */
    rc = ioctl(fd0, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        max_link_if = gmsg.info.link_if_cnt;
        max_line_cnt = gmsg.info.line_cnt;
        host_buffer_mmap_len = gmsg.info.host_buffer_memsize;

        /* initialize host module */
        rc = init_host_module(fd0);
        if( rc < 0 ) {
            fprintf(stderr, "Host Module init failed\n");
            close(fd0);
            return -2;
        }
    }

    if( cim_detect ) {
        poll_line_status_change(fd0);
    }
    else {
        poll_events(fd0, timeout);
    }

    cleanup_host_module();

    close(fd0);
	
    return 1;
}

static void poll_events( int fd, int timeout )
{
    struct pollfd fds[MAX_DESCRIPTORS];
    FPD_event_t event;
    int rc;
    int i;

    for (i = 0; i < MAX_DESCRIPTORS; i++) {
        fds[i].fd = -1;
    }
    
    fds[0].fd = fd;
    fds[0].events = POLLIN;

    while(1) {
        fprintf(stderr, "waiting for events ... ");

        rc = poll(fds, MAX_DESCRIPTORS, timeout);
        if((rc == 0) || (rc < 0 && errno == EINTR)) {
            printf("timedout\n");
            continue;
        }
        else if( rc < 0 ) {
            err("poll() returned %d\n", rc);
            break;
        }

        fprintf(stderr, "receive notifications\n");
        fprintf(stderr, "getting events ... ");
        rc = ioctl(fd, FPD_CHECK_EVENTS, &event);
        fprintf(stderr, "done\n");
        if( rc == 0 ) {
            handle_events(fd, &event);
        }
    }

    return;
}

static void poll_line_status_change( int fd )
{
    int rc;

    while(1) {
        fprintf(stderr, "\nwaiting for Line Status Change ... ");

        rc = ioctl(fd, FPD_POLL_LINE_STATUS, &gmsg);
        fprintf(stderr, "got it\n\n");
        if (rc == 0) {
            /* display info */
            display_line_status(&gmsg.linestat);
        }
        else {
            if(errno) {
                printf("FPD_GET_LINE_STATUS ioctl failed : %s\n", strerror(errno));
            }
            return;
        }
    }

    return;
}

void usage( void )
{
    printf( "Use (" FPD_EVENTS_REV "):\n"
            "fpd-events [-t msecs] [-p size] [-c] [-h] [-v] [devname]\n\n"
            "    -t msecs   set polling timeout in milliseconds\n"
            "    -p size    read packet size\n"
            "    -c         only wait for line status change\n"
            "    -h         help\n"
            "    -v         verbose\n"
            " [devname] is /dev/fpd\n\n");
}

void fpdlog( char *fmt, ... )
{
    va_list ap;
    if( g_verbose ) {
        printf( "[fpd-events] " );
        va_start(ap,fmt);
        vprintf( fmt, ap );
        va_end(ap);
    }	
}

void msg( char *fmt, ... )
{
   va_list ap;
   printf( "[fpd-events] " );
   va_start(ap,fmt);
   vprintf( fmt, ap );
   va_end(ap);
}

void err( char *fmt, ... )
{
   va_list ap;
   fprintf( stderr, "[fpd-events] " );
   va_start(ap,fmt);
   vfprintf( stderr, fmt, ap );
   va_end(ap);	
}

static int fpd_read_protocol_data( int fd, int link_if, int btype )
{
    int rc;
    int more_data_available = 1;
    FPD_data_t *pdata = &gmsg.pdata;
    unsigned char protocol_buffer[512];

    pdata->link_if = link_if;
    pdata->type = btype;
    pdata->requested_len = packet_size;
    pdata->buf = &protocol_buffer[0];

    do {
        printf("\t\tReading %d bytes of data from %s buffer....",
               pdata->requested_len, pdata->type?"Priority":"CIM");

        rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, &gmsg);
        if( rc == 0 ) {
            int i;

            if( gmsg.pdata.actual_len == gmsg.pdata.requested_len ) {
                printf("passed.\n");
            }
            else {
                printf("incomplete.\n");
                printf("\t\t\tRequested %d, Got %d\n",
                       gmsg.pdata.requested_len, gmsg.pdata.actual_len);
                more_data_available = 0;
            }

            if( gmsg.pdata.actual_len > 0 ) {
                printf("\t\t\tPacket: ");
            }

            for( i = 0; i < gmsg.pdata.actual_len; i++ ) {
                printf("%02x ", gmsg.pdata.buf[i]);
            }
            printf("\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_RECEIVE_PROTOCOL_DATA ioctl failed : %s\n", strerror(errno));
            }
            more_data_available = 0;
        }
    } while( more_data_available );

    return 0;
}

static int fpd_read_host_data( int fd, int link_if, int host_chan )
{
    FPD_hostdata_t host_data, *pkt = &host_data;
    FPD_host_rxparam_t *prxinfo;
    FPD_host_rxclear_t rxclear, *prx = &rxclear;
    unsigned char *bufaddr;
    int more_data_available = 1;
    int rc;


    pkt->link_if = link_if;
    pkt->host_chan = host_chan;
    pkt->requested_len = packet_size;

    do {
        printf("\t\tReading %d bytes of data from Host buffer %d....",
               pkt->requested_len, host_chan);

        rc = ioctl(fd, FPD_RECEIVE_HOST_DATA, pkt);
        if( rc == 0 ) {
            int i;

            if( pkt->actual_len == pkt->requested_len ) {
                printf("passed.\n");
            }
            else {
                more_data_available = 0;
                printf("incomplete.\n");
                printf("\t\tRequested %d, Got %d\n",
                       pkt->requested_len, pkt->actual_len);
            }

            /* compute buffer address */
            prxinfo = &pkt->info.rx;
            bufaddr = (unsigned char *)host_buffer + (prxinfo->buf_id * prxinfo->buf_size) + prxinfo->offset;
            fprintf(stderr, "\nbufID=%d, baseAddr=%p, bufAddr=%p\n", prxinfo->buf_id, host_buffer, bufaddr);

            /* dump data */
            if( pkt->actual_len > 0 ) {
                printf("\t\t\tPacket: ");
                for( i = 0; i < pkt->actual_len; i++ ) {
                    printf("%02x ", bufaddr[i]);
                }
                printf("\n");

                /* release buffer */
                prx->link_if = link_if;
                prx->host_chan = host_chan;
                prx->info = pkt->info.rx;
                printf("\t\tReleasing Host buffer %d....", host_chan);
                rc = ioctl(fd, FPD_RELEASE_RX_HOST_DATA, prx);
                if( rc == 0 ) {
                    printf("passed.\n");
                }
                else {
                    printf("FAILED.\n");
                    if(errno) {
                        printf("FPD_RELEASE_RX_HOST_DATA ioctl failed : %s\n", strerror(errno));
                    }
                    more_data_available = 0;
                }
            }
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_RECEIVE_HOST_DATA ioctl failed : %s\n", strerror(errno));
            }
            more_data_available = 0;
        }
    } while( more_data_available );

    return 0;
}

static int fpd_get_invalid_cmd( int fd, int link_if )
{
    FPD_rxinvcmd_t invalid_cmd;
    int rc;

    invalid_cmd.link_if = link_if;
    rc = ioctl(fd, FPD_GET_RX_INVALID_CMD, &invalid_cmd);
    if (rc == 0) {
        if( invalid_cmd.n_entries > 0 ) {
            int i;

            for( i = 0; i < invalid_cmd.n_entries; i++ ) {
                printf("\t\t\tCMD 0x%02x\n", invalid_cmd.inv_cmd[i]);
            }
        }
        else {
            printf("\t\t\tWEIRD!! No entries found.\n");
        }
    }
    else {
        if(errno) {
            printf("FPD_GET_RX_INVALID_CMD ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static void display_error( int fd, FPD_error_t *pinfo )
{
    switch( pinfo->type ) {
        case FPD_ERR_TYPE_PCI:
            printf("\t\tPCI Error: %08x\n", pinfo->error);
            if( pinfo->error ) {
                if( pinfo->error & PCI_ERR_PARITY ) {
                    printf("       Data Parity Detected\n");
                }
                else if( pinfo->error & PCI_ERR_SIG_TARGET_ABORT ) {
                    printf("       Signaled Target Abort\n");
                }
                else if( pinfo->error & PCI_ERR_RCV_TARGET_ABORT ) {
                    printf("       Received Target Abort\n");
                }
                else if( pinfo->error & PCI_ERR_RCV_MASTER_ABORT ) {
                    printf("       Received Master Abort\n");
                }
                else if( pinfo->error & PCI_ERR_SIG_SYSTEM_ERROR ) {
                    printf("       Signaled System Error\n");
                }
                else if( pinfo->error & PCI_ERR_DETECTED_PARITY ) {
                    printf("       Detected Parity Error\n");
                }
            }
            else {
                printf("       NONE\n");
            }
            break;

        case FPD_ERR_TYPE_LINK_IF:
            if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
                printf("\t\tBackground Link IF Error: %08x\n", pinfo->error);
            }
            else {
                printf("\t\tLink IF %d Error: %08x\n",
                       pinfo->link_if, pinfo->error);
            }

            if( pinfo->error ) {
                if( pinfo->error & FPD_ERR_RX_INV_CMD ) {
                    printf("\t\tFPD_ERR_RX_INV_CMD\n");

                    /* retrieve the invalid command */
                    fpd_get_invalid_cmd(fd, pinfo->link_if);
                }
                if( pinfo->error & FPD_ERR_UPDATE_INV_PKT ) {
                    printf("\t\tFPD_ERR_UPDATE_INV_PKT\n");
                }
                if( pinfo->error & FPD_ERR_RX_PRI_RD_EMPTY_BUF ) {
                    printf("\t\tFPD_ERR_RX_PRI_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_SYNC ) {
                    printf("\t\tFPD_ERR_TX_PRI_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_NO_EOP ) {
                    printf("\t\tFPD_ERR_TX_PRI_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_WR_FULL_BUF ) {
                    printf("\t\tFPD_ERR_TX_PRI_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_RX_CIM_RD_EMPTY_BUF ) {
                    printf("\t\tFPD_ERR_RX_CIM_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_SYNC ) {
                    printf("\t\tFPD_ERR_TX_CIM_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_NO_EOP ) {
                    printf("\t\tFPD_ERR_TX_CIM_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_WR_FULL_BUF ) {
                    printf("\t\tFPD_ERR_TX_CIM_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_TIMEOUT ) {
                    printf("\t\tFPD_ERR_TX_PRI_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_TIMEOUT ) {
                    printf("\t\tFPD_ERR_TX_CIM_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_0 ) {
                    printf("\t\tFPD_ERR_TX_HOST_TIMEOUT_0\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_1 ) {
                    printf("\t\tFPD_ERR_TX_HOST_TIMEOUT_1\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_0 ) {
                    printf("\t\tFPD_ERR_RX_DMA_NO_BUF_0\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_1 ) {
                    printf("\t\tFPD_ERR_RX_DMA_NO_BUF_1\n");
                }
            }
            else {
                printf("\t\tNONE\n");
            }
            break;

        default:
            break;
    }

    return;
}

static int fpd_get_link_errors( int fd, int link_if )
{
    int rc;

    gmsg.error.type = FPD_ERR_TYPE_LINK_IF;
    gmsg.error.link_if = link_if;
    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
    if (rc == 0) {
        /* display info */
        display_error(fd, &gmsg.error);
    }
    else {
        if(errno) {
            printf("FPD_GET_ERROR ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static int fpd_get_pci_errors( int fd )
{
    int rc;

    gmsg.error.type = FPD_ERR_TYPE_PCI;
    gmsg.error.link_if = 0; /* don't care */
    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
    if (rc == 0) {
        /* display info */
        display_error(fd, &gmsg.error);
    }
    else {
        if(errno) {
            printf("FPD_GET_ERROR ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static void display_line_status( FPD_linestat_t *pinfo )
{
    int i, id, line;
    unsigned int status;
    unsigned char linestat;

    i = 0;
    line = 0;
    do {
        status = pinfo->line_status[i];
        for( id = 0; id < 16; id++ ) {
            if( (line + id) >= max_line_cnt ) {
                return;
            }

            printf("\t\tLine %02d: ", line + id);
            linestat = status & 0x3;
            switch( linestat ) {
                case 0:
                    printf("-");
                    break;
                case 1:
                    printf("Paragon CIM");
                    break;
                case 2:
                    printf("KX2.0 CIM");
                    break;
                default:
                    printf("Unknown!");
                    break;
            }

            if( line < 32 ) {
                if( pinfo->status_change[0] & (1 << (line+id)) ) {
                    printf(" (Status changed)\n");
                }
                else {
                    printf("\n");
                }
            }
            else {
                if( pinfo->status_change[1] & (1 << ((line - 32) + id)) ) {
                    printf(" (Status changed)\n");
                }
                else {
                    printf("\n");
                }
            }

            status >>= 2;
        }
        i++;
        line += 16;
    } while( line < max_line_cnt );

    return;
}

static int fpd_get_line_status( int fd )
{
    int rc;

    rc = ioctl(fd, FPD_GET_LINE_STATUS, &gmsg);
    if (rc == 0) {
        /* display info */
        display_line_status(&gmsg.linestat);
    }
    else {
        if(errno) {
            printf("FPD_GET_LINE_STATUS ioctl failed : %s\n", strerror(errno));
        }
    }


    return 0;
}

static int fpd_get_echo_rsp( int fd, unsigned int link_if )
{
    int rc;

    gmsg.echo_rsp.link_if = link_if;
    rc = ioctl(fd, FPD_GET_ECHO_RSP, &gmsg);
    if( rc == 0 ) {
        printf("\t\tDetected ");
        switch( gmsg.echo_rsp.echo_rsp ) {
            case DEVICE_ID_POWER_CIM_OLD:
                printf("old Power CIM\n");
                break;
            case DEVICE_ID_POWER_CIM_NEW:
                printf("KX2.0 Power CIM\n");
                break;
            case DEVICE_ID_PC_CIM:
                printf("PC CIM\n");
                break;
            case DEVICE_ID_P2CIM_PS2DUAL:
                printf("P2CIM-PS2DUAL\n");
                break;
            case DEVICE_ID_SUN_CIM:
                printf("SUN CIM\n");
                break;
            default:
                printf("Unknown device [%02x]\n", gmsg.echo_rsp.echo_rsp);
                return -2;
        }
    }
    else {
        printf("ERROR: FPD_GET_ECHO_RSP ioctl failed (%s)\n", strerror(errno));
    }

    return rc;
}

static void handle_events( int fd, FPD_event_t *pevt )
{
    int link;
    int id;

    printf("\n");

    /* Link Interfaces */
    for( link = 0; link < max_link_if; link++ ) {
        if( !(pevt->event_mask & (FPD_EVENT_MASK_LINK0 << link)) )
            continue;

        /* Host buffer */
        for( id = 0; id < FPD_HOST_CNT; id++ ) {
            if( pevt->link_if[link].host[id].has_data ) {
                printf("\tLink IF %d require Host Buffer %d processing\n",
                       link, id);
                fpd_read_host_data( fd, link, id );
            }
        }

        /* Priority buffer */
        if( pevt->link_if[link].rx_priority ) {
            printf("\tLink IF %d receive Priority Buffer data\n", link);
            fpd_read_protocol_data( fd, link, FPD_PRIORITY_BUFFER );
        }

        /* CIM buffer */
        if( pevt->link_if[link].rx_cim ) {
            printf("\tLink IF %d receive CIM Buffer data\n", link);
            fpd_read_protocol_data( fd, link, FPD_CIM_BUFFER );
        }

        /* error detected */
        if( pevt->link_if[link].error ) {
            printf("\tLink IF %d receive errors\n", link);
            fpd_get_link_errors( fd, link );
        }

        /* echo response detected */
        if( pevt->link_if[link].echo_rsp ) {
            printf("\tLink IF %d receive echo response\n", link);
            fpd_get_echo_rsp( fd, link );
        }
    }

    /* Background Link Interface events */
    if( pevt->event_mask & (FPD_EVENT_MASK_BGND_LINK) ) {
        if( pevt->bgnd_link_if.rx_cim ) {
            printf("\tBacground Link IF receive CIM Buffer data\n");
            fpd_read_protocol_data( fd, FPD_LINK_IF_ID_BGND, FPD_CIM_BUFFER );
        }

        if( pevt->bgnd_link_if.error ) {
            printf("\tBackground Link IF receive errors\n");
            fpd_get_link_errors( fd, FPD_LINK_IF_ID_BGND );
        }

        if( pevt->bgnd_link_if.echo_rsp ) {
            printf("\tBackground Link IF receive echo response\n");
            fpd_get_echo_rsp( fd, FPD_LINK_IF_ID_BGND );
        }
    }

    if( pevt->pci_error ) {
        printf("\tDetected PCI error\n");
        fpd_get_pci_errors( fd );
    }

    if( pevt->cim_detect ) {
        printf("\tDetected changes to CIM topology\n");
        fpd_get_line_status( fd );
    }

    printf("\n");

    return;
}
