/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Application to switch the communication and video path
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"

#define FPD_SWITCH_REV        "$Revision: 1.4 $"
#define FPD_APPS_PASS         0
#define FPD_APPS_FAIL        -1

/* CIM types */
#define VMCIM                 1
#define PCIM                  2
#define DCIM                  3

#define LOCAL_VIDEO_DISABLE   0
#define LOCAL_VIDEO_PCIM      1
#define LOCAL_VIDEO_VMCIM     2
#define LOCAL_VIDEO_GUI       3

#define FPD_LINK_IF_BASE0     0x00000180
#define FPD_LINK_IF_BASE1     0x00000200
#define FPD_LINK_IF_BASE2     0x00000280
#define FPD_LINK_IF_BASE3     0x00000300
#define FPD_LINK_IF_BASE4     0x00000380

static int host_buffer_mmap_len = 0;
static int max_link_if = 0;
static int max_line_cnt = 0;
static char bgnd_link_if_present = 0;

enum rx_paragon_cmd {
    PCMD_AUTO_SKEW              = 0x03,
    PCMD_VIDEO_PHSYNC_PVSYNC    = 0x04,
    PCMD_VIDEO_PHSYNC_NVSYNC    = 0x05,
    PCMD_VIDEO_NHSYNC_PVSYNC    = 0x06,
    PCMD_VIDEO_NHSYNC_NVSYNC    = 0x07,
    PCMD_PS2KBD_DATA1           = 0x09,
    PCMD_PS2KBD_DATA2           = 0x0A,
    PCMD_SUNKBD_DATA1           = 0x0D,
    PCMD_SUNKBD_DATA2           = 0x0E,
    PCMD_SUNKBD_DATA3           = 0x0F,
    PCMD_CIM_CONFIGURATION      = 0x10,
    PCMD_PS2MOUSE_CMD1          = 0x11,
    PCMD_PS2MOUSE_CMD2          = 0x12,
    PCMD_BEEP                   = 0x15,
    PCMD_CIM_EEPROM_REPORT      = 0x64,

    /* CIM FW Update */
    PCMD_CIM_UPDATE_DEV_TYPE    = 0xC9,
    PCMD_CIM_UPDATE_CONT_SEND   = 0xF4,
    PCMD_CIM_UPDATE_ERASE_ERR   = 0xF6,
    PCMD_CIM_UPDATE_PROG_ERR    = 0xF7,
    PCMD_CIM_UPDATE_BLKCHK_ERR  = 0xF8,
    PCMD_CIM_UPDATE_ACK         = 0xFA,
    PCMD_CIM_UPDATE_NACK        = 0xFE,
};

static FPD_cmdprog_t pcmds[] = {
  /* link,           cmd,            len, valid, ram, chksum, dest */
    {   0, PCMD_AUTO_SKEW,             1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_PHSYNC_PVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_PHSYNC_NVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_NHSYNC_PVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_VIDEO_NHSYNC_NVSYNC,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_CONFIGURATION,     5,     1,   1,      1, FPD_CIM_BUFFER},
    {   0, PCMD_PS2KBD_DATA1,          2,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_PS2KBD_DATA2,          3,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_SUNKBD_DATA1,          2,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_SUNKBD_DATA2,          3,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_SUNKBD_DATA3,          4,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_PS2MOUSE_CMD1,         2,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_PS2MOUSE_CMD2,         3,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_EEPROM_REPORT,     4,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_DEV_TYPE,  27,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_CONT_SEND,  1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_ERASE_ERR,  1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_PROG_ERR,   1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_BLKCHK_ERR, 1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_ACK,        1,     1,   1,      0, FPD_CIM_BUFFER},
    {   0, PCMD_CIM_UPDATE_NACK,       1,     1,   1,      0, FPD_CIM_BUFFER},
};

static void usage( void )
{
    char answer;

    printf( "Use (" FPD_SWITCH_REV "):\n"
            "fpd-switch [options] [devname ~ /dev/fpd]\n"
            "    -l link        link interface\n"
            "    -t target_port target # to switch\n"
            "    -c cim_type    CIM type [1=VM-CIM, 2=PCIM, 3=DCIM]\n"
            "    -k kvm_chan    KVM channel to use (VSC)\n"
            "    -h hsync_pol   H-sync polarity (if PCIM or DCIM)\n"
            "                    [0=positive, 1=negative]\n"
            "    -v vsync_pol   V-sync polarity (if PCIM or DCIM)\n"
            "                    [0=positive, 1=negative]\n"
            "    -d             Disconnect Communication & Video Path\n"
            "    -x             Local port switching\n"
            "    -z             Local GUI mode (use with -x)\n"
            "    -g             Green mode (use with -x)\n"
            "    -?             help\n\n");
    printf( "NOTE: link, target_port, and kvm_chan are all 0-based.\n");
    printf( "      Do not connect same target port to multiple Link IF/VSC.\n\n");

    printf("Do you want to see configuration examples (y/n) : ");
    scanf("%c", &answer);

    /* CIM connection */
    if((answer == 'y') || (answer == 'Y')) {
        printf( "1. To connect Link IF 0 to Target 0 using VSC 0 and VM-CIM, use:\n");
        printf( "\t> fpd-switch -l 0 -t 0 -c 1 -k 0\n");
        printf( "2. To connect Link IF 1 to Target 31 using VSC 2 and DCIM\n"
                "   with a 1024x768@60Hz (H/V-sync are negative), use:\n");
        printf( "\t> fpd-switch -l 1 -t 31 -c 3 -k 2 -h 1 -v 1\n");
        printf( "3. To disconnect Link IF 0 using VSC 0, use:\n");
        printf( "\t> fpd-switch -d -l 0 -k 0\n");
        printf( "4. Switching local port to Target 10 using Link IF 5 and\n"
                "   PCIM with a 800x600@75Hz (H/V-sync are positive), use:\n");
        printf( "\t> fpd-switch -x -l 0 -t 10 -c 2 -h 0 -v 0\n");
        printf( "5. Switching local port to Local GUI\n");
        printf( "\t> fpd-switch -x -l 0 -z\n");
        printf( "6. Switching local port to Green Mode\n");
        printf( "\t> fpd-switch -x -g\n\n");
    }
}

static void err( char *fmt, ... )
{
   va_list ap;
   fprintf( stderr, "[fpd-switch] " );
   va_start(ap,fmt);
   vfprintf( stderr, fmt, ap );
   va_end(ap);	
}

static int create_switch_params( FPD_switch_t *pswitch,
                                 int link,
                                 int line,
                                 int protocol,
                                 int txpar,
                                 int rxpar )
{
    if( link < max_link_if ) {
        pswitch->link_if = link;
    }
    else if ( link == FPD_LINK_IF_ID_BGND ) {
        if( bgnd_link_if_present ) {
            pswitch->link_if = link;
        }
        else {
            printf("\nERROR: Background Link Interface is not present\n");
            printf("Please re-do!!!\n");
            return -1;
        }
    }
    else {
        printf("\nERROR: Invalid Link Interface ID %d\n", link);
        printf("Please re-do!!!\n");
        return -1;
    }

    pswitch->protocol  = protocol;
    pswitch->tx_parity = txpar;
    pswitch->rx_parity = rxpar;

    if( line < max_line_cnt ) {
        pswitch->target_port = line;
    }
    else {
        printf("\nERROR: Invalid Line ID %d\n", line);
        printf("Please re-do!!!\n");
        return -3;
    }

    pswitch->driver_txbuf_wait_time = 0;
    pswitch->fpga_txbuf_wait_time = 0;

    return 0;
}

static int create_remote_video_params( FPD_video_switch_t *pvideo,
                                       int link,
                                       int line,
                                       int kvm_chan,
                                       int protocol,
                                       int hsync,
                                       int vsync )
{
    pvideo->link_if     = link;
    pvideo->kvm_chan    = kvm_chan;
    pvideo->target_port = line;
    pvideo->protocol    = protocol;
    pvideo->hsync_pol   = hsync;
    pvideo->vsync_pol   = vsync;

    return 0;
}

static int create_local_video_params( FPD_local_video_switch_t *pvideo,
                                      int src,
                                      int link,
                                      int line,
                                      int hsync,
                                      int vsync )
{
    pvideo->local_video_src = src;
    if( pvideo->local_video_src != FPD_LOCAL_VIDEO_SRC_DISABLE ) {
        pvideo->link_if = link;
        if( pvideo->local_video_src != FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE ) {
            pvideo->target_port = line;
            if( pvideo->local_video_src == FPD_LOCAL_VIDEO_SRC_PARAGON_CIM ) {
                pvideo->hsync_pol = hsync;
                pvideo->vsync_pol = vsync;
            }
        }
    }

    return 0;
}

static int disconnect_link( int fd, int link_id )
{
    FPD_disc_t disc;

    disc.link_if = link_id;
    disc.driver_txbuf_wait_time = 0;
    disc.fpga_txbuf_wait_time = 0;

    return( ioctl(fd, FPD_DISCONNECT_CHANNEL, &disc) );
}

static int invalidate_all_cmds( int fd, int link_if )
{
    FPD_cmdinv_t cmd;
    int rc;

    cmd.link_if = link_if;
    rc = ioctl(fd, FPD_INVALIDATE_ALL_CMDS, &cmd);

    return rc;
}

static int configure_lookup_table( int fd, int link_if )
{
    int i, rc;
    int n_cmds = sizeof(pcmds)/sizeof(pcmds[0]);

    /* invalidate all commands in the lookup table */
    if( invalidate_all_cmds(fd, link_if) != 0 ) {
        err("ERROR: Failed to invalidate all cmds!\n");
        return -1;
    }

    /* write commands */
    for( i = 0; i < n_cmds; i++ ) {
        pcmds[i].link_if = link_if;
        rc = ioctl(fd, FPD_WRITE_ENTRY_CMD_TABLE, &pcmds[i]);
        if( rc != 0 ) {
            err("ERROR: FPD_WRITE_ENTRY_CMD_TABLE ioctl failed (%s)\n", strerror(errno));
            if( invalidate_all_cmds(fd, link_if) != 0 ) {
                err("ERROR: Failed to invalidate all cmds!\n");
                return -1;
            }
            return -2;
        }
    }

    return 0;
}

int main( int argc, char * const argv[] )
{
    int fd;
    int ch;
    char dev[128] = "/dev/fpd";
    FPD_msg_t gmsg;
    int link = -1;
    int target_port = -1;
    int cim_type = -1;
    int kvm_chan = -1;
    int protocol = -1, tx_parity = -1, rx_parity = -1;
    int hsync_pol = -1, vsync_pol = -1;
    int disconnect = 0;
    char local_switch = 0;
    char green_mode = 0;
    char local_gui = 0;
    int local_video_src = -1;
    int rc = 0;

    while( (ch = getopt(argc, argv, "l:t:c:k:h:v:dxzg?")) != EOF ) {
        switch( ch ) {
            case 'l':
                link = atoi(optarg);
                break;

            case 't':
                target_port = atoi(optarg);
                break;

            case 'c':
                cim_type = atoi(optarg);
                break;

            case 'k':
                kvm_chan = atoi(optarg);
                break;

            case 'h':
                hsync_pol = atoi(optarg);
                break;

            case 'v':
                vsync_pol = atoi(optarg);
                break;

            case 'd':
                disconnect = 1;
                break;

            case 'x':
                local_switch = 1;
                break;

            case 'g':
                green_mode = 1;
                break;

            case 'z':
                local_gui = 1;
                break;

            case '?':
            default:
                usage();
                return 1;
        }
    }

    if( optind == argc-1 ) {
        strcpy(dev, argv[optind++]);
    }

    if( disconnect && local_switch ) {
        err("Do not use -d and -x options together\n");
        return -1;
    }

    if( disconnect ) {
        /* disconnection */
        if((link < 0) || (kvm_chan < 0)) {
            err("To disconnect, specify both Link IF and KVM Channel\n");
            return -2;
        }
    }
    else if( local_switch ) {
        /* local port connection */
        if( green_mode ) {
            local_video_src = FPD_LOCAL_VIDEO_SRC_DISABLE;
        }

        if( local_gui ) {
            if( local_video_src == -1 ) { 
                local_video_src = FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE;
                if( link < 0 ) {
                    err("Specify Link IF using -l\n");
                    return -3;
                }
            }
            else {
                err("Do not use -g, -z, and -c options together\n");
                return -4;
            }
        }

        if( cim_type > 0 ) {
            if( local_video_src != -1 ) { 
                err("Do not use -g, -z, and -c options together\n");
                return -5;
            }

            switch( cim_type ) {
                case VMCIM:
                    local_video_src = FPD_LOCAL_VIDEO_SRC_VM_CIM;
                    break;
                case PCIM:
                case DCIM:
                    local_video_src = FPD_LOCAL_VIDEO_SRC_PARAGON_CIM;
                    break;
                default:
                    err("Invalid CIM type specified\n");
                    return -6;
            }
        }

        if( local_video_src < 0 ) {
            err("Specify Local video source using either -g, -z, or -c\n");
            return -7;
        }

        switch( local_video_src ) {
            case FPD_LOCAL_VIDEO_SRC_DISABLE:
                break;
            case FPD_LOCAL_VIDEO_SRC_PARAGON_CIM:
                if( link < 0 ) {
                    err("Specify Link IF using -l\n");
                    rc = -1;
                }
                if( target_port < 0 ) {
                    err("Specify Target Port using -t\n");
                    rc = -1;
                }
                if( hsync_pol < 0 ) {
                    err("Specify H-sync polarity using -h\n");
                    rc = -1;
                }
                if( vsync_pol < 0 ) {
                    err("Specify V-sync polarity using -v\n");
                    rc = -1;
                }
                break;
            case FPD_LOCAL_VIDEO_SRC_VM_CIM:
                if( link < 0 ) {
                    err("Specify Link IF using -l\n");
                    rc = -1;
                }
                if( target_port < 0 ) {
                    err("Specify Target Port using -t\n");
                    rc = -1;
                }
                break;
            case FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE:
                if( link < 0 ) {
                    err("Specify Link IF using -l\n");
                    rc = -1;
                }
                break;
            default:
                rc = -1;
                break;
        }

        if( rc < 0 ) {
            usage();
            return -8;
        }
    }
    else {
        /* remote connection */
        if( link < 0 ) {
            err("Specify Link IF using -l\n");
            rc = -1;
        }
        if( target_port < 0 ) {
            err("Specify Target Port using -t\n");
            rc = -1;
        }
        if( kvm_chan < 0 ) {
            err("Specify KVM Channel using -k\n");
            rc = -1;
        }
        if( cim_type < 0 ) {
            err("Specify CIM Type using -c\n\n");
            rc = -1;
        }

        if( rc < 0 ) {
            usage();
            return -9;
        }
    }

    if( (fd = open(dev, O_RDWR)) < 0 ) {
        err("Cannot open %s : %s\n", dev, strerror(errno));
        return -10;
    }

    /* determine Link and Line Interface count */
    rc = ioctl(fd, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        max_link_if = gmsg.info.link_if_cnt;
        max_line_cnt = gmsg.info.line_cnt;
        host_buffer_mmap_len = gmsg.info.host_buffer_memsize;
        bgnd_link_if_present = gmsg.info.bgnd_link_if_present;
    }
    else {
        err("Failed to get info.\n");
        goto switch_exit;
    }

    if( disconnect ) {
        /* DISCONNECTION */

        if( link >= max_link_if ) {
            err("Invalid Link Interface specified\n");
            rc = -11;
            goto switch_exit;
        }

        printf("Disconnecting CommPath Link IF %d.................", link);
        if( disconnect_link(fd, link) == 0 ) {
            printf("passed\n");
        }
        else {
            printf("FAILED\n");
        }

        printf("Disconnecting VideoPath Link IF %d, VSC %d.........", link, kvm_chan);
        gmsg.video_switch.link_if = link;
        gmsg.video_switch.kvm_chan = kvm_chan;
        rc = ioctl(fd, FPD_DISABLE_REMOTE_VIDEO, &gmsg);
        if (rc == 0) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
        }

        goto switch_exit;
    }
    else if(local_switch && 
            ((local_video_src == FPD_LOCAL_VIDEO_SRC_DISABLE) ||
             ((local_video_src == FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE)))) {
        create_local_video_params(&gmsg.local_video_switch, local_video_src, link, target_port, hsync_pol, vsync_pol);
        switch( gmsg.local_video_switch.local_video_src ) {
            case FPD_LOCAL_VIDEO_SRC_DISABLE:
                printf("Disabling Local Video............................");
                break;
            default:
                if( link >= max_link_if ) {
                    err("Invalid Link Interface specified\n");
                    rc = -11;
                    goto switch_exit;
                }

                printf("Switching Local Video to Local GUI...............");
                break;
        }

        rc = ioctl(fd, FPD_SWITCH_LOCAL_VIDEO, &gmsg);
        if (rc == 0) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
        }

        if( local_video_src == FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE ) {
            printf("Disconnecting CommPath Link IF %d.................", link);
            if( disconnect_link(fd, link) == 0 ) {
                printf("passed\n");
            }
            else {
                printf("FAILED\n");
            }
        }

        goto switch_exit;
    }
    else {
        /* check for valid args */
        if( link >= max_link_if ) {
            err("Invalid Link Interface specified\n");
            rc = -12;
            goto switch_exit;
        }

        if( target_port >= max_line_cnt ) {
            err("Invalid Link Interface specified\n");
            rc = -13;
            goto switch_exit;
        }

        switch( cim_type ) {
            case VMCIM:
                protocol = FPD_PROTOCOL_VM;
                break;
            case PCIM:
                protocol  = FPD_PROTOCOL_PARAGON;
                tx_parity = FPD_PARAGON_PARITY_ODD;
                rx_parity = FPD_PARAGON_PARITY_ODD;
                break;
            case DCIM:
                protocol  = FPD_PROTOCOL_PARAGON;
                tx_parity = FPD_PARAGON_PARITY_EVEN;
                rx_parity = FPD_PARAGON_PARITY_EVEN;
                break;
            default:
                err("Invalid CIM type specified\n");
                rc = -14;
                goto switch_exit;
        }

        /* setup command table if paragon */
        if( protocol == FPD_PROTOCOL_PARAGON ) {
            /* required to disconnect link before re-configuring */
            rc = disconnect_link(fd, link);
            if( rc != 0 ) {
                err("ERROR: Failed to disconnect Link %d to configure Paragon table\n", link);
                goto switch_exit;
            }

            if( configure_lookup_table(fd, link) != 0 ) {
                rc = -15;
                goto switch_exit;
            }
        }

        if( create_switch_params(&gmsg.conn, link, target_port, protocol, tx_parity, rx_parity) == 0 ) {
            printf("Switching CommPath Link IF %d to Target %d.........",
                   gmsg.conn.link_if, gmsg.conn.target_port);
            rc = ioctl(fd, FPD_SWITCH_CHANNEL, &gmsg);
            if (rc == 0) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                goto switch_exit;
            }
        }
        else {
            rc = -16;
            goto switch_exit;
        }

        if( local_switch ) {
            create_local_video_params(&gmsg.local_video_switch, local_video_src, link, target_port, hsync_pol, vsync_pol);
            switch( local_video_src ) {
                case FPD_LOCAL_VIDEO_SRC_PARAGON_CIM:
                    printf("Switching Local Video to PCIM Target %d...........",
                           gmsg.local_video_switch.target_port);
                    break;
                case FPD_LOCAL_VIDEO_SRC_VM_CIM:
                    printf("Switching Local Video to VMCIM Target %d..........",
                           gmsg.local_video_switch.target_port);
                    break;
            }

            rc = ioctl(fd, FPD_SWITCH_LOCAL_VIDEO, &gmsg);
            if (rc == 0) {
                printf("passed.\n");
                goto switch_exit;
            }
            else {
                printf("FAILED.\n");
            }
        }
        else {
            /* remote switching */
            create_remote_video_params(&gmsg.video_switch, link, target_port, kvm_chan, protocol, hsync_pol, vsync_pol);
            printf("Switching VideoPath Link %d, VSC %d to Target %d....",
                   gmsg.video_switch.link_if,
                   gmsg.video_switch.kvm_chan,
                   gmsg.video_switch.target_port);
            rc = ioctl(fd, FPD_SWITCH_REMOTE_VIDEO, &gmsg);
            if (rc == 0) {
                printf("passed.\n");
                goto switch_exit;
            }
            else {
                printf("FAILED.\n");
            }
        }
    }

    printf("Disconnecting CommPath Link IF %d.................", link);
    if( disconnect_link(fd, link) == 0 ) {
        printf("passed\n");
    }
    else {
        printf("FAILED\n");
    }

 switch_exit:
    close(fd);

    return rc;
}
