/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Interface
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_ioctl.h"
#include "fpd_reg.h"
#include "fpd_diag.h"
#include "fpd_link.h"
#include "fpd_paragon.h"
#include "fpd_video_switch.h"
#include "debug.h"
#include "version.h"

#undef	DEBUG

#ifdef	DEBUG
# define debugk(fmt,args...)	printk(fmt ,##args)
#else
# define debugk(fmt,args...)
#endif


/******************************************************************************
 *  ROUTINE:        fpd_get_info()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves information regarding the FPGA Protocol including the 
 *      current driver version, FPGA image version, and FPGA Protocol version.
 *
 *  PARAMETERS:
 *
 *      pdev  -  FPD device info
 *      pinfo -  Ptr to a user-allocated buffer to store the information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_info(fpd_device_t *pdev, FPD_info_t *pinfo)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);
    *pinfo = pdev->info;
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_pci_cfg()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves information regarding the FPGA Protocol's PCI Configuration
 *      Space.
 *
 *  PARAMETERS:
 *
 *      pdev  -  FPD device info
 *      pcfg  -  Ptr to a user-allocated buffer to store the information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_pci_cfg(fpd_device_t *pdev, FPD_pcicfg_t *pcfg)
{
    struct pci_dev *pci = pdev->pcidev;

    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    pci_read_config_word(pci, PCI_VENDOR_ID, &pcfg->vendor_id);
    pci_read_config_word(pci, PCI_DEVICE_ID, &pcfg->device_id);
    pci_read_config_word(pci, PCI_COMMAND, &pcfg->command);
    pci_read_config_word(pci, PCI_STATUS, &pcfg->status);
    pci_read_config_byte(pci, PCI_REVISION_ID, &pcfg->revision_id);
    pci_read_config_byte(pci, PCI_CLASS_PROG, &pcfg->prog_if);
    pci_read_config_byte(pci, PCI_CLASS_DEVICE, &pcfg->sub_class_code);
    pci_read_config_byte(pci, PCI_CLASS_CODE, &pcfg->base_class_code);
    pci_read_config_byte(pci, PCI_CACHE_LINE_SIZE, &pcfg->cache_line_size);
    pci_read_config_byte(pci, PCI_LATENCY_TIMER, &pcfg->latency_timer);
    pci_read_config_byte(pci, PCI_HEADER_TYPE, &pcfg->header_type);
    pci_read_config_byte(pci, PCI_BIST, &pcfg->bist);
    pci_read_config_dword(pci, PCI_BASE_ADDRESS_0, &pcfg->bar0);
    pci_read_config_dword(pci, PCI_BASE_ADDRESS_1, &pcfg->bar1);
    pci_read_config_dword(pci, PCI_BASE_ADDRESS_2, &pcfg->bar2);
    pci_read_config_dword(pci, PCI_BASE_ADDRESS_3, &pcfg->bar3);
    pci_read_config_dword(pci, PCI_BASE_ADDRESS_4, &pcfg->bar4);
    pci_read_config_dword(pci, PCI_BASE_ADDRESS_5, &pcfg->bar5);
    pci_read_config_dword(pci, PCI_CARDBUS_CIS, &pcfg->cardbus_cis_ptr);
    pci_read_config_word(pci, PCI_SUBSYSTEM_VENDOR_ID, &pcfg->subsystem_vendor_id);
    pci_read_config_word(pci, PCI_SUBSYSTEM_ID, &pcfg->subsystem_id);
    pci_read_config_dword(pci, PCI_ROM_ADDRESS, &pcfg->rom_bar);
    pci_read_config_byte(pci, PCI_CAPABILITY_LIST, &pcfg->cap_ptr);
    pci_read_config_byte(pci, PCI_INTERRUPT_LINE, &pcfg->interrupt_line);
    pci_read_config_byte(pci, PCI_INTERRUPT_PIN, &pcfg->interrupt_pin);
    pci_read_config_byte(pci, PCI_MIN_GNT, &pcfg->min_grant);
    pci_read_config_byte(pci, PCI_MAX_LAT, &pcfg->max_latency);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_read_reg()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Read a FPGA Protocol device Register.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      preg  -  Ptr to a user-allocated buffer that contains the register 
 *               offset and a place to store the output.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_read_reg(fpd_device_t *pdev, FPD_rw_reg_t *preg)
{
    debugk("%s: dev %d base %p reg %x\n",
           __FUNCTION__, pdev->id, pdev->remap, preg->reg);

    if((preg->reg >= pdev->iolen) || (preg->reg & 0x3)) {
        return -EINVAL;
    }

    preg->value = FPD_READ(pdev->remap, preg->reg);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_write_reg()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Write to a FPGA Protocol device Register.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      preg  -  Ptr to a user-allocated buffer that contains the register 
 *               offset and the value to write.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_write_reg(fpd_device_t *pdev, FPD_rw_reg_t *preg)
{
    debugk("%s: dev %d base %p reg %x val %x\n",
           __FUNCTION__, pdev->id, pdev->remap, preg->reg, preg->value);

    if((preg->reg >= pdev->iolen) || (preg->reg & 0x3)) {
        return -EINVAL;
    }

    FPD_WRITE(pdev->remap, preg->reg, preg->value);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_check_events()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the event(s) triggered by FPGA Protocol device.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      pevt  -  Ptr to a user-allocated buffer to store the events.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_check_events(fpd_device_t *pdev, FPD_event_t *pevt)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    /* get lock */
    fpd_lock_acquire(&pdev->event_lock);

    /* retrieve current set events */
    memcpy(pevt, &pdev->event, sizeof(FPD_event_t));

    /* clear events */
    memset(&pdev->event, 0, sizeof(FPD_event_t));
    fpd_lock_write_resource(&pdev->event_lock, 0);

    /* release lock */
    fpd_lock_release(&pdev->event_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_send_protocol_data()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Sends a keyboard, mouse, or CIM data to the FPGA Protocol device to
 *      be transmitted to the Far-end (i.e., CIM device).
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      ppkt  -  Ptr to a user-allocated buffer containing the link interface, 
 *               routing info, and packets.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_send_protocol_data(fpd_device_t *pdev, FPD_data_t *ppkt)
{
    fpd_linkif_t *plink;
    int bytes_written;
    int result;

    debugk("%s: dev %d linkIF %d type %d len %d\n", __FUNCTION__, pdev->id,
           ppkt->link_if, ppkt->type, ppkt->requested_len);

    if(( ppkt->link_if >= pdev->info.link_if_cnt && ppkt->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( ppkt->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present &&
         ppkt->type != FPD_CIM_BUFFER )) {
        return -EINVAL;
    }

    if( ppkt->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[ppkt->link_if];
    }

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        ppkt->actual_len = 0;
        return -ENOLINK;
    }

    /* it is not allowed to send data when in the middle of disconnection */
    if( atomic_read(&plink->close_comm) ) {
        ppkt->actual_len = 0;
        return -ENOLINK;
    }

    /* check for valid packet length */
    if(( ppkt->requested_len > FPD_MAX_PACKET_SIZE ) ||
       ( ppkt->requested_len < 0 )) {
        ppkt->actual_len = 0;
        return -EINVAL;
    }

    switch( ppkt->type ) {
        case FPD_CIM_BUFFER:
            bytes_written = pdata_write(&plink->cim, ppkt->requested_len,
                                        ppkt->buf_const, USER_SPACE);
            break;
        case FPD_PRIORITY_BUFFER:
            bytes_written = pdata_write(&plink->priority, ppkt->requested_len,
                                        ppkt->buf_const, USER_SPACE);
            break;
        default:
            ppkt->actual_len = 0;
            return -EINVAL;
    }

    if( bytes_written < 0 ) {
        ppkt->actual_len = 0;
        result = bytes_written;
    }
    else {
        ppkt->actual_len = bytes_written;
        result = 0;
    }

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_receive_protocol_data()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Get a keyboard, mouse, or CIM data received from the FPGA Protocol
 *      device.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      ppkt  -  Ptr to a user-allocated buffer containing the link interface, 
 *               routing info, and packets.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_receive_protocol_data(fpd_device_t *pdev, FPD_data_t *ppkt)
{
    fpd_linkif_t *plink;
    int bytes_read;
    int result;

    debugk("%s: dev %d linkIF %d type %d len %d\n", __FUNCTION__, pdev->id,
           ppkt->link_if, ppkt->type, ppkt->requested_len);

    if(( ppkt->link_if >= pdev->info.link_if_cnt && ppkt->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( ppkt->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present &&
         ppkt->type != FPD_CIM_BUFFER )) {
        return -EINVAL;
    }

    if( ppkt->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[ppkt->link_if];
    }

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        ppkt->actual_len = 0;
        return -ENOLINK;
    }

    switch( ppkt->type ) {
        case FPD_CIM_BUFFER:
            bytes_read = pdata_read(&plink->cim,
                                    ppkt->requested_len, ppkt->buf);
            break;
        case FPD_PRIORITY_BUFFER:
            bytes_read = pdata_read(&plink->priority,
                                    ppkt->requested_len, ppkt->buf);
            break;
        default:
            ppkt->actual_len = 0;
            return -EINVAL;
    }

    if( bytes_read < 0 ) {
        ppkt->actual_len = 0;
        result = bytes_read;
    }
    else {
        ppkt->actual_len = bytes_read;
        result = 0;
    }

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_switch_channel()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Switches the keyboard and mouse signals of a KVM user channel to the
 *      specified target port.
 *
 *  PARAMETERS:
 *
 *      pdev     -  Ptr to the device operated on
 *      pswitch  -  Ptr to a user-allocated buffer containing the switch
 *                  information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_switch_channel(fpd_device_t *pdev, FPD_switch_t *pswitch)
{
    fpd_linkif_t *plink;
    u32 reg;
    u32 link_base;
    u32 id;

    debugk("%s: dev %d linkIF %d port %d\n",
           __FUNCTION__, pdev->id, pswitch->link_if, pswitch->target_port);

    if(( pswitch->link_if >= pdev->info.link_if_cnt && pswitch->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pswitch->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    if( pswitch->target_port >= pdev->info.line_cnt ) {
        return -EINVAL;
    }

    switch( pswitch->protocol ) {
        case FPD_PROTOCOL_VM:
            break;
        case FPD_PROTOCOL_PARAGON:
            switch( pswitch->tx_parity ) {
                case FPD_PARAGON_PARITY_ODD:
                case FPD_PARAGON_PARITY_EVEN:
                    break;
                default:
                    return -EINVAL;
            }

            switch( pswitch->rx_parity ) {
                case FPD_PARAGON_PARITY_ODD:
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_IGNORE:
                    break;
                default:
                    return -EINVAL;
            }
            break;
        default:
            return -EINVAL;
    }

    /*
     * No other Link Interface is allowed to switch or disconnect until the
     * device's switch lock is obtained. This lock is per FPGA Protocol device
     * and not per Link Interface and it is necessary this way to verify 
     * other Link Interfaces are not connected to the same target port.
     */
    fpd_lock_acquire(&pdev->switch_lock);

    /* verify that no other Link Interface is connected to this channel */
    for( id = 0; id < pdev->info.link_if_cnt; id++ ) {
        plink = pdev->link[id];

        if( pswitch->link_if == plink->link_id )
            continue;

        if( plink->enabled && (plink->target_port == pswitch->target_port) ) {
            fpd_lock_release(&pdev->switch_lock);
            return -EINVAL;
        }
    }

    /* verify that the Background Link Interface is not connected to this channel */
    if( pswitch->link_if != FPD_LINK_IF_ID_BGND ) {
        if( pdev->info.bgnd_link_if_present ) {
            plink = pdev->bgnd_link;

            if( plink->enabled && (plink->target_port == pswitch->target_port) ) {
                fpd_lock_release(&pdev->switch_lock);
                return -EINVAL;
            }
        }
    }

    if( pswitch->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[pswitch->link_if];
    }
    link_base = plink->base;

    /* close data communication with user-app */
    atomic_set(&plink->close_comm, 1);

    /* try to wait for a clean disconnect */
    if(( pswitch->driver_txbuf_wait_time > 0 ) ||
       ( pswitch->fpga_txbuf_wait_time > 0 )) {
        /* adjust timeout value, if necessary */
        if( pswitch->driver_txbuf_wait_time < 100 ) {
            atomic_set(&pdev->driver_txbuf_wait_time, 100);
        }
        else if( pswitch->driver_txbuf_wait_time > 60000 ) {
            atomic_set(&pdev->driver_txbuf_wait_time, 60000);
        }
        else {
            atomic_set(&pdev->driver_txbuf_wait_time, pswitch->driver_txbuf_wait_time);
        }

        if( pswitch->fpga_txbuf_wait_time < 100 ) {
            atomic_set(&pdev->fpga_txbuf_wait_time, 100);
        }
        else if( pswitch->fpga_txbuf_wait_time > 60000 ) {
            atomic_set(&pdev->fpga_txbuf_wait_time, 60000);
        }
        else {
            atomic_set(&pdev->fpga_txbuf_wait_time, pswitch->fpga_txbuf_wait_time);
        }

        if( pswitch->link_if != FPD_LINK_IF_ID_BGND ) {
            pdata_disconnect_check(&plink->priority);
        }
        pdata_disconnect_check(&plink->cim);
    }

    /* reset Link Interface */
    fpd_link_reset(pdev, plink);

    /* program the new line number */
    reg = FPD_READ(pdev->remap, link_base + LIF_LINK_CONTROL);
    reg &= ~LIFCR_LINE_TO_IF;
    reg |= (pswitch->target_port << 15);

    /* program protocol type */
    if( pswitch->protocol == FPD_PROTOCOL_VM ) {
        /* VM */
        reg &= ~LIFCR_PROTOCOL;
    }
    else {
        /* Paragon */
        reg |= LIFCR_PROTOCOL;

        /* program parity */
        if( pswitch->tx_parity == FPD_PARAGON_PARITY_EVEN ) {
            /* even parity */
            reg &= ~LIFCR_TX_PAR;
        }
        else {
            /* odd parity */
            reg |= LIFCR_TX_PAR;
        }

        if( pswitch->rx_parity != FPD_PARAGON_PARITY_IGNORE ) {
            /* do not ignore rx parity */
            reg &= ~LIFCR_IGNORE_RX_PAR;

            if( pswitch->rx_parity == FPD_PARAGON_PARITY_EVEN ) {
                /* even parity */
                reg &= ~LIFCR_RX_PAR;
            }
            else {
                /* odd parity */
                reg |= LIFCR_RX_PAR;
            }
        }
        else {
            /* ignore rx parity */
            reg |= LIFCR_IGNORE_RX_PAR;
        }
    }

    /* take Link IF out of reset */
    reg &= ~LIFCR_LINK_IF_RST;

    /* write the new value to the register */
    FPD_WRITE(pdev->remap, link_base + LIF_LINK_CONTROL, reg);

    plink->target_port = pswitch->target_port;
    plink->protocol = pswitch->protocol;
    plink->tx_parity = pswitch->tx_parity;
    plink->rx_parity = pswitch->rx_parity;
    plink->enabled = 1;

    /* setup Link Interface interrupts */
    fpd_link_setup_interrupt(pdev, plink);

    /* open data communication with user-app */
    atomic_set(&plink->close_comm, 0);

    /* release lock */
    fpd_lock_release(&pdev->switch_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_disconnect_channel()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Disconnects the keyboard and mouse data line by disabling the Link
 *      Interface.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pdisc  -  Ptr to a user-allocated buffer containing the disconnect
 *                information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_disconnect_channel(fpd_device_t *pdev, FPD_disc_t *pdisc)
{
    fpd_linkif_t *plink;

    debugk("%s: dev %d linkIF %d\n", __FUNCTION__, pdev->id, pdisc->link_if);

    if(( pdisc->link_if >= pdev->info.link_if_cnt && pdisc->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pdisc->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    if( pdisc->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[pdisc->link_if];
    }

    /*
     * No other Link Interface is allowed to switch or disconnect until the
     * device's switch lock is obtained. This lock is per FPGA Protocol device
     * and not per Link Interface and it is necessary this way to verify 
     * other Link Interfaces are not connected to the same target port.
     */
    fpd_lock_acquire(&pdev->switch_lock);

    if( pdisc->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[pdisc->link_if];
    }

    /* close data communication with user-app */
    atomic_set(&plink->close_comm, 1);

    /* try to wait for a clean disconnect */
    if(( pdisc->driver_txbuf_wait_time > 0 ) ||
       ( pdisc->fpga_txbuf_wait_time > 0 )) {
        /* adjust timeout value, if necessary */
        if( pdisc->driver_txbuf_wait_time < 100 ) {
            atomic_set(&pdev->driver_txbuf_wait_time, 100);
        }
        else if( pdisc->driver_txbuf_wait_time > 60000 ) {
            atomic_set(&pdev->driver_txbuf_wait_time, 60000);
        }
        else {
            atomic_set(&pdev->driver_txbuf_wait_time, pdisc->driver_txbuf_wait_time);
        }

        if( pdisc->fpga_txbuf_wait_time < 100 ) {
            atomic_set(&pdev->fpga_txbuf_wait_time, 100);
        }
        else if( pdisc->fpga_txbuf_wait_time > 60000 ) {
            atomic_set(&pdev->fpga_txbuf_wait_time, 60000);
        }
        else {
            atomic_set(&pdev->fpga_txbuf_wait_time, pdisc->fpga_txbuf_wait_time);
        }

        if( pdisc->link_if != FPD_LINK_IF_ID_BGND ) {
            pdata_disconnect_check(&plink->priority);
        }
        pdata_disconnect_check(&plink->cim);
    }

    /* reset Link Interface */
    fpd_link_reset(pdev, plink);

    /* release lock */
    fpd_lock_release(&pdev->switch_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_switch_info()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current switch information (Link IF to Line IF).
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pinfo  -  Ptr to a user-allocated buffer to store the switch info
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_switch_info(fpd_device_t *pdev, FPD_switchinfo_t *pinfo)
{
    fpd_linkif_t *plink;
    u32 link;

    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    memset(pinfo, 0, sizeof(FPD_switchinfo_t));
    pinfo->link_if_cnt = pdev->info.link_if_cnt;
    pinfo->bgnd_link_if_present = pdev->info.bgnd_link_if_present;

    /* get lock */
    fpd_lock_acquire(&pdev->switch_lock);

    for( link = 0; link < pdev->info.link_if_cnt; link++ ) {
        plink = pdev->link[link];

        pinfo->link[link].link_if = plink->link_id;
        pinfo->link[link].target_port = plink->target_port;
        pinfo->link[link].protocol = plink->protocol;
        pinfo->link[link].tx_parity = plink->tx_parity;
        pinfo->link[link].rx_parity = plink->rx_parity;
        pinfo->link[link].enabled = plink->enabled;
    }

    if( pdev->info.bgnd_link_if_present ) {
        plink = pdev->bgnd_link;
        pinfo->bgnd_link.target_port = plink->target_port;
        pinfo->bgnd_link.protocol = plink->protocol;
        pinfo->bgnd_link.tx_parity = plink->tx_parity;
        pinfo->bgnd_link.rx_parity = plink->rx_parity;
        pinfo->bgnd_link.enabled = plink->enabled;
    }

    /* release lock */
    fpd_lock_release(&pdev->switch_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_line_status()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current line status of all the target ports.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pstat  -  Ptr to a user-allocated buffer to store the line status
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_line_status(fpd_device_t *pdev, FPD_linestat_t *pstat)
{
    int i;

    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    fpd_lock_acquire(&pdev->linestat_lock);

    /* read from the driver's current status */
    for( i = 0; i < 2; i++ ) {
        pstat->status_change[i] = pdev->line_status_chg[i];
    }

    /* read directly from the FPGA register */
    pstat->line_status[0] = FPD_READ(pdev->remap, FPD_LINE_STATUS0);
    pstat->line_status[1] = FPD_READ(pdev->remap, FPD_LINE_STATUS1);
    pstat->line_status[2] = FPD_READ(pdev->remap, FPD_LINE_STATUS2);
    pstat->line_status[3] = FPD_READ(pdev->remap, FPD_LINE_STATUS3);

    fpd_lock_release(&pdev->linestat_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_error()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the error condition detected by the FPGA Protocol Device.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      perr   -  Ptr to a user-allocated buffer to store the error condition
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_error(fpd_device_t *pdev, FPD_error_t *perr)
{
    fpd_linkif_t *plink;

    debugk("%s: dev %d type %d LinkIF %d\n", __FUNCTION__, pdev->id,
           perr->type, perr->link_if);

    switch( perr->type ) {
        case FPD_ERR_TYPE_PCI:
            fpd_lock_acquire(&pdev->pci_error);
            perr->error = fpd_lock_read_resource(&pdev->pci_error);
            fpd_lock_write_resource(&pdev->pci_error, 0);
            fpd_lock_release(&pdev->pci_error);
            break;
        case FPD_ERR_TYPE_LINK_IF:
            if(( perr->link_if >= pdev->info.link_if_cnt && perr->link_if != FPD_LINK_IF_ID_BGND ) ||
               ( perr->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
                return -EINVAL;
            }

            if( perr->link_if == FPD_LINK_IF_ID_BGND ) {
                plink = pdev->bgnd_link;
            }
            else {
                plink = pdev->link[perr->link_if];
            }

            fpd_lock_acquire(&plink->error);
            perr->error = fpd_lock_read_resource(&plink->error);
            fpd_lock_write_resource(&plink->error, 0);
            fpd_lock_release(&plink->error);
            break;
        default:
            return -EINVAL;
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_statistics()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current statistics stored in the FPGA Protocol Device.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pstats -  Ptr to a user-allocated buffer to store the statistics.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_statistics(fpd_device_t *pdev, FPD_stats_t *pstats)
{
    fpd_linkif_t *plink;
    u32 link_base, reg;

    debugk("%s: dev %d LinkIF %d\n", __FUNCTION__, pdev->id, pstats->link_if);

    if(( pstats->link_if >= pdev->info.link_if_cnt && pstats->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pstats->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    if( pstats->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[pstats->link_if];
    }
    link_base = plink->base;

    reg = FPD_READ(pdev->remap, link_base + LIF_RX_STATISTICS1);
    pstats->rx.buf_full_cnt = (reg & 0xff000000) >> 24;
    pstats->rx.protocol_err_cnt = (reg & 0x00ff0000) >> 16;
    pstats->rx.crc_err_cnt = (reg & 0x0000ff00) >> 8;
    pstats->rx.inv_pkt_cnt = (reg & 0x000000ff);

    reg = FPD_READ(pdev->remap, link_base + LIF_RX_STATISTICS2);
    pstats->rx.parity_err_cnt = (reg & 0xff000000) >> 24;
    pstats->rx.seq_err_cnt = (reg & 0x00ff0000) >> 16;
    pstats->rx.timeout_cnt = (reg & 0x0000ff00) >> 8;
    pstats->rx.noise_cnt = (reg & 0x000000ff);

    reg = FPD_READ(pdev->remap, link_base + LIF_TX_STATISTICS);
    pstats->tx.max_retry = (reg & 0x0f000000) >> 24;
    pstats->tx.buf_full_cnt = (reg & 0x00ff0000) >> 16;
    pstats->tx.protocol_err_cnt = (reg & 0x0000ff00) >> 8;
    pstats->tx.crc_err_cnt = (reg & 0x000000ff);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_run_diagnostics()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Executes diagnostics.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pdiag  -  Ptr to a user-allocated buffer that contains information
 *                required to execute the diagnostics
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_run_diagnostics(fpd_device_t *pdev, FPD_diag_t *pdiag)
{
    FPD_data_t *pdata = &pdiag->pdata;

    debugk("%s: dev %d code %d\n", __FUNCTION__, pdev->id, pdiag->code);

    /* verify parameters */
    if(( pdata->link_if >= pdev->info.link_if_cnt && pdata->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pdata->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        pdiag->result = FPD_DIAG_ERROR_INVALID_LINKIF;
        return 0;
    }

    switch( pdata->type ) {
        case FPD_CIM_BUFFER:
            break;
        case FPD_PRIORITY_BUFFER:
            if( pdata->link_if == FPD_LINK_IF_ID_BGND ) {
                pdiag->result = FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
                return 0;
            }
            break;
        default:
            pdiag->result = FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
            return 0;
    }

    if( pdata->requested_len > FPD_MAX_PACKET_SIZE ) {
        pdiag->result = FPD_DIAG_ERROR_INVALID_PKT_LENGTH;
        return 0;
    }

    switch( pdiag->diag_code ) {
        case FPD_DIAG_TEST_CRC_ERROR:
            pdiag->result = fpd_diag_test_crc_error(pdev, pdata);
            break;

        case FPD_DIAG_TEST_PROTOCOL_ERROR:
            pdiag->result = fpd_diag_test_protocol_error(pdev, pdata);
            break;

        case FPD_DIAG_TEST_INVALID_PKT:
            pdiag->result = fpd_diag_test_invalid_pkt(pdev, pdata);
            break;

        case FPD_DIAG_TEST_BUFFER_FULL:
            /* param1 contains the direction */
            if( pdiag->param1 == 0 ) {
                /* receiver-side */
                pdiag->result = fpd_diag_test_buffer_full_rxside(pdev, pdiag->param2);
            }
            else {
                /* transmitter-side */
                pdiag->result = fpd_diag_test_buffer_full_txside(pdev, pdata);
            }
            break;

        default:
            pdiag->result = FPD_DIAG_ERROR_INVALID_TEST;
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_config_host()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Configures the FPGA Protocol device's Host Buffer.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      pcfg  -  Ptr to a user-allocated buffer containing the host buffer  
 *               configuration.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_config_host(fpd_device_t *pdev, FPD_hostcfg_t *pcfg)
{
    fpd_linkif_t *plink;
    int result;

    debugk("%s: dev %d linkIF %d chan %d dmasize %d len %d\n", __FUNCTION__, pdev->id,
           pcfg->link_if, pcfg->host_chan, pcfg->dma_size, pcfg->function);

    if( pcfg->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    plink = pdev->link[pcfg->link_if];

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        return -ENOLINK;
    }

    switch( pcfg->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            return -EINVAL;
    }

    result = fpd_host_config(&plink->host, pcfg->host_chan,
                             (u8)pcfg->function, pcfg->dma_size,
                             pcfg->rx_pkt_size_notification);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_config_host()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Gets the the FPGA Protocol device's Host Buffer configuration.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      pcfg  -  Ptr to a user-allocated buffer to store the host buffer  
 *               configuration.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_config_host(fpd_device_t *pdev, FPD_hostcfg_t *pcfg)
{
    fpd_linkif_t *plink;
    fpd_host_t *phost;
    fpd_host_chan_t *pchan;

    debugk("%s: dev %d linkIF %d chan %d dmasize %d len %d\n", __FUNCTION__,
           pdev->id, pcfg->link_if, pcfg->host_chan);

    if( pcfg->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    switch( pcfg->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            return -EINVAL;
    }

    plink = pdev->link[pcfg->link_if];
    phost = &plink->host;
    pchan = phost->channel[pcfg->host_chan];
    pcfg->function = pchan->function;
    pcfg->dma_size = pchan->dma_size;
    pcfg->rx_pkt_size_notification = pchan->rx_pkt_size_notification;

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_host_buffer()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Requests for a Host Buffer memory for doing DMA Transmit transactions.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      pcfg  -  Ptr to a user-allocated buffer to store the host buffer  
 *               configuration.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_host_buffer(fpd_device_t *pdev, FPD_hostbuf_t *pbuf)
{
    fpd_linkif_t *plink;

    debugk("%s: dev %d linkIF %d chan %d\n", __FUNCTION__,
           pdev->id, pbuf->link_if, pbuf->host_chan);

    if( pbuf->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    switch( pbuf->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            return -EINVAL;
    }

    plink = pdev->link[pbuf->link_if];

    return( fpd_host_get_txbuffer(&plink->host, pbuf ) );
}

/******************************************************************************
 *  ROUTINE:        fpd_send_host_data()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Sends data directly to the FPGA Protocol device's Host Buffer (without
 *      using the USB function emulation) to be transmitted to the Far-end
 *      (i.e., CIM device).
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      ppkt  -  Ptr to a user-allocated buffer containing the link interface, 
 *               routing info, and data.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_send_host_data(fpd_device_t *pdev, FPD_hostdata_t *ppkt)
{
    fpd_linkif_t *plink;
    int bytes_written;
    int result;

    debugk("%s: dev %d linkIF %d chan %d dmasize %d len %d\n", __FUNCTION__, pdev->id,
           ppkt->link_if, ppkt->host_chan, ppkt->pkt_size, ppkt->requested_len);

    if( ppkt->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    plink = pdev->link[ppkt->link_if];

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        ppkt->actual_len = 0;
        return -ENOLINK;
    }

    if( ppkt->requested_len > FPD_HOST_BUFFER_SIZE ) {
        ppkt->actual_len = 0;
        return -EINVAL;
    }

    switch( ppkt->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            ppkt->actual_len = 0;
            return -EINVAL;
    }

    bytes_written = fpd_host_write(&plink->host, ppkt->host_chan,
                                   ppkt->requested_len, ppkt->info.tx.buf_id);

    if( bytes_written < 0 ) {
        ppkt->actual_len = 0;
        result = bytes_written;
    }
    else {
        ppkt->actual_len = bytes_written;
        result = 0;
    }

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_receive_host_data()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Receives data directly to the FPGA Protocol device's Host Buffer
 *      (without using the USB function emulation) to be transmitted to the
 *      Far-end (i.e., CIM device).
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      ppkt  -  Ptr to a user-allocated buffer containing the link interface, 
 *               routing info, and data.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_receive_host_data(fpd_device_t *pdev, FPD_hostdata_t *ppkt)
{
    fpd_linkif_t *plink;
    int result;

    debugk("%s: dev %d linkIF %d chan %d dmasize %d len %d\n", __FUNCTION__, pdev->id,
           ppkt->link_if, ppkt->host_chan, ppkt->pkt_size, ppkt->requested_len);

    if( ppkt->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    plink = pdev->link[ppkt->link_if];

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        ppkt->actual_len = 0;
        return -ENOLINK;
    }

    if( ppkt->requested_len > FPD_HOST_BUFFER_SIZE ) {
        ppkt->actual_len = 0;
        return -EINVAL;
    }

    switch( ppkt->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            ppkt->actual_len = 0;
            return -EINVAL;
    }

    result = fpd_host_read(&plink->host, ppkt);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_release_rx_host_data()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Releases the RX buffer used for the Host Buffer so that the driver
 *      will be able to use it again at a later time.  This is used in
 *      for Transparent Mode.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pclear -  Ptr to a user-allocated buffer containing the link interface, 
 *                host buffer channel, and RX buffer state information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_release_rx_host_data(fpd_device_t *pdev, FPD_host_rxclear_t *pclear)
{
    fpd_linkif_t *plink;

    if( pclear->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    plink = pdev->link[pclear->link_if];

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        return -ENOLINK;
    }

    switch( pclear->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            return -EINVAL;
    }

    return( fpd_host_read_complete(&plink->host, pclear->host_chan, &pclear->info) );
}

/******************************************************************************
 *  ROUTINE:        fpd_set_dma_burst_size()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Sets the burst size for the PCI bus transactions mastered by the DMA
 *      engine.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pburst -  Ptr to a user-allocated buffer containing the dma burst size
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_set_dma_burst_size(fpd_device_t *pdev, FPD_dma_burst_size_t *pburst)
{
    u32 gcr;
    u32 bitset;

    switch( pburst->size ) {
        case 8:
            bitset = 0;
            break;
        case 16:
            bitset = 1;
            break;
        case 32:
            bitset = 2;
            break;
        case 64:
            bitset = 3;
            break;
        default:
            return -EINVAL;
    }

    gcr = FPD_READ(pdev->remap, FPD_GLOBAL_CONTROL);
    gcr &= ~GCR_DMA_BURST_SIZE;
    gcr |= bitset;
    FPD_WRITE(pdev->remap, FPD_GLOBAL_CONTROL, gcr);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_dma_burst_size()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the burst size for the PCI bus transactions mastered by the
 *      DMA engine.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pburst -  Ptr to a user-allocated buffer for storing the dma burst size
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_dma_burst_size(fpd_device_t *pdev, FPD_dma_burst_size_t *pburst)
{
    u32 gcr;
    u32 bitset;

    gcr = FPD_READ(pdev->remap, FPD_GLOBAL_CONTROL);
    bitset = gcr & GCR_DMA_BURST_SIZE;

    switch( bitset ) {
        case 0:
            pburst->size = 8;
            break;
        case 1:
            pburst->size = 16;
            break;
        case 2:
            pburst->size = 32;
            break;
        default:
            pburst->size = 64;
            break;
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_set_tx_timeout()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Sets the transmit timeout value before the FPGA notifies the driver
 *      of its failure to transmit a packet.  This timeout is global to all
 *      the FPGA buffers for all Link Interfaces.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pval   -  Ptr to a user-allocated buffer containing the timeout value
 *                in seconds.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_set_tx_timeout(fpd_device_t *pdev, FPD_txtimeout_t *pval)
{
    u32 gcr;

    if( pval->timeout == 0 || pval->timeout > 15 ) {
        return -EINVAL;
    }

    gcr = FPD_READ(pdev->remap, FPD_GLOBAL_CONTROL);
    gcr &= ~GCR_TX_TIMEOUT;
    gcr |= (pval->timeout << 7);
    FPD_WRITE(pdev->remap, FPD_GLOBAL_CONTROL, gcr);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_tx_timeout()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current transmit timeout value that is used by each of
 *      the 3 transmit buffers to time a successful transmission of a packet.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pval   -  Ptr to a user-allocated buffer for storing the transmit
 *                timeout in seconds.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_tx_timeout(fpd_device_t *pdev, FPD_txtimeout_t *pval)
{
    u32 gcr;

    gcr = FPD_READ(pdev->remap, FPD_GLOBAL_CONTROL);
    pval->timeout = (gcr & GCR_TX_TIMEOUT) >> 7;

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_invalidate_cmd_table()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Invalidates the Command Lookup Table for the specified Link Interface.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pinv   -  Ptr to a user-allocated buffer containing the Link 
 *                Interface.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_invalidate_all_cmds(fpd_device_t *pdev, FPD_cmdinv_t *pinv)
{
    debugk("%s: dev %d Link %d\n", __FUNCTION__, pdev->id, pinv->link_if);

    if(( pinv->link_if >= pdev->info.link_if_cnt && pinv->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pinv->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    return( fpd_paragon_invalidate_table(pdev, pinv->link_if) );
}

/******************************************************************************
 *  ROUTINE:        fpd_write_entry_in_cmd_table()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Programs a single entry in the Command Lookup Table for the specified
 *      Link Interface.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pcmd   -  Ptr to a user-allocated buffer containing the information
 *                to be programmed.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_write_entry_in_cmd_table(fpd_device_t *pdev, FPD_cmdprog_t *pcmd)
{
    debugk("%s: dev %d Link %d CMD %02x\n", __FUNCTION__, pdev->id,
           pcmd->link_if, pcmd->cmd);

    if(( pcmd->link_if >= pdev->info.link_if_cnt && pcmd->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pcmd->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present &&
         pcmd->dest != FPD_CIM_BUFFER )) {
        return -EINVAL;
    }

    return( fpd_paragon_write_cmd(pdev, pcmd) );
}

/******************************************************************************
 *  ROUTINE:        fpd_read_entry_in_cmd_table()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the properties of a single entry in the Command Lookup Table.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pcmd   -  Ptr to a user-allocated buffer that will be used to store
 *                the information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_read_entry_in_cmd_table(fpd_device_t *pdev, FPD_cmdprog_t *pcmd)
{
    debugk("%s: dev %d Link %d CMD %02x\n", __FUNCTION__, pdev->id,
           pcmd->link_if, pcmd->cmd);

    if(( pcmd->link_if >= pdev->info.link_if_cnt && pcmd->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pcmd->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    return( fpd_paragon_read_cmd(pdev, pcmd) );
}

/******************************************************************************
 *  ROUTINE:        fpd_get_rx_invalid_cmd()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the invalid commands received by the Link Interface.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pcmd   -  Ptr to a user-allocated buffer that will be used to store
 *                the information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_rx_invalid_cmd(fpd_device_t *pdev, FPD_rxinvcmd_t *pcmd)
{
    debugk("%s: dev %d Link %d\n", __FUNCTION__, pdev->id, pcmd->link_if);

    if(( pcmd->link_if >= pdev->info.link_if_cnt && pcmd->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pcmd->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    return( fpd_paragon_get_invalid_cmd(pdev, pcmd) );
}

/******************************************************************************
 *  ROUTINE:        fpd_get_echo_rsp()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the Echo Response received by the Link Interface.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pcmd   -  Ptr to a user-allocated buffer that will be used to store
 *                the information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_echo_rsp(fpd_device_t *pdev, FPD_echorsp_t *pcmd)
{
    debugk("%s: dev %d Link %d\n", __FUNCTION__, pdev->id, pcmd->link_if);

    if(( pcmd->link_if >= pdev->info.link_if_cnt && pcmd->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pcmd->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    return( fpd_paragon_get_echo_rsp(pdev, pcmd) );
}

/******************************************************************************
 *  ROUTINE:        fpd_enter_cim_update()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Place the Link IF in CIM FW Update mode.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pcimu  -  Ptr to a user-allocated buffer that contains information
 *                required to do the action.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_enter_cim_update(fpd_device_t *pdev, FPD_cimupdate_t *pcimu)
{
    debugk("%s: dev %d Link %d\n", __FUNCTION__, pdev->id, pcimu->link_if);

    if(( pcimu->link_if >= pdev->info.link_if_cnt && pcimu->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pcimu->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    return( fpd_paragon_enter_cim_update(pdev, pcimu->link_if, pcimu->mode_chg_option) );
}

/******************************************************************************
 *  ROUTINE:        fpd_exit_cim_update()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Take the Link IF out of CIM FW Update mode.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pcimu  -  Ptr to a user-allocated buffer that contains information
 *                required to do the action.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_exit_cim_update(fpd_device_t *pdev, FPD_cimupdate_t *pcimu)
{
    debugk("%s: dev %d Link %d\n", __FUNCTION__, pdev->id, pcimu->link_if);

    if(( pcimu->link_if >= pdev->info.link_if_cnt && pcimu->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pcimu->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    return( fpd_paragon_exit_cim_update(pdev, pcimu->link_if) );
}

/******************************************************************************
 *  ROUTINE:        fpd_set_link0_mode()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Sets the behavior of Link Interface 0 when using VM Protocol.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pmode  -  Ptr to a user-allocated buffer that contains the specified
 *                mode.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_set_link0_mode(fpd_device_t *pdev, FPD_link0mode_t *pmode)
{
    u32 gcr;

    debugk("%s: dev %d mode %x\n", __FUNCTION__, pdev->id, pmode->mode);

    gcr = FPD_READ(pdev->remap, FPD_GLOBAL_CONTROL);
    switch( pmode->mode ) {
        case FPD_LINK0_MODE_MASTER:
            gcr &= ~GCR_LINK0_MODE;
            break;
        case FPD_LINK0_MODE_SLAVE:
            gcr |= GCR_LINK0_MODE;
            break;
        default:
            return -EINVAL;
    }
    FPD_WRITE(pdev->remap, FPD_GLOBAL_CONTROL, gcr);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_link0_mode()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the behavior of Link Interface 0 when using VM Protocol.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pmode  -  Ptr to a user-allocated buffer to store the current Link 0
 *                mode.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_link0_mode(fpd_device_t *pdev, FPD_link0mode_t *pmode)
{
    u32 gcr;

    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    gcr = FPD_READ(pdev->remap, FPD_GLOBAL_CONTROL);
    if( gcr & GCR_LINK0_MODE ) {
        pmode->mode = FPD_LINK0_MODE_SLAVE;
    }
    else {
        pmode->mode = FPD_LINK0_MODE_MASTER;
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_switch_remote_video()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Remote video switching.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pvs    -  Ptr to a user-allocated buffer containing the remote video
 *                switching information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_switch_remote_video(fpd_device_t *pdev, FPD_video_switch_t *pvs)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    return( fpd_vs_select(&pdev->vs, pvs) );
}

/******************************************************************************
 *  ROUTINE:        fpd_disable_remote_video()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Remote video disconnection
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pvs    -  Ptr to a user-allocated buffer containing the remote video
 *                disconnection information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_disable_remote_video(fpd_device_t *pdev, FPD_video_switch_disc_t *pvs)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    return( fpd_vs_disable(&pdev->vs, pvs->link_if, pvs->kvm_chan, 0) );
}

/******************************************************************************
 *  ROUTINE:        fpd_switch_local_video()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Local video switching.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pvs    -  Ptr to a user-allocated buffer containing the local video
 *                switching information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_switch_local_video(fpd_device_t *pdev, FPD_local_video_switch_t *pvs)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    return( fpd_vs_local_select(&pdev->vs, pvs) );
}

/******************************************************************************
 *  ROUTINE:        fpd_write_vschip_reg()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Write to a register of the Video Switch chip.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pchip  -  Ptr to a user-allocated buffer that contains information on
 *                chip id, register offset, value to write.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_write_vschip_reg(fpd_device_t *pdev, FPD_vschip_write_t *pchip)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    return( fpd_vs_write_chip(&pdev->vs, pchip->chip_id, pchip->reg, pchip->data) );
}

/******************************************************************************
 *  ROUTINE:        fpd_get_vschip_contents()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves all the register contents of the Video Switch chip.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pchip  -  Ptr to a user-allocated buffer that will be used to store
 *                the information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_vschip_contents(fpd_device_t *pdev, FPD_vschip_contents_t *pchip)
{
    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    return( fpd_vs_get_vschip_contents(&pdev->vs, pchip) );
}

/******************************************************************************
 *  ROUTINE:        fpd_reset_link_interface()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Resets the specified Link Interface.  Flushes CIM, Priority and Host
 *      buffers in the FPGA and Driver.
 *
 *  PARAMETERS:
 *
 *      pdev    -  Ptr to the device operated on
 *      preset  -  Ptr to a user-allocated buffer containing the reset
 *                 information.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_reset_link_interface(fpd_device_t *pdev, FPD_reset_link_t *preset)
{
    fpd_linkif_t *plink;
    u32 reg;
    u32 link_base;
    u8  link_is_enabled;

    debugk("%s: dev %d linkIF %d\n", __FUNCTION__, pdev->id, preset->link_if);

    if(( preset->link_if >= pdev->info.link_if_cnt && preset->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( preset->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present )) {
        return -EINVAL;
    }

    /*
     * No other Link Interface is allowed to switch or disconnect until the
     * device's switch lock is obtained. This lock is per FPGA Protocol device
     * and not per Link Interface and it is necessary this way to verify 
     * other Link Interfaces are not connected to the same target port.
     * In this case, it is necessary so the state of the Link Interface will
     * not change in the middle of the reset.
     */
    fpd_lock_acquire(&pdev->switch_lock);

    if( preset->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[preset->link_if];
    }
    link_base = plink->base;

    /* save the current state of the Link Interface */
    link_is_enabled = plink->enabled;

    if( link_is_enabled ) {
        /* close data communication with user-app */
        atomic_set(&plink->close_comm, 1);
    }

    /* reset Link Interface */
    fpd_link_reset(pdev, plink);

    /* take Link IF out of reset */
    reg = FPD_READ(pdev->remap, link_base + LIF_LINK_CONTROL);
    reg &= ~LIFCR_LINK_IF_RST;
    FPD_WRITE(pdev->remap, link_base + LIF_LINK_CONTROL, reg);

    if( link_is_enabled ) {
        plink->enabled = 1;

        /* setup Link Interface interrupts */
        fpd_link_setup_interrupt(pdev, plink);

        /* open data communication with user-app */
        atomic_set(&plink->close_comm, 0);
    }

    /* release lock */
    fpd_lock_release(&pdev->switch_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_poll_line_status()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Poll the current line status of all the target ports.  This will 
 *      return only when there is a change in the line status or a signal
 *      has been received.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pstat  -  Ptr to a user-allocated buffer to store the line status
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_poll_line_status(fpd_device_t *pdev, FPD_linestat_t *pstat)
{
    int i;

    debugk("%s: dev %d\n", __FUNCTION__, pdev->id);

    /*
     * Wait for interrupt
     */
    wait_event_interruptible(pdev->cim_detect_queue,
                             (atomic_read(&pdev->cim_detect_event) ||
                              signal_pending(current)));

    /* clear the Link IF request */
    atomic_set(&pdev->cim_detect_event, 0);

    FPD_DEBUG(2, "%s%x: awakened\n", __FUNCTION__, pdev->id);

    if (signal_pending(current)) {
        FPD_DEBUG(2, "%s: Got a signal\n", __FUNCTION__);
        return -EINTR;
    }

    fpd_lock_acquire(&pdev->linestat_lock);

    /* read from the driver's current status */
    for( i = 0; i < 2; i++ ) {
        pstat->status_change[i] = pdev->line_status_chg[i];
    }

    /* read directly from the FPGA register */
    pstat->line_status[0] = FPD_READ(pdev->remap, FPD_LINE_STATUS0);
    pstat->line_status[1] = FPD_READ(pdev->remap, FPD_LINE_STATUS1);
    pstat->line_status[2] = FPD_READ(pdev->remap, FPD_LINE_STATUS2);
    pstat->line_status[3] = FPD_READ(pdev->remap, FPD_LINE_STATUS3);

    fpd_lock_release(&pdev->linestat_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_get_buffer_info()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current CIM/Priority Buffer information.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pbinfo -  Ptr to a user-allocated buffer to store the info.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_buffer_info(fpd_device_t *pdev, FPD_buffer_t *pbinfo)
{
    fpd_linkif_t *plink;
    fpd_pdata_t *pcim;

    /* verify parameters */
    if(( pbinfo->link_if >= pdev->info.link_if_cnt && pbinfo->link_if != FPD_LINK_IF_ID_BGND ) ||
       ( pbinfo->link_if == FPD_LINK_IF_ID_BGND && !pdev->info.bgnd_link_if_present &&
         pbinfo->type != FPD_CIM_BUFFER )) {
        return -EINVAL;
    }

    switch( pbinfo->type ) {
        case FPD_CIM_BUFFER:
        case FPD_PRIORITY_BUFFER:
            break;
        default:
            return -EINVAL;
    }

    if( pbinfo->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pdev->bgnd_link;
    }
    else {
        plink = pdev->link[pbinfo->link_if];
    }

    if( pbinfo->type == FPD_CIM_BUFFER ) {
        pcim = &plink->cim;
    }
    else {
        pcim = &plink->priority;
    }

    pbinfo->txbuf_start = (u32)pcim->txbuf;
    pbinfo->txbuf_end = (u32)pcim->txbuf_end;
    pbinfo->txbuf_rdptr = (u32)pcim->txbuf_rdptr;
    pbinfo->txbuf_wrptr = (u32)pcim->txbuf_wrptr;
    pbinfo->txbuf_bytes_left = pcim->txbuf_bytes_left;
    pbinfo->rxbuf_start = (u32)pcim->rxbuf;
    pbinfo->rxbuf_end = (u32)pcim->rxbuf_end;
    pbinfo->rxbuf_rdptr = (u32)pcim->rxbuf_rdptr;
    pbinfo->rxbuf_wrptr = (u32)pcim->rxbuf_wrptr;
    pbinfo->rxbuf_bytes_unused = pcim->rxbuf_bytes_unused;
    pbinfo->rxbuf_no_space = atomic_read(&pcim->rxbuf_no_space);

    return 0;
}


/******************************************************************************
 *  ROUTINE:        fpd_get_host_rxbdtbl()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current Host Buffer's Receive Buffer Descriptor Table.
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pbdtbl -  Ptr to a user-allocated buffer to store the info.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_host_rxbdtbl(fpd_device_t *pdev, FPD_hostrxbdtbl_t *pbdtbl)
{
    fpd_linkif_t *plink;

    debugk("%s: dev %d linkIF %d HostChan %d BufID %d\n", __FUNCTION__,
           pdev->id, pbdtbl->link_if, pbdtbl->host_chan, pbdtbl->buf_index);

    if( pbdtbl->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    switch( pbdtbl->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            return -EINVAL;
    }

    plink = pdev->link[pbdtbl->link_if];

    return( fpd_host_get_rxbdtbl(&plink->host, pbdtbl->host_chan, pbdtbl) );
}

/******************************************************************************
 *  ROUTINE:        fpd_get_host_txbuf_info()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current Host Buffer's Transmit Buffer information
 *
 *  PARAMETERS:
 *
 *      pdev   -  Ptr to the device operated on
 *      pbinfo -  Ptr to a user-allocated buffer to store the info.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_get_host_txbuf_info(fpd_device_t *pdev, FPD_host_txbuf_info_t *pinfo)
{
    fpd_linkif_t *plink;

    debugk("%s: dev %d linkIF %d HostChan %d\n", __FUNCTION__, pdev->id,
           pinfo->link_if, pinfo->host_chan);

    if( pinfo->link_if >= pdev->info.link_if_cnt ) {
        return -EINVAL;
    }

    switch( pinfo->host_chan ) {
        case 0:
        case 1:
            break;
        default:
            return -EINVAL;
    }

    plink = pdev->link[pinfo->link_if];

    return( fpd_host_get_txbuf_info(&plink->host, pinfo->host_chan, pinfo) );
}

/******************************************************************************
 *  ROUTINE:        fpd_driver_ioctl()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Translates ioctl calls to calls to the Driver functions.
 *
 *  PARAMETERS:
 *
 *      pdev  -  Ptr to the device operated on
 *      cmd   -  IO control command
 *      arg   -  Argument pointer
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_driver_ioctl(fpd_device_t *pdev, unsigned int cmd, unsigned long arg)
{
    FPD_msg_t fpd_msg;
    int	size = _IOC_SIZE( cmd );
    int rc = 0;

    debugk("%s: ioctl %08x\n", __FUNCTION__, cmd);

    /* verify the structure of the command */
    if(size > sizeof(fpd_msg))
        return - ENOTTY;

    /* Copy the data from the user space */
    if ( _IOC_DIR(cmd) & _IOC_WRITE ) {
        debugk("%s: Copy from user space %08X %08X %d\n", __FUNCTION__,
               (int)&fpd_msg, (int)arg, size );
        copy_from_user( &fpd_msg, (void *)arg, size );
    }

    /* dispatch the call */
    switch (cmd) {
        case FPD_GET_INFO:
            rc = fpd_get_info(pdev, &fpd_msg.info);
            break;

        case FPD_GET_PCI_CFG:
            rc = fpd_get_pci_cfg(pdev, &fpd_msg.pcicfg);
            break;

        case FPD_READ_REG:
            rc = fpd_read_reg(pdev, &fpd_msg.rwreg);
            break;

        case FPD_WRITE_REG:
            rc = fpd_write_reg(pdev, &fpd_msg.rwreg);
            break;

        case FPD_CHECK_EVENTS:
            rc = fpd_check_events(pdev, &fpd_msg.event);
            break;

        case FPD_SEND_PROTOCOL_DATA:
            rc = fpd_send_protocol_data(pdev, &fpd_msg.pdata);
            break;

        case FPD_RECEIVE_PROTOCOL_DATA:
            rc = fpd_receive_protocol_data(pdev, &fpd_msg.pdata);
            break;

        case FPD_SWITCH_CHANNEL:
            rc = fpd_switch_channel(pdev, &fpd_msg.conn);
            break;

        case FPD_DISCONNECT_CHANNEL:
            rc = fpd_disconnect_channel(pdev, &fpd_msg.disc);
            break;

        case FPD_GET_SWITCH_INFO:
            rc = fpd_get_switch_info(pdev, &fpd_msg.switch_info);
            break;

        case FPD_GET_LINE_STATUS:
            rc = fpd_get_line_status(pdev, &fpd_msg.linestat);
            break;

        case FPD_GET_ERROR:
            rc = fpd_get_error(pdev, &fpd_msg.error);
            break;

        case FPD_GET_STATISTICS:
            rc = fpd_get_statistics(pdev, &fpd_msg.stats);
            break;

        case FPD_RUN_DIAGNOSTICS:
            rc = fpd_run_diagnostics(pdev, &fpd_msg.diag);
            break;

        case FPD_SET_HOST_BUFFER_CFG:
            rc = fpd_config_host(pdev, &fpd_msg.host_cfg);
            break;

        case FPD_GET_HOST_BUFFER_CFG:
            rc = fpd_get_config_host(pdev, &fpd_msg.host_cfg);
            break;

        case FPD_GET_HOST_BUFFER:
            rc = fpd_get_host_buffer(pdev, &fpd_msg.host_buf);
            break;

        case FPD_SEND_HOST_DATA:
            rc = fpd_send_host_data(pdev, &fpd_msg.hdata);
            break;

        case FPD_RECEIVE_HOST_DATA:
            rc = fpd_receive_host_data(pdev, &fpd_msg.hdata);
            break;

        case FPD_RELEASE_RX_HOST_DATA:
            rc = fpd_release_rx_host_data(pdev, &fpd_msg.host_rxclear);
            break;

        case FPD_SET_DMA_BURST_SIZE:
            rc = fpd_set_dma_burst_size(pdev, &fpd_msg.dma_burst);
            break;

        case FPD_GET_DMA_BURST_SIZE:
            rc = fpd_get_dma_burst_size(pdev, &fpd_msg.dma_burst);
            break;

        case FPD_SET_TX_TIMEOUT:
            rc = fpd_set_tx_timeout(pdev, &fpd_msg.tx_timeout);
            break;

        case FPD_GET_TX_TIMEOUT:
            rc = fpd_get_tx_timeout(pdev, &fpd_msg.tx_timeout);
            break;

        case FPD_INVALIDATE_ALL_CMDS:
            rc = fpd_invalidate_all_cmds(pdev, &fpd_msg.invalidate_cmd);
            break;

        case FPD_WRITE_ENTRY_CMD_TABLE:
            rc = fpd_write_entry_in_cmd_table(pdev, &fpd_msg.paragon_cmd);
            break;

        case FPD_READ_ENTRY_CMD_TABLE:
            rc = fpd_read_entry_in_cmd_table(pdev, &fpd_msg.paragon_cmd);
            break;

        case FPD_GET_RX_INVALID_CMD:
            rc = fpd_get_rx_invalid_cmd(pdev, &fpd_msg.invalid_cmd);
            break;

        case FPD_GET_ECHO_RSP:
            rc = fpd_get_echo_rsp(pdev, &fpd_msg.echo_rsp);
            break;

        case FPD_ENTER_CIM_UPDATE_MODE:
            rc = fpd_enter_cim_update(pdev, &fpd_msg.cim_update);
            break;

        case FPD_EXIT_CIM_UPDATE_MODE:
            rc = fpd_exit_cim_update(pdev, &fpd_msg.cim_update);
            break;

        case FPD_SET_LINK0_MODE:
            rc = fpd_set_link0_mode(pdev, &fpd_msg.link0mode);
            break;

        case FPD_GET_LINK0_MODE:
            rc = fpd_get_link0_mode(pdev, &fpd_msg.link0mode);
            break;

        case FPD_SWITCH_REMOTE_VIDEO:
            rc = fpd_switch_remote_video(pdev, &fpd_msg.video_switch);
            break;

        case FPD_DISABLE_REMOTE_VIDEO:
            rc = fpd_disable_remote_video(pdev, &fpd_msg.video_switch_disc);
            break;

        case FPD_SWITCH_LOCAL_VIDEO:
            rc = fpd_switch_local_video(pdev, &fpd_msg.local_video_switch);
            break;

        case FPD_WRITE_VSCHIP_REG:
            rc = fpd_write_vschip_reg(pdev, &fpd_msg.vschip_write);
            break;

        case FPD_READ_ALL_VSCHIP_REG:
            rc = fpd_get_vschip_contents(pdev, &fpd_msg.vschip_contents);
            break;

        case FPD_RESET_LINK_INTERFACE:
            rc = fpd_reset_link_interface(pdev, &fpd_msg.reset_link);
            break;

        case FPD_POLL_LINE_STATUS:
            rc = fpd_poll_line_status(pdev, &fpd_msg.linestat);
            break;

        case FPD_TRACE_ON:
            fpd_trace_on = 1;
            break;

        case FPD_TRACE_OFF:
            fpd_trace_on = 0;
            break;

        case FPD_SET_LOG_LVL:
            printk("SETTING LOG LEVEL to %d\n", fpd_msg.set_loglvl.level);
            fpd_debug_level = fpd_msg.set_loglvl.level;
            break;

        case FPD_GET_LOG_LVL:
            fpd_msg.get_loglvl.level = fpd_debug_level;
            break;

        case FPD_GET_BUFFER_INFO:
            rc = fpd_get_buffer_info(pdev, &fpd_msg.buffer);
            break;

        case FPD_GET_HOST_RXBDTBL:
            rc = fpd_get_host_rxbdtbl(pdev, &fpd_msg.host_rxbdtbl);
            break;

        case FPD_GET_HOST_TXBUF_INFO:
            rc = fpd_get_host_txbuf_info(pdev, &fpd_msg.host_txinfo);
            break;

        case FPD_DRIVER_RESET:
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
            while(MOD_IN_USE) {
                MOD_DEC_USE_COUNT;
            }
            MOD_INC_USE_COUNT;
#endif
            break;

        default:
            return -ENOTTY;	
    }

    /* copy the data back to the user space */
    if (rc == 0 && (_IOC_DIR(cmd) & _IOC_READ)) {
        debugk("%s: Copy to user space %08X %08X %d\n", __FUNCTION__,
               (int)&fpd_msg, (int)arg, size );
        copy_to_user( (void *)arg, &fpd_msg, size );
    }

    return rc;
}
