/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Simulation of FPGA Protocol CIM/Priority Data
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#ifdef SIMULATE_FPGA_CIM_BUFFER

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_reg.h"
#include "fpd_ioctl.h"
#include "fpd_thread.h"
#include "debug.h"

#ifndef MIN
#define MIN(a,b)                               ((a < b)?a:b)
#endif


/******************************************************************************
 *  ROUTINE:        fpd_simulate_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the simulation Data interface
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *      parg    Pointer to data to be stored in private member for later
 *              access 
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
fpd_simulate_init( fpd_simulate_buf_t *psim, u32 link_if, u32 btype, void *parg )
{
    /* allocate buffer */
    psim->buf = (u8 *)kmalloc(FPD_SIMULATED_BUFSIZE, GFP_KERNEL);
    if( !psim->buf ) {
        FPD_ERROR("Failed to allocate Simulated buffer\n");
        return -ENOMEM;
    }

    memset(psim->buf, 0, FPD_SIMULATED_BUFSIZE);
    psim->link_if          = link_if;
    psim->btype            = btype;
    psim->buf_rdptr        = psim->buf;
    psim->buf_wrptr        = psim->buf;
    psim->buf_end          = (u8 *)(psim->buf + FPD_SIMULATED_BUFSIZE);
    psim->buf_bytes_left   = FPD_SIMULATED_BUFSIZE;
    fpd_lock_init(&psim->lock, 0);

    psim->private = parg;

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_simulate_cleanup()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleans up the simulation Data interface module.
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *
 *  RETURNS:
 *
 *     	0       Cleanup is successful.
 *     -1       Cleanup failed.
 *
 *****************************************************************************/
int
fpd_simulate_cleanup( fpd_simulate_buf_t *psim )
{
    kfree(psim->buf);
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_simulate_read_GISR()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current status of the buffer that simulates the FPGA
 *      Global Interrupt Status Register.
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *
 *  RETURNS:
 *
 *     	Status
 *
 *****************************************************************************/
u32
fpd_simulate_read_GISR( fpd_simulate_buf_t *psim )
{
    u32 gisr = 0;

    if( psim->buf_bytes_left != FPD_SIMULATED_BUFSIZE ) {
        switch( psim->link_if ) {
            case 0:
                gisr = GISR_LINK_IF_0;
                break;
            case 1:
                gisr = GISR_LINK_IF_1;
                break;
            case 2:
                gisr = GISR_LINK_IF_2;
                break;
            case 3:
                gisr = GISR_LINK_IF_3;
                break;
            case 4:
                gisr = GISR_LINK_IF_4;
                break;
            case 5:
                gisr = GISR_BG_LINK_IF;
                break;
            default:
                break;
        }
    }

    return gisr;
}

/******************************************************************************
 *  ROUTINE:        fpd_simulate_read_LIFISR()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current status of the buffer that simulates the FPGA
 *      Link Interface Interrupt Status Register.
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *
 *  RETURNS:
 *
 *     	Status
 *
 *****************************************************************************/
u32
fpd_simulate_read_LIFISR( fpd_simulate_buf_t *psim )
{
    u32 lifisr = 0;

    if( psim->buf_bytes_left != FPD_SIMULATED_BUFSIZE ) {
        switch( psim->btype ) {
            case FPD_CIM_BUFFER:
                lifisr = LIFISR_RX_CIM_BUF;
                break;
            case FPD_PRIORITY_BUFFER:
                lifisr = LIFISR_RX_PRI_BUF;
                break;
            default:
                break;
        }
    }

    return lifisr;
}

/******************************************************************************
 *  ROUTINE:        fpd_simulate_read_status()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the current status of the buffer that simulates the CIM
 *      Buffer Status Register.
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *
 *  RETURNS:
 *
 *     	Status
 *
 *****************************************************************************/
u32
fpd_simulate_read_status( fpd_simulate_buf_t *psim )
{
    if( psim->buf_bytes_left != FPD_SIMULATED_BUFSIZE ) {
        return( LIFCBSR_RX_PKT_AVAIL | LIFCBSR_RX_SOP );
    }
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_simulate_write()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Store data to the simulated fpga data buffer that pdata_receive will
 *      read from.
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *      len     Data length.
 *      appbuf  Data buffer.
 *
 *  RETURNS:
 *
 *     	0       Succeeded.
 *     -ENOSPC  Run out of space in driver's RX Buffer.
 *     -EFAULT  Failed to copy data from user space memory.
 *
 *****************************************************************************/
int
fpd_simulate_write( fpd_simulate_buf_t *psim, int len, u8 *appbuf, u32 do_notify )
{
    fpd_device_t *pfpd = (fpd_device_t *)psim->private;
    u32 addr_mask;
    u32 remainder;
    u32 size;
    u8 *pbuf = appbuf;
    u8 *orig_wrptr = psim->buf_wrptr;
    u32 orig_bytes_left = psim->buf_bytes_left;
    int bytes_wrote = 0;
    int round = 1;

    /**********************************************/
    /* packet shall be stored in 4-byte alignment */
    /**********************************************/

    /* get lock */
    fpd_lock_acquire(&psim->lock);

    /* verify that write will occur on a 4-byte alignment */
    if( (u32)psim->buf_wrptr & 0x03 ) {
        FPD_ERROR("Simulated Buffer Write Ptr Alignment error!\n");

        /* Damn it! Adjust it then */
        addr_mask = (u32)psim->buf_wrptr & 0x03;
        switch( addr_mask ) {
            case 1:
                psim->buf_wrptr += 3;
                psim->buf_bytes_left -= 3;
                break;
            case 2:
                psim->buf_wrptr += 2;
                psim->buf_bytes_left -= 2;
                break;
            case 3:
                psim->buf_wrptr += 1;
                psim->buf_bytes_left -= 1;
                break;
            default:
                /* yipee! it is aligned */
                break;
        }
    }

    for( ; len > 0; len -= size, pbuf += size, round++ ) {
        FPD_DDEBUG(pfpd->id, 3,
                   "Simulated Write wrptr=%p, rdptr=%p, start=%p, end=%p\n",
                   psim->buf_wrptr, psim->buf_rdptr,
                   psim->buf, psim->buf_end);


        FPD_DDEBUG(pfpd->id, 3,
                   "Simulated Write Len %d Avail %d Wrote %d Round %d\n",
                   len, psim->buf_bytes_left, bytes_wrote, round);

        /* handle condition: no space left to store data */
        if( (len + sizeof(u32)) > psim->buf_bytes_left ) {
            FPD_ERROR("No space for Simulated Data Write!\n");
            fpd_lock_release(&psim->lock);
            return -ENOSPC;
        }

        /* prepend the packet length */
        if( round == 1 ) {
            *(u32 *)psim->buf_wrptr = len & 0x3FF;
            psim->buf_wrptr += sizeof(u32);
            psim->buf_bytes_left -= sizeof(u32);
            if( (u32)psim->buf_wrptr >= (u32)psim->buf_end ) {
                /* reached end, reset to start of buf */
                psim->buf_wrptr = psim->buf;
            }
        }

        if( (u32)psim->buf_wrptr >= (u32)psim->buf_rdptr ) {
            FPD_DDEBUG(pfpd->id, 3, "Simulated Write wrptr >= rdptr\n");
            /* get remaining size from the wrptr to the end of the buffer */
            remainder = (u32)psim->buf_end - (u32)psim->buf_wrptr;
        }
        else {
            FPD_DDEBUG(pfpd->id, 3, "Simulated Write wrptr < rdptr\n");
            /* available space start from wrptr and ends at rdptr */
            remainder = (u32)psim->buf_rdptr - (u32)psim->buf_wrptr;
        }

        /* determine the length of the copy for this round */
        size = MIN(len, remainder);
        FPD_DDEBUG(pfpd->id, 3, "Simulated Write size %d (LEN %d, REM %d)\n",
                   size, len, remainder);

        if( copy_from_user((void *)psim->buf_wrptr, (const void *)pbuf, size) != 0 )
        {
            FPD_ERROR("Simulated Write copy_from_user failed\n");

            /* 
             * Do not allow partial writes!
             * Revert back to original location of wrptr
             */
            psim->buf_wrptr = orig_wrptr;
            psim->buf_bytes_left = orig_bytes_left;

            fpd_lock_release(&psim->lock);
            return -EFAULT;
        }

        /* move wrptr */
        psim->buf_wrptr += size;
        psim->buf_bytes_left -= size;
        if( (u32)psim->buf_wrptr >= (u32)psim->buf_end ) {
            /* reached end, reset to start of buf */
            psim->buf_wrptr = psim->buf;
        }

        bytes_wrote += size;
    }

    FPD_DDEBUG(pfpd->id, 3,
               "Simulated Write completed Avail %d Wrote %d TXbuf wrptr %p\n",
               psim->buf_bytes_left, bytes_wrote, psim->buf_wrptr);

    /* make sure that write ends on a 4-byte alignment */
    addr_mask = (u32)psim->buf_wrptr & 0x03;
    switch( addr_mask ) {
        case 1:
            psim->buf_wrptr += 3;
            psim->buf_bytes_left -= 3;
            break;
        case 2:
            psim->buf_wrptr += 2;
            psim->buf_bytes_left -= 2;
            break;
        case 3:
            psim->buf_wrptr += 1;
            psim->buf_bytes_left -= 1;
            break;
        default:
            /* yipee! it is aligned */
            break;
    }

    if( (u32)psim->buf_wrptr >= (u32)psim->buf_end ) {
        /* reached end, reset to start of buf */
        psim->buf_wrptr = psim->buf;
    }

    FPD_DDEBUG(pfpd->id, 3,
               "Simulated Write fix wrptr Avail %d Wrote %d TXbuf wrptr %p\n",
               psim->buf_bytes_left, bytes_wrote, psim->buf_wrptr);

    /* release lock */
    fpd_lock_release(&psim->lock);

    /* notify FPGA Protocol Thread of new packets */
    if(( bytes_wrote > 0 ) && do_notify) {
        fpd_thread_wakeup(pfpd);
    }

    return bytes_wrote;
}

/******************************************************************************
 *  ROUTINE:        fpd_simulate_read()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Read data from the simulated FPGA CIM/Priority Data buffer 32-bit
 *      word at a time.
 *
 *  PARAMETERS:
 *
 *      psim    Pointer to the simulated buffer data structure.
 *
 *  RETURNS:
 *
 *     	> 0     Number of bytes successfully read.
 *     	0       No data is available at this time.
 *     -EFAULT  Failed to read
 *
 *  NOTE:
 *
 *      Read pointer is always 4-byte aligned.
 *      Write pointer has to make sure that it ends in a 4-byte alignment.
 *
 *****************************************************************************/
u32
fpd_simulate_read( fpd_simulate_buf_t *psim )
{
    fpd_device_t *pfpd = (fpd_device_t *)psim->private;
    u32 addr_mask;
    int bytes_unread;
    int data = 0;

    /* get lock */
    fpd_lock_acquire(&psim->lock);

    /******************************************/
    /* packets are stored in 4-byte alignment */
    /******************************************/

    /* verify that the read will occur on a 4-byte alignment */
    if( (u32)psim->buf_rdptr & 0x03 ) {
        FPD_ERROR("Simulated Buffer Read Ptr Alignment error!\n");

        /* Damn it! Adjust it then */
        addr_mask = (u32)psim->buf_rdptr & 0x03;
        switch( addr_mask ) {
            case 1:
                psim->buf_rdptr += 3;
                psim->buf_bytes_left += 3;
                break;
            case 2:
                psim->buf_rdptr += 2;
                psim->buf_bytes_left += 2;
                break;
            case 3:
                psim->buf_rdptr += 1;
                psim->buf_bytes_left += 1;
                break;
            default:
                /* yipee! it is aligned */
                break;
        }
    }

    /* determine # of bytes not yet read */
    bytes_unread = FPD_SIMULATED_BUFSIZE - psim->buf_bytes_left;

    /* handle condition: no data in RX buffer */
    if( bytes_unread <= 0 ) {
        FPD_DDEBUG(pfpd->id, 3, "Simulated Read has no data available\n");
        goto simulated_read_error;
    }

    FPD_DDEBUG(pfpd->id, 3,
               "Simulated Read rdptr=%p, wrptr=%p, start=%p, end=%p, unread %d\n",
               psim->buf_rdptr, psim->buf_wrptr,
               psim->buf, psim->buf_end, bytes_unread);

    data = *(u32 *)psim->buf_rdptr;

    /* move rdptr */
    psim->buf_rdptr += sizeof(u32);
    psim->buf_bytes_left += sizeof(u32);
    if( (u32)psim->buf_rdptr >= (u32)psim->buf_end ) {
        /* reached end, reset to start of buf */
        psim->buf_rdptr = psim->buf;
    }


    FPD_DDEBUG(pfpd->id, 3,
               "Simulated Read completed Unread %d rdptr %p\n",
               bytes_unread - sizeof(u32), psim->buf_rdptr);

 simulated_read_error:
    /* release lock */
    fpd_lock_release(&psim->lock);

    return data;
}
#endif /* SIMULATE_FPGA_CIM_BUFFER */
