/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device TX CIM Data Thread
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>	/* For (un)lock_kernel */
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <linux/wrapper.h> /* wrapper for compatibility with future versions */
#endif
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/semaphore.h>

#include "fpd.h"
#include "fpd_pdata.h"
#include "debug.h"

#undef	DEBUG

#ifdef	DEBUG
# define debugk(fmt,args...)	printk(fmt ,##args)
#else
# define debugk(fmt,args...)
#endif

/* ==================================================================== */

/* ==================================================================== */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void txcim_thread_setup(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    /* lock the kernel */
    lock_kernel();

    /* release insmod's memory map, use the kernel one's */
    exit_mm(current);

    /* set session leader and process group to init process */
    current->session = 1;
    current->pgrp = 1;

    /* set name of this process (max 15 chars + 0 !) */
    switch( plink->link_id ) {
        case 0:
            strcpy(current->comm, "txcim0");
            break;
        case 1:
            strcpy(current->comm, "txcim1");
            break;
        case 2:
            strcpy(current->comm, "txcim2");
            break;
        case 3:
            strcpy(current->comm, "txcim3");
            break;
        case 4:
            strcpy(current->comm, "txcim4");
            break;
        case FPD_LINK_IF_ID_BGND:
            strcpy(current->comm, "BGND-txcim");
            break;
        default:
            strcpy(current->comm, "CRAPPY-txcim");
            break;
    }

    /* disconnect from the fs usage */
    exit_files(current);
    exit_fs(current);

    /* fill in thread structure */
    plink->tx_thread = current;

    /* set signal mask to what we want to respond */
    siginitsetinv(&current->blocked, sigmask(SIGKILL));

    /* let others run */
    unlock_kernel();

    /* tell the creator that we are ready and let him run */
    up(&plink->tx_sem);
}
#endif

static void txcim_thread_leave(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    /* lock the kernel, the exit will unlock it */
    plink->tx_thread = NULL;

    /* notify the kill_thread() routine that we are terminating. */
    up(&plink->tx_sem);
}

static int txcim_thread(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;
    fpd_pdata_t *pcim;

    debugk("%s: called\n", __FUNCTION__);

    init_waitqueue_head(&plink->tx_queue);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    /* Setup the thread environment */
    txcim_thread_setup(dev_arg);
#else
    lock_kernel();
    switch( plink->link_id ) {
        case 0:
            daemonize("txcim0");
            break;
        case 1:
            daemonize("txcim1");
            break;
        case 2:
            daemonize("txcim2");
            break;
        case 3:
            daemonize("txcim3");
            break;
        case 4:
            daemonize("txcim4");
            break;
        case FPD_LINK_IF_ID_BGND:
            daemonize("BGND-txcim");
            break;
        default:
            daemonize("CRAPPY-txcim");
            break;
    }
    unlock_kernel();

    /* fill in thread structure */
    plink->tx_thread = current;

    allow_signal(SIGKILL);

    /* tell the creator that we are ready and let him run */
    up(&plink->tx_sem);
#endif

    FPD_INFO("TX CIM %x Thread started\n", plink->link_id);

    while (1) {
        /*
         * Wait for interrupt
         */
        wait_event_interruptible(plink->tx_queue,
                                 (atomic_read(&plink->req_service) ||
                                  signal_pending(current)));

        debugk("%s%x: awakened\n", __FUNCTION__, plink->link_id);

        if (signal_pending(current)) {
            debugk("%s: Got a signal\n", __FUNCTION__);
            break;
        }

        /* clear the Link IF request */
        atomic_set(&plink->req_service, 0);

        /* check request for priority buffer */
        pcim = &plink->priority;
        if( atomic_read(&pcim->req_tx) ) {
            FPD_DEBUG(2, "Req TX Priority Buffer\n");

            /* clear request */
            atomic_set(&pcim->req_tx, 0);

            /* service request */
            pdata_transmit(pcim);
        }

        /* check request for cim buffer */
        pcim = &plink->cim;
        if( atomic_read(&pcim->req_tx) ) {
            FPD_DEBUG(2, "Req TX CIM Buffer\n");

            /* clear request */
            atomic_set(&pcim->req_tx, 0);

            /* service request */
            pdata_transmit(pcim);
        }
    }

    txcim_thread_leave(plink);

    return 0;
}

static void txcim_thread_start(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    init_MUTEX_LOCKED(&plink->tx_sem);

    kernel_thread(txcim_thread, dev_arg, 0);

    /* Wait till it has reached the setup_thread routine */
    down(&plink->tx_sem);
}

static void txcim_thread_stop(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;

    debugk("%s: called\n", __FUNCTION__);

    if (!plink->tx_thread) {
        printk ("Can't terminate non-existant txcim thread\n");
        return;
    }

    init_MUTEX_LOCKED(&plink->tx_sem);

    kill_proc(plink->tx_thread->pid, SIGKILL, 1);

    /* block till thread terminated */
    down(&plink->tx_sem);
}

/* ==================================================================== */

int txcim_thread_init(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;

    debugk ("%s: called\n", __FUNCTION__);
    atomic_set(&plink->req_service, 0);
    txcim_thread_start(dev_arg);

    return 0;
}

void txcim_thread_cleanup(void *dev_arg)
{
    debugk ("%s: called\n", __FUNCTION__);
    txcim_thread_stop(dev_arg);
}

void txcim_thread_wakeup(void *dev_arg)
{
    fpd_linkif_t *plink = (fpd_linkif_t *)dev_arg;

    debugk ("%s: called\n", __FUNCTION__);
    atomic_set(&plink->req_service, 1);
    wake_up_interruptible(&plink->tx_queue);
}
