/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Driver I/O control definitions
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#ifndef _FPD_IOCTL_H_
#define	_FPD_IOCTL_H_

/*-----------*
 * Constants *
 *-----------*/

#define	FPD_MAGIC                                       0xCE

/* maximum Link Interface */
#define FPD_LINK_IF_CNT                                 5

/* maximum host buffer */
#define FPD_HOST_CNT                                    2

/* maximum KX2.0 CIM/Priority packet size */
#define FPD_MAX_PACKET_SIZE                             (512 - sizeof(int))

/* Background Link Interface ID */
#define FPD_LINK_IF_ID_BGND                             1010

/* Protocol types */
#define FPD_PROTOCOL_VM                                 0x00
#define FPD_PROTOCOL_PARAGON                            0x01

/* Paragon Parity types */
#define FPD_PARAGON_PARITY_EVEN                         0x00
#define FPD_PARAGON_PARITY_ODD                          0x01
#define FPD_PARAGON_PARITY_IGNORE                       0x02

/* maximum Paragon packet size */
#define FPD_MAX_PARAGON_PACKET_SIZE                     255

/* Data Buffer types */
#define FPD_CIM_BUFFER                                  0
#define FPD_PRIORITY_BUFFER                             1
#define FPD_HOST_BUFFER                                 2

/* Error types */
#define FPD_ERR_TYPE_PCI                                0x00000000
#define FPD_ERR_TYPE_LINK_IF                            0x00000001

/* PCI Error */
#define PCI_ERR_PARITY                                  0x00000100
#define PCI_ERR_SIG_TARGET_ABORT                        0x00000800
#define PCI_ERR_RCV_TARGET_ABORT                        0x00001000
#define PCI_ERR_RCV_MASTER_ABORT                        0x00002000
#define PCI_ERR_SIG_SYSTEM_ERROR                        0x00004000
#define PCI_ERR_DETECTED_PARITY                         0x00008000

/* Link Interface Error */
#define FPD_ERR_RX_DMA_NO_BUF_0                         0x00000001
#define FPD_ERR_RX_DMA_NO_BUF_1                         0x00000002
#define FPD_ERR_TX_HOST_TIMEOUT_0                       0x00000004
#define FPD_ERR_TX_HOST_TIMEOUT_1                       0x00000008
#define FPD_ERR_TX_CIM_TIMEOUT                          0x00000010
#define FPD_ERR_TX_PRI_TIMEOUT                          0x00000020
#define FPD_ERR_TX_CIM_WR_FULL_BUF                      0x00000040
#define FPD_ERR_TX_CIM_NO_EOP                           0x00000080
#define FPD_ERR_TX_CIM_SYNC                             0x00000100
#define FPD_ERR_RX_CIM_RD_EMPTY_BUF                     0x00000200
#define FPD_ERR_TX_PRI_WR_FULL_BUF                      0x00000400
#define FPD_ERR_TX_PRI_NO_EOP                           0x00000800
#define FPD_ERR_TX_PRI_SYNC                             0x00001000
#define FPD_ERR_RX_PRI_RD_EMPTY_BUF                     0x00002000
#define FPD_ERR_RX_INV_CMD                              0x00004000
#define FPD_ERR_UPDATE_INV_PKT                          0x00008000

/* Link 0 Mode Behavior - VM Protocol only */
#define FPD_LINK0_MODE_MASTER                           0
#define FPD_LINK0_MODE_SLAVE                            1

/* Diagnostics */
#define FPD_DIAG_TEST_CRC_ERROR                         0x00000001
#define FPD_DIAG_TEST_PROTOCOL_ERROR                    0x00000002
#define FPD_DIAG_TEST_INVALID_PKT                       0x00000003
#define FPD_DIAG_TEST_BUFFER_FULL                       0x00000004

#define FPD_DIAG_ERROR_NONE                             0x00000000
#define FPD_DIAG_ERROR_INVALID_PKT_LENGTH               0x00000001
#define FPD_DIAG_ERROR_INVALID_BUFFER_TYPE              0x00000002
#define FPD_DIAG_ERROR_INVALID_LINKIF                   0x00000003
#define FPD_DIAG_ERROR_INVALID_TEST                     0x00000004
#define FPD_DIAG_ERROR_MEM_ALLOC_FAILED                 0x00000005
#define FPD_DIAG_ERROR_USER_MEM_FAULT                   0x00000006
#define FPD_DIAG_ERROR_FPGA_BUFFER_NO_SPACE             0x00000007
#define FPD_DIAG_ERROR_TX_TIMEOUT_NOT_SET               0x00000008
#define FPD_DIAG_ERROR_INCORRECT_TX_BYTE_AVAIL          0x00000009
#define FPD_DIAG_ERROR_WRONG_TX_CRC_CNT                 0x0000000a
#define FPD_DIAG_ERROR_TX_CRC_CNT_NOT_CLEARED           0x0000000b
#define FPD_DIAG_ERROR_WRONG_MAX_RETRY                  0x0000000c
#define FPD_DIAG_ERROR_MAX_RETRY_NOT_CLEARED            0x0000000d
#define FPD_DIAG_ERROR_WRONG_TX_PROT_ERR_CNT            0x0000000e
#define FPD_DIAG_ERROR_TX_PROT_ERR_CNT_NOT_CLEARED      0x0000000f
#define FPD_DIAG_ERROR_WRONG_RX_TIMEOUT_CNT             0x00000010
#define FPD_DIAG_ERROR_RX_TIMEOUT_CNT_NOT_CLEARED       0x00000011
#define FPD_DIAG_ERROR_TX_BUF_NOT_SET                   0x00000012
#define FPD_DIAG_ERROR_TX_BYTE_AVAIL_NOT_512            0x00000013
#define FPD_DIAG_ERROR_WRONG_TX_BUF_FULL_CNT            0x00000014
#define FPD_DIAG_ERROR_TX_BUF_FULL_CNT_NOT_CLEARED      0x00000015

/* Host Buffer functionality */
#define FPD_HOST_FUNCTION_NONE                          0x00
#define FPD_HOST_FUNCTION_TRANSPARENT                   0x01
#define FPD_HOST_FUNCTION_VIRTUAL_MEDIA                 0x02

#define FPD_HOST_MAX_RX_BD                              256
#define FPD_HOST_MAX_RX_BD_TABLE_SIZE                   (8 * FPD_HOST_MAX_RX_BD)

/* Maximum logged RX invalid Paragon command */
#define FPD_MAX_INVALID_PARAGON_CMD                     15

/* Legacy CIM Update mode */
#define FPD_CHANGE_MODE_AFTER_PKT_SENT                  0
#define FPD_CHANGE_MODE_IMMEDIATELY                     1

/* Local Video Source */
#define FPD_LOCAL_VIDEO_SRC_DISABLE                     0
#define FPD_LOCAL_VIDEO_SRC_PARAGON_CIM                 1
#define FPD_LOCAL_VIDEO_SRC_VM_CIM                      2
#define FPD_LOCAL_VIDEO_SRC_GRAPHIC_ENGINE              3

/* Event Masks */
#define FPD_EVENT_MASK_CIM_DETECT                       0x00000001
#define FPD_EVENT_MASK_PCI_ERROR                        0x00000002
#define FPD_EVENT_MASK_BGND_LINK                        0x00000010
#define FPD_EVENT_MASK_LINK0                            0x00000020
#define FPD_EVENT_MASK_LINK1                            0x00000040
#define FPD_EVENT_MASK_LINK2                            0x00000080
#define FPD_EVENT_MASK_LINK3                            0x00000100
#define FPD_EVENT_MASK_LINK4                            0x00000200

/*------------*
 * Data Types *
 *------------*/

typedef struct
{
    u_int32_t  major;
    u_int32_t  minor;
    u_int32_t  patch;
} FPD_version_t;

typedef struct
{
    FPD_version_t               fpga_version;
    FPD_version_t               driver_version;
    FPD_version_t               protocol_version;
    u_int32_t                   link_if_cnt;
    u_int32_t                   bgnd_link_if_present;
    u_int32_t                   line_cnt;
    u_int32_t                   host_buffer_cnt;
    u_int32_t                   host_buffer_memsize;
    u_int32_t                   host_buffer_pagecnt;
    u_int32_t                   host_buffer_pagesize;
} FPD_info_t;

typedef struct
{
    u_int16_t  vendor_id;
    u_int16_t  device_id;
    u_int16_t  command;
    u_int16_t  status;
    u_int8_t   revision_id;
    u_int8_t   prog_if;
    u_int8_t   sub_class_code;
    u_int8_t   base_class_code;
    u_int8_t   cache_line_size;
    u_int8_t   latency_timer;
    u_int8_t   header_type;
    u_int8_t   bist;
    u_int32_t  bar0, bar1, bar2, bar3, bar4, bar5;
    u_int32_t  cardbus_cis_ptr;
    u_int16_t  subsystem_vendor_id;
    u_int16_t  subsystem_id;
    u_int32_t  rom_bar;
    u_int8_t   cap_ptr;
    u_int8_t   interrupt_line;
    u_int8_t   interrupt_pin;
    u_int8_t   min_grant;
    u_int8_t   max_latency;
} FPD_pcicfg_t;

typedef struct
{
    u_int32_t  reg;
    u_int32_t  value;
} FPD_rw_reg_t;

typedef struct
{
    struct {
        struct {
            u_int8_t  function;
            u_int8_t  has_data;
        } host[FPD_HOST_CNT];
        u_int8_t   rx_priority;
        u_int8_t   rx_cim;
        u_int8_t   error;
        u_int8_t   echo_rsp;
    } link_if[FPD_LINK_IF_CNT];

    struct {
        u_int8_t   rx_cim;
        u_int8_t   error;
        u_int8_t   echo_rsp;
    } bgnd_link_if;

    u_int8_t   pci_error;
    u_int8_t   cim_detect;
    u_int32_t  event_mask;
} FPD_event_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  type;
    int        requested_len;
    int        actual_len;
    union {
	u_int8_t  *buf;
	const u_int8_t  *buf_const;
    };
} FPD_data_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  target_port;
    u_int32_t  protocol;  /* VM or Paragon mode    */
    u_int32_t  tx_parity; /* Paragon-specific mode */
    u_int32_t  rx_parity; /* Paragon-specific mode */
    u_int32_t  driver_txbuf_wait_time; /* in milliseconds, 100ms granularity */
    u_int32_t  fpga_txbuf_wait_time;   /* in milliseconds, 100ms granularity */
} FPD_switch_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  driver_txbuf_wait_time; /* in milliseconds, 100ms granularity */
    u_int32_t  fpga_txbuf_wait_time;   /* in milliseconds, 100ms granularity */
} FPD_disc_t;

typedef struct
{
    u_int32_t  link_if_cnt; /* excluding the background Link */
    u_int32_t  bgnd_link_if_present;

    struct {
        u_int32_t  link_if;
        u_int32_t  target_port;
        u_int8_t   protocol;  /* VM or Paragon mode    */
        u_int8_t   tx_parity; /* Paragon-specific mode */
        u_int8_t   rx_parity; /* Paragon-specific mode */
        u_int8_t   enabled;
    } link[FPD_LINK_IF_CNT];

    struct {
        u_int32_t  target_port;
        u_int8_t   protocol;  /* VM or Paragon mode    */
        u_int8_t   tx_parity; /* Paragon-specific mode */
        u_int8_t   rx_parity; /* Paragon-specific mode */
        u_int8_t   enabled;
    } bgnd_link;
} FPD_switchinfo_t;

typedef struct
{
    /* status_change[0] = channels 0-31, status_change[1] = channels 32-63*/
    u_int32_t  status_change[2];

    /* line_status[0] = chan 0-15, line_status[1] = chan 16-31
     * line_status[2] = chan 32-47, line_status[3] = chan 48-63
     */
    u_int32_t  line_status[4];
} FPD_linestat_t;

typedef struct
{
    u_int32_t  type;
    u_int32_t  link_if;
    u_int32_t  error;
} FPD_error_t;

typedef struct
{
    u_int8_t   buf_full_cnt;
    u_int8_t   protocol_err_cnt;
    u_int8_t   crc_err_cnt;
    u_int8_t   inv_pkt_cnt;
    u_int8_t   parity_err_cnt;
    u_int8_t   seq_err_cnt;
    u_int8_t   timeout_cnt;
    u_int8_t   noise_cnt;
} FPD_rxstats_t;

typedef struct
{
    u_int8_t   max_retry;
    u_int8_t   buf_full_cnt;
    u_int8_t   protocol_err_cnt;
    u_int8_t   crc_err_cnt;
} FPD_txstats_t;

typedef struct
{
    u_int32_t       link_if;
    FPD_rxstats_t   rx;
    FPD_txstats_t   tx;
} FPD_stats_t;

typedef struct
{
    u_int32_t  diag_code;
    u_int32_t  result;
    u_int32_t  param1;
    u_int32_t  param2;
    FPD_data_t pdata;
} FPD_diag_t;

typedef struct {
    u_int32_t  link_if;
    u_int32_t  host_chan;
    u_int32_t  dma_size;
    u_int32_t  function;
    u_int32_t  rx_pkt_size_notification;
} FPD_hostcfg_t;

typedef struct
{
    u_int32_t  buf_id;
    u_int32_t  buf_size;
} FPD_host_txparam_t;

typedef struct
{
    u_int32_t  buf_id;
    u_int32_t  buf_size;
    u_int32_t  offset;
    u_int16_t  short_pkt;
    u_int16_t  reached_page_end;
    u_int32_t  len;
    u_int16_t  startbd;
    u_int16_t  endbd;
} FPD_host_rxparam_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  host_chan; /* 0 or 1 */
    int        requested_len;
    int        actual_len;
    union txrxparam {
        FPD_host_txparam_t tx;
        FPD_host_rxparam_t rx;
    } info;
} FPD_hostdata_t;

typedef struct {
    u_int32_t  link_if;
    u_int32_t  host_chan;
    u_int32_t  buf_id;
    u_int32_t  buf_size;
} FPD_hostbuf_t;

typedef struct {
    u_int32_t  link_if;
    u_int32_t  host_chan; /* 0 or 1 */
    FPD_host_rxparam_t info;
} FPD_host_rxclear_t;

typedef struct
{
    u_int32_t  size;
} FPD_dma_burst_size_t;

typedef struct
{
    u_int8_t   timeout;
} FPD_txtimeout_t;

typedef struct {
    u_int32_t  link_if;
} FPD_cmdinv_t;

typedef struct {
    u_int32_t  link_if;
    u_int8_t   cmd;
    u_int8_t   byte_cnt;
    u_int8_t   valid;
    u_int8_t   in_ram;
    u_int8_t   chksum;
    u_int8_t   dest;
} FPD_cmdprog_t;

typedef struct {
    u_int32_t  link_if;
    u_int8_t   inv_cmd[FPD_MAX_INVALID_PARAGON_CMD];
    u_int8_t   n_entries;
} FPD_rxinvcmd_t;

typedef struct
{
    u_int32_t  link_if;
    u_int8_t   echo_rsp;
} FPD_echorsp_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  mode_chg_option;
} FPD_cimupdate_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  kvm_chan;        /* chan 0 - 3  */
    u_int32_t  target_port;     /* port 0 - 63 */
    u_int8_t   protocol;        /* 0 = VM, 1 = Paragon              */
    u_int8_t   hsync_pol;       /* PCIM Horizontal Sync Polarity    */
    u_int8_t   vsync_pol;       /* PCIM Vertical Sync Polarity      */
} FPD_video_switch_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  kvm_chan;        /* chan 0 - 3  */
} FPD_video_switch_disc_t;

typedef struct
{
    u_int32_t  local_video_src; /* Paragon CIM, VM-CIM, Z7, Disable */
    u_int32_t  link_if;         /* not necessary for Disable mode   */
    u_int32_t  target_port;     /* port 0 - 63 if connected to CIM  */
    u_int8_t   hsync_pol;       /* PCIM Horizontal Sync Polarity    */
    u_int8_t   vsync_pol;       /* PCIM Vertical Sync Polarity      */
} FPD_local_video_switch_t;

typedef struct
{
    u_int32_t  link_if;
} FPD_reset_link_t;

typedef struct
{
    u_int8_t   chip_id;
    u_int8_t   Ai;
    u_int8_t   Bi;
    u_int8_t   Ci;
    u_int8_t   Di;
    u_int8_t   enable_input_mux;
    u_int8_t   Ti;
    u_int8_t   test_mux;
    u_int8_t   sync_detectors;
    u_int8_t   Ax;
    u_int8_t   Bx;
    u_int8_t   Cx;
    u_int8_t   Dx;
    u_int8_t   Sx;
    u_int8_t   soc;
    u_int8_t   gain_input_mux;
} FPD_vschip_reg_t;

typedef struct {
    u_int32_t        dev;
    FPD_vschip_reg_t reg;
} FPD_vschip_contents_t;

typedef struct
{
    u_int8_t   chip_id;
    u_int8_t   reg;
    u_int8_t   data;
} FPD_vschip_write_t;

typedef struct
{
    u_int32_t  mode;  /* 0 = master, 1 = slave */
} FPD_link0mode_t;

typedef struct
{
    u_int32_t  level;
} FPD_loglvl_t;

typedef struct
{
    u_int32_t  link_if;
    u_int32_t  type;
    u_int32_t  txbuf_start;
    u_int32_t  txbuf_end;
    u_int32_t  txbuf_rdptr;
    u_int32_t  txbuf_wrptr;
    u_int32_t  txbuf_bytes_left;
    u_int32_t  rxbuf_start;
    u_int32_t  rxbuf_end;
    u_int32_t  rxbuf_rdptr;
    u_int32_t  rxbuf_wrptr;
    u_int32_t  rxbuf_bytes_unused;
    u_int32_t  rxbuf_no_space;
} FPD_buffer_t;

typedef struct {
    u_int32_t  link_if;
    u_int32_t  host_chan;
    u_int32_t  buf_index;
    int        overall_len;  
    u_int32_t  phys_addr;
    u_int16_t  state;
    u_int16_t  is_current_buffer;
    u_int32_t  bdnum;
    u_int16_t  startbd;
    u_int16_t  endbd;
    u_int8_t   bdtbl[FPD_HOST_MAX_RX_BD_TABLE_SIZE];
} FPD_hostrxbdtbl_t;

typedef struct {
    u_int32_t  link_if;
    u_int32_t  host_chan;
    u_int32_t  buf_cnt;
    struct txinfo {
        u_int16_t state;
        u_int16_t len;
    } info[32];
} FPD_host_txbuf_info_t;

typedef struct
{
    FPD_data_t                  pdata;
    u_int32_t                   do_notify;
} FPD_simulate_t;

typedef union
{
    FPD_info_t                  info;
    FPD_pcicfg_t                pcicfg;
    FPD_rw_reg_t                rwreg;
    FPD_event_t                 event;
    FPD_data_t                  pdata;
    FPD_switch_t                conn;
    FPD_disc_t                  disc;
    FPD_switchinfo_t            switch_info;
    FPD_linestat_t              linestat;
    FPD_error_t                 error;
    FPD_stats_t                 stats;
    FPD_diag_t                  diag;
    FPD_hostcfg_t               host_cfg;
    FPD_hostdata_t              hdata;
    FPD_hostbuf_t               host_buf;
    FPD_host_rxclear_t          host_rxclear;
    FPD_dma_burst_size_t        dma_burst;
    FPD_txtimeout_t             tx_timeout;
    FPD_cmdinv_t                invalidate_cmd;
    FPD_cmdprog_t               paragon_cmd;
    FPD_rxinvcmd_t              invalid_cmd;
    FPD_echorsp_t               echo_rsp;
    FPD_cimupdate_t             cim_update;
    FPD_link0mode_t             link0mode;
    FPD_video_switch_t          video_switch;
    FPD_video_switch_disc_t     video_switch_disc;
    FPD_local_video_switch_t    local_video_switch;
    FPD_vschip_write_t          vschip_write;
    FPD_vschip_contents_t       vschip_contents;
    FPD_reset_link_t            reset_link;
    FPD_loglvl_t                set_loglvl;
    FPD_loglvl_t                get_loglvl;
    FPD_buffer_t                buffer;
    FPD_hostrxbdtbl_t           host_rxbdtbl;
    FPD_host_txbuf_info_t       host_txinfo;
    FPD_simulate_t              simulate;
} FPD_msg_t;

/*------------------------*
 * IO Control Definitions *
 *------------------------*/

#define FPD_GET_INFO              _IOR (FPD_MAGIC,  1, FPD_info_t)
#define FPD_GET_PCI_CFG           _IOR (FPD_MAGIC,  2, FPD_pcicfg_t)
#define FPD_READ_REG              _IOWR(FPD_MAGIC,  3, FPD_rw_reg_t)
#define FPD_WRITE_REG             _IOW (FPD_MAGIC,  4, FPD_rw_reg_t)
#define FPD_CHECK_EVENTS          _IOR (FPD_MAGIC,  5, FPD_event_t)
#define FPD_SEND_PROTOCOL_DATA    _IOWR(FPD_MAGIC,  6, FPD_data_t)
#define FPD_RECEIVE_PROTOCOL_DATA _IOWR(FPD_MAGIC,  7, FPD_data_t)
#define FPD_SWITCH_CHANNEL        _IOW (FPD_MAGIC,  8, FPD_switch_t)
#define FPD_DISCONNECT_CHANNEL    _IOW (FPD_MAGIC,  9, FPD_disc_t)
#define FPD_GET_SWITCH_INFO       _IOR (FPD_MAGIC, 10, FPD_switchinfo_t)
#define FPD_GET_LINE_STATUS       _IOR (FPD_MAGIC, 11, FPD_linestat_t)
#define FPD_GET_ERROR             _IOWR(FPD_MAGIC, 12, FPD_error_t)
#define FPD_GET_STATISTICS        _IOWR(FPD_MAGIC, 13, FPD_stats_t)
#define FPD_RUN_DIAGNOSTICS       _IOWR(FPD_MAGIC, 14, FPD_diag_t)
#define FPD_SET_HOST_BUFFER_CFG   _IOW (FPD_MAGIC, 15, FPD_hostcfg_t)
#define FPD_GET_HOST_BUFFER_CFG   _IOWR(FPD_MAGIC, 16, FPD_hostcfg_t)
#define FPD_GET_HOST_BUFFER       _IOWR(FPD_MAGIC, 17, FPD_hostbuf_t)
#define FPD_SEND_HOST_DATA        _IOWR(FPD_MAGIC, 18, FPD_hostdata_t)
#define FPD_RECEIVE_HOST_DATA     _IOWR(FPD_MAGIC, 19, FPD_hostdata_t)
#define FPD_RELEASE_RX_HOST_DATA  _IOW (FPD_MAGIC, 20, FPD_host_rxclear_t)
#define FPD_SET_DMA_BURST_SIZE    _IOW (FPD_MAGIC, 21, FPD_dma_burst_size_t)
#define FPD_GET_DMA_BURST_SIZE    _IOR (FPD_MAGIC, 22, FPD_dma_burst_size_t)
#define FPD_SET_TX_TIMEOUT        _IOW (FPD_MAGIC, 23, FPD_txtimeout_t)
#define FPD_GET_TX_TIMEOUT        _IOR (FPD_MAGIC, 24, FPD_txtimeout_t)
#define FPD_INVALIDATE_ALL_CMDS   _IOW (FPD_MAGIC, 25, FPD_cmdinv_t)
#define FPD_WRITE_ENTRY_CMD_TABLE _IOW (FPD_MAGIC, 26, FPD_cmdprog_t)
#define FPD_READ_ENTRY_CMD_TABLE  _IOWR(FPD_MAGIC, 27, FPD_cmdprog_t)
#define FPD_GET_RX_INVALID_CMD    _IOWR(FPD_MAGIC, 28, FPD_rxinvcmd_t)
#define FPD_SET_LINK0_MODE        _IOW (FPD_MAGIC, 29, FPD_link0mode_t)
#define FPD_GET_LINK0_MODE        _IOR (FPD_MAGIC, 30, FPD_link0mode_t)
#define FPD_GET_ECHO_RSP          _IOWR(FPD_MAGIC, 31, FPD_echorsp_t)
#define FPD_ENTER_CIM_UPDATE_MODE _IOW (FPD_MAGIC, 32, FPD_cimupdate_t)
#define FPD_EXIT_CIM_UPDATE_MODE  _IOW (FPD_MAGIC, 33, FPD_cimupdate_t)
#define FPD_SWITCH_REMOTE_VIDEO   _IOW (FPD_MAGIC, 34, FPD_video_switch_t)
#define FPD_DISABLE_REMOTE_VIDEO  _IOW (FPD_MAGIC, 35, FPD_video_switch_disc_t)
#define FPD_SWITCH_LOCAL_VIDEO    _IOW (FPD_MAGIC, 36, FPD_local_video_switch_t)
#define FPD_WRITE_VSCHIP_REG      _IOW (FPD_MAGIC, 37, FPD_vschip_write_t)
#define FPD_READ_ALL_VSCHIP_REG   _IOWR(FPD_MAGIC, 38, FPD_vschip_contents_t)
#define FPD_RESET_LINK_INTERFACE  _IOW (FPD_MAGIC, 39, FPD_reset_link_t)
#define FPD_POLL_LINE_STATUS      _IOR (FPD_MAGIC, 40, FPD_linestat_t)

/* debugging tools */
#define FPD_TRACE_ON              _IO  (FPD_MAGIC, 50)
#define FPD_TRACE_OFF             _IO  (FPD_MAGIC, 51)
#define FPD_SET_LOG_LVL           _IOW (FPD_MAGIC, 52, FPD_loglvl_t)
#define FPD_GET_LOG_LVL           _IOR (FPD_MAGIC, 53, FPD_loglvl_t)
#define FPD_GET_BUFFER_INFO       _IOWR(FPD_MAGIC, 54, FPD_buffer_t)
#define FPD_GET_HOST_RXBDTBL      _IOWR(FPD_MAGIC, 55, FPD_hostrxbdtbl_t)
#define FPD_GET_HOST_TXBUF_INFO   _IOWR(FPD_MAGIC, 56, FPD_host_txbuf_info_t)

#define FPD_DRIVER_RESET          _IO  (FPD_MAGIC, 255)

#endif /* _FPD_IOCTL_H_ */
