/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Locking mechanism Hader File
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#ifndef _FPD_LOCK_H_
#define _FPD_LOCK_H_

#include <asm/semaphore.h>

typedef struct {
    struct semaphore       sem;
    u32                    val;
} fpd_lock_t;


/******************************************************************************
 *  ROUTINE:        fpd_lock_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the lock mechanism
 *
 *  PARAMETERS:
 *
 *      plock     Pointer to the lock structure.
 *      init_val  Initial value assigned to the resource data.
 *
 *  RETURNS:
 *
 *     	0         Initialization is successful.
 *     -1         Initialization failed.
 *
 *****************************************************************************/
static inline int fpd_lock_init( fpd_lock_t *plock, u32 init_val )
{
    init_MUTEX(&plock->sem);
    plock->val = init_val;
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_lock_acquire()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Acquires the lock.
 *
 *  PARAMETERS:
 *
 *      plock   Pointer to the lock structure.
 *
 *  RETURNS:
 *
 *     	0       Successful.
 *     -1       Failed.
 *
 *****************************************************************************/
static inline int fpd_lock_acquire( fpd_lock_t *plock )
{
#if 0
    if( down_interruptible(&plock->sem) )
        return -1;
#else
    down(&plock->sem);
#endif
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_lock_release()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Release the lock.
 *
 *  PARAMETERS:
 *
 *      plock   Pointer to the lock structure.
 *
 *  RETURNS:
 *
 *     	0       Successful.
 *     -1       Failed.
 *
 *****************************************************************************/
static inline int fpd_lock_release( fpd_lock_t *plock )
{
    up(&plock->sem);
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_lock_read_resource()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Read the value of the resource guarded by the lock. The lock should
 *      already have been acquired prior to this call.
 *
 *  PARAMETERS:
 *
 *      plock   Pointer to the lock structure.
 *
 *  RETURNS:
 *
 *     Value of the resource.
 *
 *****************************************************************************/
static inline u32 fpd_lock_read_resource( fpd_lock_t *plock )
{
    return( plock->val );
}

/******************************************************************************
 *  ROUTINE:        fpd_lock_write_resource()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Write a value to the resource guarded by the lock. The lock should
 *      already have been acquired prior to this call.
 *
 *  PARAMETERS:
 *
 *      plock   Pointer to the lock structure.
 *      val     Value assigned for the resource.
 *
 *  RETURNS:
 *
 *     None.
 *
 *****************************************************************************/
static inline void fpd_lock_write_resource( fpd_lock_t *plock, u32 val )
{
    plock->val = val;
}

#endif /* _FPD_LOCK_H_ */
