/**
 * gpio_core_faraday.c
 *
 * Low-level driver for the Faraday/KIRA100 GPIO core
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 *
*/

#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/string.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/version.h>
#include <asm/io.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/arch-cpe/cpe/kira100.h>
# include <asm/arch-cpe/cpe/peppercon.h>
#else 
# include <asm/arch/platform-a320/kira100.h>
# include <asm/arch/platform-a320/peppercon.h>
#endif
#include "gpio_common.h"
#include "gpio_core_faraday.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define GPIO_FARADAY_IO_SIZE		0x48
#define GPIO_FARADAY_IRQ		30
#define GPIO_FARADAY_NR_BITS		32
#define GPIO_FARADAY_NAME		"faraday"

#define GPIO_DATA_OUT			0x00
#define GPIO_DATA_IN			0x01
#define GPIO_PIN_DIR			0x02
#define GPIO_PIN_BYPASS			0x03
#define GPIO_DATA_SET			0x04
#define GPIO_DATA_CLEAR			0x05
#define GPIO_PIN_PULL_ENABLE		0x06
#define GPIO_PIN_PULL_TYPE		0x07
#define GPIO_INTR_ENABLE		0x08
#define GPIO_INTR_RAW_STATE		0x09
#define GPIO_INTR_MASKED_STATE		0x0A
#define GPIO_INTR_MASK			0x0B
#define GPIO_INTR_CLEAR			0x0C
#define GPIO_INTR_TRIGGER		0x0D
#define GPIO_INTR_BOTH			0x0E
#define GPIO_INTR_RISE_NEG		0x0F
#define GPIO_INTR_BOUNCE_ENABLE		0x10
#define GPIO_INTR_BOUNCE_PRESCALE	0x11

#define GPIO_MULTIPLEXER_MASK		0xFFFF0000

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    u32			io_base;
    volatile u32 *	regs;
    int			isr_registered;
    int			mem_region_requested;
} ft_data_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

ft_data_t ft_data[] = {
    { .io_base			= CPE_GPIO_BASE,
      .regs			= NULL,
      .isr_registered		= 0,
      .mem_region_requested	= 0
    },
};

static uint32_t kira_rev = 0;

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void faraday_cleanup(gpio_core_t *core);

static int faraday_get_input(gpio_core_t *core, u_long *value);
static int faraday_get_output(gpio_core_t *core, u_long *value);
static int faraday_get_enable(gpio_core_t *core, u_long *value);
static int faraday_set_output(gpio_core_t *core, u_long set, u_long clr);
static int faraday_set_enable(gpio_core_t *core, u_long set, u_long clr);
static int faraday_irq_enable(gpio_core_t *core, u_long set, u_long clr);
static int faraday_irq_edge(gpio_core_t *core, u_long level, u_long edge,
			    u_long pos, u_long neg);
static int faraday_irq_ack(gpio_core_t *core, u_long mask);
static int faraday_get_alternate(gpio_core_t *core, u_long *value);
static int faraday_set_alternate(gpio_core_t *core, u_long set, u_long clr);

static irqreturn_t  faraday_isr(int irq, void *dev_id, struct pt_regs *regs);

#ifdef ENABLE_CHECKS
static void check_multiplexer(ft_data_t *data, u_long mask);
#endif

/* ------------------------------------------------------------------------- *
 * register access inline functions
 * ------------------------------------------------------------------------- */

static __inline u32
gpio_get_reg(ft_data_t *gpio, int reg)
{
    u_long flags;
    u32 ret;
    
    spin_lock_irqsave(&pp_kira_gpio_lock, flags);
    ret = readl(&gpio->regs[reg]);    
    spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);

    return ret;
}

static __inline void
gpio_set_reg(ft_data_t *gpio, int reg, u32 val)
{
    u_long flags;
    spin_lock_irqsave(&pp_kira_gpio_lock, flags);
    writel(val, &gpio->regs[reg]);
    spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);
}

static __inline u32
gpio_get_reg_bits(ft_data_t *gpio, int reg, u32 mask)
{
    u_long flags;
    u_char ret;

    spin_lock_irqsave(&pp_kira_gpio_lock, flags);
    ret = gpio_get_reg(gpio, reg) & mask;
    spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);

    return ret;
}

static __inline void
gpio_set_reg_bits(ft_data_t *gpio, int reg, u32 set_mask, u32 clr_mask)
{
    u_long flags;

    spin_lock_irqsave(&pp_kira_gpio_lock, flags);
    writel((readl(&gpio->regs[reg]) & ~clr_mask) | set_mask, &gpio->regs[reg]);
    spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);
}

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */

int
gpio_core_faraday_init(gpio_core_t *core, u8 core_nr)
{
    int ret = -1, r;
    int nr_cores = sizeof(ft_data) / sizeof(ft_data_t);
    ft_data_t *pdata;
    
    if (core_nr >= nr_cores) {
	ERR("Unsupported core nr %d (max is %d)", core_nr, nr_cores - 1);	
	return ret;
    }

    core->private_data = pdata = &ft_data[core_nr];
    core->cleanup      = faraday_cleanup;
    
    if (request_mem_region(pdata->io_base,
			   GPIO_FARADAY_IO_SIZE, NAME) == NULL) {
	
	ERR("memory region @ 0x%08x (size %d) already in use",
	    pdata->io_base, GPIO_FARADAY_IO_SIZE);
	
	goto bail;
    }
    pdata->mem_region_requested = 1;

    if ((pdata->regs = ioremap_nocache(pdata->io_base,
				       GPIO_FARADAY_IO_SIZE)) == NULL) {

	ERR("could not remap GPIO regs @ 0x%08x (size %d)",
	    pdata->io_base, GPIO_FARADAY_IO_SIZE);
	goto bail;	
    }

    if ((r = request_irq(GPIO_FARADAY_IRQ, faraday_isr, SA_SHIRQ | SA_SAMPLE_RANDOM,
			 NAME, core)) < 0) {
	
	ERR("failed to request irq %d (err %d)", GPIO_FARADAY_IRQ, r);
	goto bail;
    }
    pdata->isr_registered = 1;
    
    core->caps = GPIO_CAPS_IRQ | GPIO_CAPS_ALTERNATE_FUNCTIONS;
    core->nr_bits = GPIO_FARADAY_NR_BITS;
    strncpy(core->name, GPIO_FARADAY_NAME, sizeof(core->name) - 1);
    core->name[sizeof(core->name) - 1] = '\0';

    /* get KIRA100 revision */
    kira_rev = pp_kira_get_revision();

    core->get_input  = faraday_get_input;
    core->get_output = faraday_get_output;
    core->get_enable = faraday_get_enable;
    core->set_output = faraday_set_output;
    core->set_enable = faraday_set_enable;
    core->irq_enable = faraday_irq_enable;
    core->irq_edge   = faraday_irq_edge;
    core->irq_ack    = faraday_irq_ack;

    /* optional alternate functions switching */
    core->get_alternate = faraday_get_alternate;
    core->set_alternate = faraday_set_alternate;

    /* set defaults (edge trigger, both edges, disable all irqs) */
    gpio_set_reg_bits(pdata, GPIO_INTR_TRIGGER, 0, 0xFFFFFFFF);
    gpio_set_reg_bits(pdata, GPIO_INTR_BOTH, 0xFFFFFFFF, 0);
    gpio_set_reg_bits(pdata, GPIO_INTR_ENABLE, 0, 0xFFFFFFFF);
    
    ret = SUCCESS;
    INFO("GPIO core '%s' (%d) initialized", core->name, core_nr);
    
 bail:
    if (ret != SUCCESS) {
	faraday_cleanup(core);
    }
    return ret;
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
faraday_cleanup(gpio_core_t *core)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

    if (pdata == NULL) {	
	return; // we were never really initialized
    }
    
    if (pdata->isr_registered) {
	free_irq(GPIO_FARADAY_IRQ, core); pdata->isr_registered = 0;
    }
    
    if (pdata->regs != NULL) {
	iounmap((void*)pdata->regs);
	pdata->regs = NULL;
    }

    if (pdata->mem_region_requested) {
	release_mem_region(pdata->io_base, GPIO_FARADAY_IO_SIZE);
    }

    INFO("GPIO core '%s' cleanup done", core->name);
}

static int
faraday_get_input(gpio_core_t *core, u_long *value)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    
    if (value) *value = gpio_get_reg(pdata, GPIO_DATA_IN);

    return 0;
}

static int
faraday_get_output(gpio_core_t *core, u_long *value)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

    if (value) *value = gpio_get_reg(pdata, GPIO_DATA_OUT);

    return 0;
}

static int
faraday_get_enable(gpio_core_t *core, u_long *value)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

    if (value) *value = gpio_get_reg(pdata, GPIO_PIN_DIR);

    return 0;
}

static int
faraday_set_output(gpio_core_t *core, u_long set, u_long clr)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

#ifdef ENABLE_CHECKS
    check_multiplexer(pdata, set | clr);
#endif
    
    gpio_set_reg_bits(pdata, GPIO_DATA_OUT, set, clr);

    return 0;
}

static int
faraday_set_enable(gpio_core_t *core, u_long set, u_long clr)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

#ifdef ENABLE_CHECKS
    check_multiplexer(pdata, set);
#endif
    
    gpio_set_reg_bits(pdata, GPIO_PIN_DIR, set, clr);

    return 0;
}

static int
faraday_irq_enable(gpio_core_t *core, u_long set, u_long clr)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

    u_long irq = (gpio_get_reg(pdata, GPIO_INTR_ENABLE) | set) & ~clr;
    u_long oe  = gpio_get_reg(pdata, GPIO_PIN_DIR);

#ifdef ENABLE_CHECKS
    check_multiplexer(pdata, set);
#endif
    
    if (irq & oe) {
	WARN("Not all irq bits are set to input (INTR_ENABLE=0x%08lx, PIN_DIR=0x%08lx)\n",
	     irq, oe);
    }
    
    gpio_set_reg_bits(pdata, GPIO_INTR_ENABLE, set, clr);

    /* unmask all enabled IRQs */
    gpio_set_reg_bits(pdata, GPIO_INTR_MASK, 0, set);
    
    return 0;
}

static int
faraday_irq_edge(gpio_core_t *core, u_long level, u_long edge,
		 u_long pos, u_long neg)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    u_long both = edge & (pos & neg);
    u_long single = edge & (pos ^ neg);
    
#ifdef ENABLE_CHECKS
    check_multiplexer(pdata, level | edge | pos | neg);
#endif
   
    gpio_set_reg_bits(pdata, GPIO_INTR_BOTH, both, single);
    gpio_set_reg_bits(pdata, GPIO_INTR_TRIGGER, level, edge);
    gpio_set_reg_bits(pdata, GPIO_INTR_RISE_NEG, neg, pos);
    
    return 0;
}

static int
faraday_irq_ack(gpio_core_t *core, u_long mask)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    u_long level = gpio_get_reg(pdata, GPIO_INTR_TRIGGER);

    gpio_set_reg_bits(pdata, GPIO_INTR_MASK, 0, mask & level);
    gpio_set_reg(pdata, GPIO_INTR_CLEAR, mask & level);
    
    return 0;
}

static int
faraday_get_alternate(gpio_core_t *core, u_long *value)
{
    //ft_data_t *pdata = (ft_data_t*) core->private_data;

    if (value) {
        *value = pp_kira_get_pmu_reg(KIRA_PMU_REG_PORTMUX);
        if (KIRA_MAJOR(kira_rev) < 2) *value = ~(*value);
        *value &= GPIO_MULTIPLEXER_MASK;
    }

    return 0;
}

static int
faraday_set_alternate(gpio_core_t *core, u_long set, u_long clr)
{
    //ft_data_t *pdata = (ft_data_t*) core->private_data;

    if (KIRA_MAJOR(kira_rev) > 1) {
        /* portmux bit 0 -> gpio, 1 -> alternate */
        pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_PORTMUX, set, clr);
    } else {
        /* portmux bit 1 -> gpio, 0 -> alternate */
        pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_PORTMUX, clr, set);
    }
    
    return 0;
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */

static irqreturn_t
faraday_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    gpio_core_t *core = (gpio_core_t*) dev_id;
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    u_long intr_state = gpio_get_reg(pdata, GPIO_INTR_MASKED_STATE);
    u_long level = gpio_get_reg(pdata, GPIO_INTR_TRIGGER);

    if (!intr_state) return IRQ_NONE; /* not our intr */

    /* acknowledge the irq bits for edge triggered pins */
    gpio_set_reg(pdata, GPIO_INTR_CLEAR, intr_state & ~level);

    /* mask the irq bits for level triggered pins */
    gpio_set_reg(pdata, GPIO_INTR_MASK, intr_state & level);
    
    if (core->isr_cb) core->isr_cb(core, intr_state);

    return IRQ_HANDLED;
}

#ifdef ENABLE_CHECKS
static void
check_multiplexer(ft_data_t *data, u_long mask)
{
    uint32_t reg;
    
    mask &= GPIO_MULTIPLEXER_MASK;
    
    if (mask == 0) return;

    reg = pp_kira_get_pmu_reg(KIRA_PMU_REG_PORTMUX);

    if (KIRA_MAJOR(kira_rev) >= 2) {
	reg = ~reg;
    }
    
    if ((reg & mask) != mask) {
	WARN("Port Multiplexer bits not all set to GPIO! (was 0x%08lx, need 0x%08lx)",
	     reg & mask, mask);
    }
}
#endif
