#ifndef __I2C_COMMON_H
#define __I2C_COMMON_H

/*
   common definitions for the Peppercon I2C devices
   
   We define some logical I2C devices for erla. They can operate
   either in Master or Slave mode. These devices use the underlying
   device driver for the bus operations.
*/

#include <linux/poll.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>

#include "pp_i2c.h"
#include "ringbuffer.h"

#define I2C_MASTER_BUFFER_SIZE	512

struct i2c_device_data {
	struct i2c_client*	client;
};

/* our device structure */
struct i2c_logical_device {
	/* name of the device */
	char			name[20];
	/* name of desired physical adapter (empty: ignore name) */
	char			desired_adapter_name[15];
	/* i2c adapter */
	struct i2c_adapter*	adapter;
	/* semaphore */
	struct semaphore	sem;
	/* operation mode */
	int			transmitter_mode;
	int			receiver_mode;
	/* wait queues for poll */
	wait_queue_head_t	wq_read;
	wait_queue_head_t	wq_write;
	/* i2c bus addresses for master mode */
	u16			slave_address;
	u16			master_address;
	/* flags */
	char			i2c_initialized;
	/* master data buffer */
	char			master_buffer[I2C_MASTER_BUFFER_SIZE];
	int			master_bufsize;
	/* slave data buffers */
	struct ringbuffer*	slave_write_buffer;
	struct ringbuffer*	slave_read_buffer;
	/* locks */
	spinlock_t		slave_write_lock;
	spinlock_t		slave_read_lock;
	/* device structure */
	struct i2c_device_data*	data;
};

/* functions */
int i2c_common_init(struct i2c_logical_device* dev,
                    struct i2c_driver* driver,
                    struct i2c_slave_device* slave_dev);
int i2c_common_cleanup(struct i2c_logical_device* dev,
                       struct i2c_driver* driver,
                       struct i2c_slave_device* slave_dev);
int i2c_common_i2c_detect_client(struct i2c_logical_device* dev,
                                 struct i2c_driver* driver,
				 struct i2c_adapter *adapter,
                                 int address,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
                                 unsigned short flags,
#endif
				 int kind);
int i2c_common_i2c_detach_client(struct i2c_logical_device* dev,
				 struct i2c_client *client);
int i2c_common_open(struct i2c_logical_device* dev,
		    struct inode * inode, struct file * file);
int i2c_common_release(struct i2c_logical_device* dev,
		       struct inode * inode, struct file * file);
int i2c_common_ioctl(struct i2c_logical_device* dev,
		     struct inode * inode, struct file * file,
		     uint cmd, ulong arg);
loff_t i2c_common_llseek(struct i2c_logical_device* dev,
			 struct file * file, loff_t offset, int whence);
int i2c_common_read(struct i2c_logical_device* dev,
		    struct file* file, char* buffer,
		    size_t size, loff_t* offset);
int i2c_common_write(struct i2c_logical_device* dev,
		     struct file* file, const char* buffer,
		     size_t size, loff_t* offset);
int i2c_common_poll(struct i2c_logical_device* dev,
		    struct file * file, poll_table * wait);
int i2c_common_slave_write_cb(struct i2c_logical_device* dev,
			      char* buf, int size);
int i2c_common_slave_read_cb(struct i2c_logical_device* dev,
			     char* buf, int size);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
int i2c_common_attach_adapter(struct i2c_logical_device* dev,
			      struct i2c_adapter *adapter,
			      int (*found_proc) (struct i2c_adapter *, int, int));
#else
int i2c_common_attach_adapter(struct i2c_logical_device* dev,
			      struct i2c_adapter *adapter,
			      i2c_client_found_addr_proc *found_proc);
#endif

#endif /* __I2C_COMMON_H */

