#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/delay.h>

//#define DBG_LEVEL 3

#include "debug.h"
#include "i2c_common.h"

#define I2C_IPMB_TEST_NAME	"i2c-ipmb_test"

#define IPMB_TEST_REQUEST_STRING "12345678 - Peppercon IPMB Test Request - 87654321"
#define IPMB_TEST_REPLY_STRING   "ABCDEFGH - Peppercon IPMB Test Reply - HGFEDCBA"

static int ipmb_perform_test(struct i2c_logical_device* dev);
static int ipmb_start_slave_test_thread(void);
static void ipmb_stop_slave_test_thread(void);

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */

#define DBG(fmt, x...)	I2C_DBG(I2C_IPMB_TEST_NAME, fmt, ##x)
#define DBG2(fmt, x...)	I2C_DBG2(I2C_IPMB_TEST_NAME, fmt, ##x)
#define WARN(fmt, x...)	I2C_WARN(I2C_IPMB_TEST_NAME, fmt, ##x)

/* ------------------------------------------------------------------------- *
 * Constants
 * ------------------------------------------------------------------------- */

/* ------------------------------------------------------------------------- *
 * Function prototypes
 * ------------------------------------------------------------------------- */

static int i2c_ipmb_test_init(struct i2c_device_plugin *dev);
static void i2c_ipmb_test_cleanup(struct i2c_device_plugin *dev);
static int i2c_ipmb_test_i2c_attach_adapter(struct i2c_adapter *adapter);
static int i2c_ipmb_test_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
				           unsigned short flags,
#endif
				           int kind);
static int i2c_ipmb_test_i2c_detach_client(struct i2c_client *client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void i2c_ipmb_test_i2c_inc_use (struct i2c_client *client);
static void i2c_ipmb_test_i2c_dec_use (struct i2c_client *client);
#endif
static int i2c_ipmb_test_i2c_command(struct i2c_client *client,
			             unsigned int cmd, void *arg);
static int i2c_ipmb_test_open(struct inode * inode, struct file * file);
static int i2c_ipmb_test_release(struct inode * inode, struct file * file);
static int i2c_ipmb_test_ioctl(struct inode * inode, struct file * file,
			       uint cmd, ulong arg);
static loff_t i2c_ipmb_test_llseek(struct file * file, loff_t offset, int whence);
static int i2c_ipmb_test_read(struct file* file, char* buffer,
			      size_t size, loff_t* offset);
static int i2c_ipmb_test_write(struct file* file, const char* buffer,
			       size_t size, loff_t* offset);
static unsigned int i2c_ipmb_test_poll(struct file * file, poll_table * wait);
static int i2c_ipmb_test_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_ipmb_test_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_ipmb_test_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

struct file_operations i2c_ipmb_test_ops = {
	owner:			THIS_MODULE,
	ioctl:			i2c_ipmb_test_ioctl,
	open:			i2c_ipmb_test_open,
	release:		i2c_ipmb_test_release,
	read:			i2c_ipmb_test_read,
	write:			i2c_ipmb_test_write,
	llseek:			i2c_ipmb_test_llseek,
	poll:			i2c_ipmb_test_poll,
};

static struct i2c_device_plugin i2c_ipmb_test_plugin = {
	name:		I2C_IPMB_TEST_NAME,
	minor:		I2C_IPMB_TEST_MINOR,
	address:	I2C_IPMB_TEST_ADDRESS,
	initialized:	0,
	fops:		&i2c_ipmb_test_ops,
	init_func:	i2c_ipmb_test_init,
	cleanup_func:	i2c_ipmb_test_cleanup,
};

/* ------------------------------------------------------------------------- *
 * Module structures
 * ------------------------------------------------------------------------- */

static struct i2c_logical_device i2c_ipmb_test_device = {
	name:			I2C_IPMB_TEST_NAME,
	desired_adapter_name:   "FIA320 I2C 1",
	transmitter_mode:	I2C_MODE_MASTER,
	receiver_mode:		I2C_MODE_SLAVE,
	slave_address:		I2C_IPMB_TEST_ADDRESS,
	master_address:		I2C_OWN_ADDRESS,
	master_bufsize:		I2C_MASTER_BUFFER_SIZE,
	i2c_initialized:	0,
	data:			NULL,
};

/* ------------------------------------------------------------------------- *
 * driver structure
 * ------------------------------------------------------------------------- */

static struct i2c_driver i2c_ipmb_test_driver = {
	name:			"i2c_ipmb_test",
	id:			0xFFF6,	/* f000-ffff for local use */
	flags:			I2C_DF_NOTIFY,
	attach_adapter:		i2c_ipmb_test_i2c_attach_adapter,
	detach_client:		i2c_ipmb_test_i2c_detach_client,
	command:		i2c_ipmb_test_i2c_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	inc_use:		i2c_ipmb_test_i2c_inc_use,
	dec_use:		i2c_ipmb_test_i2c_dec_use,
#endif
};

/* ------------------------------------------------------------------------- *
 * Slave device structure
 * ------------------------------------------------------------------------- */

static struct i2c_slave_device i2c_ipmb_test_slave = {
	name:			"i2c_ipmb_test",
	adapter:		NULL,	/* later filled up by init function */
	slave_write:		i2c_ipmb_test_slave_write_cb,
	slave_read:		i2c_ipmb_test_slave_read_cb,
	is_responsible:		i2c_ipmb_test_slave_responsible_cb,
};

/* ------------------------------------------------------------------------- *
 * Initialization
 * ------------------------------------------------------------------------- */

int
i2c_ipmb_test_init_plugin(struct i2c_device_plugin ** plugins,
		          int * no_plugins, int max_no)
{
	if (*no_plugins >= max_no) {
		return -ENOMEM;
	}
	
	plugins[(*no_plugins)++] = &i2c_ipmb_test_plugin;
	return 0;
}

static int
i2c_ipmb_test_init(struct i2c_device_plugin *dev)
{
	DBG("init.\n");
	
	if (i2c_common_init(&i2c_ipmb_test_device,
	    &i2c_ipmb_test_driver, &i2c_ipmb_test_slave) != 0) {
		goto fail;
	}
	
	if (ipmb_start_slave_test_thread()) {
	    goto fail;
	}
	
	DBG("initialization finished.\n");
	return 0;
fail:
	WARN("initialization failed.\n");
	return -1;
}

static void
i2c_ipmb_test_cleanup(struct i2c_device_plugin *dev)
{
	DBG("cleanup.\n");
	
	ipmb_stop_slave_test_thread();
	i2c_common_cleanup(&i2c_ipmb_test_device, &i2c_ipmb_test_driver, &i2c_ipmb_test_slave);
	
	DBG("cleanup finished.\n");
}

/* ------------------------------------------------------------------------- *
 * I2C functions
 * ------------------------------------------------------------------------- */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void
i2c_ipmb_test_i2c_inc_use (struct i2c_client *client)
{
}

static void
i2c_ipmb_test_i2c_dec_use (struct i2c_client *client)
{
}
#endif

static int
i2c_ipmb_test_i2c_command(struct i2c_client *client,
		          unsigned int cmd, void *arg)
{
	/* no commands defined */
	return 0;
}

static int
i2c_ipmb_test_i2c_attach_adapter(struct i2c_adapter *adapter)
{
	return i2c_common_attach_adapter(&i2c_ipmb_test_device,
		adapter, &i2c_ipmb_test_i2c_detect_client);
}

static int
i2c_ipmb_test_i2c_detach_client(struct i2c_client *client)
{
	return i2c_common_i2c_detach_client(&i2c_ipmb_test_device, client);
}


static int
i2c_ipmb_test_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			        unsigned short flags,
#endif
			        int kind)
{
	return i2c_common_i2c_detect_client(&i2c_ipmb_test_device,
		&i2c_ipmb_test_driver, adapter, address,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	       	flags,
#endif
	       	kind);
}


/* ------------------------------------------------------------------------- *
 * Device functions
 * ------------------------------------------------------------------------- */

static int
i2c_ipmb_test_open(struct inode * inode, struct file * file)
{
	return i2c_common_open(&i2c_ipmb_test_device, inode, file);
}

static int
i2c_ipmb_test_release(struct inode * inode, struct file * file)
{
	return i2c_common_release(&i2c_ipmb_test_device, inode, file);
}

static loff_t
i2c_ipmb_test_llseek(struct file * file, loff_t offset, int whence)
{
	return i2c_common_llseek(&i2c_ipmb_test_device,
		file, offset, whence);
}

static int
i2c_ipmb_test_ioctl(struct inode * inode, struct file * file,
	            uint cmd, ulong arg)
{
	if (cmd == IOCTL_I2C_IPMB_EVALBOARD_TEST) {
	    return ipmb_perform_test(&i2c_ipmb_test_device);
	}
	return i2c_common_ioctl(&i2c_ipmb_test_device,
		inode, file, cmd, arg);
}

static int
i2c_ipmb_test_read(struct file* file, char* buffer,
	           size_t size, loff_t* offset)
{
	return i2c_common_read(&i2c_ipmb_test_device,
		file, buffer, size, offset);
}

static int
i2c_ipmb_test_write(struct file* file, const char* buffer,
	            size_t size, loff_t* offset)
{
	return i2c_common_write(&i2c_ipmb_test_device,
		file, buffer, size, offset);
}

static unsigned int
i2c_ipmb_test_poll(struct file * file, poll_table * wait)
{
	return i2c_common_poll(&i2c_ipmb_test_device,
		file, wait);
}

/* ------------------------------------------------------------------------- *
 * I2C Slave mode callbacks
 * ------------------------------------------------------------------------- */

static int
i2c_ipmb_test_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size)
{
	return i2c_common_slave_write_cb(&i2c_ipmb_test_device, buf, size);
}

static int
i2c_ipmb_test_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size)
{
	return i2c_common_slave_read_cb(&i2c_ipmb_test_device, buf, size);
}

static int
i2c_ipmb_test_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size)
{
	if (size >= 3 && (!strcmp(buf, IPMB_TEST_REQUEST_STRING) || !strcmp(buf, IPMB_TEST_REPLY_STRING))) {
		return 1;
	}

	return 0;
}

/* ------------------------------------------------------------------------- *
 * The test implementation
 * ------------------------------------------------------------------------- */

static volatile int ipmb_test_reply_received = 0;
static wait_queue_head_t ipmb_test_reply_wq;
static volatile int ipmb_test_thread_should_run;
static volatile int ipmb_test_thread_running;
struct semaphore * ipmb_test_thread_notify;

static int ipmb_test_thread(void * arg);

/* --- commonly used stuff --- */
static inline int ringbuffer_is_empty_locked(struct i2c_logical_device* dev) {
    unsigned long flags;
    int ret;
    
    spin_lock_irqsave(&dev->slave_write_lock, flags);
    ret = ringbuffer_is_empty(dev->slave_write_buffer);
    spin_unlock_irqrestore(&dev->slave_write_lock, flags);
    
    return ret;
}

static int ipmb_send_string(struct i2c_logical_device* dev, const char *s) {
    char b[256];
    snprintf(b, sizeof(b), "%s", s);
    struct i2c_msg msg = { dev->slave_address, 0, strlen(s), b };
    
    if (i2c_transfer(dev->adapter, &msg, 1) < 0) {
        WARN("Could not transfer message string.\n");
        return -1;
    }
    
    return 0;
}

/* --- master part --- */
static int ipmb_perform_test(struct i2c_logical_device* dev) {
    int ret = 0;
    
    // clear the condition
    ipmb_test_reply_received = 0;
    
    // send the string
    if (ipmb_send_string(dev, IPMB_TEST_REQUEST_STRING)) {
        ret = -EIO;
        goto bail;
    }
    
    // wait for the reply
    ret = wait_event_interruptible_timeout(ipmb_test_reply_wq, ipmb_test_reply_received, HZ);
    if (ret < 0) {
        ret = -EINTR;
        goto bail;
    }
    if (!ipmb_test_reply_received) {
        WARN("Timeout waiting for slave reply.");
        ret = -ETIMEDOUT;
        goto bail;
    }
    
    ret = 0;

bail:
    return ret;
}

/* --- slave part --- */
static int ipmb_start_slave_test_thread(void) {
    int r;
    
    DBG("Initializing IPMB test slave part\n");
    
    init_waitqueue_head(&ipmb_test_reply_wq);
    ipmb_test_reply_received = 0;
    
    DECLARE_MUTEX_LOCKED(mtx);
    ipmb_test_thread_notify = &mtx;
    ipmb_test_thread_should_run = 1;
    if ((r = kernel_thread(ipmb_test_thread, NULL, 0)) < 0) {
        WARN("failed to initialize IPMB test slave thread (err %d)\n", r);
        return -EBUSY;
    }
    down(&mtx);
    ipmb_test_thread_notify = NULL;

    return 0;
}

static void ipmb_stop_slave_test_thread(void) {
    DBG("Cleaning up IPMB test slave part\n");
    
    if (ipmb_test_thread_running) {
        DECLARE_MUTEX_LOCKED(mtx);
        ipmb_test_thread_notify = &mtx;
        
        ipmb_test_thread_should_run = 0;
        wake_up_interruptible(&i2c_ipmb_test_device.wq_read);
        down(&mtx);
        ipmb_test_thread_notify = NULL;
    }
}

static void ipmb_send_reply_if_requested(struct i2c_logical_device* dev) {
    unsigned long flags;
    int found_request = 0;
    int len;
    
    spin_lock_irqsave(&dev->slave_write_lock, flags);
    while (!ringbuffer_is_empty(dev->slave_write_buffer)) {
        unsigned char buf[BUFFER_ENTRYSIZE];
        size_t mysize = sizeof(buf);
        if (ringbuffer_get_entry(dev->slave_write_buffer, buf, &mysize) < 0) {
            WARN("Could not get ringbuffer entry!\n");
            break;
        }
        
        if (!found_request) {
            if (mysize == (len = strlen(IPMB_TEST_REQUEST_STRING)) &&
                !strncmp((const char *)buf, IPMB_TEST_REQUEST_STRING, len)) {
                DBG("Found request.\n");
                found_request = 1;
            }
        }
        
        if (!ipmb_test_reply_received) {
            if (mysize == (len = strlen(IPMB_TEST_REPLY_STRING)) &&
                !strncmp((const char *)buf, IPMB_TEST_REPLY_STRING, len)) {
                DBG("Found reply.\n");
                ipmb_test_reply_received = 1;
                wake_up_interruptible(&ipmb_test_reply_wq);
            }
        }
    }
    spin_unlock_irqrestore(&dev->slave_write_lock, flags);
    
    if (found_request) {
        DBG("Sending reply\n");
        msleep(20);
        ipmb_send_string(dev, IPMB_TEST_REPLY_STRING);
    } else {
        DBG("Didn't find request in buffer.\n");
    }
}

static int ipmb_test_thread(void * arg) {
    DBG("ipmb_test_thread started\n");
    ipmb_test_thread_running = 1;

    daemonize(I2C_IPMB_TEST_NAME);

    if (ipmb_test_thread_notify != NULL) {
        up(ipmb_test_thread_notify);
    }
    
    while (ipmb_test_thread_should_run) {
        int ret = wait_event_interruptible(i2c_ipmb_test_device.wq_read, !ipmb_test_thread_should_run || !ringbuffer_is_empty_locked(&i2c_ipmb_test_device));
        if (ret < 0) {
            if (signal_pending(current)) {
                flush_signals(current);
                continue;
            }
            WARN("Waiting for event failed!\n");
            break;
        }

        DBG("ipmb_test_thread - woken up\n");
        
        if (!ipmb_test_thread_should_run) {
            DBG("Have to break");
            break;
        }
        if (!ringbuffer_is_empty_locked(&i2c_ipmb_test_device)) {
            DBG("Have data, sending reply if needed.\n");
            ipmb_send_reply_if_requested(&i2c_ipmb_test_device);
        }
        cond_resched();
    }
    DBG("ipmb_test_thread finished\n");

    ipmb_test_thread_running = 0;
    if (ipmb_test_thread_notify != NULL) {
        up(ipmb_test_thread_notify);
    }
    
    return 0;
}

