#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/slab.h>
#include <linux/poll.h>

#include "debug.h"
#include "i2c_common.h"

#define I2C_KME_NAME	"i2c-kme"

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */

#define DBG(fmt, x...)	I2C_DBG(I2C_KME_NAME, fmt, ##x)
#define DBG2(fmt, x...)	I2C_DBG2(I2C_KME_NAME, fmt, ##x)
#define WARN(fmt, x...)	I2C_WARN(I2C_KME_NAME, fmt, ##x)

/* ------------------------------------------------------------------------- *
 * Constants
 * ------------------------------------------------------------------------- */

/* ------------------------------------------------------------------------- *
 * Function prototypes
 * ------------------------------------------------------------------------- */

static int i2c_kme_init(struct i2c_device_plugin *dev);
static void i2c_kme_cleanup(struct i2c_device_plugin *dev);
static int i2c_kme_i2c_attach_adapter(struct i2c_adapter *adapter);
static int i2c_kme_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
				     unsigned short flags,
#endif
				     int kind);
static int i2c_kme_i2c_detach_client(struct i2c_client *client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void i2c_kme_i2c_inc_use (struct i2c_client *client);
static void i2c_kme_i2c_dec_use (struct i2c_client *client);
#endif
static int i2c_kme_i2c_command(struct i2c_client *client,
			       unsigned int cmd, void *arg);
static int i2c_kme_open(struct inode * inode, struct file * file);
static int i2c_kme_release(struct inode * inode, struct file * file);
static int i2c_kme_ioctl(struct inode * inode, struct file * file,
			 uint cmd, ulong arg);
static loff_t i2c_kme_llseek(struct file * file, loff_t offset, int whence);
static int i2c_kme_read(struct file* file, char* buffer,
			size_t size, loff_t* offset);
static int i2c_kme_write(struct file* file, const char* buffer,
			 size_t size, loff_t* offset);
static unsigned int i2c_kme_poll(struct file * file, poll_table * wait);
static int i2c_kme_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_kme_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size);
static int i2c_kme_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

struct file_operations i2c_kme_ops = {
	owner:			THIS_MODULE,
	ioctl:			i2c_kme_ioctl,
	open:			i2c_kme_open,
	release:		i2c_kme_release,
	read:			i2c_kme_read,
	write:			i2c_kme_write,
	llseek:			i2c_kme_llseek,
	poll:			i2c_kme_poll,
};

/* ------------------------------------------------------------------------- *
 * Module structures
 * ------------------------------------------------------------------------- */

static struct i2c_logical_device i2c_kme_device = {
	name:			I2C_KME_NAME,
#ifdef PP_BOARD_KIRA
	desired_adapter_name:   "FIA320 I2C 1",
#else
	desired_adapter_name:   "IBM_IIC I2C 0",
#endif	
	transmitter_mode:	I2C_MODE_MASTER,
	receiver_mode:		I2C_MODE_SLAVE,
	slave_address:		I2C_KME_ADDRESS,
	master_address:		I2C_OWN_ADDRESS,
	master_bufsize:		I2C_MASTER_BUFFER_SIZE,
	i2c_initialized:	0,
	data:			NULL,
};

static struct i2c_device_plugin i2c_kme_plugin = {
	name:		I2C_KME_NAME,
	minor:		I2C_KME_MINOR,
	address:	I2C_KME_ADDRESS,
	initialized:	0,
	fops:		&i2c_kme_ops,
	init_func:	i2c_kme_init,
	cleanup_func:	i2c_kme_cleanup,
};

/* ------------------------------------------------------------------------- *
 * driver structure
 * ------------------------------------------------------------------------- */

static struct i2c_driver i2c_kme_driver = {
	name:			"i2c_kme",
	id:			0xFFF4,	/* f000-ffff for local use */
	flags:			I2C_DF_NOTIFY,
	attach_adapter:		i2c_kme_i2c_attach_adapter,
	detach_client:		i2c_kme_i2c_detach_client,
	command:		i2c_kme_i2c_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	inc_use:		i2c_kme_i2c_inc_use,
	dec_use:		i2c_kme_i2c_dec_use,
#endif
};

/* ------------------------------------------------------------------------- *
 * Slave device structure
 * ------------------------------------------------------------------------- */

static struct i2c_slave_device i2c_kme_slave = {
	name:			"i2c_kme",
	adapter:		NULL,	/* later filled up by init function */
	slave_write:		i2c_kme_slave_write_cb,
	slave_read:		i2c_kme_slave_read_cb,
	is_responsible:		i2c_kme_slave_responsible_cb,
};

/* ------------------------------------------------------------------------- *
 * Initialization
 * ------------------------------------------------------------------------- */

int
i2c_kme_init_plugin(struct i2c_device_plugin ** plugins,
		    int * no_plugins, int max_no)
{
	if (*no_plugins >= max_no) {
		return -ENOMEM;
	}
	
	plugins[(*no_plugins)++] = &i2c_kme_plugin;
	return 0;
}

static int
i2c_kme_init(struct i2c_device_plugin *dev)
{
	DBG("init.\n");
	
	if (i2c_common_init(&i2c_kme_device,
	    &i2c_kme_driver, &i2c_kme_slave) != 0) {
		goto fail;
	}
	
	DBG("initialization finished.\n");
	return 0;
fail:
	WARN("initialization failed.\n");
	return -1;
}

static void
i2c_kme_cleanup(struct i2c_device_plugin *dev)
{
	DBG("cleanup.\n");
	
	i2c_common_cleanup(&i2c_kme_device, &i2c_kme_driver, &i2c_kme_slave);
	
	DBG("cleanup finished.\n");
}

/* ------------------------------------------------------------------------- *
 * I2C functions
 * ------------------------------------------------------------------------- */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void
i2c_kme_i2c_inc_use (struct i2c_client *client)
{
}

static void
i2c_kme_i2c_dec_use (struct i2c_client *client)
{
}
#endif

static int
i2c_kme_i2c_command(struct i2c_client *client,
		    unsigned int cmd, void *arg)
{
	/* no commands defined */
	return 0;
}

static int
i2c_kme_i2c_attach_adapter(struct i2c_adapter *adapter)
{
	return i2c_common_attach_adapter(&i2c_kme_device,
		adapter, &i2c_kme_i2c_detect_client);
}

static int
i2c_kme_i2c_detach_client(struct i2c_client *client)
{
	return i2c_common_i2c_detach_client(&i2c_kme_device, client);
}


static int
i2c_kme_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			  unsigned short flags,
#endif
			  int kind)
{
	return i2c_common_i2c_detect_client(&i2c_kme_device,
		&i2c_kme_driver, adapter, address,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	       	flags,
#endif
	       	kind);
}


/* ------------------------------------------------------------------------- *
 * Device functions
 * ------------------------------------------------------------------------- */

static int
i2c_kme_open(struct inode * inode, struct file * file)
{
	return i2c_common_open(&i2c_kme_device, inode, file);
}

static int
i2c_kme_release(struct inode * inode, struct file * file)
{
	return i2c_common_release(&i2c_kme_device, inode, file);
}

static loff_t
i2c_kme_llseek(struct file * file, loff_t offset, int whence)
{
	return i2c_common_llseek(&i2c_kme_device,
		file, offset, whence);
}

static int
i2c_kme_ioctl(struct inode * inode, struct file * file,
	      uint cmd, ulong arg)
{
	return i2c_common_ioctl(&i2c_kme_device,
		inode, file, cmd, arg);
}

static int
i2c_kme_read(struct file* file, char* buffer,
	     size_t size, loff_t* offset)
{
	int i;
	unsigned char buf[BUFFER_ENTRYSIZE];
	int ret = i2c_common_read(&i2c_kme_device,
		file, buffer, size, offset);
	
	/* rearrange data for the kme device,
	   the kme sends 3 times its own i2c address
	   in front of a message; we are only interested
	   in the latter one */
	if (ret < 4)
		return ret;
	
	if (copy_from_user(buf, buffer, ret)) {
		ret = -EFAULT;
		goto bail;
	}
	
	for (i = 0; i < ret; i++)
		buf[i] = buf[i + 3];
	
	if (copy_to_user((char *)buffer, buf, ret - 3)) {
		ret = -EFAULT;
		goto bail;
	}

	return ret - 3;
 bail:
 	return ret;
}

static int
i2c_kme_write(struct file* file, const char* buffer,
	      size_t size, loff_t* offset)
{
	return i2c_common_write(&i2c_kme_device,
		file, buffer, size, offset);
}

static unsigned int
i2c_kme_poll(struct file * file, poll_table * wait)
{
	return i2c_common_poll(&i2c_kme_device,
		file, wait);
}

/* ------------------------------------------------------------------------- *
 * I2C Slave mode callbacks
 * ------------------------------------------------------------------------- */

static int
i2c_kme_slave_write_cb(struct i2c_slave_device *dev, char* buf, int size)
{
	return i2c_common_slave_write_cb(&i2c_kme_device, buf, size);
}

static int
i2c_kme_slave_read_cb(struct i2c_slave_device *dev, char* buf, int size)
{
	return i2c_common_slave_read_cb(&i2c_kme_device, buf, size);
}

static int
i2c_kme_slave_responsible_cb(struct i2c_slave_device *dev, unsigned int slave_addr, char* buf, int size)
{
	/* we can only recognize KME operations if at least 3 bytes are transmitted */
	if (size < 3)  return 0;
	
	/* KME operations start with 3 times the KME's slave address. */
	if (buf[0] == buf[1] &&
	    buf[1] == buf[2] &&
	    buf[2] == I2C_KME_ADDRESS) return 1;
	
	return 0;
}

