#ifdef __KERNEL__
#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/version.h>
#else
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#endif

#include "ringbuffer.h"

#ifndef min
#define min(a, b)  (((a) < (b)) ? (a) : (b))
#endif
#ifndef max
#define max(a, b)  (((a) > (b)) ? (a) : (b))
#endif

/* returns the position of the next ring buffer entry */
inline int
next_pos(int old_pos) {
	return (old_pos + 1) == BUFFER_ENTRIES ? 0 : old_pos + 1;
}

static inline int
ringbuffer_is_really_full(struct ringbuffer* buffer)
{
	return (buffer->no_buf >= BUFFER_ENTRIES);
}

int
ringbuffer_is_full(struct ringbuffer* buffer)
{
	if (buffer->ignore_overflow) return 0;
	return ringbuffer_is_really_full(buffer);
}

int
ringbuffer_is_empty(struct ringbuffer* buffer)
{
	return (buffer->no_buf <= 0);
}


/* read the current entry from the ring buffer
   returns the position of the put element or
   < 0 if the buffer is full */
int
ringbuffer_get_entry(struct ringbuffer* buffer, char* data, int* len)
{
	if (!ringbuffer_is_empty(buffer)) {
		int pos = buffer->buf_out;
		buffer->buf_out = next_pos(buffer->buf_out);
		buffer->no_buf--;
		*len = min(*len, buffer->len[pos]);
		memcpy(data, buffer->buffer[pos], *len);
		return pos;
	}

	/* buffer empty, return an error */
	return -1;
}

/* put an entry in the ringbuffer
   returns the position of the put element or
   < 0 if the buffer is full */
static inline int do_put_entry(struct ringbuffer* buffer, char* data, int* len)
{
	int pos = buffer->buf_in;
	buffer->buf_in = next_pos(buffer->buf_in);
	*len = min(*len, BUFFER_ENTRYSIZE);
	buffer->len[pos] = *len;
	memcpy(buffer->buffer[pos], data, *len);
	return pos;
}

int
ringbuffer_put_entry(struct ringbuffer* buffer, char* data, int* len)
{
	if (!ringbuffer_is_really_full(buffer)) {
		/* increase the number of buffer entries... */
		buffer->no_buf++;
		/* ... and put the data into the buffer */
		return do_put_entry(buffer, data, len);
	}
	else if (buffer->ignore_overflow) {
		/* discard oldest entry in the buffer... */
		buffer->buf_out = next_pos(buffer->buf_out);
		/* ... and put the data into the buffer */
		return do_put_entry(buffer, data, len);
	}

	/* buffer full, return an error */
	return -1;
}

/* initializes/clears the ring buffer */
void
ringbuffer_init(struct ringbuffer* buffer, int allow_overflow)
{
	buffer->buf_in = 0;
	buffer->buf_out = 0;
	buffer->no_buf = 0;
	buffer->ignore_overflow = allow_overflow;
}

