/**
 * lpc_core.c
 *
 * KIRA100 (slave) LPC IP core low-level access functions
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/version.h>

#ifdef __powerpc__
#include <asm/ppc405_pcimcs.h>
#endif

#ifdef __arm__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <asm/arch/cpe/kira100.h>
#include <asm/arch/cpe/peppercon.h>
#else
#include <asm/arch/platform/kira100.h>
#include <asm/arch/platform/peppercon.h>
#endif
#include <asm/arch/net_lock.h>
#endif

#include <linux/delay.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/ioctl.h>
#include <linux/poll.h>
#include <linux/pci.h>
#include <linux/wait.h>
#include <asm/io.h>
#include <asm/atomic.h>

#include "lpc.h"
#include "lpc_core.h"
#include "lpc_kcs.h"
#include "lpc_bt.h"
#include "lpc_smic.h"

#define LOG_COLORED
#define noLOG_BLACK_BG
#define noLOG_DBG
#include "log.h"

#define SUCCESS 0

#ifdef __arm__
/***
    here is everything needed for the DMA/MMU hack
***/

void fixed_read_block(u32 *buf, lpc_t *lpcx, u32 offset, int cnt)
{
    ahbfix_read_block(buf, CPE_LPC_BASE, offset, cnt, 1);
}

u8 fixed_read(lpc_t *lpcx, int offset)
{
    u32 val;
    ahbfix_read_block(&val, CPE_LPC_BASE, offset, 1, 0);
    return (u8)val;
}

void fixed_write_block(u32 *vals, lpc_t *lpcx, u32 offset, int cnt)
{
    ahbfix_write_block(vals, CPE_LPC_BASE, offset, cnt, 1);
}

void fixed_write(lpc_t *lpcx, int offset, uint8_t val)
{
    u32 temp = val;
    
    ahbfix_write_block(&temp, CPE_LPC_BASE, offset, 1, 0);
    return;
}

#endif 

/***
    the normal register read/write functions
***/
    
u8 direct_read(lpc_t *lpcx, int offset)
{ 
    return readb((uint8_t*)lpcx->io_base + (offset << 2)); 
}

void direct_write(lpc_t *lpcx, int offset, uint8_t val)
{ 
    writeb(val, (uint8_t*)lpcx->io_base + (offset << 2)); 
}

void direct_read_block(u32 *buf, lpc_t *lpcx, u32 offset, int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) buf[i] = (u32)direct_read(lpcx, offset);
}

void direct_write_block(u32 *vals, lpc_t *lpcx, u32 offset, int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) direct_write(lpcx, offset, (u8)(vals[i]));
}


/*
 * Public functions
 */

int lpc_init(void *io_base, lpc_t **plpc)
{

    lpc_t *lpc = kmalloc(sizeof(lpc_t), GFP_KERNEL);
    memset(lpc, 0, sizeof(lpc_t));

    lpc->io_base = io_base;

    /* get KIRA revision, init DMA/MMU hack if needed and fill the function pointers */
#ifdef __arm__    
    lpc->kira_rev = pp_kira_get_revision();
    DBG("LPC: detected KIRA revision %d.%d\n", KIRA_MAJOR(lpc->kira_rev), KIRA_MINOR(lpc->kira_rev));
    if (KIRA_MAJOR(lpc->kira_rev) < 2) {
        /* we need the dma or mmu hack in this case */
        WARN("using AHB workaround!");
        lpc->set_reg = fixed_write;
        lpc->get_reg = fixed_read;
        lpc->get_block = fixed_read_block;
        lpc->set_block = fixed_write_block;
    } else
#endif
    {
        lpc->set_reg = direct_write;
        lpc->get_reg = direct_read;
        lpc->get_block = direct_read_block;
        lpc->set_block = direct_write_block;
    }        
    
#ifdef __powerpc__
    {
	unsigned char hicr1;
	int acc_err;
	/* check if io range is (machine check save) accessible */
	hicr1 = mcs_in_8((uint8_t*)lpc->io_base + (LPC_HICR1 << 2), &acc_err);
	if (acc_err != 0) {
	    lpc->io_base = 0;
	    return -EIO;
	}
	mcs_out_8(hicr1, (uint8_t*)lpc->io_base + (LPC_HICR1 << 2), &acc_err);
	if (acc_err != 0) {
	    lpc->io_base = 0;
	    return -EIO;
	}
    }
#endif

    lpc_soft_reset(lpc);

    /* check LPC core presence (LADR1 should be always 0x0060 after reset) */
    if (lpc_get_reg(lpc, LPC_LADR12L) != 0x60 || lpc_get_reg(lpc, LPC_LADR12H) != 0x00) {
        lpc->io_base = 0;
        return -ENODEV;
    }

    *plpc = lpc;
    return SUCCESS;
}

int lpc_cleanup(lpc_t *lpc)
{
    lpc_soft_reset(lpc);
    kfree(lpc);
    return SUCCESS;
}

int lpc_soft_reset(lpc_t *lpc)
{
    /* ensure that reset bit =0 */
    lpc_set_reg_bits(lpc, LPC_HICR1, 0, LPC_LRSTB);

    /* set reset bit to commit reset */
    lpc_set_reg_bits(lpc, LPC_HICR1, LPC_LRSTB, 0);

    /* clear reset bit to start operation */
    lpc_set_reg_bits(lpc, LPC_HICR1, 0, LPC_LRSTB);

    return SUCCESS;
}
