/**
 * lpc_core.h
 *
 * KIRA100 (slave) LPC IP core low-level access functions
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#ifndef __LPC_CORE_H__
#define __LPC_CORE_H__

#include <linux/kernel.h>
#include <asm/io.h>

/*
 * LPC register offsets
 */

/* host interface control registers */
#define LPC_HICR0       0x30 
#define LPC_HICR1       0x31
#define LPC_HICR2       0x32
#define LPC_HICR3       0x33
#define LPC_HICR4       0x00

/* LPC address registers */
#define LPC_LADR12H     0x38 
#define LPC_LADR12L     0x39
#define LPC_LADR3H      0x24
#define LPC_LADR3L      0x25

/* input data/output data/status registers */
#define LPC_IDR1        0x28 
#define LPC_ODR1        0x29
#define LPC_STR1        0x2a
#define LPC_IDR2        0x2c
#define LPC_ODR2        0x2d
#define LPC_STR2        0x2e
#define LPC_IDR3        0x20
#define LPC_ODR3        0x21
#define LPC_STR3        0x22

/* bidirectional data registers */
#define LPC_TWR0        0x10 
#define LPC_TWR1        0x11
#define LPC_TWR2        0x12
#define LPC_TWR3        0x13
#define LPC_TWR4        0x14
#define LPC_TWR5        0x15
#define LPC_TWR6        0x16
#define LPC_TWR7        0x17
#define LPC_TWR8        0x18
#define LPC_TWR9        0x19
#define LPC_TWR10       0x1a
#define LPC_TWR11       0x1b
#define LPC_TWR12       0x1c
#define LPC_TWR13       0x1d
#define LPC_TWR14       0x1e
#define LPC_TWR15       0x1f

/* serial IRQ control registers */
#define LPC_SIRQCR0     0x26 
#define LPC_SIRQCR1     0x27
#define LPC_SIRQCR2     0x34

/* host interface select register */
#define LPC_HISEL       0x2f

/* BT registers */
#define LPC_BTSR0       0x02
#define LPC_BTSR1       0x03
#define LPC_BTCSR0      0x04
#define LPC_BTCSR1      0x05
#define LPC_BTCR        0x06
#define LPC_BTIMSR      0x07
#define LPC_BTDTR       0x35
#define LPC_BTFVSR0     0x36
#define LPC_BTFVSR1     0x37

/* SMIC registers */
#define LPC_SMICFLG     0x08
#define LPC_SMICCSR     0x0a
#define LPC_SMICDTR     0x0b
#define LPC_SMICIR0     0x0c
#define LPC_SMICIR1     0x0e

/* host (master) mode registers */
#define LPC_HOST_STATUS     0x4A
#define LPC_HOST_CMD        0x49
#define LPC_HOST_ADDR0      0x47
#define LPC_HOST_ADDR1      0x46
#define LPC_HOST_ADDR2      0x45
#define LPC_HOST_ADDR3      0x44
#define LPC_HOST_DATA_IN    0x48
#define LPC_HOST_DATA_OUT   0x43

/* POST code snooping */
#define LPC_SNOOP_ADDRH     0x40
#define LPC_SNOOP_ADDRL     0x41
#define LPC_SNOOP_FIFO      0x42

/* UART emulation */
#define LPC_UART_BASE               0x50 // reg addr base of local UART
#define LPC_UART_HOST_BASE_ADDRH    0x60 // UART base address on LPC bus
#define LPC_UART_HOST_BASE_ADDRL    0x61
#define LPC_UART_SIRQ_MASKH         0x62 // UART SIRQ(s) used on host
#define LPC_UART_SIRQ_MASKL         0x63

/*
 * LPC bits
 */

/* bits of reg HICR0 */
#define LPC_LPC3E       ((uint8_t)(1 << 7))
#define LPC_LPC2E       ((uint8_t)(1 << 6))
#define LPC_LPC1E       ((uint8_t)(1 << 5))

/* bits of reg HICR1 */
#define LPC_LRSTB       ((uint8_t)(1 << 4))

/* bits of reg HICR2 */
#define LPC_LRST        ((uint8_t)(1 << 6))
#define LPC_SDWN        ((uint8_t)(1 << 5))
#define LPC_ABRT        ((uint8_t)(1 << 4))
#define LPC_IBFIE3      ((uint8_t)(1 << 3))
#define LPC_IBFIE2      ((uint8_t)(1 << 2))
#define LPC_IBFIE1      ((uint8_t)(1 << 1))
#define LPC_ERRIE       ((uint8_t)(1 << 0))

/* bits of reg HICR4 */
#define LPC_LADR12SEL   ((uint8_t)(1 << 7))
#define LPC_KCS1ENBL    ((uint8_t)(1 << 5))
#define LPC_KCS2ENBL    ((uint8_t)(1 << 4))
#define LPC_SW3ENBL     ((uint8_t)(1 << 3))
#define LPC_KCS3ENBL    ((uint8_t)(1 << 2))
#define LPC_SMIC3ENBL   ((uint8_t)(1 << 1))
#define LPC_BT3ENBL     ((uint8_t)(1 << 0))

/* bits of reg STR1 */
#define LPC_CD1         ((uint8_t)(1 << 3))
#define LPC_IBF1        ((uint8_t)(1 << 1))
#define LPC_OBF1        ((uint8_t)(1 << 0))

/* bits of reg STR2 */
#define LPC_CD2         ((uint8_t)(1 << 3))
#define LPC_IBF2        ((uint8_t)(1 << 1))
#define LPC_OBF2        ((uint8_t)(1 << 0))

/* bits of reg STR3 */
#define LPC_IBF3B       ((uint8_t)(1 << 7))
#define LPC_OBF3B       ((uint8_t)(1 << 6))
#define LPC_CD3         ((uint8_t)(1 << 3))
#define LPC_IBF3A       ((uint8_t)(1 << 1))
#define LPC_OBF3A       ((uint8_t)(1 << 0))

/* bits of reg HISEL */
#define LPC_SELSTR3     ((uint8_t)(1 << 7))

/* bits of reg BTSR0 */
#define LPC_FRDI        ((uint8_t)(1 << 4))
#define LPC_HRDI        ((uint8_t)(1 << 3))
#define LPC_HWRI        ((uint8_t)(1 << 2))
#define LPC_HBTWI       ((uint8_t)(1 << 1))
#define LPC_HBTRI       ((uint8_t)(1 << 0))
#define LPC_BTI_MASK0   ((uint8_t)0x1f)

/* bits of reg BTSR1 */
#define LPC_HRSTI       ((uint8_t)(1 << 6))
#define LPC_IRQCRI      ((uint8_t)(1 << 5))
#define LPC_BEVTI       ((uint8_t)(1 << 4))
#define LPC_B2HI        ((uint8_t)(1 << 3))
#define LPC_H2BI        ((uint8_t)(1 << 2))
#define LPC_CRRPI       ((uint8_t)(1 << 1))
#define LPC_CRWPI       ((uint8_t)(1 << 0))
#define LPC_BTI_MASK1   ((uint8_t)0x7f)

/* bits of reg BTCSR0 */
#define LPC_FSEL1       ((uint8_t)(1 << 6))
#define LPC_FSEL0       ((uint8_t)(1 << 5))
#define LPC_FRDIE       ((uint8_t)(1 << 4))
#define LPC_HRDIE       ((uint8_t)(1 << 3))
#define LPC_HWRIE       ((uint8_t)(1 << 2))
#define LPC_HBTWIE      ((uint8_t)(1 << 1))
#define LPC_HBTRIE      ((uint8_t)(1 << 0))

/* bits of reg BTCSR1 */
#define LPC_RSTRENBL    ((uint8_t)(1 << 7))
#define LPC_HRSTIE      ((uint8_t)(1 << 6))
#define LPC_IRQCRIE     ((uint8_t)(1 << 5))
#define LPC_BEVTIE      ((uint8_t)(1 << 4))
#define LPC_B2HIE       ((uint8_t)(1 << 3))
#define LPC_H2BIE       ((uint8_t)(1 << 2))
#define LPC_CRRPIE      ((uint8_t)(1 << 1))
#define LPC_CRWPIE      ((uint8_t)(1 << 0))

/* bits of reg BTCR */
#define LPC_B_BUSY      ((uint8_t)(1 << 7))
#define LPC_H_BUSY      ((uint8_t)(1 << 6))
#define LPC_BEVT_ATN    ((uint8_t)(1 << 4))
#define LPC_B2H_ATN     ((uint8_t)(1 << 3))
#define LPC_H2B_ATN     ((uint8_t)(1 << 2))
#define LPC_CLR_RD_PTR  ((uint8_t)(1 << 1))
#define LPC_CLR_WR_PTR  ((uint8_t)(1 << 0))

/* bits of reg LPC_SMICFLG */
#define LPC_RX_DATA_RDY ((uint8_t)(1 << 7))
#define LPC_TX_DATA_RDY ((uint8_t)(1 << 6))
#define LPC_SMI         ((uint8_t)(1 << 4))
#define LPC_SEVT_ATN    ((uint8_t)(1 << 3))
#define LPC_SMS_ATN     ((uint8_t)(1 << 2))
#define LPC_BUSY        ((uint8_t)(1 << 0))

/* bits of reg LPC_SMICIR0 */
#define LPC_HDTWI       ((uint8_t)(1 << 4))
#define LPC_HDTRI       ((uint8_t)(1 << 3))
#define LPC_STARI       ((uint8_t)(1 << 2))
#define LPC_CTLWI       ((uint8_t)(1 << 1))
#define LPC_BUSYI       ((uint8_t)(1 << 0))
#define LPC_SMICI_MASK  ((uint8_t)0x1f)

/* bits of reg LPC_SMICIR1 */
#define LPC_HDTWIE      ((uint8_t)(1 << 4))
#define LPC_HDTRIE      ((uint8_t)(1 << 3))
#define LPC_STARIE      ((uint8_t)(1 << 2))
#define LPC_CTLWIE      ((uint8_t)(1 << 1))
#define LPC_BUSYIE      ((uint8_t)(1 << 0))

/* bits of LPC_HOST_STATUS */
#define LPC_SNOOP_EMPTY     ((uint8_t)(1 << 7))
#define LPC_HOST_BUSY_VALID ((uint8_t)(1 << 2))
#define LPC_HOST_ERR        ((uint8_t)(1 << 1))
#define LPC_HOST_BUSY       ((uint8_t)(1 << 0))

/* bits of LPC_HOST_COMMAND */
#define LPC_HOST_ENABLE     ((uint8_t)(1 << 7))
#define LPC_HOST_GO         ((uint8_t)(1 << 5))
#define LPC_HOST_READ_CMD   ((uint8_t)(1 << 4))
#define LPC_HOST_WRITE_CMD  ((uint8_t)(0 << 4))

struct lpc_s;
typedef struct lpc_s lpc_t;

struct lpc_s {
    void *io_base;
    uint8_t (* get_reg)(lpc_t *, int);
    void (* set_reg)(lpc_t *, int, uint8_t);
    void (* set_block)(uint32_t *vals, lpc_t* fml_priv, uint32_t offset, int cnt);
    void (* get_block)(uint32_t *buf, lpc_t* fml_priv, uint32_t offset, int cnt);
    uint32_t kira_rev;
    
};

int lpc_init(void *io_base, lpc_t **plpc);
int lpc_cleanup(lpc_t *lpc);
int lpc_soft_reset(lpc_t *lpc);

static __inline unsigned char lpc_get_reg(lpc_t *lpc, int reg)
{ return lpc->get_reg(lpc, reg); }

static __inline void lpc_set_reg(lpc_t *lpc, int reg, uint8_t val)
{ lpc->set_reg(lpc, reg, val); }

static __inline unsigned char lpc_get_reg_bits(lpc_t *lpc, int reg, uint8_t mask)
{ return lpc->get_reg(lpc, reg) & mask; }

static __inline void lpc_set_reg_bits(lpc_t *lpc, int reg, uint8_t set_mask, uint8_t clr_mask)
{ lpc->set_reg(lpc, reg, (lpc->get_reg(lpc, reg) & ~clr_mask) | set_mask); }

#ifdef __arm__
#define LOCK_NET if (KIRA_MAJOR(dev->lpc->kira_rev) < 2) ahbfix_lock_net()
#define UNLOCK_NET if (KIRA_MAJOR(dev->lpc->kira_rev) < 2) ahbfix_unlock_net()
#else 
#define LOCK_NET
#define UNLOCK_NET
#endif

#endif /* !__LPC_CORE_H__ */
