/**
 * lpc_mod.h
 *
 * KIRA100 (slave) LPC kernel module
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

/*
 * Device file (major: 246, minor: don't care):
 * /dev/lpc c 246 0
*
 * Following modes can be used in channels:
 *          KCS  BT   SMIC
 * LPC1      *    -    -
 * LPC2      *    -    -
 * LPC3      *    *    *
 * (Each channel can be used independently and in parallel.)
 *
 * Snooping and Master Mode do not use any channels.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/version.h>

#include "lpc.h"
#include "lpc_core.h"
#include "lpc_kcs.h"
#include "lpc_bt.h"
#include "lpc_smic.h"
#include "lpc_snoop.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
# define MOD_DEC_USE_COUNT
# define MOD_INC_USE_COUNT
#endif

#ifdef __arm__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <asm/arch/cpe/peppercon.h>
#else
#include <asm/arch/platform/peppercon.h>
#endif
#include <asm/arch/net_lock.h>
#endif

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define SUCCESS 0

#define LPC_DMA_TIMEOUT		HZ

#define NAME                    "lpc"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

#define LPC_KCS_MAJOR ((uint8_t)246)

// PCI dev and INTMUX not used by KIRA
#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
#  define LPC_USE_PCI
#  define LPC_USE_INTMUX
#endif

#ifdef LPC_USE_PCI
#  define KIRA100_PCI_VENDOR_ID  0x1743
#  define KIRA100_PCI_DEVICE_ID  0x1003
#  define KIRA100_PCI_BAR_NO     1
#endif

/* KIRA100 IO addresses/bits */

uint32_t kira100_io_base = 0;
#define LPC_IO_SIZE             (0x40 * sizeof(uint32_t)) /* regs are dword alligned */

#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
#  define LPC_IO_BASE_OFFS        0x950000
#  define DEF_IRQ                 27
#endif

#ifdef PP_BOARD_KIRA
#  define LPC_IO_BASE_OFFS        CPE_LPC_BASE
#  define DEF_IRQ                 13
#endif

#ifdef LPC_USE_INTMUX
#  define INTMUX_IO_BASE_OFFS     0x9a0000
#  define INTMUX_IO_SIZE          (1 * sizeof(uint32_t)) /* 1 dword */
#  define LPC_INT     ((uint16_t)(1 << 11)) // interrupt flag bits (INTMUX register)
#endif

/* ------------------------------------------------------------------------- *
 * private data
 * ------------------------------------------------------------------------- */

 /* special protocol function table (entries may be NULL!) */
typedef struct {
    int (*cleanup)(void* priv);
    int (*stop)(void* priv);
    int (*read)(void* priv, uint8_t *buf, int size);
    int (*write)(void* priv, const uint8_t *buf, int size);
    int (*can_read)(void* priv);
    int (*can_write)(void* priv);
    int (*host_atn)(void* priv, int set);
    int (*event)(void* priv);
    void *priv;
    struct list_head isr_anchor;
} lpc_drv_t;

typedef struct {
    atomic_t            refcnt;
    void                *lpc_addr;
    void                *intmux_addr;
    int                 isr_registered;

    lpc_t               *lpc;

    wait_queue_head_t   wait;
    struct list_head    isr_list;

    lpc_ioctl_hk_t      hk;
} lpc_dev_t;

static lpc_dev_t lpc_dev;

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static int  lpc_mod_init(void);
static void lpc_mod_cleanup(void);
static int  lpc_mod_ioctl(struct inode *, struct file *, uint, ulong);
static int  lpc_mod_open(struct inode *, struct file *);
static int  lpc_mod_release(struct inode *, struct file *);
static int  lpc_mod_read(struct file *, char *, size_t, loff_t *);
static int  lpc_mod_write(struct file *, const char *, size_t, loff_t *);
static unsigned int lpc_mod_poll(struct file *, poll_table *);
static irqreturn_t lpc_mod_isr(int irq, void *dev_id, struct pt_regs *regs);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Ralf Guenther <rgue@peppercon.de>");
MODULE_DESCRIPTION("Linux kernel module for KIRA100 slave LPC interface");
MODULE_SUPPORTED_DEVICE("/dev/lpc");

module_init(lpc_mod_init);
module_exit(lpc_mod_cleanup);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations lpc_mod_ops;

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

int __init
lpc_mod_init(void)
{
    lpc_dev_t *dev = &lpc_dev;
    int ret = SUCCESS, r;
    uint32_t lpc_io_base;
#ifdef LPC_USE_INTMUX
    uint32_t intmux_io_base;
#endif

    /* init jump table */
    memset(&lpc_mod_ops, 0, sizeof(lpc_mod_ops));
    lpc_mod_ops.ioctl   = lpc_mod_ioctl;
    lpc_mod_ops.open    = lpc_mod_open;
    lpc_mod_ops.release = lpc_mod_release;
    lpc_mod_ops.read    = lpc_mod_read;
    lpc_mod_ops.write   = lpc_mod_write;
    lpc_mod_ops.poll    = lpc_mod_poll;
    lpc_mod_ops.owner   = THIS_MODULE;

    /* init dev's private data */
    memset(dev, 0, sizeof(lpc_dev_t));
    init_waitqueue_head(&dev->wait);
    INIT_LIST_HEAD(&dev->isr_list);
    atomic_set(&dev->refcnt, 0);
    
    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(LPC_KCS_MAJOR, NAME, &lpc_mod_ops)) < 0) {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto fail;
    }

#ifdef LPC_USE_PCI
    {
        /* find hw */
        struct pci_dev *pci;
        pci = pci_find_device(KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID, NULL);
        if (pci == NULL) {
            ERR("failed to find PCI device (%04x:%04x) containing LPC core",
                KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID);
            ret = -ENODEV;
            goto fail;
        }
        kira100_io_base = pci_resource_start(pci, KIRA100_PCI_BAR_NO);
    }
#endif

    lpc_io_base = kira100_io_base + LPC_IO_BASE_OFFS;
    if ((dev->lpc_addr = ioremap_nocache(lpc_io_base, LPC_IO_SIZE)) == NULL) {
        ERR("failed to map LPC IO space %08x-%08x to memory",
            lpc_io_base, lpc_io_base + LPC_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }

#ifdef LPC_USE_INTMUX
    intmux_io_base = kira100_io_base + INTMUX_IO_BASE_OFFS;
    if ((dev->intmux_addr = ioremap_nocache(intmux_io_base, INTMUX_IO_SIZE)) == NULL) {
        ERR("failed to map INTMUX IO space %08x-%08x to memory",
            intmux_io_base, intmux_io_base + INTMUX_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }
#endif

    /* check/request mem */
    if ((r = check_mem_region(lpc_io_base, LPC_IO_SIZE)) < 0) {
        ERR("LPC: IO space %08x-%08x already in use (err %d)",
            lpc_io_base, lpc_io_base + LPC_IO_SIZE - 1, r);
        ret = -EBUSY;
        goto fail;
    }
    request_mem_region(lpc_io_base, LPC_IO_SIZE, NAME);
    
    /* check/init LPC core */
    if ((r = lpc_init(dev->lpc_addr, &dev->lpc)) < 0) {
        ERR("LPC: failed to init LPC - core not present? (err %d)", r);
        ret = -EIO;
        goto fail;
    }

    /* request IRQ */
    if ((r = request_irq(DEF_IRQ, lpc_mod_isr, SA_SHIRQ, NAME, dev)) < 0) {
        ERR("LPC: failed to request irq %d (err %d)", DEF_IRQ, r);
        ret = -EBUSY;
        goto fail;
    }
    dev->isr_registered = 1;

#ifdef LPC_USE_INTMUX
    INFO("successfully loaded. (io: %08x, intmux: %08x, irq: %d)",
        lpc_io_base, intmux_io_base, DEF_IRQ);
#else // !LPC_USE_INTMUX
    INFO("successfully loaded. (io: %08x, irq: %d)",
        lpc_io_base, DEF_IRQ);
#endif


    return ret;
fail:
    lpc_mod_cleanup();
    return ret;
}

static void
lpc_mod_cleanup(void)
{
    lpc_dev_t *dev = &lpc_dev;
    int r;
    uint32_t lpc_io_base = kira100_io_base + LPC_IO_BASE_OFFS;

    if (dev->isr_registered) free_irq(DEF_IRQ, dev); dev->isr_registered = 0;
    if (dev->lpc) lpc_cleanup(dev->lpc); dev->lpc = NULL;
    release_mem_region(lpc_io_base, LPC_IO_SIZE);

    if ((r = unregister_chrdev(LPC_KCS_MAJOR, NAME)) < 0) {
        ERR("failed to unregister device (err %d)", r);
    }
#ifdef LPC_USE_INTMUX
    if (dev->intmux_addr) iounmap(dev->intmux_addr); dev->intmux_addr = NULL;
#endif
    if (dev->lpc_addr) iounmap(dev->lpc_addr); dev->lpc_addr = NULL;
}

/* ------------------------------------------------------------------------- *
 * the driver operations
 * ------------------------------------------------------------------------- */

static int
lpc_mod_open(struct inode *inode, struct file *file)
{
    lpc_drv_t *drv = (lpc_drv_t*)kmalloc(sizeof(lpc_drv_t), GFP_KERNEL);
    memset(drv, 0, sizeof(lpc_drv_t));
    INIT_LIST_HEAD(&drv->isr_anchor);
    file->private_data = drv;

    /* do nothing else, LPC_IOCTL_INIT does any neccessary initialization */

    DBG("LPC: opened");

    MOD_INC_USE_COUNT;
    
    return SUCCESS;
}

static int
lpc_mod_release(struct inode *inode, struct file *file)
{
    lpc_drv_t *drv = file->private_data;

    if (drv->stop) drv->stop(drv->priv);
    if (!list_empty(&drv->isr_anchor)) list_del(&drv->isr_anchor);
    if (drv->cleanup) drv->cleanup(drv->priv);
    file->private_data = NULL;
    kfree(drv);

    DBG("LPC: closed");
    MOD_DEC_USE_COUNT;
    return SUCCESS;
}

static int
lpc_mod_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
    lpc_dev_t *dev = &lpc_dev;
    lpc_drv_t *drv = file->private_data;

    switch (cmd) {
    case LPC_IOCTL_INIT:
    {
        int r;
        lpc_ioctl_init_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;

        /* priv exists: driver must already be initialized */
        if (drv->priv) return -EBUSY;

        switch (d.mode) {
        case LPC_MODE_KCS:
            if ((r = kcs_init(dev->lpc, d.chan, d.host_port, &dev->wait, &drv->priv)) < 0) {
                ERR("LPC%d: failed to init KCS mode (err %d)", d.chan, r);
                return r;
            }

            drv->cleanup   = kcs_cleanup;
            drv->stop      = kcs_stop;
            drv->read      = kcs_read;
            drv->write     = kcs_write;
            drv->can_read  = kcs_can_read;
            drv->can_write = kcs_can_write;
            drv->host_atn  = kcs_host_atn;
            drv->event     = kcs_event;
            list_add(&drv->isr_anchor, &dev->isr_list);
            kcs_start(drv->priv);

            INFO("LPC%d: mode KCS initialized successfully. (host base: 0x%x)",
                 d.chan, d.host_port);
            return SUCCESS;
        case LPC_MODE_BT:
            if ((r = bt_init(dev->lpc, d.chan, d.host_port, &dev->wait, &drv->priv)) < 0) {
                ERR("LPC%d: failed to init BT mode (err %d)", d.chan, r);
                return r;
            }

            drv->cleanup   = bt_cleanup;
            drv->stop      = bt_stop;
            drv->read      = bt_read;
            drv->write     = bt_write;
            drv->can_read  = bt_can_read;
            drv->can_write = bt_can_write;
            drv->host_atn  = bt_host_atn;
            drv->event     = bt_event;
            list_add(&drv->isr_anchor, &dev->isr_list);
            bt_start(drv->priv);

            INFO("LPC%d: mode BT initialized successfully. (host base: 0x%x)",
                 d.chan, d.host_port);
            return SUCCESS;
        case LPC_MODE_SMIC:
            if ((r = smic_init(dev->lpc, d.chan, d.host_port, &dev->wait, &drv->priv)) < 0) {
                ERR("LPC%d: failed to init SMIC mode (err %d)", d.chan, r);
                return r;
            }

            drv->cleanup   = smic_cleanup;
            drv->stop      = smic_stop;
            drv->read      = smic_read;
            drv->write     = smic_write;
            drv->can_read  = smic_can_read;
            drv->can_write = smic_can_write;
            drv->host_atn  = smic_host_atn;
            drv->event     = smic_event;
            list_add(&drv->isr_anchor, &dev->isr_list);
            smic_start(drv->priv);

            INFO("LPC%d: mode SMIC initialized successfully. (host base: 0x%x)",
                 d.chan, d.host_port);
            return SUCCESS;
        case LPC_MODE_SNOOP:
            if ((r = snoop_init(dev->lpc, d.host_port, &drv->priv)) < 0) {
                ERR("LPC: failed to init Snoop mode (err %d)", r);
                return r;
            }

            drv->cleanup   = snoop_cleanup;
            drv->read      = snoop_read;
            drv->can_read  = snoop_can_read;

            INFO("LPC: mode Snoop initialized successfully. (snooped host io port: 0x%x)",
                 d.host_port);
            return SUCCESS;
        default:
            return -EINVAL;
        }
    }
    case LPC_IOCTL_SET_HOST_ATN:
    {
        return drv->host_atn ? drv->host_atn(drv->priv, 1) : -EFAULT;
    }
    case LPC_IOCTL_CLR_HOST_ATN:
    {
        return drv->host_atn ? drv->host_atn(drv->priv, 0) : -EFAULT;
    }
    case LPC_IOCTL_MEM_READ:
    {
        lpc_ioctl_mem_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;

        // program addr
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR0, (d.addr >>  0) & 0xff);
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR1, (d.addr >>  8) & 0xff);
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR2, (d.addr >> 16) & 0xff);
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR3, (d.addr >> 24) & 0xff);

        // enable master and start reading
        lpc_set_reg(dev->lpc, LPC_HOST_CMD, 0);
        lpc_set_reg(dev->lpc, LPC_HOST_CMD, LPC_HOST_ENABLE | LPC_HOST_GO | LPC_HOST_READ_CMD);

        // wait for command completion
        while (!(lpc_get_reg(dev->lpc, LPC_HOST_STATUS) & LPC_HOST_BUSY_VALID)) udelay(50);
        while (lpc_get_reg(dev->lpc, LPC_HOST_STATUS) & LPC_HOST_BUSY) udelay(50);

        // check for error
        if (lpc_get_reg(dev->lpc, LPC_HOST_STATUS) & LPC_HOST_ERR) {
            lpc_set_reg(dev->lpc, LPC_HOST_CMD, 0);
            return -EIO;
        }

        // get val
        d.val = lpc_get_reg(dev->lpc, LPC_HOST_DATA_OUT);

        // disable master mode
        lpc_set_reg(dev->lpc, LPC_HOST_CMD, 0);

        if (copy_to_user((void*)arg, &d, sizeof(d))) return -EFAULT;
        return SUCCESS;
    }
    case LPC_IOCTL_MEM_WRITE:
    {
        lpc_ioctl_mem_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;

        // program addr
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR0, (d.addr >>  0) & 0xff);
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR1, (d.addr >>  8) & 0xff);
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR2, (d.addr >> 16) & 0xff);
        lpc_set_reg(dev->lpc, LPC_HOST_ADDR3, (d.addr >> 24) & 0xff);

        // program value
        lpc_set_reg(dev->lpc, LPC_HOST_DATA_IN, d.val);

        // enable master and start reading
        lpc_set_reg(dev->lpc, LPC_HOST_CMD, 0);
        lpc_set_reg(dev->lpc, LPC_HOST_CMD, LPC_HOST_ENABLE | LPC_HOST_GO | LPC_HOST_WRITE_CMD);

        // wait for command completion
        while (!(lpc_get_reg(dev->lpc, LPC_HOST_STATUS) & LPC_HOST_BUSY_VALID)) udelay(50);
        while (lpc_get_reg(dev->lpc, LPC_HOST_STATUS) & LPC_HOST_BUSY) udelay(50);

        // check for error
        if (lpc_get_reg(dev->lpc, LPC_HOST_STATUS) & LPC_HOST_ERR) {
            lpc_set_reg(dev->lpc, LPC_HOST_CMD, 0);
            return -EIO;
        }

        // disable master mode
        lpc_set_reg(dev->lpc, LPC_HOST_CMD, 0);

        return SUCCESS;
    }
    case LPC_IOCTL_SET_UART_CFG:
    {
        lpc_ioctl_uart_cfg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;

        if (d.irq > 15) return -EINVAL;

        // program port
        lpc_set_reg(dev->lpc, LPC_UART_HOST_BASE_ADDRL, (d.port >>  0) & 0xff);
        lpc_set_reg(dev->lpc, LPC_UART_HOST_BASE_ADDRH, (d.port >>  8) & 0xff);

        // program irq
        {
            /* bit0: general enable IRQ - for d.irq==0 (no irq) bit0 is set to 0 */
            unsigned short irq = (1 << d.irq) ^ 1;
            lpc_set_reg(dev->lpc, LPC_UART_SIRQ_MASKL, (irq >>  0) & 0xff);
            lpc_set_reg(dev->lpc, LPC_UART_SIRQ_MASKH, (irq >>  8) & 0xff);
        }

        return SUCCESS;
    }
    case LPC_IOCTL_GET_REG:
    {
        lpc_ioctl_reg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        d.val = lpc_get_reg(dev->lpc, d.reg);
        if (copy_to_user((void*)arg, &d, sizeof(d))) return -EFAULT;
        return SUCCESS;
    }
    case LPC_IOCTL_SET_REG:
    {
        lpc_ioctl_reg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        lpc_set_reg(dev->lpc, d.reg, d.val);
        return SUCCESS;
    }
#ifdef __arm__    
    case LPC_IOCTL_LOCK_NET:
    {
        int lock;
        if (copy_from_user(&lock, (void*)arg, sizeof(lock))) return -EFAULT;
        if (lock) ahbfix_lock_net();
        else ahbfix_unlock_net();
        return SUCCESS;
    }
#endif    
    case LPC_IOCTL_GET_HK:
    {
        if (copy_to_user((void*)arg, &dev->hk, sizeof(lpc_ioctl_hk_t))) return -EFAULT;
        return SUCCESS;
    }
    default:
        /* invalid cmd */
        return -EINVAL;
    }
}

static int
lpc_mod_read(struct file* file, char *buffer, size_t size, loff_t *offset)
{
    lpc_drv_t *drv = file->private_data;
    return drv->read ? drv->read(drv->priv, buffer, size) : -EINVAL;
}

static int
lpc_mod_write(struct file* file, const char *buffer, size_t size, loff_t *offset)
{
    lpc_drv_t *drv = file->private_data;
    return drv->write ? drv->write(drv->priv, buffer, size) : -EINVAL;
}

static unsigned int
lpc_mod_poll(struct file *file, poll_table *wait)
{
    lpc_dev_t *dev = &lpc_dev;
    lpc_drv_t *drv = file->private_data;

    unsigned int mask = 0;
    poll_wait(file, &dev->wait, wait);
    if (drv->can_read && drv->can_read(drv->priv)) mask |= POLLIN | POLLRDNORM;
    if (drv->can_write && drv->can_write(drv->priv)) mask |= POLLOUT | POLLWRNORM;
    return mask;
}

/* ------------------------------------------------------------------------- *
 * interrupt service routine
 * ------------------------------------------------------------------------- */

static irqreturn_t
lpc_mod_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    lpc_dev_t *dev = &lpc_dev;
    struct list_head *ptr;

    if (!dev->lpc) return IRQ_NONE; /* not yet init, irq is not for us */

    /* statistics */
    dev->hk.intr_cnt++;
#ifdef LPC_USE_INTMUX
    {
        uint32_t intmux = readl(dev->intmux_addr);
        if (intmux & LPC_INT) dev->hk.lpc_intr_cnt++;
    }
#endif

    LOCK_NET;
    
    /* iterate registered sinks */
    for (ptr = dev->isr_list.next; ptr != &dev->isr_list; ptr = ptr->next) {
        lpc_drv_t *drv = list_entry(ptr, lpc_drv_t, isr_anchor);
        if (drv->event) drv->event(drv->priv);
    }

    UNLOCK_NET;

    return IRQ_HANDLED;
}
