/**
 * pwmtach_core_faraday.c
 *
 * Low-level driver for the Faraday/KIRA100 PWM/Tacho core
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 *
 */

#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/vmalloc.h>
#include <linux/version.h>
#include <asm/io.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/arch-cpe/cpe/kira100.h>
# include <asm/arch-cpe/cpe/peppercon.h>
#else
# include <asm/arch/platform-a320/kira100.h>
# include <asm/arch/platform-a320/peppercon.h>
#endif
#include <asm/arch/cpe_clk.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/arch/cpe_bd.h>
#else
# include <asm/arch/pp_bd.h>
#endif

#include "pwmtach_common.h"
#include "pwmtach_core_faraday.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

/* for some reason beyond imagination the least significant nibble of the
   tacho control register (Offset 0x00) can't be read on Kira 100-R02 */
#define KIRA_R02_WORKAROUND

#define PWM_IO_SIZE		0x7C
#define TACHO_IO_SIZE		0x50

#define PWM_COUNT		8
#define TACHO_COUNT		8

#define PWMTACH_FARADAY_NAME	"faraday"
#define MAX_PERIOD		1024
#define BASE_CLOCK_HZ		12288000
#define PWM_CLOCK_DIVIDER	1	  /* original clock is 12.288 MHz */
#define MAX_PRESCALE		64
#define MIN_PRESCALE		1
#define MIN_FREQ		(BASE_CLOCK_HZ / (MAX_PRESCALE * MAX_PERIOD))
#define MAX_FREQ		(BASE_CLOCK_HZ / (MIN_PRESCALE * 2))
#define PWM_DUTY_CYCLE_DEFAULT	50	  /* in % */
#define TACHO_ACCU_PERIOD	0x02000000

/* PWM registers */
#define REG_PWM_CONTROL		0x00
#define REG_PWM_DUTY		0x01
#define REG_PWM_PERIOD		0x02

#define REGS_PER_PWM		4
#define PWM_REG(index, reg)	((index) * REGS_PER_PWM + (reg))

/* PWM register bits */
#define PWM_CONTROL_PRESCALE	0x0000003F
#define PWM_DUTY_FULLDUTY	0x00000400
#define PWM_PERIOD_DUTY_MASK	0x000003FF

/* Tacho registers */
#define REG_TACHO_GLOBAL_CONTROL	0x00
#define REG_TACHO_GLOBAL_TIMER		0x01
#define REG_TACHO_GLOBAL_STATUS		0x02

#define REG_TACHO_COUNTER	0x00
#define REG_TACHO_ALERT		0x01
#define REG_TACHO_CHANNEL_BASE	0x04
#define REGS_PER_TACHO		2
#define TACHO_REG(index, reg)	(REG_TACHO_CHANNEL_BASE + (index) * REGS_PER_TACHO + (reg))

/* Tacho register bits */
#define TACHO_CONTROL_ENABLE_BIT(index)	(0x01 << (index))
#define TACHO_CONTROL_ALERT_BIT(index)	(0x01 << (16 + (index)))
#define TACHO_STATUS_OFLOW_BIT(index)	(0x01 << (index))

/* Pins multiplexed in PMU with GPIO 16-31 */
#define PWM_MUX_BIT(index)	(0x01 << (16 + (index)))
#define TACHO_MUX_BIT(index)	(0x01 << (24 + (index)))

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    uint32_t		pwm_io_base;
    uint32_t		tacho_io_base;
    volatile uint32_t  *pwm_regs;
    volatile uint32_t  *tacho_regs;
    int			isr_registered;
    int			pmu_mux_pwmtach_is_1;
    u_int		duty_cycle[PWM_COUNT];
    
    /* shadow registers as they cannot be read back completely from core */
    u_int		shadow_period[PWM_COUNT];
    u_int		shadow_prescale[PWM_COUNT];

#ifdef KIRA_R02_WORKAROUND
    /* control register least significant nibble */
    uint8_t		tc_low;
#endif
} ft_data_t;


/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

ft_data_t ft_data[] = {
    { .pwm_io_base		= CPE_PWM_BASE,
      .tacho_io_base		= CPE_TACHO_BASE,
      .pwm_regs			= NULL,
      .tacho_regs		= NULL,
      .isr_registered		= 0,
      .pmu_mux_pwmtach_is_1	= 0,
#ifdef KIRA_R02_WORKAROUND
      .tc_low			= 0
#endif
    },
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void cleanup(pwmtach_core_t *core);


static int tach_get_rpm(struct pwmtach_core_s *core, uint8_t index,
			uint8_t pulses_per_rev, uint16_t *revs_per_minute);   
static int pwm_set_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int freq);
static int pwm_get_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int *freq);
static int pwm_set_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t duty_cycle);
static int pwm_get_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t *duty_cycle);

static u_char pwm_get_alternate(struct pwmtach_core_s *core, uint8_t index);
static void pwm_set_alternate(struct pwmtach_core_s *core, uint8_t index, u_char alternate);
static u_char tach_get_alternate(struct pwmtach_core_s *core, uint8_t index);
static void tach_set_alternate(struct pwmtach_core_s *core, uint8_t index, u_char alternate);

static void tach_activate_channel(struct pwmtach_core_s *core, uint8_t index);

static int check_multiplexer(ft_data_t *data, u_long mask);


/* ------------------------------------------------------------------------- *
 * register access inline functions
 * ------------------------------------------------------------------------- */

static __inline uint32_t get_pwm_reg(ft_data_t *data, int reg)
{ return readl(&data->pwm_regs[reg]); }
static __inline void set_pwm_reg(ft_data_t *data, int reg, uint32_t val)
{ writel(val, &data->pwm_regs[reg]); }
static __inline uint32_t get_pwm_reg_bits(ft_data_t *data, int reg, uint32_t mask)
{ return get_pwm_reg(data, reg) & mask; }
static __inline void set_pwm_reg_bits(ft_data_t *data, int reg, uint32_t set_mask,
				      uint32_t clr_mask)
{ set_pwm_reg(data, reg, (get_pwm_reg(data, reg) & ~clr_mask) | set_mask); }

#ifdef KIRA_R02_WORKAROUND

static __inline uint32_t get_tacho_reg(ft_data_t *data, int reg)
{
    uint32_t val = readl(&data->tacho_regs[reg]);
    if (reg == 0) val = (val & 0xfffffff0) | data->tc_low;
    return val;
}
static __inline void set_tacho_reg(ft_data_t *data, int reg, uint32_t val)
{
    writel(val, &data->tacho_regs[reg]);
    if (reg == 0) data->tc_low = val & 0x0f;
}

#else

static __inline uint32_t get_tacho_reg(ft_data_t *data, int reg)
{ return readl(&data->tacho_regs[reg]); }
static __inline void set_tacho_reg(ft_data_t *data, int reg, uint32_t val)
{ writel(val, &data->tacho_regs[reg]); }

#endif

static __inline uint32_t get_tacho_reg_bits(ft_data_t *data, int reg, uint32_t mask)
{ return get_tacho_reg(data, reg) & mask; }
static __inline void set_tacho_reg_bits(ft_data_t *data, int reg, uint32_t set_mask,
					uint32_t clr_mask)
{ set_tacho_reg(data, reg, (get_tacho_reg(data, reg) & ~clr_mask) | set_mask); }

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */

int
pwmtach_core_faraday_init(pwmtach_core_t *core, u_char core_nr)
{
    int ret = -1, i;
    int nr_cores = sizeof(ft_data) / sizeof(ft_data_t);
    ft_data_t *pdata;
    uint32_t kira_version;
    
    if (core_nr >= nr_cores) {
	ERR("Unsupported core nr %d (max is %d)", core_nr, nr_cores - 1);	
	return ret;
    }

    core->private_data = pdata = &ft_data[core_nr];
    core->cleanup      = cleanup;

    if (request_mem_region(pdata->pwm_io_base,
			   PWM_IO_SIZE, NAME) == NULL) {
	
	ERR("memory region @ 0x%08x (size %d) already in use",
	    pdata->pwm_io_base, PWM_IO_SIZE);
	
	goto bail;
    }

    if (request_mem_region(pdata->tacho_io_base,
			   TACHO_IO_SIZE, NAME) == NULL) {
	
	ERR("memory region @ 0x%08x (size %d) already in use",
	    pdata->tacho_io_base, TACHO_IO_SIZE);
	
	goto bail;
    }
    
    if ((pdata->pwm_regs = ioremap_nocache(pdata->pwm_io_base,
					   PWM_IO_SIZE)) == NULL) {

	ERR("could not remap GPIO regs @ 0x%08x (size %d)",
	    pdata->pwm_io_base, PWM_IO_SIZE);
	goto bail;	
    }

    if ((pdata->tacho_regs = ioremap_nocache(pdata->tacho_io_base,
					     TACHO_IO_SIZE)) == NULL) {

	ERR("could not remap GPIO regs @ 0x%08x (size %d)",
	    pdata->tacho_io_base, TACHO_IO_SIZE);
	goto bail;	
    }
    
    core->caps = PWMTACH_CAPS_HAVE_PWM | PWMTACH_CAPS_HAVE_TACHO | PWMTACH_CAPS_PWM_FREQUENCY_SEPARATE;
    core->pwm_count = PWM_COUNT;
    core->tacho_count = TACHO_COUNT;
    strcpy(core->name, PWMTACH_FARADAY_NAME);
    
    core->tach_get_rpm = tach_get_rpm;
    core->pwm_set_freq = pwm_set_freq;
    core->pwm_get_freq = pwm_get_freq;
    core->pwm_set_duty = pwm_set_duty;
    core->pwm_get_duty = pwm_get_duty;

    core->tach_activate_channel = tach_activate_channel;

    core->pwm_get_alternate	= pwm_get_alternate;
    core->pwm_set_alternate	= pwm_set_alternate;
    core->tach_get_alternate	= tach_get_alternate;
    core->tach_set_alternate	= tach_set_alternate;

    /* get KIRA version and check how the PMU mux behaves */
    kira_version = pp_kira_get_revision();
    if (KIRA_MAJOR(kira_version) >= 2) {
	DBG("Found KIRA R02 or higher, swapping PMU mux values");
	pdata->pmu_mux_pwmtach_is_1 = 1;
    }
   
    /* initialize duty cycle to default */
    for (i = 0; i < PWM_COUNT; i++) {
	pdata->duty_cycle[i] = PWM_DUTY_CYCLE_DEFAULT;
    }
    
    /* set clock divider in PMU */
    pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_MISC,
			     (PWM_CLOCK_DIVIDER - 1) << KIRA_PMU_REG_MISC_PWMCLKDIV_SHIFT,
			     KIRA_PMU_REG_MISC_PWMCLKDIV);
    
    /* enable tachos and set acculmulation period */
    for (i = 0; i < TACHO_COUNT; i++) {
	set_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL,
		      get_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL) | TACHO_CONTROL_ENABLE_BIT(i));
    }    
    set_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL, 0x0);
    set_tacho_reg(pdata, REG_TACHO_GLOBAL_TIMER, TACHO_ACCU_PERIOD);

    
    ret = SUCCESS;
    INFO("PWM/Tacho core '%s' (%d) initialized", core->name, core_nr);

 bail:
    if (ret != SUCCESS) {
	cleanup(core);
    }
    return ret;    
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
cleanup(pwmtach_core_t *core)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    int i;
    
    if (pdata == NULL) {	
	return; // we were never really initialized
    }

    /* disable tachos */
    for (i = 0; i < TACHO_COUNT; i++) {
	set_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL,
		      get_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL) & ~TACHO_CONTROL_ENABLE_BIT(i));
    }

    if (pdata->pwm_regs != NULL) {
	iounmap((void*)pdata->pwm_regs);
	pdata->pwm_regs = NULL;
    }

    if (pdata->tacho_regs != NULL) {
	iounmap((void*)pdata->tacho_regs);
	pdata->tacho_regs = NULL;
    }

    release_mem_region(pdata->pwm_io_base, PWM_IO_SIZE);
    release_mem_region(pdata->tacho_io_base, TACHO_IO_SIZE);

    INFO("PWM/Tacho core '%s' cleanup done", core->name);    
}

static int
tach_get_rpm(struct pwmtach_core_s *core, uint8_t index,
	     uint8_t pulses_per_rev, uint16_t *revs_per_minute)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    bd_arm_t * bd = (bd_arm_t *)__res;
    u32 divisor, tacho_value, accu_period;

    if (revs_per_minute == NULL) return -1;
    
    if (check_multiplexer(data, TACHO_MUX_BIT(index)) != 0) {
	*revs_per_minute = 0;
	goto bail;
    }
    
    if (get_tacho_reg(data, REG_TACHO_GLOBAL_STATUS) & TACHO_STATUS_OFLOW_BIT(index)) {
	DBG("Tacho Overflow");
	*revs_per_minute = 0;
    } else {
#define TACH_SCALE 1000
	accu_period = get_tacho_reg(data, REG_TACHO_GLOBAL_TIMER) / TACH_SCALE;
	tacho_value = get_tacho_reg(data, TACHO_REG(index, REG_TACHO_COUNTER));
	/* HACK: for an unknown reason tacho channels with no input have a reading of 1 sometimes.
	   have to figure our why */
	if (tacho_value == 0 || tacho_value == 1) {
	    *revs_per_minute = 0;
	    goto bail;
	}

	divisor = accu_period * pulses_per_rev;
	*revs_per_minute = (BD_APB_CLK(bd) / TACH_SCALE * tacho_value + divisor / 2) / divisor * 60;

	DBG("APB_CLK=%u, divisor=%u, pulses=%u, value=%u, Accu=%u, RPM=%u", BD_APB_CLK(bd),
	    divisor, pulses_per_rev, tacho_value, accu_period, *revs_per_minute);
    }    

 bail:
    return 0;
}

static int
pwm_set_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int freq)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    u_int prescale;
    u_int period;
    u_int scaled_freq;
    int ret = -1;

    check_multiplexer(data, PWM_MUX_BIT(index));
    
    if (freq <= MIN_FREQ || freq > MAX_FREQ) {
	ERR("Frequency out of range (min %u, max %u)", MIN_FREQ, MAX_FREQ);
	goto bail;
    }

    ret = 0;
	
    if (freq == 0) {
	// special case, disable PWM
	set_pwm_reg(data, PWM_REG(index, REG_PWM_PERIOD), 0);
	goto bail;
    }
    
    prescale = (BASE_CLOCK_HZ + (freq * MAX_PERIOD - 1)) / (freq * MAX_PERIOD);
    if (prescale == 0) {
	prescale = 1;
    }
    
    scaled_freq = (BASE_CLOCK_HZ + prescale/2) / prescale;
    period      = (scaled_freq + freq/2) / freq;
   
    DBG("Wanted Freq=%u, ScaledFreq=%u, Prescale=%u, Period=%u\n", freq, scaled_freq,
	prescale, period);

    data->shadow_prescale[index] = prescale - 1;
    data->shadow_period[index]   = (period - 1) & PWM_PERIOD_DUTY_MASK;
    
    set_pwm_reg(data, PWM_REG(index, REG_PWM_CONTROL),
		(get_pwm_reg(data, PWM_REG(index, REG_PWM_CONTROL)) & ~PWM_CONTROL_PRESCALE) | data->shadow_prescale[index]);
    set_pwm_reg(data, PWM_REG(index, REG_PWM_PERIOD), data->shadow_period[index]);
   
    // recalculate duty cycle for new frequency
    pwm_set_duty(core, index, data->duty_cycle[index]);

    ret = 0;
    
 bail:
    return ret;
}

static int
pwm_get_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int *freq)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    u_int current_freq;
    
    check_multiplexer(data, PWM_MUX_BIT(index));

    if (data->shadow_period[index] == 0) {
	// special case (disabled PWM), return 0;
	if (freq) *freq = 0;
	goto bail;
    }
    
    current_freq = (BASE_CLOCK_HZ + (data->shadow_period[index] * data->shadow_prescale[index]) / 2) /
	(data->shadow_period[index] * data->shadow_prescale[index]);
   
    if (freq) *freq = current_freq;

 bail:
    return 0;
}

static int
pwm_set_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t duty_cycle)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    u_int real_duty;

    check_multiplexer(data, PWM_MUX_BIT(index));

    /* special cases, duty cycle 0 and 100 */
    switch (duty_cycle) {
      case 0:
	  set_pwm_reg(data, PWM_REG(index, REG_PWM_PERIOD), 0);
	  set_pwm_reg(data, PWM_REG(index, REG_PWM_DUTY), 0);
	  goto bail;
      case 100:
	  set_pwm_reg(data, PWM_REG(index, REG_PWM_PERIOD), 0);
	  set_pwm_reg(data, PWM_REG(index, REG_PWM_DUTY), PWM_DUTY_FULLDUTY);
	  goto bail;
    }
    
    real_duty = (duty_cycle * data->shadow_period[index] + 100 / 2) / 100;
    
    DBG("Wanted Duty=%u, Period=%u, Duty-Period=%u\n", duty_cycle,
	data->shadow_period[index], real_duty);
    
    data->duty_cycle[index] = duty_cycle;
    
    set_pwm_reg(data, PWM_REG(index, REG_PWM_DUTY), real_duty & PWM_PERIOD_DUTY_MASK);

 bail:
    return 0;
}

static int
pwm_get_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t *duty_cycle)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    u_int period;
    u_int duty, real_duty;

    check_multiplexer(data, PWM_MUX_BIT(index));
    
    period    = get_pwm_reg(data, PWM_REG(index, REG_PWM_PERIOD)) & PWM_PERIOD_DUTY_MASK;
    real_duty = get_pwm_reg(data, PWM_REG(index, REG_PWM_DUTY)) & PWM_PERIOD_DUTY_MASK;

    duty = (real_duty * 100 + period / 2) / period;
    
    if (*duty_cycle) *duty_cycle = duty;

    return 0;
}

static u_char
pwm_get_alternate(struct pwmtach_core_s *core, uint8_t index)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    uint32_t reg;

    reg = pp_kira_get_pmu_reg(KIRA_PMU_REG_PORTMUX);

    if (data->pmu_mux_pwmtach_is_1) {
	return (reg & PWM_MUX_BIT(index)) ? 0 : 1;
    } else {
	return (reg & PWM_MUX_BIT(index)) ? 1 : 0;
    }
}


static void
pwm_set_alternate(struct pwmtach_core_s *core, uint8_t index, u_char alternate)
{
    ft_data_t *data = (ft_data_t*) core->private_data;

    if (data->pmu_mux_pwmtach_is_1) {
	pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_PORTMUX,
				 alternate ? 0 : PWM_MUX_BIT(index),
				 alternate ? PWM_MUX_BIT(index) : 0);
    } else {
	pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_PORTMUX,
				 alternate ? PWM_MUX_BIT(index) : 0,
				 alternate ? 0 : PWM_MUX_BIT(index));
    }
}

static u_char
tach_get_alternate(struct pwmtach_core_s *core, uint8_t index)
{
    ft_data_t *data = (ft_data_t*) core->private_data;
    uint32_t reg;

    reg = pp_kira_get_pmu_reg(KIRA_PMU_REG_PORTMUX);

    if (data->pmu_mux_pwmtach_is_1) {
	return (reg & TACHO_MUX_BIT(index)) ? 0 : 1;
    } else {
	return (reg & TACHO_MUX_BIT(index)) ? 1 : 0;
    }
}

static void
tach_set_alternate(struct pwmtach_core_s *core, uint8_t index, u_char alternate)
{
    ft_data_t *data = (ft_data_t*) core->private_data;

    if (data->pmu_mux_pwmtach_is_1) {
	pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_PORTMUX,
				 alternate ? 0 : TACHO_MUX_BIT(index),
				 alternate ? TACHO_MUX_BIT(index) : 0);
    } else {
	pp_kira_set_pmu_reg_bits(KIRA_PMU_REG_PORTMUX,
				 alternate ? TACHO_MUX_BIT(index) : 0,
				 alternate ? 0 : TACHO_MUX_BIT(index));
    }
}

static void
tach_activate_channel(struct pwmtach_core_s *core, uint8_t index)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;

    set_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL,
		      get_tacho_reg(pdata, REG_TACHO_GLOBAL_CONTROL) | TACHO_CONTROL_ENABLE_BIT(index));
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */

static int
check_multiplexer(ft_data_t *data, u_long mask)
{
    uint32_t reg;
    
    if (mask == 0) return 0;

    reg = pp_kira_get_pmu_reg(KIRA_PMU_REG_PORTMUX);

    if (data->pmu_mux_pwmtach_is_1) {
	reg = ~reg;
    }
    
    /* portmux bit 1 -> gpio, 0 -> pwm/tacho */
    if ((reg & mask) != 0) {
	WARN("Port Multiplexer bits not all set to PWM/Tacho! (was 0x%08x, mask 0x%08lx)",
	     reg, mask);
	return -1;
    }

    return 0;
}
