/**
 * tco_fml_core.h
 *
 * Low level funktions for Intel's TCO Pass-Thru via FML
 *
 * (c) 2006 Peppercon AG, Thomas Rendelmann <thre@peppercon.de>
 */

#include <linux/threads.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/pci.h>
#include <linux/delay.h>

#include "tco_core.h"
#include "fmlcore.h"

#define NAME "TCO_FML"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

/* ------------------------------------------------------------------------- *
 * constants
 * ------------------------------------------------------------------------- */

#define DEV_NAME                "tco_fml"
#define FML_MAX_DATA_SIZE       240
#define FML_SLAVE_ADDR          ((uint8_t)0x64)

#define DELAY() udelay(50)
#define RETRIES 3

#define SUCCESS 0

/* KIRA100 IO addresses/bits */

// PCI dev and INTMUX not used by KIRA
#ifndef PP_BOARD_KIRA
#  define FML_USE_PCI
#  define FML_USE_INTMUX
#endif

#ifdef FML_USE_PCI
#  define KIRA100_PCI_VENDOR_ID 0x1743
#  define KIRA100_PCI_DEVICE_ID 0x1003
#  define KIRA100_PCI_BAR_NO    1
#  define FML_IO_BASE_OFFS      0x9b0000
#  define DEF_IRQ               27
#else
#  define FML_IO_BASE_OFFS      CPE_FML_BASE
#  define DEF_IRQ               IRQ_FML
#endif

#ifdef FML_USE_INTMUX
#  define INTMUX_IO_BASE_OFFS   0x9a0000
#  define INTMUX_IO_SIZE        (1 * sizeof(uint32_t)) /* 1 dword */
#  define FML_INT               ((uint16_t)(1 << 14))
#endif

uint32_t kira100_io_base = 0;
#define FML_IO_SIZE             0x2000 // == 8kbyte

/* ------------------------------------------------------------------------- *
 * function prototypes
 * ------------------------------------------------------------------------- */

static int  recv_cb(struct fmlcore_private *fml);
static void fml_isr(int irq, void *dev_id, struct pt_regs *regs);

static int  fml_init(struct net_device *dev, tco_t *tco);
static void fml_cleanup(tco_t *tco);
static int  fml_core_init(struct net_device *dev, tco_t *tco);
static void fml_core_release(struct net_device *dev, tco_t *tco);
static void fml_enable_client_int(tco_t *tco, int en);
static u32  fml_read_reg(tco_t *tco, u32 offset);
static void fml_write_reg(tco_t *tco, u32 offset, u32 val);
static int  fml_write_block(tco_t *tco, u8 command,
                            u8 byte_count, const u8 * data);
static int  fml_read_block(tco_t *tco, u8 command,
                           u8 *byte_count, u8 *data);

/* ------------------------------------------------------------------------- *
 * private data
 * ------------------------------------------------------------------------- */

typedef struct fml_priv_s {
    tco_t                   tco;

    /* fml specific stuff */
    void                    *fml_addr;
#ifdef FML_USE_INTMUX
    void                    *intmux_addr;
#endif
    struct fmlcore_private  fml;
    int                     isr_registered;
} fml_priv_t;

static fml_priv_t fml_priv = {
    tco: {
        name:               DEV_NAME,
        initialized:        0,
        max_data_size:      FML_MAX_DATA_SIZE,
        slave_addr:         FML_SLAVE_ADDR,

        init:               fml_init,
        cleanup:            fml_cleanup,
        core_init:          fml_core_init,
        core_release:       fml_core_release,
        enable_client_int:  fml_enable_client_int,
        check_int_source:   NULL,
        read_reg:           fml_read_reg,
        write_reg:          fml_write_reg,
        write_block:        fml_write_block,
        read_block:         fml_read_block,
    },
};

tco_t *tco = (tco_t*)&fml_priv;

/* ------------------------------------------------------------------------- *
 * exported functions
 * ------------------------------------------------------------------------- */

/* initialize on module loading */
static int fml_init(struct net_device *dev, tco_t *tco) {
    fml_priv_t *priv = (fml_priv_t *)tco;
    int ret = SUCCESS, r;
    uint32_t fml_io_base;
#ifdef FML_USE_INTMUX
    uint32_t intmux_io_base;
#endif

#ifdef FML_USE_PCI
    {
        struct pci_dev *pci;
        pci = pci_find_device(KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID, NULL);
        if (pci == NULL) {
            ERR("failed to find PCI device (%04x:%04x) containing FML core",
                KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID);
            ret = -ENODEV;
            goto fail;
        }
        kira100_io_base = pci_resource_start(pci, KIRA100_PCI_BAR_NO);
    }
#endif

    fml_io_base = kira100_io_base + FML_IO_BASE_OFFS;
    if ((priv->fml_addr = ioremap_nocache(fml_io_base, FML_IO_SIZE)) == (unsigned long)NULL) {
        ERR("failed to map FML core IO space at %08x-%08x to memory",
            fml_io_base, fml_io_base + FML_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }

#ifdef FML_USE_INTMUX
    intmux_io_base = kira100_io_base + INTMUX_IO_BASE_OFFS;
    if ((dev->intmux_addr = ioremap_nocache(intmux_io_base, INTMUX_IO_SIZE)) == NULL) {
        ERR("failed to map INTMUX IO space %08x-%08x to memory",
            intmux_io_base, intmux_io_base + INTMUX_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }
#endif

    /* fill fml struct */
    priv->fml.name      = dev->name;
    priv->fml.io_base   = (u32)priv->fml_addr;
    priv->fml.phys_base = fml_io_base;
    //priv->fml.fml_clk   = 8333; /* 8.333Mhz */
    priv->fml.fml_clk   = 6944; /* we need Tlow to be at least 72ns */
#ifdef PP_BOARD_KIRA
    priv->fml.sys_clk   = 100000; /* 100Mhz/AHB */
#else
    priv->fml.sys_clk   = 33000; /* 33Mhz/PCI */
#endif
    priv->fml.irq       = DEF_IRQ;
    priv->fml.slave     = 0;
    priv->fml.slave_req_callback  = recv_cb;

    dev->irq            = DEF_IRQ;

    // reserve memory
    if ((r = check_mem_region(fml_io_base, FML_IO_SIZE))) {
        ERR("IO space %08x-%08x already in use (err %d)",
            fml_io_base, fml_io_base + FML_IO_SIZE - 1, r);
        ret = -EBUSY;
        goto fail;
    }
    request_mem_region(fml_io_base, FML_IO_SIZE, "fml");

    // init hardware
    if (fmlcore_init(&priv->fml) < 0) {
        ret = -ENODEV;
        goto fail;
    }
    tco->initialized = 1;

#ifdef FML_USE_INTMUX
    INFO("successfully loaded. (io: %08x, intmux: %08x, irq: %d)",
        fml_io_base, intmux_io_base, DEF_IRQ);
#else // !FML_USE_INTMUX
    INFO("successfully loaded. (io: %08x, irq: %d)",
        fml_io_base, DEF_IRQ);
#endif

    return SUCCESS;

fail:
    fml_cleanup(tco);
    return ret;
}

static void fml_cleanup(tco_t *tco) {
    fml_priv_t *priv = (fml_priv_t *)tco;

    // release hardware
    if (tco->initialized) {
        fmlcore_release(&priv->fml);
        tco->initialized = 0;
    }
    release_mem_region(kira100_io_base + FML_IO_BASE_OFFS, FML_IO_SIZE);

#ifdef FML_USE_INTMUX
    if (priv->intmux_addr) {
        iounmap(priv->intmux_addr);
        priv->intmux_addr = NULL;
    }
#endif
    if (priv->fml_addr) {
        iounmap(priv->fml_addr);
        priv->fml_addr = NULL;
    }
}

/* initialize core on first usage */
static int fml_core_init(struct net_device *dev, tco_t *tco) {
    fml_priv_t *priv = (fml_priv_t *)tco;
    int ret = SUCCESS, r;

    // disable IRQ source
    fml_enable_client_int(tco, 0);

    // request IRQ
    if ((r = request_irq(DEF_IRQ, fml_isr, SA_SHIRQ, DEV_NAME, dev)) < 0) {
        ERR("failed to request irq %d (err %d)", DEF_IRQ, r);
        ret = -EBUSY;
        goto bail;
    }
    priv->isr_registered = 1;

 bail:
    return ret;
}

/* cleanup core on last usage */
static void fml_core_release(struct net_device *dev, tco_t *tco) {
    fml_priv_t *priv = (fml_priv_t *)tco;

    // release IRQ
    if (priv->isr_registered) {
        free_irq(DEF_IRQ, dev);
        priv->isr_registered = 0;
    }
}

/* enable/disable client interrupt */
static void fml_enable_client_int(tco_t *tco, int en) {
    fml_priv_t *priv = (fml_priv_t *)tco;

    // workaround: FML core generates IRQ on low SINTEX during master read
    if (en) priv->fml.write_reg(MFMLIACK, &priv->fml, MIACK_ADDR);

    priv->fml.write_reg(
        (priv->fml.read_reg(&priv->fml, MGR_ADDR) & ~MFMLIEN) | (en ? MFMLIEN : 0),
        &priv->fml, MGR_ADDR);
}

/* read/write a register */
static u32 fml_read_reg(tco_t *tco, u32 offset) {
    fml_priv_t *priv = (fml_priv_t *)tco;
    return priv->fml.read_reg(&priv->fml, offset);
}

static void fml_write_reg(tco_t *tco, u32 offset, u32 val) {
    fml_priv_t *priv = (fml_priv_t *)tco;
    priv->fml.write_reg(val, &priv->fml, offset);
}

/* block read/write */
static int fml_write_block(tco_t *tco,
		               u8 command,
		               u8 byte_count,
		               const u8 * data) {
    fml_priv_t *priv = (fml_priv_t *)tco;
    int ret = 0, retry = RETRIES;
    do {
        ret = fmlcore_write_block(&priv->fml, command, FML_SLAVE_ADDR, 
                                  byte_count, data);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during FML master write transfer: %d", ret);
        //else WARN("Error during FML master write transfer: %d", ret); // TODO: happens often, investigate!
    } while (retry-- > 0);
    return ret;
}

static int fml_read_block(tco_t *tco,
		              u8 command,
		              u8 *byte_count,
		              u8 *data) {
    fml_priv_t *priv = (fml_priv_t *)tco;
    int ret = 0, retry = RETRIES;
    do {
        ret = fmlcore_read_block(&priv->fml, command, FML_SLAVE_ADDR, 
                                 byte_count, data);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during FML master read transfer: %d", ret);
        else WARN("Error during FML master read transfer: %d", ret);
    } while (retry-- > 0);
    return ret;
}

/* ------------------------------------------------------------------------- *
 * interrupt handling
 * ------------------------------------------------------------------------- */

static int recv_cb(struct fmlcore_private *fml)
{
    return tco_recv_cb(tco);
}

static void fml_isr(int irq, void *dev_id, struct pt_regs *regs)
{
//WARN("enter irq");
    fml_priv_t *priv = (fml_priv_t*)tco;

#ifdef FML_USE_INTMUX
    {
        uint32_t intmux = readl(priv->intmux_addr);
        if (!(intmux & FML_INT)) return;
    }
#endif

    if (!tco->initialized) return;

    fmlcore_irq_handler(irq, &priv->fml, regs);
//WARN("leave irq");
}
