/***************************************************************************
 *   Copyright (C) 2004 by Ralf Guenther                                   *
 *   rgue@peppercon.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <sys/time.h>
#include <stdint.h>

#include <gpio.h>

static const char *dev[2] = { "/dev/gpio0", "/dev/gpio1" };

static void usage()
{
    printf("gpio_test ... status\n");
    printf("gpio_test <bit-nr (0-33)> ... read bit\n");
    printf("gpio_test <bit-nr (0-33)> <val (0|1|z)> ... write bit\n");
    printf("gpio_test -w [<bit-nr (0-33)>] ... watch one or all bits\n");
    exit(1);
}

void get_state()
{
    int i;
    unsigned long in, set, clr, tri;
    for (i = 0; i < 2; i++)
    {
        int fd = open(dev[i], O_RDWR);
        if (fd == -1) {
            printf("%s: ERR open failed (%s)\n", dev[i], strerror(errno));
            exit(-1);
        }
        gpio_ioctl_in(fd, &in);
        gpio_ioctl_is_set(fd, &set);
        gpio_ioctl_is_clr(fd, &clr);
        gpio_ioctl_is_tri(fd, &tri);
        printf("%s: in=%08lx tri=%08lx set=%08lx clr=%08lx\n", dev[i], in, tri, set, clr);

        close(fd);
    }
}

static void watch(int bit)
{
    int i;
    int fd[2];
    unsigned long mask[2];
    unsigned long val[2];

    if (bit == -1) mask[0] = mask[1] = 0xffffffff;
    else {
        mask[0] = mask[1] = 0;
        if (bit < 18) mask[0] |= 1 << bit;
        else mask[1] |= 1 << (bit - 18);
    }

    for (i = 0; i < 2; i++)
    {
        fd[i] = open(dev[i], O_RDWR);
        if (fd[i] == -1) {
            printf("%s: ERR open failed (%s)\n", dev[i], strerror(errno));
            exit(-1);
        }
        gpio_ioctl_watch(fd[i], mask[i]);
    }

    gpio_ioctl_in(fd[0], &val[0]);
    gpio_ioctl_in(fd[1], &val[1]);

    printf(" 0:%08x  1:%08x\n", val[0], val[1]);

    while (1) {
        struct timeval tv = { .tv_sec = 10, .tv_usec = 0, };
        fd_set set;
        FD_ZERO(&set);
        FD_SET(fd[0], &set);
        FD_SET(fd[1], &set);
        FD_SET(STDIN_FILENO, &set);
        select(FD_SETSIZE, &set, NULL, NULL, &tv);

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            printf("key hit: terminating\n");
            getchar();
            break;
        }

	gpio_ioctl_in(fd[0], &val[0]);
	gpio_ioctl_in(fd[1], &val[1]);

        printf("%c0:%08x %c1:%08x\n",
            FD_ISSET(fd[0], &set) ? '*' : ' ',
            val[0],
            FD_ISSET(fd[1], &set) ? '*' : ' ',
            val[1]);
    }

    for (i = 0; i < 2; i++) close(fd[i]);
}

int main(int argc, char *argv[])
{
    argc = argc; argv = argv;

    if (argc <= 1) get_state();
    else if (argc > 3 || !strcmp(argv[1], "-?")) usage();
    else if (strcmp(argv[1], "-w") == 0) watch(argc < 3 ? -1 : atoi(argv[2]));
    else {
        int bit = atoi(argv[1]);
        if (bit < 0 || bit > 33) usage();
        int fd = open(dev[bit / 18], O_RDWR);
        if (fd == -1) {
            printf("%s: ERR open failed (%s)\n", dev[bit / 18], strerror(errno));
            exit(-1);
        }
        if (argc == 2) {
	    unsigned long val;
	    gpio_ioctl_in(fd, &val);
            printf("GPIO%d is %d\n", bit, (val >> (bit % 18)) & 1);
        } else {
            if (!strcmp(argv[2], "z")) {
		    gpio_ioctl_set_bit_enable(fd, bit % 18, 0);		    
	    } else {
	            if (!strcmp(argv[2], "1")) gpio_ioctl_set_bit(fd, bit % 18, 1);
	            else if (!strcmp(argv[2], "0")) gpio_ioctl_set_bit(fd, bit % 18, 0);
		    else usage();		    
		    gpio_ioctl_set_bit_enable(fd, bit % 18, 1);
	    } else usage();

            printf("GPIO%d set to %s\n", bit, argv[2]);
        }
    }

    return 0;
}
