/***************************************************************************
 *   Copyright (C) 2004 by Ralf Guenther                                   *
 *   rgue@peppercon.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <sys/time.h>
#include <stdint.h>

#include "lpc.h"
#include "../lpc_core.h"

static void dump(unsigned char* b, int s)
{
    printf("%d bytes:", s);
    while (s--) printf(" %02x", *(b++));
    printf("\n");
    fflush(stdout);
}

static unsigned char *flipped(unsigned char v, unsigned char e)
{
    static unsigned char s[] = "xxxxxxxx";

    int i;
    for (i = 0; i < 8; i++)
        s[7 - i] =
            ((v >> i) & 1) > ((e >> i) & 1) ? '1' :
            ((v >> i) & 1) < ((e >> i) & 1) ? '0' :
            'x';
    
    return s;
}

static void lpc3_test(int fd)
{
    int i;

    struct {
        int reg;
        unsigned char val;
    } hicr[5] = {
        { .reg = LPC_HICR0, },
        { .reg = LPC_HICR1, },
        { .reg = LPC_HICR2, },
        { .reg = LPC_HICR3, },
        { .reg = LPC_HICR4, }
    };

    printf("Starting LPC3 test...\n");
    printf("Listening... ([ENTER] to quit)\n");
    
    /* set LPC3 host base address */
    lpc_ioctl_set_reg(fd, LPC_LADR3H, (LPC_DEFAULT_KCS_HOST_PORT_BASE >> 8) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_LADR3L, (LPC_DEFAULT_KCS_HOST_PORT_BASE & 0xfe) | 0x02);

    /* enable LPC3 */
    lpc_ioctl_set_reg(fd, LPC_HICR0, lpc_ioctl_get_reg(fd, LPC_HICR0) | LPC_LPC3E);

    lpc_ioctl_get_reg(fd, LPC_IDR1);
    lpc_ioctl_get_reg(fd, LPC_IDR2);
    lpc_ioctl_get_reg(fd, LPC_IDR3);
    lpc_ioctl_set_reg(fd, LPC_ODR3, 0xab);

    for (i = 0; i < 5; i++) hicr[i].val = lpc_ioctl_get_reg(fd, hicr[i].reg);

    {
//       int c = 0, e = 0;
        struct pollfd pfd = { .fd = STDIN_FILENO, .events = POLLIN, .revents = POLLIN, };
        while (!poll(&pfd, 1, 10)) {
/*
            if (lpc_ioctl_get_reg(fd, LPC_STR3) & LPC_IBF3A) {
                c++;
                unsigned char v = lpc_ioctl_get_reg(fd, LPC_IDR3);
                if (1 || v != 0x00) {
                    e++;
                    //printf("got %02x (bad=%s c=%d e=%d)\n", v, flipped(v, 0x00), c, e);
                    printf("IDR3 got %02x (bad=%s c=%d e=%d)\n", v, flipped(v, 0x00), c, e);
                }
            }
*/
//        printf("Idle loop %d\n", l++);
            if (lpc_ioctl_get_reg(fd, LPC_STR1) & LPC_IBF1) {
                unsigned char v = lpc_ioctl_get_reg(fd, LPC_IDR1);
                printf("IDR1 has reveived %02x\n", v);
            }
            if (lpc_ioctl_get_reg(fd, LPC_STR2) & LPC_IBF2) {
                unsigned char v = lpc_ioctl_get_reg(fd, LPC_IDR2);
                printf("IDR2 has reveived %02x\n", v);
            }
            if (lpc_ioctl_get_reg(fd, LPC_STR3) & LPC_IBF3A) {
                unsigned char v = lpc_ioctl_get_reg(fd, LPC_IDR3);
                printf("IDR3 has reveived %02x\n", v);
            }
            for (i = 0; i < 5; i++) {
                unsigned char v = lpc_ioctl_get_reg(fd, hicr[i].reg);
                if (v != hicr[i].val) {
//                    printf("HICR%d has changed from %02x to %02x\n", i, hicr[i].val, v);
                    hicr[i].val = v;
                }
            }
        }
    }
    printf("terminated\n");
    getchar();
    
    lpc_ioctl_set_reg(fd, LPC_HICR0, lpc_ioctl_get_reg(fd, LPC_HICR0) & ~LPC_LPC3E);
}

const unsigned char resp_clr_msg_flags[] = {
    0, // completion code
};

const unsigned char resp_get_dev_id[] = {
    0, // completion code
    0, // dev id (unspec)
    1, // revision
    0x00, // firmware major rev
    0x01, // firmware minor rev
    0x51, // IPMI ver
    0, // add dev supp
    0, // manufactor id (unspec)
    0,
    0,
    0, // product id (unspec)
    0,
    0, // aux firmaware rev
    0,
    0,
    0,
};

const unsigned char resp_get_msg_flags[] = {
    0, // completion code
    0, // flags
};
        
const unsigned char resp_get_chan_info[] = {
    0, // completion code
    0x0f, // chan no (SI)
    0x0c, // media type (SI)
    0x05, // prot type (KCS)
    0, // sess supp (none)
    0, // vendor id
    0,
    0,
    0xff, // aux chan info (no intr)
    0xff,
};

static void response(unsigned char netfn, unsigned char cmd, unsigned char *data, int *sz)
{
    switch (((int)netfn << 8) | cmd) {
    case 0x0601:
        printf("Sending 'Get Device ID' response...\n");
        memcpy(data, resp_get_dev_id, sizeof(resp_get_dev_id));
        *sz = sizeof(resp_get_dev_id);
        break;
    case 0x0630:
        printf("Sending 'Clear Message Flags' response...\n");
        memcpy(data, resp_clr_msg_flags, sizeof(resp_get_msg_flags));
        *sz = sizeof(resp_clr_msg_flags);
        break;
    case 0x0631:
        printf("Sending 'Get Message Flags' response...\n");
        memcpy(data, resp_get_dev_id, sizeof(resp_get_msg_flags));
        *sz = sizeof(resp_get_msg_flags);
        break;
    case 0x0642:
        printf("Sending 'Get Channel Info' response...\n");
        memcpy(data, resp_get_dev_id, sizeof(resp_get_chan_info));
        *sz = sizeof(resp_get_chan_info);
        break;
    default:
        printf("ERROR: illegal NetFn/Cmd - sending error response...\n");
        data[0] = 0xff; // completion code
        *sz = 1;
    }

}

static void kcs_engine(int fd, int chan)
{
    printf("Init KCS mode...\n");
    if (lpc_ioctl_init(fd, LPC_MODE_KCS, chan, LPC_DEFAULT_KCS_HOST_PORT_BASE)) {
        printf("ERROR: ioctl INIT (%s)\n", strerror(errno));
        exit(-1);
    }

    while (1) {
        struct {
            unsigned char netfn;
            unsigned char cmd;
            unsigned char data[1024];
        } m;
        int s, s2;

        fd_set set;
        FD_ZERO(&set);
        FD_SET(fd, &set);
        FD_SET(STDIN_FILENO, &set);

        printf("Waiting for request from remote console... ([ENTER] to quit)\n");

        //struct timeval tv = { .tv_sec = 1, .tv_usec = 0, };
        if (select (FD_SETSIZE, &set, NULL, NULL, NULL/*&tv*/) <= 0) {
            // idle
            // ...
            continue;
        }

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            getchar();
            break;
        }
        if (!FD_ISSET(fd, &set)) continue;

        printf("Receiving request...\n");

        s = read(fd, &m, sizeof(m));
        dump((void*)&m, s);

        if (s < 2) {
            printf("ERROR: request too short - ignoring it\n");
            continue;
        }

        s -= 2;
        response(m.netfn >> 2, m.cmd, m.data, &s);
        m.netfn |= 1 << 2;
        s += 2;

        if ((s2 = write(fd, &m, s)) != s)
            printf("ERROR: only %d of %d bytes written\n", s2, s);
        dump((void*)&m, s);
    }
}

static void dump_dbg(int fd, const char* names[])
{
    int ln = 90;
    uint8_t d[256];
    int c = lpc_ioctl_get_reg(fd, 0x51);
    int i, j;
    for (i = 0; i < c; i++) d[i] = lpc_ioctl_get_reg(fd, 0x50);
    printf("count=%d ... %d\n", c, lpc_ioctl_get_reg(fd, 0x52));
    if (c < ln) ln = c;

    if (ln) {
        for (i = 0; i < ln; i++) putchar('0' + (i % 10));
        putchar('\n');
        
        for (j = 0; j < 8; j++) {
            if (!names[j]) break;
            printf(names[j]);
            for (i = 0; i < ln; i++) putchar((d[i] >> j) & 1 ? '-' : '_');
            putchar('\n');
        }
    }
}

static void bt_engine(int fd)
{
    typedef struct {    
        unsigned char btsr0;
        unsigned char btsr1;
        unsigned char btfvsr0;
        unsigned char btfvsr1;
        unsigned char dbg0;
        unsigned char dbg1;
    } bt_regs_t;
    //bt_regs_t old;
    //struct pollfd pfd = { .fd = STDIN_FILENO, .events = POLLIN, .revents = POLLIN, };

    printf("Init BT mode...\n");
    if (lpc_ioctl_init(fd, LPC_MODE_BT, 3, LPC_DEFAULT_BT_HOST_PORT_BASE)) {
        printf("ERROR: ioctl INIT (%s)\n", strerror(errno));
        exit(-1);
    }

#if 0
    lpc_ioctl_set_reg(fd, LPC_BTDTR, 1); // pre-fill fifo
    lpc_ioctl_set_reg(fd, LPC_BTDTR, 2);
    lpc_ioctl_set_reg(fd, LPC_BTDTR, 3);

    lpc_ioctl_set_reg(fd, 0x51, 0); // reset dbg reg

    while (!poll(&pfd, 1, 10)) {
        bt_regs_t new = {
            .btsr0      = lpc_ioctl_get_reg(fd, LPC_BTSR0),
            .btsr1      = lpc_ioctl_get_reg(fd, LPC_BTSR1),
            .btfvsr0    = lpc_ioctl_get_reg(fd, LPC_BTFVSR0),
            .btfvsr1    = lpc_ioctl_get_reg(fd, LPC_BTFVSR1),
            .dbg0       = 0,//lpc_ioctl_get_reg(fd, 0x50),
            .dbg1       = 0,//lpc_ioctl_get_reg(fd, 0x51),
        };
        if (memcmp(&old, &new, sizeof(old)) != 0) {
            printf("btsr0=%08x btsr1=%08x btfvsr0=%02X btfvsr1=%02X dbg0=%02x dbg1=%02x\n",
                new.btsr0, new.btsr1, new.btfvsr0, new.btfvsr1, new.dbg0, new.dbg1);
            memcpy(&old, &new, sizeof(old));
        }
    }
    printf("terminated\n");
    getchar();
#endif

#if 1
    while (1) {
        struct {
                unsigned char size;
                unsigned char netfn;
                unsigned char seq;
                unsigned char cmd;
                unsigned char data[1024];
        } m;
        int s, s2;

        fd_set set;
        FD_ZERO(&set);
        FD_SET(fd, &set);
        FD_SET(STDIN_FILENO, &set);

        printf("Waiting for request from remote console... ([ENTER] to quit)\n");

        //struct timeval tv = { .tv_sec = 1, .tv_usec = 0, };
        if (select (FD_SETSIZE, &set, NULL, NULL, NULL/*&tv*/) <= 0) {
            // idle
            printf("FIFO levels: H2B=%d B2H=%d STR3=%02x\n",
                lpc_ioctl_get_reg(fd, LPC_BTFVSR0), lpc_ioctl_get_reg(fd, LPC_BTFVSR1), lpc_ioctl_get_reg(fd, LPC_STR3));
/*
            if (lpc_ioctl_get_reg(fd, LPC_BTFVSR1) < 10) {
                lpc_ioctl_set_reg(fd, LPC_BTDTR, 1);
                lpc_ioctl_set_reg(fd, LPC_BTDTR, 2);
                lpc_ioctl_set_reg(fd, LPC_BTDTR, 3);
            }
            printf("FIFO levels after : H2B=%d B2H=%d\n", lpc_ioctl_get_reg(fd, LPC_BTFVSR0), lpc_ioctl_get_reg(fd, LPC_BTFVSR1));
*/
            continue;
        }

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            lpc_ioctl_hk_t hk;

            getchar();

            ioctl(fd, LPC_IOCTL_GET_HK, &hk);
            printf("hk: intr=%d lpc_intr=%d\n",
                hk.intr_cnt, hk.lpc_intr_cnt);
            break;
        }
        if (!FD_ISSET(fd, &set)) continue;

        printf("Receiving request...\n");

        s = read(fd, &m, sizeof(m));
        dump((void*)&m, s);

        if (s < 2) {
            printf("ERROR: request too short - ignoring it\n");
            continue;
        }
        
        if (s != m.size + 1) {
            printf("ERROR: size mismatch: is %d, len %d\n", s, m.size + 1);
            continue;
        }

        s -= 4;
        response(m.netfn >> 2, m.cmd, m.data, &s);
        m.netfn |= 1 << 2;
        m.size = s + 3;
        s += 4;

        if ((s2 = write(fd, &m, s)) != s)
            printf("ERROR: only %d of %d bytes written\n", s2, s);
        dump((void*)&m, s);
    }
#endif
}

static void smic_engine(int fd)
{
    typedef struct {
        uint8_t data;
        uint8_t ctrl;
        uint8_t flags;
    } smic_regs_t;
    //smic_regs_t old = { .data = -1, };
    //struct pollfd pfd = { .fd = STDIN_FILENO, .events = POLLIN, .revents = POLLIN, };

    printf("Init SMIC mode...\n");
    if (lpc_ioctl_init(fd, LPC_MODE_SMIC, 3, LPC_DEFAULT_SMIC_HOST_PORT_BASE)) {
        printf("ERROR: ioctl INIT (%s)\n", strerror(errno));
        exit(-1);
    }

#if 0
    lpc_ioctl_set_reg(fd, LPC_SMICIR1, 0); // disable irqs

    while (1/*!poll(&pfd, 1, 10)*/) {
        smic_regs_t new = {
            .data  = lpc_ioctl_get_reg(fd, LPC_SMICDTR),
            .ctrl  = lpc_ioctl_get_reg(fd, LPC_SMICCSR),
            .flags = lpc_ioctl_get_reg(fd, LPC_SMICFLG),
        };
        if (memcmp(&old, &new, sizeof(old)) != 0) {
            printf("data=%02x ctrl=%02x flags=%02x\n", new.data, new.ctrl, new.flags);
            memcpy(&old, &new, sizeof(old));
        }

        {
            static const char *n[] = {
                "addr[0]   ",
                "addr[1]   ",
                "addr[2]   ",
                "addr[3]   ",
                "lpc_match ",
                "ch3_match ",
                "SMIC      ",
                "KCS|BT    ",
                NULL
            };
            dump_dbg(fd, n);
        }
        lpc_ioctl_set_reg(fd, 0x50, 0); // clear dbg fifo
        
        getchar();
    }
    printf("terminated\n");
    getchar();
#endif
   
#if 1
    while (1) {
        struct {
            unsigned char netfn;
            unsigned char cmd;
            unsigned char data[1024];
        } m;
        int s, s2;

        fd_set set;
        FD_ZERO(&set);
        FD_SET(fd, &set);
        FD_SET(STDIN_FILENO, &set);

        printf("Waiting for request from remote console... ([ENTER] to quit)\n");

        select(FD_SETSIZE, &set, NULL, NULL, NULL);

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            getchar();
            break;
        }
        if (!FD_ISSET(fd, &set)) continue;

        printf("Receiving request...\n");

        
        s = read(fd, &m, sizeof(m));
        dump((void*)&m, s);
        
        if (s < 2) {
            printf("ERROR: request too short - ignoring it\n");
            continue;
        }
        
        s -= 2;
        response(m.netfn >> 2, m.cmd, m.data, &s);
        m.netfn |= 1 << 2;
        s += 2;

        if ((s2 = write(fd, &m, s)) != s)
            printf("ERROR: only %d of %d bytes written\n", s2, s);
        dump((void*)&m, s);
    }
#endif
}

static int dti(struct timeval* ti2, struct timeval* ti1)
{
    return (ti2->tv_sec * 1000 + ti2->tv_usec / 1000)
         - (ti1->tv_sec * 1000 + ti1->tv_usec / 1000);
}

static void reg_stress(int fd, int chan)
{
    int i, j;

    printf("Init KCS mode for Stress Test...\n");
    if (lpc_ioctl_init(fd, LPC_MODE_KCS, chan, LPC_DEFAULT_KCS_HOST_PORT_BASE)) {
        printf("ERROR: ioctl INIT (%s)\n", strerror(errno));
        exit(-1);
    }

const int _l = 1;
struct timeval ti1, ti2;
    for (i = 0; ; i++) {
        printf("%d0000 r...\n", i);
if (_l) lpc_ioctl_lock_net(fd, 1); // lock net
gettimeofday(&ti1, NULL);
        for (j = 0; j < 10000; j++) lpc_ioctl_get_reg(fd, LPC_IDR3);
gettimeofday(&ti2, NULL);
if (_l) lpc_ioctl_lock_net(fd, 0); // unlock net
printf("%dms\n", dti(&ti2, &ti1));
        printf("%d0000 w...\n", i);
if (_l) lpc_ioctl_lock_net(fd, 1); // lock net
gettimeofday(&ti1, NULL);
        for (j = 0; j < 10000; j++) lpc_ioctl_set_reg(fd, LPC_ODR3, j);
gettimeofday(&ti2, NULL);
if (_l) lpc_ioctl_lock_net(fd, 0); // unlock net
printf("%dms\n", dti(&ti2, &ti1));
        printf("pause...\n");
        sleep(1); // short pause to prevent the lpc hack to blockage the system
    }
}

static void multi_kcs()
{
    int fd[3];
    char devname[] = "/dev/lpc";
    int i;

    for (i = 0; i < 3; i++) {
        if ((fd[i] = open(devname, O_RDWR)) == -1) {
            printf("ERROR: fauled to open KCS%d device %s: %s\n", i + 1, devname, strerror(errno));
            exit(-1);
        }

        if (lpc_ioctl_init(fd[i], LPC_MODE_KCS, i + 1, LPC_DEFAULT_KCS_HOST_PORT_BASE + i * 10)) {
            printf("ERROR: failed to init KCS%d: %s\n", i + 1, strerror(errno));
            exit(-1);
        }
    }

    while (1) {
        fd_set set;
        FD_ZERO(&set);
        FD_SET(STDIN_FILENO, &set);

        printf("Waiting for request from remote console... ([ENTER] to quit)\n");

        for (i = 0; i < 2; i++) FD_SET(fd[i], &set);

        //struct timeval tv = { .tv_sec = 1, .tv_usec = 0, };
        if (select (FD_SETSIZE, &set, NULL, NULL, NULL/*&tv*/) <= 0) {
            // idle
            // ...
            continue;
        }

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            getchar();
            break;
        }

        for (i = 0; i < 2; i++) {
            struct {
                unsigned char netfn;
                unsigned char cmd;
                unsigned char data[1024];
            } m;
            int s, s2;

            if (!FD_ISSET(fd[i], &set)) continue;

            printf("Receiving request at KCS%d...\n", i + 1);

            s = read(fd[i], &m, sizeof(m));
            dump((void*)&m, s);

            if (s < 2) {
                printf("ERROR: request too short - ignoring it\n");
                continue;
            }

            s -= 2;
            response(m.netfn >> 2, m.cmd, m.data, &s);
            m.netfn |= 1 << 2;
            s += 2;

            if ((s2 = write(fd[i], &m, s)) != s)
                printf("ERROR: only %d of %d bytes written\n", s2, s);
            dump((void*)&m, s);
        }
    }

    for (i = 0; i < 3; i++) close(fd[i]);
}

/* manual snooping
static void snoop(int fd, unsigned short port)
{
    int i = 0;

    printf("Pre-emptying FIFO...\n");
    while (!(lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_SNOOP_EMPTY)) {
        printf("  %d: 0x%02x\n", i++, lpc_ioctl_get_reg(fd, LPC_SNOOP_FIFO));
    }

    printf("Programming snoop port 0x%04x...\n", port);
    lpc_ioctl_set_reg(fd, LPC_SNOOP_ADDRH, (port >> 8) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_SNOOP_ADDRL, (port >> 0) & 0xff);

    printf("Snooping (hit [ENTER] to exit)...\n");
    {
        struct pollfd pfd = { .fd = STDIN_FILENO, .events = POLLIN, .revents = POLLIN, };
        i = 0;
        while (1) {
            if (poll(&pfd, 1, 1) == 1) {
                break;
            }

            while (!(lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_SNOOP_EMPTY)) {
                printf("  %d: 0x%02x\n", i++, lpc_ioctl_get_reg(fd, LPC_SNOOP_FIFO));
            }
        }
    }

    printf("Bye bye.\n");
}
*/

static void snoop(int fd, unsigned short port)
{
    int i = 0;

    printf("Init Snoop mode...\n");
    if (lpc_ioctl_init(fd, LPC_MODE_SNOOP, 0, port)) {
        printf("ERROR: ioctl INIT (%s)\n", strerror(errno));
        exit(-1);
    }

    printf("Snooping (hit [ENTER] to exit)...\n");
    while (1) {
        struct timeval tv = { .tv_sec = 0, .tv_usec = 0, };
        fd_set set;
        FD_ZERO(&set);
        FD_SET(fd, &set);
        FD_SET(STDIN_FILENO, &set);

        select(FD_SETSIZE, &set, NULL, NULL, &tv);

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            getchar();
            break;
        }
        if (!FD_ISSET(fd, &set)) continue;

        {
            unsigned char data[16];
            int c = read(fd, &data, sizeof(data));
            if (c > 0) {
                int j;
                printf("%3d:", i++);
                for (j = 0; j < c; j++) printf(" 0x%02x", data[j]);
                printf("\n");
            } else {
                printf("  ERROR: read failed\n");
            }
        }
    }

    printf("Bye bye.\n");
}

static void dump_host_regs(int fd)
{
    unsigned char stat = lpc_ioctl_get_reg(fd, LPC_HOST_STATUS);
    unsigned char cmd = lpc_ioctl_get_reg(fd, LPC_HOST_CMD);
    unsigned int addr =
        ((unsigned int)lpc_ioctl_get_reg(fd, LPC_HOST_ADDR0) <<  0) +
        ((unsigned int)lpc_ioctl_get_reg(fd, LPC_HOST_ADDR1) <<  8) +
        ((unsigned int)lpc_ioctl_get_reg(fd, LPC_HOST_ADDR2) << 16) +
        ((unsigned int)lpc_ioctl_get_reg(fd, LPC_HOST_ADDR3) << 24);
    unsigned char din = lpc_ioctl_get_reg(fd, LPC_HOST_DATA_IN);
    unsigned char dout = lpc_ioctl_get_reg(fd, LPC_HOST_DATA_OUT);

    printf("LPC host: stat=%02x cmd=%02x addr=%08x out=%02x in=%02x\n",
           stat, cmd, addr, dout, din);
}

static unsigned char host_read(int fd, unsigned int addr)
{
int v = lpc_ioctl_mem_read(fd, addr);
if (v < 0) printf("Transfer error occured during read of %08x\n", addr);
return v < 0 ? 0xff : (unsigned char)v;

//    printf("Read LPC mem from 0x%08x...\n", addr);

    // program addr
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR0, (addr >>  0) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR1, (addr >>  8) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR2, (addr >> 16) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR3, (addr >> 24) & 0xff);

    // enable master and start reading
    lpc_ioctl_set_reg(fd, LPC_HOST_CMD, 0);
//dump_host_regs(fd);
    lpc_ioctl_set_reg(fd, LPC_HOST_CMD, LPC_HOST_ENABLE | LPC_HOST_GO | LPC_HOST_READ_CMD);
//dump_host_regs(fd);

    // wait for command completion
//    printf("Waiting for busy flag to become valid...\n");
    while (!(lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_HOST_BUSY_VALID));
//dump_host_regs(fd);
//    printf("Waiting for busy flag to become 0...\n");
    while (lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_HOST_BUSY);

//dump_host_regs(fd);
    if (lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_HOST_ERR) {
        printf("Transfer error occured\n");
        return 0;
    } else {
        unsigned char v = lpc_ioctl_get_reg(fd, LPC_HOST_DATA_OUT);
//        printf("  0x%02x\n", v);
        return v;
    }
}

static void host_write(int fd, unsigned int addr, unsigned char val)
{
int r = lpc_ioctl_mem_write(fd, addr, val);
if (r < 0) printf("Transfer error occured during write of %02x to %08x\n", val, addr);
return;

    printf("Write 0x%02x to LPC mem @ 0x%08x...\n", val, addr);

    // program addr
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR0, (addr >>  0) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR1, (addr >>  8) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR2, (addr >> 16) & 0xff);
    lpc_ioctl_set_reg(fd, LPC_HOST_ADDR3, (addr >> 24) & 0xff);

    // program value
    lpc_ioctl_set_reg(fd, LPC_HOST_DATA_IN, val);

    // enable master and start reading
    lpc_ioctl_set_reg(fd, LPC_HOST_CMD, 0);
    lpc_ioctl_set_reg(fd, LPC_HOST_CMD, LPC_HOST_ENABLE | LPC_HOST_GO | LPC_HOST_WRITE_CMD);

    // wait for command completion
    printf("Waiting for busy flag to become valid...\n");
    while (!(lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_HOST_BUSY_VALID));
    printf("Waiting for busy flag to become 0...\n");
    while (lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_HOST_BUSY);

//dump_host_regs(fd);
    if (lpc_ioctl_get_reg(fd, LPC_HOST_STATUS) & LPC_HOST_ERR) {
        printf("Transfer error occured\n");
    } else {
        printf("  ok\n");
    }
}

#define flash_base      0xfffc0000
#define flash_gpi_reg   0xffbc0100

static unsigned char flash_get_gpi_reg(int fd)
{
    return host_read(fd, flash_gpi_reg);
}

static void flash_erase_sector(int fd, unsigned int sec)
{
    // erase 4k sector
    host_write(fd, flash_base + 0x5555, 0xaa);
    host_write(fd, flash_base + 0x2aaa, 0x55);
    host_write(fd, flash_base + 0x5555, 0x80);
    host_write(fd, flash_base + 0x5555, 0xaa);
    host_write(fd, flash_base + 0x2aaa, 0x55);
    host_write(fd, flash_base + sec, 0x30);
}

static void flash_erase_block(int fd, unsigned int blk)
{
    // erase 16k block
    host_write(fd, flash_base + 0x5555, 0xaa);
    host_write(fd, flash_base + 0x2aaa, 0x55);
    host_write(fd, flash_base + 0x5555, 0x80);
    host_write(fd, flash_base + 0x5555, 0xaa);
    host_write(fd, flash_base + 0x2aaa, 0x55);
    host_write(fd, flash_base + blk, 0x50);
}

static void flash_program_byte(int fd, unsigned int addr, unsigned char val)
{
    // flash one byte
    host_write(fd, flash_base + 0x5555, 0xaa);
    host_write(fd, flash_base + 0x2aaa, 0x55);
    host_write(fd, flash_base + 0x5555, 0xa0);
    host_write(fd, flash_base + addr, val);
}

static unsigned char flash_read_byte(int fd, unsigned int addr)
{
    return host_read(fd, flash_base + addr);
}

static void host(int fd)
{
    int i;
    unsigned char v;

    printf("Erasing sector 0...\n"); 
    flash_erase_sector(fd, 0); 
    printf("  done.\n");

    printf("Waiting for end of erasing...\n"); 
    for (i = 0; i < 10; i++) {
        printf("   GPI=%02x %08x=%02x\n", flash_get_gpi_reg(fd), 0x100, flash_read_byte(fd, 0x100));
        usleep(1000);
    }
    printf("  done.\n");

    printf("Test erasing results...\n"); 
    for (i = 0x100; i < 0x110; i++) {
        v = flash_read_byte(fd, i);
        printf("   %08x=%02x\n", i, v);
        if (v != 0xff) {
            printf("ERROR %08x is not %02x after erasing\n", i, 0xff);
            return;
        }
    }
    printf("  done.\n");

    printf("Writing data...\n"); 
    for (i = 0x100; i < 0x110; i++) {
        printf("   %08x<=%02x\n", i, i & 0xff);
        flash_program_byte(fd, i, i & 0xff);
    }
    printf("  done.\n");

    printf("Test writing results...\n"); 
    for (i = 0x100; i < 0x110; i++) {
        v = flash_read_byte(fd, i);
        printf("   %08x=%02x\n", i, v);
        if (v != (i & 0xff)) {
            printf("ERROR %08x is not %02x after writing\n", i, i & 0xff);
            return;
        }
    }
    printf("  done.\n");
}

static void vuart(int fd, unsigned short port, unsigned char irq)
{
    if (lpc_ioctl_set_uart_cfg(fd, port, irq) < 0) {
        printf("failed to config vuart (%d)\n", errno);
        return;
    }
    printf("vuart successfully configurred: host port = 0x%x, host irq = %d\n",
           port, irq);
}

int main(int argc __attribute__((unused)), char *argv[] __attribute__((unused)))
{

    char *name = "/dev/lpc";
    int fd;
    printf("Opening %s...\n", name);
    fd = open(name, O_RDWR);
    if (fd == -1) {
        printf("ERROR: open failed (%s)\n", strerror(errno));
        exit(-1);
    }

    //lpc3_test(fd);
    //kcs_engine(fd, 3);
    //multi_kcs();
    //bt_engine(fd);
    //smic_engine(fd);
    //reg_stress(fd, 3);
    //snoop(fd, 0x80);
    //host(fd);
    vuart(fd, 0xca0/*0x3e8*/, 5);

    close(fd);
    return 0;
}
