/*
 * Faraday FOTG200-Peripheral  USB Device Controller driver
 *
 * Copyright (C) 2004-2005 Lineo
 *      by John Chiang
 * Copyright (C) 2004 Faraday tech corp.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

/*
 * This device has ep0 and four bulk/interrupt endpoints.
 *
 *  - Endpoint numbering is fixed: EP1-bulk in, EP2-bulk out, EP3-interrupt in, EP4 interrupt out
 *  - EP maxpacket (if full speed:64, if high speed:512)
 *  - no DMA supporting in the first version
 *  - support AHB_DMA in the 2nd version
 */

#undef DEBUG
// #define	VERBOSE		/* extra debug messages (success too) */
// #define	USB_TRACE	/* packet-level success messages */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/version.h>

#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>

#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/pci.h>
#include <asm/system.h>
#include <asm/unaligned.h>
#include <asm/arch/cpe_int.h>
#ifdef USE_OTG
#include <linux/usb_otg.h>//For OTG
#endif

#include "FTC_FOTG200_udc.h"
#include "FTC_zero_ll.h"
#define DRV_NAME "FTC_FOTG200"
#include "../lara_common.h"

#define	DRIVER_DESC	"FOTG200 USB Peripheral Controller"
#define	DRIVER_VERSION	"15-Feb 2005"

static const char driver_name [] = "FTC_FOTG200_udc";
static const char driver_desc [] = DRIVER_DESC;

static char *names [] = {"ep0","ep1","ep2","ep3","ep4","ep5","ep6","ep7","ep8","ep9","ep10" };

static struct FTC_udc	*the_controller=0;

MODULE_AUTHOR("Faraday-SW");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");

int minor = 1;
MODULE_PARM(minor, "i");

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
typedef int gfp_flags_t;
#else
typedef unsigned gfp_flags_t;
#endif

#ifdef CONFIG_FARADAY_FOTG2XX

extern struct otg_transceiver *FOTG2XX_get_otg_transceiver(void);

#else

#ifdef REGISTER_DEVICE

#define DEV_MAJOR     41    //major device for Device
#define AP_CMD_ENABLE_DEV          5

int fotg200_usb_gadget_register_driver(struct usb_gadget_driver *driver);
int fotg200_usb_gadget_unregister_driver(struct usb_gadget_driver *driver);

extern int devfs_register_chrdev (unsigned int major, const char *name,
				  struct file_operations *fops);
static int DEVC_AP_ioctl(struct inode * inode, struct file * file
                             ,unsigned int cmd,unsigned long arg);

static struct file_operations dev_fops = {
	owner:		THIS_MODULE,
	ioctl:		DEVC_AP_ioctl,
};

//*********************************************************
// Name: DEVC_AP_ioctl
// Description: Request from user mode AP
// Input:struct inode * inode, struct file * file
//      ,unsigned int cmd,unsigned long arg
// Output: int
//*********************************************************
static int DEVC_AP_ioctl(struct inode * inode, struct file * file
                             ,unsigned int cmd,unsigned long arg)
{
   DBG_FUNCC("+DEVC_AP_ioctl()\n");

   switch(cmd){
      case AP_CMD_ENABLE_DEV:
		 mUsbChipEnSet();
		 mUsbGlobIntEnSet();
    	 return (0);
    	 break;
	  default:
		 break;
   }
   return 1;
}

#endif /* REGISTER_DEVICE */

#endif /* CONFIG_FARADAY_FOTG2XX */


static u8 get_ep_fifo(struct FTC_ep *ep)
{
    u8 result;
    if (ep->num == 0) return FIFOCX;
    result = mUsbEPMapRd(ep->num);
    return ep->is_in? result&0x0f : result>>4;
}

static void nuke(struct FTC_ep *, int status);

//****************************************************
// Name:FTC_ep_enable
// Description: Enable endpoint
//              EP0 : has been enabled while driver booting up
//              Need to give this EP's descriptor
// Input:<1>.ep structure point
// Output:none
//****************************************************
static int
FTC_ep_enable(struct usb_ep *_ep, const struct usb_endpoint_descriptor *desc)
{
    D(D_BLABLA, "FTC_ep_enable(%s)\n", _ep->name);
    struct FTC_udc *dev;
    struct FTC_ep *ep;
    u16 max;
    unsigned long flags;
    int fifo;
    u8 fifo_config = 0;

    ep = container_of(_ep, struct FTC_ep, ep);

    if (!_ep || !desc || ep->desc
	    || desc->bDescriptorType != USB_DT_ENDPOINT) {
	return -EINVAL;
    }

    dev = ep->dev;
    if (ep == &dev->ep[0]) return -EINVAL;

    if (!dev->driver || dev->gadget.speed == USB_SPEED_UNKNOWN) {
	return -ESHUTDOWN;
    }

    if (ep->num != (desc->bEndpointAddress & 0x0f)) {
	return -EINVAL;
    }

    switch (desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) {
	case USB_ENDPOINT_XFER_BULK:
	    fifo_config = BIT5 | BIT1;
	    break;
	case USB_ENDPOINT_XFER_INT:
	    fifo_config = BIT5 | BIT1 | BIT0;
	    break;
	default:
	    return -EINVAL;
    }

    spin_lock_irqsave(&ep->dev->lock, flags);

    /* Find a free FIFO buffer */
    for (fifo=0; fifo<MAX_FIFO_NUM; fifo++) {
	if ((mUsbFIFOMapRd(fifo) & 0x0f) == 0x0f) break;
    }
    if (fifo == MAX_FIFO_NUM) {
	D(D_ERROR, "Can't enable endpoint %s: No free FIFO buffer.\n", ep->ep.name);
	return -ENOSPC;
    }

    max = le16_to_cpu(get_unaligned(&desc->wMaxPacketSize));
    ep->dma = 1;
    ep->is_in = (USB_DIR_IN & desc->bEndpointAddress) != 0;
    ep->ep.maxpacket = max;
    ep->stopped = 0;
    ep->desc = desc;

    /* Configure the EP/FIFO mapping registers */
    mUsbEPMap(ep->num, ep->is_in? fifo|0xf0 : (fifo<<4)|0x0f);
    mUsbFIFOMap(fifo, (ep->is_in? BIT4 : 0) | ep->num);
    mUsbFIFOConfig(fifo, fifo_config);

    spin_unlock_irqrestore(&ep->dev->lock, flags);
    D(D_VERBOSE, "enabled %s endpoint %s (num %d, FIFO %d, maxpacket %u)\n",
	    ep->is_in ? "IN" : "OUT", ep->ep.name, ep->num, fifo, max);
    return 0;
}
//****************************************************
// Name:ep_reset
// Description: ep reset
//
// Input:<1>.ep structure point
// Output:none
//****************************************************
static void ep_reset(struct FTC_ep *ep)
{
    D(D_BLABLA, "ep_reset()\n");
    ep->ep.maxpacket = MAX_FIFO_SIZE;
    ep->desc = 0;
    ep->stopped = 1;
    ep->irqs = 0;
    ep->dma = 0;
}

static int FTC_ep_disable(struct usb_ep *_ep)
{
    D(D_BLABLA, "FTC_ep_disable(%s)\n", _ep->name);
    struct FTC_ep *ep;
    struct FTC_udc *dev;
    unsigned long flags;
    int fifo;

    ep = container_of(_ep, struct FTC_ep, ep);
    if (!_ep || !ep->desc) return -ENODEV;

    dev = ep->dev;
    if (ep == &dev->ep[0]) return -EINVAL;

    spin_lock_irqsave(&dev->lock, flags);

    fifo = get_ep_fifo(ep);
    nuke(ep, -ESHUTDOWN);
    ep_reset(ep);
    mUsbEPMap(ep->num, 0xff);
    mUsbFIFOMap(fifo, 0x0f);
    mUsbFIFOConfig(fifo, 0);

    spin_unlock_irqrestore(&dev->lock, flags);
    return 0;
}

/*-------------------------------------------------------------------------*/
//****************************************************
// Name:FTC_alloc_request
// Description: allocate request
//
// Input:<1>.ep structure point
//       <2>.gfp flag
// Output:none
//****************************************************
static struct usb_request *
FTC_alloc_request(struct usb_ep *_ep, gfp_flags_t gfp_flags)
{
	struct FTC_request	*req;

	DBG_FUNCC("+FTC_alloc_request\n");

  //<1>.Checking input
	if (!_ep)
		return 0;

  //<2>.Malloc memory
	req = kmalloc(sizeof *req, gfp_flags);
	if (!req)
		return 0;

	memset(req, 0, sizeof *req);
	req->req.dma = DMA_ADDR_INVALID;

  //<3>.Add to list
	INIT_LIST_HEAD(&req->queue);

	DBG_FUNCC("-FTC_alloc_request\n");

	return &req->req;
}
//****************************************************
// Name:FTC_free_request
// Description: free request
//
// Input:<1>.ep structure point
//       <2>.gfp flag
// Output:none
//****************************************************
static void
FTC_free_request(struct usb_ep *_ep, struct usb_request *_req)
{
	struct FTC_request	*req;

    DBG_FUNCC("+FTC_free_request()\n");

	if (!_ep || !_req)
		return;

	req = container_of(_req, struct FTC_request, req);
	WARN_ON(!list_empty(&req->queue));
	kfree(req);
}

/*-------------------------------------------------------------------------*/

/* many common platforms have dma-coherent caches, which means that it's
 * safe to use kmalloc() memory for all i/o buffers without using any
 * cache flushing calls.  (unless you're trying to share cache lines
 * between dma and non-dma activities, which is a slow idea in any case.)
 *
 * other platforms need more care, with 2.6 having a moderately general
 * solution except for the common "buffer is smaller than a page" case.
 */
#undef USE_KMALLOC //Please use the "pci_alloc_consistent" function call
//#define USE_KMALLOC


//****************************************************
// Name:FTC_alloc_buffer
// Description: Allocate the buffer for FOTG200 DMA
//
// Input:<1>.ep structure point
//       <2>.bytes number
//       <3>.dma address pointer
//       <4>.gfp flag
// Output:none
//****************************************************
/* allocating buffers this way eliminates dma mapping overhead, which
 * on some platforms will mean eliminating a per-io buffer copy.  with
 * some kinds of system caches, further tweaks may still be needed.
 */
static void *
FTC_alloc_buffer(struct usb_ep *_ep, unsigned bytes,
			dma_addr_t *dma, gfp_flags_t gfp_flags)
{
	void		*retval;
	struct FTC_ep	*ep;

	DBG_FUNCC("+FTC_alloc_buffer():ep = 0x%x\n",(u32) _ep);

	ep = container_of(_ep, struct FTC_ep, ep);
	if (!_ep)
		return 0;

	*dma = DMA_ADDR_INVALID;

#if	defined(USE_KMALLOC)
	retval = kmalloc(bytes, gfp_flags);

	if (retval)
		*dma = virt_to_phys(retval);



#else
	//if (ep->dma) {
		/* one problem with this call is that it wastes memory on
		 * typical 1/N page allocations: it allocates 1-N pages.
		 * another is that it always uses GFP_ATOMIC.
		 */
//#warning Using pci_alloc_consistent even with buffers smaller than a page.

		retval = consistent_alloc(gfp_flags, bytes, dma);//Bruce;;05032005;;
	//}
	//else
	//	retval = kmalloc(bytes, gfp_flags);




#endif
	return retval;
}
//****************************************************
// Name:FTC_free_buffer
// Description: Free the buffer
//
// Input:<1>.ep structure point
//       <2>.bytes number
//       <3>.dma address pointer
//       <4>.gfp flag
// Output:none
//****************************************************
static void
FTC_free_buffer(struct usb_ep *_ep, void *buf, dma_addr_t dma, unsigned bytes)
{
    struct FTC_ep	*ep;

	DBG_FUNCC("+FTC_free_buffer()\n");

	/* free memory into the right allocator */
#ifndef	USE_KMALLOC
	//if (dma != DMA_ADDR_INVALID) {
		ep = container_of(_ep, struct FTC_ep, ep);
		if (!_ep)
			return;
		/* one problem with this call is that some platforms
		 * don't allow it to be used in_irq().
		 */

		consistent_free(buf, bytes, dma);//Bruce;;05032005;;
	//}
    //else
	//	kfree (buf);
#else
		kfree (buf);
#endif
}

/*-------------------------------------------------------------------------*/
//****************************************************
// Name:done
// Description: <1>.DMA memory unmap
//              <2>.Call back complete function
//
// Input:<1>.ep structure point
//       <2>.req structure point
//       <3>.status
// Output:none
//****************************************************
static void
done(struct FTC_ep *ep, struct FTC_request *req, int status)
{
	struct FTC_udc		*dev;
	unsigned stopped = ep->stopped;
    u32 temp;


	DBG_FUNCC("+done()\n");

  //<1>.Check the status

	list_del_init(&req->queue);

	if (likely(req->req.status == -EINPROGRESS))
		req->req.status = status;
	else
		status = req->req.status;

	dev = ep->dev;




  //<2>.Unmap DMA memory
	if (req->mapped) {
		DBG_CTRLL("....pci_unmap_single len = %d\n",req->req.length);

		// important : DMA length will set as 16*n bytes
		temp = req->req.length / 16;
		if (req->req.length % 16)
           temp++;
		pci_unmap_single((void *)dev, req->req.dma, temp*16,  //USB_EPX_BUFSIZ,  //req->req.length+32,
			ep->is_in ? PCI_DMA_TODEVICE : PCI_DMA_FROMDEVICE);
		req->req.dma = DMA_ADDR_INVALID;
		req->mapped = 0;
	}

#ifndef USB_TRACE
	if (status && status != -ESHUTDOWN)
#endif
		VDBG(dev, "complete %s req %p stat %d len %u/%u\n",
			ep->ep.name, &req->req, status,
			req->req.actual, req->req.length);

	/* don't modify queue heads during completion callback */
	ep->stopped = 1;





  //<3>.call back the complete function call

	spin_unlock(&dev->lock);
	req->req.complete(&ep->ep, &req->req);
	spin_lock(&dev->lock);


  //<4>.For EP0 => Set done to HW
    if (ep->num==0)
       mUsbEP0DoneSet();
	ep->stopped = stopped;


    DBG_CTRLL(">>> (Interrupt Register=0x%x)(IntGroupMask=0x%x)...\n",mUsbIntSrc1MaskRd(),mUsbIntGroupMaskRd());
	DBG_FUNCC("-done() stopped=%d  \n",stopped);
}

/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
//****************************************************
// Name:CX_dma_Directly
// Description: Start DMA directly
//              <1>.For Control Command - Get Stayus (Only for ep0)
// Input:<1>.For Control Command - Get Stayus
//       <2>.status
// Output:none
//****************************************************
static int CX_dma_Directly(u8 *pu8Buffer, u32 u8Num,u8 bdir)
{

	u32            FIFO_Sel,wTemp,wDMABuffer,temp;
    u8             u8fifo_n;
	struct FTC_udc	*dev = the_controller;


	DBG_FUNCC("+CX_dma_Directly, start addr = 0x%x\n",(u32)pu8Buffer);


  //<1>.Get the FIFO Select
        u8fifo_n=0;
        FIFO_Sel=FOTG200_DMA2CxFIFO;



  //<2>.Map the DMA Buffer

		temp = u8Num / 16;
		if (u8Num % 16)
           temp++;
		wDMABuffer = pci_map_single((void *)dev, pu8Buffer,  temp*16, //USB_EPX_BUFSIZ
			bdir ? PCI_DMA_TODEVICE : PCI_DMA_FROMDEVICE);



  //<3>.Init DMA start register
       	// EP=0,1,2,3,4
        if (bdir)
	        mUsbDmaConfig(u8Num,DIRECTION_IN);
	    else
	        mUsbDmaConfig(u8Num,DIRECTION_OUT);

	    mUsbDMA2FIFOSel(FIFO_Sel);
	    mUsbDmaAddr((u32)wDMABuffer);

  //<4>.Enable the DMA
  	    mUsbDmaStart();

  //<5>.Waiting for DMA complete
	while(1)
	{
		wTemp = mUsbIntSrc2Rd();
		if(wTemp & BIT8)
		{
			mUsbIntDmaErrClr();
			printk("??? Cx IN DMA error..");
			break;
		}
		if(wTemp & BIT7)
		{
			mUsbIntDmaFinishClr();
			break;
		}
		if((wTemp & 0x00000007)>0)//If (Resume/Suspend/Reset) exit
		{
			mUsbIntDmaFinishClr();
			printk("???? Cx IN DMA stop because USB Resume/Suspend/Reset..");
			break;
		}
	}
	mUsbDMA2FIFOSel(FOTG200_DMA2FIFO_Non);


  //<6>.Unmap the DMA
		pci_unmap_single((void *)dev,wDMABuffer, temp*16,  //USB_EPX_BUFSIZ,  //req->req.length+32,
			bdir ? PCI_DMA_TODEVICE : PCI_DMA_FROMDEVICE);






	return 0;
}

static int start_dma(struct FTC_ep *ep, struct FTC_request *req)
{
    D(D_BLABLA, "start_dma()\n");
    struct FTC_udc *dev = ep->dev;
    u32 start = req->req.dma;
    u8 fifo = get_ep_fifo(ep);
    u32 fifo_mask = fifo==FIFOCX? FOTG200_DMA2CxFIFO : 1<<fifo;

    if (ep->num == 0) {
	if (req->req.length <= 64) {
	    ep->wDMA_Set_Length = req->req.length;
	} else {
	    ep->wDMA_Set_Length = 64;
	    mUsbIntEP0InEn();
	}
	mUsbDmaConfig(ep->wDMA_Set_Length, ep->is_in? DIRECTION_IN : DIRECTION_OUT);
    } else {
	if (likely(ep->is_in)) {
	    ep->wDMA_Set_Length=req->req.length;
	    mUsbDmaConfig(req->req.length,DIRECTION_IN);
	} else {
	    int fifo = get_ep_fifo(ep);
	    if ( mUsbFIFOOutByteCount(fifo) < mUsbEPMxPtSzRd(ep->num, DIRECTION_OUT) ) {
		ep->wDMA_Set_Length = mUsbFIFOOutByteCount(fifo);
	    } else {
		ep->wDMA_Set_Length = req->req.length;
	    }
	    mUsbDmaConfig(ep->wDMA_Set_Length, DIRECTION_OUT);
	}
    }

    dev->EPUseDMA = ep->num;

    mUsbDMA2FIFOSel(fifo_mask);
    mUsbDmaAddr(start);
    mUsbIntDmaErrEn();
    mUsbIntDmaFinishEn();
    mUsbDmaStart();

    return 0;
}

static void dma_ep0_continue(struct FTC_udc *dev, struct FTC_ep *ep)
{
    struct FTC_request *req;
    req = list_entry(ep->queue.next, struct FTC_request, queue);
    int length = req->req.length - req->req.actual;

    if (length <= 0) return;
    if (length <= 64) {
	ep->wDMA_Set_Length = length;
	mUsbIntEP0InDis();
    } else {
	ep->wDMA_Set_Length = 64;
	mUsbIntEP0InEn();
    }

    mUsbDmaConfig(ep->wDMA_Set_Length, ep->is_in? DIRECTION_IN : DIRECTION_OUT);
    mUsbDmaAddr(req->req.dma + req->req.actual);
    mUsbIntDmaErrEn();
    mUsbIntDmaFinishEn();
    mUsbDmaStart();
}

static void dma_complete(struct FTC_udc *dev, struct FTC_ep *ep)
{
    struct FTC_request *req;

    D(D_BLABLA, "dma_complete()\n");

    if (likely(!list_empty(&ep->queue))) {
	req = list_entry(ep->queue.next, struct FTC_request, queue);

	if (mUsbIntDmaErrRd() == 0) {
	    req->req.actual += ep->wDMA_Set_Length;
	    if (ep->wDMA_Set_Length == ep->ep.maxpacket
		    && req->req.actual < req->req.length) {
		return;
	    }
	} else {
	    D(D_ERROR, "DMA error\n");
	    req->req.actual=0;
	}

	done(ep, req, 0);
    }

    if (ep->num > 0 && !list_empty(&ep->queue)) {
	/* EP0 -> get next request from queue */
	req = list_entry(ep->queue.next, struct FTC_request, queue);
	start_dma(ep, req);
    } else {
	int fifo = get_ep_fifo(ep);
	mUsbDmaStop();
	mUsbIntDmaErrDis();
	mUsbIntDmaFinishDis();
	mUsbDMA2FIFOSel(FOTG200_DMA2FIFO_Non);
	dev->EPUseDMA =DMA_CHANNEL_FREE;
	ep->is_in? mUsbIntFXINDis(fifo) : mUsbIntFXOUTDis(fifo);
    }
}

//****************************************************
// Name:abort_dma
// Description: In FOTG200 abort_dma = reset dma
// Input:<1>.ep structure pointer
//       <2>.Status
// Output:none
//****************************************************
static void abort_dma(struct FTC_ep *ep, int status)
{
	struct FTC_request	*req;
	struct FTC_udc	*dev;
	u8 u8fifo_n;

	DBG_FUNCC("+abort_dma\n");

	req = list_entry(ep->queue.next, struct FTC_request, queue);

	/* FIXME using these resets isn't usably documented. this may
	 * not work unless it's followed by disabling the endpoint.
	 *
	 * FIXME the OUT reset path doesn't even behave consistently.
	 */

 //<1>.Checking => Finish
	if (mUsbIntDmaFinishRd()>0)
           goto finished;
	req->req.status = status;

	VDBG(ep->dev, "%s %s %s %d/%d\n", __FUNCTION__, ep->ep.name,
		ep->is_in ? "IN" : "OUT",
		req->req.actual, req->req.length);

    mUsbDMAResetSet();

    u8fifo_n=get_ep_fifo(ep);
	if (u8fifo_n==FIFOCX)
       mUsbCxFClr();
	else
        mUsbFIFOReset(u8fifo_n);

	return;

finished:
	///* dma already completed; no abort needed */
	//command(regs, COMMAND_FIFO_ENABLE, ep->num);
	req->req.actual = req->req.length;
	req->req.status = 0;

	mUsbDmaStop();
    mUsbIntDmaErrDis();
    mUsbIntDmaFinishDis();
    mUsbDMA2FIFOSel(FOTG200_DMA2FIFO_Non);


    dev = ep->dev;
    dev->EPUseDMA =DMA_CHANNEL_FREE;


}

void handle_fifo_irq(struct FTC_udc *dev, u8 fifo)
{
    D(D_BLABLA, "handle_fifo_irq(FIFO #%d)\n", fifo);
    int ep_num = mUsbFIFOMapRd(fifo) & 0x0f;
    struct FTC_ep *ep = &(dev->ep[ep_num]);

    if (dev->EPUseDMA == DMA_CHANNEL_FREE && !list_empty(&ep->queue)) {
	struct FTC_request *req;
	req = list_entry(ep->queue.next, struct FTC_request, queue);
	start_dma(ep, req);
    }
    /* No more requests or DMA busy; disable FIFO IRQs and wait */
    ep->is_in? mUsbIntFXINDis(fifo) : mUsbIntFXOUTDis(fifo);
}


//****************************************************
// Name:FTC_queue
// Description:
// Input:<1>.ep structure point
//       <2>.status
//       <3>.flag
// Output:none
//****************************************************
static int
FTC_queue(struct usb_ep *_ep, struct usb_request *_req, gfp_flags_t gfp_flags)
{
    struct FTC_request *req;
    struct FTC_ep *ep;
    struct FTC_udc *dev;
    unsigned long  flags;
    int status;
    u32 temp;

    D(D_BLABLA, "FTC_queue()\n");

    req = container_of(_req, struct FTC_request, req);
    if (unlikely(!_req || !_req->complete
		|| !_req->buf || !list_empty(&req->queue))) {
	return -EINVAL;
    }
    ep = container_of(_ep, struct FTC_ep, ep);
    if (unlikely(!_ep || (!ep->desc && ep->num != 0))) {
	return -EINVAL;
    }

    if (ep->num==0 && req->req.length==0) {
	mUsbEP0DoneSet();
	mUsbCfgSet(); //Temp Solution for Set Configuration
	return 0;
    }

    dev = ep->dev;
    if (unlikely(!dev->driver || dev->gadget.speed == USB_SPEED_UNKNOWN)) {
	DBG_CTRLL("??? FTC_queue => return -ESHUTDOWN\n");
	return -ESHUTDOWN;
    }
    if (dev->ep0state == EP0_SUSPEND) {
	D(D_ERROR, "Can't queue request while EP0 suspended.\n");
	return -EBUSY;
    }

    if (ep->dma && _req->dma == DMA_ADDR_INVALID) {
	// important : DMA length will set as 16*n bytes
	temp = _req->length / 16;
	if (_req->length % 16) temp++;
	_req->dma = pci_map_single((void *)dev, _req->buf, temp*16, //USB_EPX_BUFSIZ,
		ep->is_in ? PCI_DMA_TODEVICE : PCI_DMA_FROMDEVICE);
	req->mapped = 1;
    }

    spin_lock_irqsave(&dev->lock, flags);

    _req->status = -EINPROGRESS;
    _req->actual = 0;

    /* for ep0 IN without premature status, zlp is required and
     * writing EOP starts the status stage (OUT).
     */
#if 0
    if (unlikely(ep->num == 0 && ep->is_in)) {
	_req->zero = 1;
    }
#endif

    /* kickstart this i/o queue? */
    status = 0;

    if (list_empty(&ep->queue) && likely(!ep->stopped)) {
	// if (dev->EPUseDMA == DMA_CHANNEL_FREE)
	if (ep->num == 0) {
	    status = start_dma(ep, req);
	    if (unlikely(status != 0)) {
		if (status > 0) status = 0;
		req = 0;
	    }
	}
    }

    //Add request to queue
    if (likely(req != 0)) list_add_tail(&req->queue, &ep->queue);

    //Enable the FIFO Interrupt
    if (likely(ep->num > 0)) {
	u8 fifo = get_ep_fifo(ep);
	ep->is_in? mUsbIntFXINEn(fifo) : mUsbIntFXOUTEn(fifo);
    }

    spin_unlock_irqrestore(&dev->lock, flags);
    return status;
}

//****************************************************
// Name:nuke
// Description: dequeue ALL requests
// Input:<1>.ep structure point
//       <2>.status
// Output:none
//****************************************************
/* dequeue ALL requests */
static void nuke(struct FTC_ep *ep, int status)
{
	struct FTC_request	*req;

    DBG_FUNCC("+nuke() ep addr= 0x%x\n", (u32) ep);

	ep->stopped = 1;
	if (list_empty(&ep->queue))
		return;

	while (!list_empty(&ep->queue)) {
		req = list_entry(ep->queue.next, struct FTC_request, queue);
		done(ep, req, status);
	}
}
//****************************************************
// Name:FTC_dequeue
// Description: dequeue JUST ONE request
// Input:<1>.ep structure point
//       <2>.request structure pointer
// Output:none
//****************************************************
/* dequeue JUST ONE request */
static int FTC_dequeue(struct usb_ep *_ep, struct usb_request *_req)
{
	struct FTC_request	*req;
	struct FTC_ep		*ep;
	struct FTC_udc		*dev;
	unsigned long		flags;

	DBG_FUNCC("+FTC_dequeue()\n");

  //<1>.Checking the input
	ep = container_of(_ep, struct FTC_ep, ep);
	if (!_ep || !_req || (!ep->desc && ep->num != 0))
		return -EINVAL;
	dev = ep->dev;
	if (!dev->driver)
		return -ESHUTDOWN;

	/* we can't touch (dma) registers when suspended */
	if (dev->ep0state == EP0_SUSPEND)
		return -EBUSY;

	VDBG(dev, "%s %s %s %s %p\n", __FUNCTION__, _ep->name,
		ep->is_in ? "IN" : "OUT",
		ep->dma ? "dma" : "pio",
		_req);

	spin_lock_irqsave(&dev->lock, flags);

	/* make sure it's actually queued on this endpoint */
	list_for_each_entry (req, &ep->queue, queue) {
		if (&req->req == _req)
			break;
	}
	if (&req->req != _req) {
		spin_unlock_irqrestore (&dev->lock, flags);
		return -EINVAL;
	}

  //<2>.dequeue the current req
    if (ep->dma && ep->queue.next == &req->queue && !ep->stopped) {
		abort_dma(ep, -ECONNRESET);
		done(ep, req, -ECONNRESET);
		dma_complete(dev, ep);
	}
	else
	 if (!list_empty(&req->queue))
		done(ep, req, -ECONNRESET);
	else
		req = 0;
	spin_unlock_irqrestore(&dev->lock, flags);

	return req ? 0 : -EOPNOTSUPP;
}
//****************************************************
// Name:FTC_clear_halt
// Description: clear the halt ep
// Input:ep structure point
// Output:none
//****************************************************
/*-------------------------------------------------------------------------*/
static void FTC_clear_halt(struct FTC_ep *ep)
{
   struct FTC_request	*req;

	DBG_FUNCC("+FTC_clear_halt()(ep->num=%d)\n",ep->num);

	VDBG(ep->dev, "%s clear halt\n", ep->ep.name);

 //<1>.Set register(Rst_Toggle)
    if (ep->num>0)
       { if (ep->is_in)	// IN direction ?
	    {
          DBG_BULKK("FTC_udc==>FTC_clear_halt()==>IN direction \n");

	       mUsbEPinRsTgSet(ep->num);	// Set Rst_Toggle Bit
	       mUsbEPinRsTgClr(ep->num);	// Clear Rst_Toggle Bit
	       mUsbEPinStallClr(ep->num);	// Clear Stall Bit
	    }
	    else
	    {
           DBG_BULKK("FTC_udc==>FTC_clear_halt()==>OUT direction \n");

	       mUsbEPoutRsTgSet(ep->num);	// Set Rst_Toggle Bit
	       mUsbEPoutRsTgClr(ep->num);	// Clear Rst_Toggle Bit
	       mUsbEPoutStallClr(ep->num);	// Clear Stall Bit
	    }
	   }
	DBG_BULKK("FTC_udc==>FTC_clear_halt()==>ep->stopped = %d\n",ep->stopped);

  //<2>.Start next request
	if (ep->stopped) {
		ep->stopped = 0;

		if (list_empty(&ep->queue))
			return;
		req = list_entry(ep->queue.next, struct FTC_request,
						queue);
		(void) start_dma(ep, req);
	}


}
//****************************************************
// Name:FTC_set_halt
// Description:set halt
// Input:ep structure point
// Output:
//****************************************************
static int FTC_set_halt(struct usb_ep *_ep, int value)
{
	struct FTC_ep	*ep;
	unsigned long	flags;
	int		retval = 0;

	DBG_FUNCC("+FTC_set_halt()\n");

  //<1>.Checking input
	if (!_ep)
		return -ENODEV;
	ep = container_of (_ep, struct FTC_ep, ep);

	DBG_BULKK("FTC_set_halt()===> (ep->num=%d)(Value=%d)\n",ep->num,value);

  //<2>.Processing ep=0
	if (ep->num == 0) {
		if (value) {
			ep->dev->ep0state = EP0_STALL;
			ep->dev->ep[0].stopped = 1;
			mUsbEP0StallSet();
			return 0;
		} else
			return -EINVAL;

	/* don't change EPxSTATUS_EP_INVALID to READY */
	} else if (!ep->desc) {
		VDBG(ep->dev, "%s %s inactive?\n", __FUNCTION__, ep->ep.name);
		return -EINVAL;
	}

	spin_lock_irqsave(&ep->dev->lock, flags);
	if (!list_empty(&ep->queue))
		retval = -EAGAIN;
	else if (!value)
		FTC_clear_halt(ep);
	else {
		ep->stopped = 1;
		VDBG(ep->dev, "%s set halt\n", ep->ep.name);
	    if (ep->is_in)	// IN direction ?
	        mUsbEPinStallSet(ep->num);		// Set in Stall Bit
	    else
	        mUsbEPoutStallSet(ep->num);		// Set out Stall Bit
	}
	spin_unlock_irqrestore(&ep->dev->lock, flags);
	return retval;

}
//****************************************************
// Name:FTC_fifo_status
// Description:Get the size of data in FIFO-X
// Input:ep structure point
// Output:size
//****************************************************
int FTC_fifo_status(struct usb_ep *_ep)
{
	struct FTC_ep *ep;

	u32	   size;

	DBG_FUNCC("+FTC_fifo_status() => In FOTG200 => Only support DMA mode... \n");

	if (!_ep)
		return -ENODEV;
	ep = container_of(_ep, struct FTC_ep, ep);

    size =0 ;
	return size;
}
//****************************************************
// Name:FTC_fifo_flush
// Description:Clear the FIFO
// Input:ep structure pointer
// Output:none
//****************************************************
static void FTC_fifo_flush(struct usb_ep *_ep)
{
	struct FTC_ep *ep;
    u8     u8fifo_n;   //john

	DBG_FUNCC("+FTC_fifo_flush()\n");

	if (!_ep)
		return;
	ep = container_of(_ep, struct FTC_ep, ep);
	VDBG(ep->dev, "%s %s\n", __FUNCTION__, ep->ep.name);

	/* don't change EPxSTATUS_EP_INVALID to READY */
	if (!ep->desc && ep->num != 0) {
		VDBG(ep->dev, "%s %s inactive?\n", __FUNCTION__, ep->ep.name);
		return;
	}

	//For OTG200
	if (ep->num ==0) {   //EP0
       mUsbCxFClr();
	}
    else {
       u8fifo_n = mUsbEPMapRd(ep->num);		// get the relatived FIFO number
	   if (ep->is_in)
	      u8fifo_n &= 0x0F;
	   else
	      u8fifo_n >>= 4;
       if (u8fifo_n >= MAX_FIFO_NUM)	// over the Max. fifo count ?
          return;

	   // Check the FIFO had been enable ?
	   if ((mUsbFIFOConfigRd(u8fifo_n) & FIFOEnBit) == 0)
	      return;

       mUsbFIFOReset(u8fifo_n);   //reset FIFO
       udelay(10);
       mUsbFIFOResetOK(u8fifo_n);   //reset FIFO finish
	}
	return;
}

static struct usb_ep_ops FTC_ep_ops = {
	.enable		= FTC_ep_enable,
	.disable	= FTC_ep_disable,

	.alloc_request	= FTC_alloc_request,
	.free_request	= FTC_free_request,

	.alloc_buffer	= FTC_alloc_buffer,
	.free_buffer	= FTC_free_buffer,

	.queue		= FTC_queue,
	.dequeue	= FTC_dequeue,

	.set_halt	 = FTC_set_halt,
	.fifo_status = FTC_fifo_status,
	.fifo_flush	 = FTC_fifo_flush,
};

/*-------------------------------------------------------------------------*/

static int FTC_get_frame(struct usb_gadget *_gadget)
{
	struct FTC_udc	*dev;
	u16 retval;
	unsigned long	flags;

	DBG_FUNCC("+FTC_get_frame()\n");

	if (!_gadget)
		return -ENODEV;
	dev = container_of (_gadget, struct FTC_udc, gadget);
	spin_lock_irqsave (&dev->lock, flags);
	retval = mUsbFrameNo();
	spin_unlock_irqrestore (&dev->lock, flags);

	return retval;
}

static int FTC_wakeup(struct usb_gadget *_gadget)
{
	struct FTC_udc	*dev;
	unsigned long	flags;

    DBG_FUNCC("+FTC_wakeup()\n");

	if (!_gadget)
		return -ENODEV;
	dev = container_of (_gadget, struct FTC_udc, gadget);
	spin_lock_irqsave (&dev->lock, flags);

        DBG_TEMP_FUN(">>> dev->devstat=%d\n",dev->devstat);

#ifdef USE_OTG
	if (mUsb_TGC_Control_A_VBUS_VLD_Rd()) {
		/* NOTE:  OTG spec erratum says that OTG devices may
		 * issue wakeups without host enable.
		 */
		//if (dev->devstat & (UDC_B_HNP_ENABLE|UDC_R_WK_OK)) {
		INFO(dev,"remote wakeup...\n");
 	        // Set "Device_Remote_Wakeup", Turn on the"RMWKUP" bit in Mode Register
	        mUsbRmWkupSet();



	/* NOTE:  non-OTG systems may use SRP TOO... */
	} else if (dev->transceiver)
		   otg_start_srp(dev->transceiver);
#endif

	spin_unlock_irqrestore (&dev->lock, flags);

	return 0;
}

static const struct usb_gadget_ops FTC_ops = {
	.get_frame	     = FTC_get_frame,
	.wakeup		     = FTC_wakeup,
	.set_selfpowered = NULL,
	.ioctl           = NULL
};

/*-------------------------------------------------------------------------*/

static void vFOTG200_Dev_Init(void)
{
    D(D_BLABLA, "vFOTG200_Dev_Init()\n");

    /* Clear interrupt bits (bus reset, suspend and resume) */
    mUsbIntBusRstClr();
    mUsbIntSuspClr();
    mUsbIntResmClr();

    /* Disable all fifo interrupts */
    mUsbIntFIFO0_3OUTDis();
    mUsbIntFIFO0_3INDis();

    /* Soft Reset */
    mUsbSoftRstSet();
    mUsbSoftRstClr();

    /* Initial values for FIFO/EP maps */
    mUsbFIFOMap(0, 0x0f);
    mUsbFIFOMap(1, 0x0f);
    mUsbFIFOMap(2, 0x0f);
    mUsbFIFOMap(3, 0x0f);

    /* Clear all FIFO buffers */
    mUsbClrAllFIFOSet();

    /* Disable interrupts for EP0 */
    mUsbIntEP0EndDis();
    mUsbIntEP0InDis();
    mUsbIntEP0OutDis();
}

static void vUsbInit(struct FTC_udc *dev)
{
    D(D_BLABLA, "vUsbInit()\n");
    dev->u16TxRxCounter = 0;
    dev->eUsbCxCommand = CMD_VOID;
    dev->u8UsbConfigValue = 0;
    dev->u8UsbInterfaceValue = 0;
    dev->u8UsbInterfaceAlternateSetting = 0;
    vFOTG200_Dev_Init();
}

static void udc_reinit(struct FTC_udc *dev)
{
    D(D_BLABLA, "udc_reinit()\n");
    unsigned i;

    INIT_LIST_HEAD(&dev->gadget.ep_list);
    dev->gadget.ep0 = &dev->ep[0].ep;
    dev->gadget.speed = USB_SPEED_UNKNOWN;
    dev->ep0state = EP0_DISCONNECT;
    dev->irqs = 0;

    for (i=0; i<=MAX_EP_NUM; i++) {
	struct FTC_ep *ep = &dev->ep[i];
	ep->num = i;
	ep->ep.name = names[i];
	ep->ep.ops = &FTC_ep_ops;
	list_add_tail (&ep->ep.ep_list, &dev->gadget.ep_list);
	ep->dev = dev;
	INIT_LIST_HEAD (&ep->queue);
	ep_reset(ep);
	dev->ep[i].irqs = 0;
    }

    dev->ep[0].ep.maxpacket = MAX_EP0_SIZE;
    list_del_init (&dev->ep[0].ep.ep_list);
}

static void udc_reset(struct FTC_udc *dev)
{
    D(D_BLABLA, "udc_reset()\n");
    u8 i;

    dev->Dma_Status = DMA_Mode;
    dev->u8LineCount = 0;
    mUsbDMAResetSet();

    for (i=0; i<MAX_FIFO_NUM; i++) mUsbFIFOReset(i);

    vUsbInit(dev);
    dev->EPUseDMA = DMA_CHANNEL_FREE;
}

static void udc_enable(struct FTC_udc *dev)
{
    D(D_BLABLA, "udc_enable()\n");
    mUsbGlobIntEnSet();
    mUsbChipEnSet();
    /* Clear unplug bit; host will now detect the device */
    mUsbUnPLGClr();

}

/*-------------------------------------------------------------------------*/

/* keeping it simple:
 * - one bus driver, initted first;
 * - one function driver, initted second
 */

/* when a driver is successfully registered, it will receive
 * control requests including set_configuration(), which enables
 * non-control requests.  then usb traffic follows until a
 * disconnect is reported.  then a host may connect again, or
 * the driver might get unbound.
 */
int fotg200_usb_gadget_register_driver(struct usb_gadget_driver *driver)
{
	struct FTC_udc	*dev = the_controller;
	int	   retval;

	DBG_FUNCC("+fotg200_usb_gadget_register_driver()\n");

	if (!driver
			|| driver->speed != USB_SPEED_HIGH
			|| !driver->bind
			|| !driver->unbind
			|| !driver->disconnect
			|| !driver->setup)
		return -EINVAL;
	if (!dev)
		return -ENODEV;
	if (dev->driver)
		return -EBUSY;

	/* hook up the driver */
	dev->driver = driver;
	retval = driver->bind(&dev->gadget);
	if (retval) {
		DBG(dev, "bind to driver %s --> error %d\n",
				driver->driver.name, retval);
		dev->driver = 0;
		return retval;
	}

//For OTG;;Start
#ifdef CONFIG_FARADAY_FOTG2XX
	if (dev->transceiver) {
	    int	 status;
		status = otg_set_peripheral(dev->transceiver, &dev->gadget);
		if (status < 0) {
			ERROR(dev,"can't bind to transceiver\n");
			driver->unbind (&dev->gadget);

			dev->driver = 0;
		    return status;
		}
	}
#endif
//For OTG;;End

	/* then enable host detection and ep0; and we're ready
	 * for set_configuration as well as eventual disconnect.
	 */
	INFO(dev,"enable USB device by user program after system bootup\n");
	udc_enable(dev);

	DBG(dev, "registered gadget driver '%s'\n", driver->driver.name);

	DBG_FUNCC("-fotg200_usb_gadget_register_driver()\n");

	return 0;
}
//EXPORT_SYMBOL(fotg200_usb_gadget_register_driver);

static void
stop_activity(struct FTC_udc *dev, struct usb_gadget_driver *driver)
{
    unsigned i;

    DBG (dev, "%s\n", __FUNCTION__);

    if (dev->gadget.speed == USB_SPEED_UNKNOWN) {
	driver = NULL;
    }

    //udc_reset (dev);
    for (i = 0; i <= MAX_EP_NUM; i++) {
	if (dev->ep[i].desc) nuke(&dev->ep [i], -ESHUTDOWN);
    }
    if (driver) {
	spin_unlock(&dev->lock);
	//driver->disconnect(&dev->gadget);
	spin_lock(&dev->lock);
    }
    if (dev->driver) udc_enable(dev);
}

int fotg200_usb_gadget_unregister_driver(struct usb_gadget_driver *driver)
{
	struct FTC_udc	*dev = the_controller;
	unsigned long	flags;

	DBG_FUNCC("+fotg200_usb_gadget_unregister_driver()\n");

	if (!dev)
		return -ENODEV;
	if (!driver || driver != dev->driver)
		return -EINVAL;

	spin_lock_irqsave(&dev->lock, flags);

	/* Disconnect the device from USB */
	mUsbUnPLGSet();
	mdelay(20);

#ifdef USE_OTG
    if (dev->transceiver)
		(void) otg_set_peripheral(dev->transceiver, 0);
#endif

	dev->driver = 0;
	spin_unlock_irqrestore(&dev->lock, flags);

	driver->unbind(&dev->gadget);
	stop_activity(dev, driver);

	DBG(dev, "unregistered driver '%s'\n", driver->driver.name);
	return 0;
}
//EXPORT_SYMBOL(fotg200_usb_gadget_unregister_driver);


/*-------------------------------------------------------------------------*/

///////////////////////////////////////////////////////////////////////////////
//		vUsb_rst(struct FTC_udc	*dev)
//		Description:
//			1. Change descriptor table (High or Full speed).
//		input: none
//		output: none
///////////////////////////////////////////////////////////////////////////////
static void vUsb_rst(struct FTC_udc	*dev)
{

	DBG_FUNCC("+vUsb_rst()\n");

// stop
	INFO(dev,"L%x, Bus reset\n", dev->u8LineCount ++);

	mUsbIntBusRstClr();
	dev->gadget.speed = USB_SPEED_UNKNOWN;




}

///////////////////////////////////////////////////////////////////////////////
//		vUsb_suspend(dev)
//		Description:
//			1. .
//		input: none
//		output: none
///////////////////////////////////////////////////////////////////////////////
static void vUsb_suspend(struct FTC_udc	*dev)
{
	DBG_FUNCC("+vUsb_suspend()\n");

	INFO(dev,"L%x, Bus suspend\n", dev->u8LineCount ++);
	// uP must do-over everything it should handle and do before into the suspend mode
	// Go Suspend status



	mUsbIntSuspClr();
	//Bruce;;mUsbGoSuspend();
#ifdef USE_OTG
    dev->gadget.b_hnp_enable = 0;
#endif
    dev->ep0state = EP0_SUSPEND;
}

///////////////////////////////////////////////////////////////////////////////
//		vUsb_resm(struct FTC_udc	*dev)
//		Description:
//			1. Change descriptor table (High or Full speed).
//		input: none
//		output: none
///////////////////////////////////////////////////////////////////////////////
static void vUsb_resm(struct FTC_udc	*dev)
{
	DBG_FUNCC("+vUsb_resm()\n");

	INFO(dev,"L%x, Bus resume\n", dev->u8LineCount ++);
	// uP must do-over everything it should handle and do before into the suspend mode
	// uP must wakeup immediately
	mUsbIntResmClr();

    dev->ep0state = EP0_IDLE;
}



void vUsbClrEPx(void)
{
	u8 u8ep;

	DBG_FUNCC("+vUsbClrEPx()\n");

	// Clear All EPx Toggle Bit
	for (u8ep = 1; u8ep <= MAX_EP_NUM; u8ep ++)
	{
		mUsbEPinRsTgSet(u8ep);
		mUsbEPinRsTgClr(u8ep);
	}
	for (u8ep = 1; u8ep <= MAX_EP_NUM; u8ep ++)
	{
		mUsbEPoutRsTgSet(u8ep);
		mUsbEPoutRsTgClr(u8ep);
	}
}


///////////////////////////////////////////////////////////////////////////////
//		bGet_status(struct FTC_udc *dev)
//		Description:
//			1. Send 2 bytes status to host.
//		input: none
//		output: TRUE or FALSE (u8)
///////////////////////////////////////////////////////////////////////////////
static u8 bGet_status(struct FTC_udc *dev,const struct usb_ctrlrequest *ctrl)
{

	u8 u8ep_n,u8fifo_n,RecipientStatusLow, RecipientStatusHigh;
	u8 u8Tmp[2];
	u8 bdir;

	DBG_FUNCC("+bGet_status()  \n");

	RecipientStatusLow = 0;
	RecipientStatusHigh = 0;
	switch((ctrl->bRequestType)&0x3)  // Judge which recipient type is at first
	{
    		case 0:					// Device
        	// Return 2-byte's Device status (Bit1:Remote_Wakeup, Bit0:Self_Powered) to Host
        	// Notice that the programe sequence of RecipientStatus
			RecipientStatusLow = mUsbRmWkupST() << 1;
			// Bit0: Self_Powered--> DescriptorTable[0x23], D6(Bit 6)
			// Now we force device return data as self power. (Andrew)
			RecipientStatusLow |= ((USB_CONFIG_ATT_SELFPOWER >> 6) & 0x01);
        	break;
 		case 1:					// Interface
			// Return 2-byte ZEROs Interface status to Host
    		break;

		case 2:					// Endpoint
			if(ctrl->wIndex == 0x00)
			{
				if(dev->ep0state == EP0_STALL)
	       			RecipientStatusLow = TRUE;
			}
			else
			{
				u8ep_n = (u8)ctrl->wIndex  & 0x7F;		// which ep will be clear
				bdir = (u8)ctrl->wIndex >> 7;			// the direction of this ep
				if (u8ep_n > MAX_EP_NUM)			// over the Max. ep count ?
					{return FALSE;

					}
				else
				{
					u8fifo_n = mUsbEPMapRd(u8ep_n);		// get the relatived FIFO number
					if (bdir == 1)
						u8fifo_n &= 0x0F;
					else
						u8fifo_n >>= 4;
					if (u8fifo_n >= MAX_FIFO_NUM)	// over the Max. fifo count ?
						{

							return FALSE;
						}

														// Check the FIFO had been enable ?

					if (bdir == 1)						// IN direction ?
						RecipientStatusLow = mUsbEPinStallST(u8ep_n);
					else
						RecipientStatusLow = mUsbEPoutStallST(u8ep_n);
				}
			}
        	break;
		default :

			return FALSE;
	}

	// return RecipientStatus;
	u8Tmp[0] = RecipientStatusLow;
	u8Tmp[1] = RecipientStatusHigh;

	//Use DMA to transfer data
	CX_dma_Directly(u8Tmp,2,1);


	mUsbEP0DoneSet();

	return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
//		bClear_feature(struct FTC_udc *dev)
//		Description:
//			1. Send 2 bytes status to host.
//		input: none
//		output: TRUE or FALSE (u8)
///////////////////////////////////////////////////////////////////////////////
static u8 bClear_feature(struct FTC_udc *dev,const struct usb_ctrlrequest *ctrl)
{

   u8 u8ep_n;
   u8 u8fifo_n;
   u8 bdir;

   DBG_FUNCC("+bClear_feature()\n");

   switch (ctrl->wValue)		// FeatureSelector
   {
      case 0:		// ENDPOINT_HALE
		 // Clear "Endpoint_Halt", Turn off the "STALL" bit in Endpoint Control Function Register
		 if(ctrl->wIndex == 0x00)
		   u8ep_n=0;  //Sp0 ed clear feature
		 else
		 {
		    u8ep_n = ctrl->wIndex & 0x7F;		// which ep will be clear
			bdir = ctrl->wIndex >> 7;			// the direction of this ep
			if (u8ep_n > MAX_EP_NUM)			// over the Max. ep count ?
			   return FALSE;
			else
			{
		 	   u8fifo_n = get_ep_fifo(&dev->ep[u8ep_n]);		// get the relatived FIFO number

			   if (u8fifo_n<MAX_FIFO_NUM)
			       if ((mUsbFIFOConfigRd(u8fifo_n) & FIFOEnBit) == 0) // Check the FIFO had been enable ?
			           return FALSE;

			}


		 }
			FTC_clear_halt(&dev->ep[u8ep_n]);


         break;

 	  case 1 :   		// Device Remote Wakeup
	 	 // Clear "Device_Remote_Wakeup", Turn off the"RMWKUP" bit in Main Control Register
		 mUsbRmWkupClr();
         break;

	  case 2 :   		// Test Mode
	 	 return FALSE;



	  default :
		 return FALSE;
	}

	mUsbEP0DoneSet();


	return TRUE;
}



///////////////////////////////////////////////////////////////////////////////
//		bSet_feature(struct FTC_udc *dev)
//		Description:
//			1. Send 2 bytes status to host.
//		input: none
//		output: TRUE or FALSE (u8)
///////////////////////////////////////////////////////////////////////////////
static u8 bSet_feature(struct FTC_udc *dev,const struct usb_ctrlrequest *ctrl)
{


	u8 i;
	u8 u8ep_n;
	u8 u8fifo_n;
	u8 u8Tmp[52];
	u8 * pp;
	u8 bdir;

	DBG_FUNCC("+bSet_feature()\n");


	switch (ctrl->wValue)		// FeatureSelector
	{
		case 0:	// ENDPOINT_HALE
			// Set "Endpoint_Halt", Turn on the "STALL" bit in Endpoint Control Function Register
			if(ctrl->wIndex == 0x00)
			   FTC_set_halt(dev->gadget.ep0, 1);  // Return EP0_Stall
			else
			{
 			 u8ep_n = ctrl->wIndex & 0x7F;		// which ep will be clear
 		     bdir = ctrl->wIndex >> 7;			// the direction of this ep
			 u8fifo_n = get_ep_fifo(&dev->ep[u8ep_n]);		// get the relatived FIFO number
			 if (u8fifo_n<MAX_FIFO_NUM)											// Check the FIFO had been enable ?
			    if ((mUsbFIFOConfigRd(u8fifo_n) & FIFOEnBit) == 0)
   			       return FALSE;


			if (bdir == 1)						// IN direction ?
				mUsbEPinStallSet(u8ep_n);		// Clear Stall Bit
			else
				mUsbEPoutStallSet(u8ep_n);		// Set Stall Bit

			}
			break;
 		case 1 :   		// Device Remote Wakeup
 			// Set "Device_Remote_Wakeup", Turn on the"RMWKUP" bit in Mode Register
			mUsbRmWkupSet();

            break;



		case 2 :   		// Test Mode
			switch ((ctrl->wIndex >> 8))	// TestSelector
			{
				case 0x1:	// Test_J
					mUsbTsMdWr(TEST_J);
					break;

				case 0x2:	// Test_K
					mUsbTsMdWr(TEST_K);
					break;

				case 0x3:	// TEST_SE0_NAK
					mUsbTsMdWr(TEST_SE0_NAK);
					break;

				case 0x4:	// Test_Packet
					mUsbTsMdWr(TEST_PKY);
					mUsbEP0DoneSet();			// special case: follow the test sequence
					//////////////////////////////////////////////
					// Jay ask to modify, 91-6-5 (Begin)		//
					//////////////////////////////////////////////
					pp = u8Tmp;
					for (i=0; i<9; i++)			// JKJKJKJK x 9
					{
						(*pp) = (0x00);
						pp ++;
					}

					(*pp) = (0xAA);
					pp ++;
					(*pp) = (0x00);
					pp ++;

					for (i=0; i<8; i++)			// 8*AA
					{
						(*pp) = (0xAA);
						pp ++;
					}

					for (i=0; i<8; i++)			// 8*EE
					{
						(*pp) = (0xEE);
						pp ++;
					}
					(*pp) = (0xFE);
					pp ++;

					for (i=0; i<11; i++)		// 11*FF
					{
						(*pp) = (0xFF);
						pp ++;
					}

					(*pp) = (0x7F);
					pp ++;
					(*pp) = (0xBF);
					pp ++;
					(*pp) = (0xDF);
					pp ++;
					(*pp) = (0xEF);
					pp ++;
					(*pp) = (0xF7);
					pp ++;
					(*pp) = (0xFB);
					pp ++;
					(*pp) = (0xFD);
					pp ++;
					(*pp) = (0xFC);
					pp ++;
					(*pp) = (0x7E);
					pp ++;
					(*pp) = (0xBF);
					pp ++;
					(*pp) = (0xDF);
					pp ++;
					(*pp) = (0xFB);
					pp ++;
					(*pp) = (0xFD);
					pp ++;
					(*pp) = (0xFB);
					pp ++;
					(*pp) = (0xFD);
					pp ++;
					(*pp) = (0x7E);
		            CX_dma_Directly(u8Tmp,52,1);
					//////////////////////////////////////////////
					// Jay ask to modify, 91-6-5 (End)			//
					//////////////////////////////////////////////

					// Turn on "r_test_packet_done" bit(flag) (Bit 5)
					mUsbTsPkDoneSet();
					break;

				case 0x5:	// Test_Force_Enable
					//FUSBPort[0x08] = 0x20;	//Start Test_Force_Enable
					break;

				default:
					return FALSE;
			}
	         	break;

#ifdef USE_OTG
 		case 3 :   		//For OTG => b_hnp_enable

	          dev->gadget.b_hnp_enable = 1;
  		//<1>.Set b_Bus_Request
                  mUsb_OTGC_Control_B_BUS_REQ_Set();

  	        //<2>.Set the HNP enable
	          mUsb_OTGC_Control_B_HNP_EN_Set();

            break;

 		case 4 :   		//For OTG => b_hnp_enable

	          dev->gadget.a_hnp_support = 1;

            break;
 		case 5 :   		//For OTG => b_hnp_enable

	          dev->gadget.a_alt_hnp_support = 1;
  			 printk(">>> Please Connect to an alternate port on the A-device for HNP...\n");
#endif

            break;

		default :
			return FALSE;
	}

   mUsbEP0DoneSet();

	return TRUE;

}


///////////////////////////////////////////////////////////////////////////////
//		bSynch_frame(struct FTC_udc *dev)
//		Description:
//			1. If the EP is a Iso EP, then return the 2 bytes Frame number.
//				 else stall this command
//		input: none
//		output: TRUE or FALSE
///////////////////////////////////////////////////////////////////////////////
static u8 bSynch_frame(struct FTC_udc *dev,const struct usb_ctrlrequest *ctrl)
{


	DBG_FUNCC("+bSynch_frame()  ==> add by Andrew\n");

	if((ctrl->wIndex==0)||(ctrl->wIndex>4))
		return FALSE;

	// Does the Endpoint support Isochronous transfer type?
	   mUsbEP0DoneSet();
		return TRUE;
}


///////////////////////////////////////////////////////////////////////////////
//		bSet_address(struct FTC_udc *dev)
//		Description:
//			1. Set addr to FUSB200 register.
//		input: none
//		output: TRUE or FALSE (u8)
///////////////////////////////////////////////////////////////////////////////
static u8 bSet_address(struct FTC_udc *dev,const struct usb_ctrlrequest *ctrl)
{
	DBG_FUNCC("+bSet_address() = %d\n", ctrl->wValue);

	if (ctrl->wValue >= 0x0100)
		return FALSE;
	else
	{
		mUsbDevAddrSet(ctrl->wValue);
		mUsbEP0DoneSet();
		return TRUE;
	}
}


///////////////////////////////////////////////////////////////////////////////
//		vUsb_ep0setup(struct FTC_udc *dev)
//		Description:
//			1. Read the speed
//			2. Read 8-byte setup packet.
//          3. Process the standard command:
//             <1>.bSet_address
//
//
//		input: none
//		output: none
///////////////////////////////////////////////////////////////////////////////
static void vUsb_ep0setup(struct FTC_udc *dev)
{
    D(D_BLABLA, "vUsb_ep0setup()\n");
    u8 u8UsbCmd[8];
    struct usb_ctrlrequest ctrl;
    int tmp;
    u32 u32UsbCmd[2];

    if (dev->gadget.speed == USB_SPEED_UNKNOWN) {
	/* First EP0 command after usb reset, check usb speed now. */
	if (mUsbHighSpeedST()) {
	    INFO(dev,"L%x, high speed mode\n", dev->u8LineCount ++);
	    dev->gadget.speed = USB_SPEED_HIGH;
//	    vUsbFIFO_EPxCfg_HS();//Set the FIFO information
	} else {
	    INFO(dev,"L%x, full speed mode\n", dev->u8LineCount ++);
	    dev->gadget.speed = USB_SPEED_FULL;
//	    vUsbFIFO_EPxCfg_FS();//Set the FIFO information
	}
	dev->ep0state = EP0_IDLE;
    }

    nuke(&dev->ep[0], 0);
    dev->ep[0].stopped = 0;

    // Read 8-byte setup packet from FIFO
    mUsbDMA2FIFOSel(FOTG200_DMA2CxFIFO);
    u32UsbCmd[0] = mUsbEP0CmdDataRdDWord();
    u32UsbCmd[1] = mUsbEP0CmdDataRdDWord();
    mUsbDMA2FIFOSel(FOTG200_DMA2FIFO_Non);
    memcpy(u8UsbCmd, u32UsbCmd, 8);

    D(D_BLABLA, "EP0Cmd: %02x %02x %02x %02x %02x %02x %02x %02x\n",
	    u8UsbCmd[0], u8UsbCmd[1], u8UsbCmd[2], u8UsbCmd[3],
	    u8UsbCmd[4], u8UsbCmd[5], u8UsbCmd[6], u8UsbCmd[7]);

    /* read SETUP packet and enter DATA stage */
    ctrl.bRequestType = u8UsbCmd[0];
    ctrl.bRequest = u8UsbCmd[1];
    ctrl.wValue = (u8UsbCmd[3] << 8) | u8UsbCmd[2];
    ctrl.wIndex = (u8UsbCmd[5] << 8) | u8UsbCmd[4];
    ctrl.wLength = (u8UsbCmd[7] << 8) | u8UsbCmd[6];

    if (likely(ctrl.bRequestType & USB_DIR_IN)) {
	dev->ep[0].is_in = 1;
	dev->ep0state = EP0_IN;
    } else {
	dev->ep[0].is_in = 0;
	dev->ep0state = EP0_OUT;
    }

    if ((ctrl.bRequestType & 0x60) == 0) {
	switch (ctrl.bRequest) {
	    case USB_REQ_CLEAR_FEATURE:
		if (bClear_feature(dev, &ctrl) == FALSE) goto stall;
		return;

	    case USB_REQ_SET_ADDRESS:
		if (dev->ep0state == EP0_STALL) goto stall;
		if (bSet_address(dev, &ctrl) == FALSE) goto stall;
		return;

	    case USB_REQ_SET_FEATURE:
		if (bSet_feature(dev, &ctrl) == FALSE) goto stall;
		return;

	    case USB_REQ_GET_STATUS:
		if (bGet_status(dev, &ctrl) == FALSE) goto stall;
		return;

	    case USB_REQ_SYNCH_FRAME:
		if (dev->ep0state == EP0_STALL) goto stall;
		if (bSynch_frame(dev, &ctrl) == FALSE) goto stall;
		return;

	    default:
		;
	}
    }

    /* Not handled here, pass to higher-level driver */
    if (dev->ep0state == EP0_STALL) goto stall;
    spin_unlock(&dev->lock);
    tmp = dev->driver->setup(&dev->gadget, &ctrl);
    spin_lock(&dev->lock);
    if (unlikely(tmp < 0)) goto stall;
    return;

stall:
    INFO(dev, "Set STALL in vUsb_ep0setup\n");
    FTC_set_halt(dev->gadget.ep0, 1);
    dev->ep[0].stopped = 1;
    dev->ep0state = EP0_STALL;
}


///////////////////////////////////////////////////////////////////////////////
//		vUsb_ep0end(struct FTC_udc *dev)
//		Description:
//			1. End this transfer.
//		input: none
//		output: none
///////////////////////////////////////////////////////////////////////////////
static void vUsb_ep0end(struct FTC_udc *dev)
{
	DBG_FUNCC("+vUsb_ep0end()\n");

	dev->eUsbCxCommand = CMD_VOID;


}

///////////////////////////////////////////////////////////////////////////////
//		vUsb_ep0fail(struct FTC_udc *dev)
//		Description:
//			1. Stall this transfer.
//		input: none
//		output: none
///////////////////////////////////////////////////////////////////////////////
static void vUsb_ep0fail(struct FTC_udc *dev)
{
    D(D_ERROR, "EP0: Control transfer failed, stalling.\n");
    FTC_set_halt(dev->gadget.ep0, 1);
}

static void vUsbHandler(struct FTC_udc *dev)
{
    D(D_BLABLA, "vUsbHandler()\n");
#if 0
    printk("Interrupt level bits: %d:%08x %d:%08x %d:%08x\n",
	    ((dev->usb_interrupt_level1 & 1) > 0), (u32)mUsbIntSrc0Rd(),
	    ((dev->usb_interrupt_level1 & 2) > 0), (u32)mUsbIntSrc1Rd(),
	    ((dev->usb_interrupt_level1 & 4) > 0), (u32)mUsbIntSrc2Rd());
#endif

    u32 usb_interrupt_level2;

    /* Group Byte 2 */
    if (dev->usb_interrupt_level1 & BIT2) {
	usb_interrupt_level2 = mUsbIntSrc2Rd() & ~mUsbIntSrc2MaskRd();

	if (usb_interrupt_level2 & BIT0) vUsb_rst(dev);
	if (usb_interrupt_level2 & BIT1) vUsb_suspend(dev);
	if (usb_interrupt_level2 & BIT2) vUsb_resm(dev);
	if (usb_interrupt_level2 & BIT3) {
	    mUsbIntIsoSeqErrClr();
	    D(D_VERBOSE, "ISO sequence error\n");
	}
	if (usb_interrupt_level2 & BIT4) {
	    mUsbIntIsoSeqAbortClr();
	    D(D_VERBOSE, "ISO sequence abort\n");
	}
	if (usb_interrupt_level2 & BIT5) mUsbIntTX0ByteClr();
	if (usb_interrupt_level2 & BIT6) mUsbIntRX0ByteClr();
	if (usb_interrupt_level2 & BIT7) {
	    mUsbIntDmaFinishClr();
	    dma_complete(dev, &(dev->ep[dev->EPUseDMA]));
	}
	if (usb_interrupt_level2 & BIT8) {
	    mUsbIntDmaErrClr();
	    D(D_VERBOSE, "DMA error interrupt\n");
	}
    }

    /* Group Byte 0 */
    if (dev->usb_interrupt_level1 & BIT0) {
	usb_interrupt_level2 = mUsbIntSrc0Rd() & ~mUsbIntSrc0MaskRd();
	dev->ep[0].irqs++;
	if (usb_interrupt_level2 & BIT1) {
	    dma_ep0_continue(dev, &(dev->ep[dev->EPUseDMA]));
	}
	if (usb_interrupt_level2 & BIT0) {
	    vUsb_ep0setup(dev);
	} else if (usb_interrupt_level2 & BIT3) {
	    vUsb_ep0end(dev);
	}
	if (usb_interrupt_level2 & BIT4) vUsb_ep0fail(dev);
	if (usb_interrupt_level2 & BIT5) {
	    mUsbIntEP0AbortClr();
	    D(D_VERBOSE, "Command abort interrupt\n");
	}
    }

    /* Group Byte 1 */
    if (dev->usb_interrupt_level1 & BIT1) {
	int fifo;
	usb_interrupt_level2 = mUsbIntSrc1Rd() & ~mUsbIntSrc1MaskRd();
	for (fifo=0; fifo<4; fifo++) {
	    if (usb_interrupt_level2 & ((BIT1|BIT0) << (fifo*2))) {
		/* OUT FIFO: ready to be read or short packet received */
		handle_fifo_irq(dev, fifo);
	    }
	    if (usb_interrupt_level2 & (BIT16 << fifo)) {
		/* IN FIFO: ready to be written */
		handle_fifo_irq(dev, fifo);
	    }
	}
    }
}

static irqreturn_t FTC_irq(int irq, void *_dev, struct pt_regs *r)
{
    D(D_BLABLA, "FTC_irq()\n");
    struct FTC_udc *dev = _dev;
    u32 handled = 0;
    spin_lock(&dev->lock);

    dev->usb_interrupt_level1_Save = mUsbIntGroupRegRd();
    dev->usb_interrupt_level1_Mask = mUsbIntGroupMaskRd();

    dev->usb_interrupt_level1 = dev->usb_interrupt_level1_Save & ~dev->usb_interrupt_level1_Mask;
    if (dev->usb_interrupt_level1) {
	dev->irqs++;
	handled = 1;
	vUsbHandler(dev);
	// Clear usb interrupt flags
	dev->usb_interrupt_level1 = 0;
	wFOTGPeri_Port(0xC0) |= 0x07;
    }
    spin_unlock(&dev->lock);

    return IRQ_RETVAL(handled);
}

#if 0
static irqreturn_t udc_irq_for_OTG(void)
{
    struct FTC_udc *dev = the_controller;

    DBG_FUNCC("+udc_irq_for_OTG()\n");
    spin_lock(&dev->lock);

    dev->usb_interrupt_level1_Save = mUsbIntGroupRegRd();
    dev->usb_interrupt_level1_Mask = mUsbIntGroupMaskRd();

    DBG_CTRLL("dev->usb_interrupt_level1_Save = 0x%x\n", (dev->usb_interrupt_level1_Save));
    DBG_CTRLL("dev->usb_interrupt_level1_Mask = 0x%x\n", (dev->usb_interrupt_level1_Mask));

    dev->usb_interrupt_level1 = dev->usb_interrupt_level1_Save & ~dev->usb_interrupt_level1_Mask;
    if (dev->usb_interrupt_level1) {
	dev->irqs++;
	vUsbHandler(dev);
	dev->usb_interrupt_level1 = 0;
    }

    spin_unlock(&dev->lock);
    return IRQ_RETVAL(0);
}
#endif

static void FTC_usb_remove(void)
{
    D(D_BLABLA, "FTC_usb_remove()\n");

    if (the_controller->driver) {
	/* No high-level drivers should be registered when removing this module */
	D(D_ERROR, "High-level driver '%s' still active, unregistering it.\n",
		the_controller->driver->driver.name);
	fotg200_usb_gadget_unregister_driver(the_controller->driver);
    }

    udc_reset(the_controller);
    if (the_controller->got_irq) free_irq(IRQ_OTG, the_controller);
    kfree(the_controller); //Andrew update
    the_controller = NULL;
}

static int FTC_usb_probe(void)
{
    D(D_BLABLA, "FTC_usb_probe()\n");

    int retval=0;
    u32 wFound=0;

#ifdef CONFIG_FARADAY_FOTG2XX
    struct otg_transceiver *xceiv = 0;
#endif

    if (wFOTGPeri_Port(0x00) == 0x01000010
	&& wFOTGPeri_Port(0x04) == 0x00000001
	&& wFOTGPeri_Port(0x08) == 0x00000006) {
	wFound = 1;
    }

    if (!wFound) {
	D(D_ERROR, "Unable to find FOTG200 host controller.\n");
	return(-EBUSY);
    }

    D(D_VERBOSE, "FOTG200 host controller found.\n");

#ifndef USE_OTG
    /* Disable HC and OTG interrupts for non-OTG mode */
    wFOTGPeri_Port(0xC4) |= 0x06;
#endif

    /* if you want to support more than one controller in a system,
     * usb_gadget_driver_{register,unregister}() must change.
     */
    if (the_controller) {
	D(D_ERROR, "Only one USB gadget driver can be active.\n");
	return -EBUSY;
    }

    the_controller = kmalloc (sizeof *the_controller, SLAB_KERNEL);
    if (the_controller == NULL) return -ENOMEM;

    memset(the_controller, 0, sizeof *the_controller);
    spin_lock_init(&the_controller->lock);
    the_controller->gadget.ops = &FTC_ops;
    the_controller->gadget.dev.bus_id = "gadget";
    the_controller->gadget.name = driver_name;
    // the_controller->gadget.udc_isr = udc_irq_for_OTG;
    the_controller->enabled = 1;
    the_controller->EPUseDMA = DMA_CHANNEL_FREE;

    udc_reset(the_controller);
    udc_reinit(the_controller);

    wFOTGPeri_Port(0xC4) |= 0x08; // Set system interrupt signal high-active
    cpe_int_set_irq(IRQ_OTG, LEVEL, H_ACTIVE);
    if (request_irq(IRQ_OTG, FTC_irq, SA_INTERRUPT,
		driver_name, the_controller) != 0) {
	D(D_ERROR, "Failed to register interrupt handler.\n");
	retval = -EBUSY;
	goto done;
    }
    the_controller->got_irq = 1;

#ifdef REGISTER_DEVICE
    if (devfs_register_chrdev(DEV_MAJOR, "FOTG2XX", &dev_fops)) {
	DBG(dev, "Unable to register character device (major = %d).\n", DEV_MAJOR);
	retval = -EBUSY;
	goto done;
    }
#endif


#ifdef USE_OTG
    xceiv = FOTG2XX_get_otg_transceiver();
    the_controller->transceiver = xceiv;
    the_controller->gadget.is_otg = 1;
#endif

    return 0;

done:
    if (the_controller) FTC_usb_remove();
    return retval;
}

static int __init init(void)
{
    D(D_NOTICE, "FTC_FOTG200 USB-OTG low-level driver\n");
    D(D_VERBOSE, "Base address: 0x%x\n", FOTG200_BASE_ADDRESS);
    struct usb_lowlevel_driver driver = {
	fotg200_usb_gadget_register_driver,
	fotg200_usb_gadget_unregister_driver
    };

    int ret = FTC_usb_probe();
    if (ret < 0) return ret;

    ret = FTC_zero_register_lowlevel_driver(minor, driver);
    if (ret < 0) return ret;
    D(D_NOTICE, "FTC_FOTG200: registered as USB device #%d\n", ret);

    return 0;
}
module_init (init);

static void __exit cleanup(void)
{
    D(D_NOTICE, "FTC_FOTG200 USB-OTG low-level driver exiting\n");
    FTC_zero_unregister_lowlevel_driver(minor);
    return FTC_usb_remove();
}
module_exit (cleanup);

