/*
 * FTC_zero.c -- Faraday Gadget Zero, for USB development
 *
 * Copyright (C) 2004-2005 John Chiang
 *
 * FTC_zero.c -- Faraday Gadget Zero v1.1, for FUSB220 verification
 *				compatible with FUSB220_udc driver
 *
 * Copyright (C) 2005_Feb Andrew Feng
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the above-listed copyright holders may not be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * ALTERNATIVELY, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") as published by the Free Software
 * Foundation, either version 2 of that License or (at your option) any
 * later version.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "../lara_common.h"


#include <linux/config.h>
#include <asm/uaccess.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/uts.h>
#include <linux/version.h>

#include <linux/crypto.h>

#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/unaligned.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
# include <linux/device.h>
#endif 
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>

#include <asm/arch/platform-a320/peppercon.h>

#include "FTC_zero.h"
#include "FTC_zero_ll.h"
#include "chap9.h"

#include "pphid.h"
#include "ppchap9.h"
#include "ppstorage.h"
#include "scsi.h"
#include "scsi_ipmi.h"
#include "pp_usb_strings.h"

#define DRV_NAME	"pp_peppercon_usb"
#define DRIVER_VERSION	"1.0"

//#undef DEBUGLEVEL
//#define DEBUGLEVEL D_VERBOSE

static struct usb_gadget_driver		zero_driver;

static void restart(struct FTC_zero_dev *ZeroDev);
static int pp_usb_open(struct inode *, struct file *);
static int pp_usb_ioctl(struct inode *, struct file *, uint, ulong);
static void change_usb_device(struct FTC_zero_dev *ZeroDev);

// FIXME: Move this into header file(s)
int fusb220_usb_gadget_register_driver(struct usb_gadget_driver *driver);
int fusb220_usb_gadget_unregister_driver(struct usb_gadget_driver *driver);
int fotg200_usb_gadget_register_driver(struct usb_gadget_driver *driver);
int fotg200_usb_gadget_unregister_driver(struct usb_gadget_driver *driver);

/* Number of USB controllers to use/chardev minor numbers to handle */
#define MAX_USB_DEVICES 16
struct FTC_zero_dev *device_table[MAX_USB_DEVICES];

static struct file_operations lara_usb_ops = {
    owner:   THIS_MODULE,
    open:    pp_usb_open,
    ioctl:   pp_usb_ioctl,
};

/* device descriptor */

static const struct usb_device_descriptor default_device_desc = {
    .bLength =		sizeof(default_device_desc),
    .bDescriptorType =	USB_DT_DEVICE,
    .bcdUSB =		__constant_cpu_to_le16(0x0200),
    .bDeviceClass    = 0,   /* set on interface level */
    .bDeviceSubClass = 0,   /* set on interface level */
    .bDeviceProtocol = 0,   /* set on interface level */
    
    .bMaxPacketSize0 = EP0_MAXPACKET,
    .idVendor =		__constant_cpu_to_le16(PP_VENDOR_ID),
    .idProduct =	__constant_cpu_to_le16(PP_PRODUCT_ID),
    .bcdDevice =	__constant_cpu_to_le16(PP_DEVICE_RELEASE_NO),
    
    .iManufacturer =	Dev_iManufacturer,
    .iProduct =		Dev_iProduct,
    .iSerialNumber =	Dev_iSerialNumber,
    .bNumConfigurations =	PP_CONFIGURATION_NUMBER,
};

static const struct usb_qualifier_descriptor default_dev_qualifier = {
    .bLength =		sizeof(default_dev_qualifier),
    .bDescriptorType =	USB_DT_DEVICE_QUALIFIER,
    
    .bcdUSB =		__constant_cpu_to_le16(0x0200),
    .bDeviceClass =		USB_CLASS_PER_INTERFACE,
    .bNumConfigurations =	1,
};

/* config descriptor */

static const struct usb_config_descriptor default_config_desc = {
    .bLength =		sizeof(default_config_desc),
    .bDescriptorType =	USB_DT_CONFIG, /* adjusted in zero_bind */
    .wTotalLength =     __constant_cpu_to_le16(0), /* adjusted in zero_bind */
    .bNumInterfaces =	0, /* adjusted in zero_bind, must be initialised to zero! */
    .bConfigurationValue=PP_CONFIG_VALUE,
    .iConfiguration =	0,
    .bmAttributes =	USB_CONFIG_ATT_ONE | USB_CONFIG_ATT_SELFPOWER,
    .bMaxPower =	0x32, /* bMaxPower, 100mA */
};

/* mass storage interface descriptors */

static const struct usb_interface_descriptor default_intf_mass_storage = {
    .bLength =		sizeof(default_intf_mass_storage),
    .bDescriptorType =	USB_DT_INTERFACE,

    .bInterfaceNumber = 0, /* adjusted in zero_bind */
    .bAlternateSetting = 0,

    .bNumEndpoints =	2,
    .bInterfaceClass =	USB_CLASS_MASS_STORAGE,
    .bInterfaceSubClass =	USB_SC_SCSI,
    .bInterfaceProtocol =	USB_PR_BULK,

    .iInterface = 0,
};

/* mouse hid interface descriptors */

static const struct usb_interface_descriptor default_intf_desc_mouse = {
    .bLength =		sizeof(default_intf_desc_mouse),
    .bDescriptorType =	USB_DT_INTERFACE,
    
    .bInterfaceNumber = 0, /* adjusted in zero_bind */
    .bAlternateSetting = 0,
    .bNumEndpoints =	1,
    .bInterfaceClass =	3,
    .bInterfaceSubClass =	0,
    .bInterfaceProtocol =	2,
    .iInterface = 0,
};

static const USB_HID_DESCRIPTOR default_mouse_hid_desc = {
    .bLength = USB_DT_HID_SIZE,
    .bDescriptorType = USB_DT_HID,
    .wHIDClassSpecComp = __constant_cpu_to_le16(0x0101),
    .bCountry = 0,
    .bNumDescriptors = 1,
    .b1stDescType = 0x22,
    .w1stDescLength = __constant_cpu_to_le16(0), /* adjusted in zero_bind */
};

static const struct usb_endpoint_descriptor default_mouse_ep_desc_fs = {
    .bLength = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType =	USB_DT_ENDPOINT,
    
    .bEndpointAddress =	USB_DIR_IN |  (2 * PP_FEAT_USB_MASS_STORAGE_NO + 1),
    .bmAttributes =	USB_ENDPOINT_XFER_INT,
    .wMaxPacketSize =	__constant_cpu_to_le16(FS_INTR_IN_MAXPACKET),
    .bInterval =	10 /* 10 ms */
};

static const struct usb_endpoint_descriptor default_mouse_ep_desc_hs = {
    .bLength = USB_DT_ENDPOINT_SIZE,
    .bDescriptorType =	USB_DT_ENDPOINT,
    
    .bEndpointAddress =	USB_DIR_IN |  (2 * PP_FEAT_USB_MASS_STORAGE_NO + 1),
    .bmAttributes =	USB_ENDPOINT_XFER_INT,
    .wMaxPacketSize =	__constant_cpu_to_le16(FS_INTR_IN_MAXPACKET),
    .bInterval =	4 /* 1 ms, misinterpreted as 16 ms by Linux kernels < 2.6.14 */
};

/* keyboard hid interface descriptors */

struct usb_interface_descriptor default_intf_desc_keyboard = {
    .bLength =		sizeof(default_intf_desc_keyboard),
    .bDescriptorType =	USB_DT_INTERFACE,
    .bInterfaceNumber = 0, /* adjusted in zero_bind */
    .bAlternateSetting = 0,
    .bNumEndpoints =	1,
    .bInterfaceClass =	3,
    .bInterfaceSubClass =	1,
    .bInterfaceProtocol =	1,
    .iInterface = 0
};

static const USB_HID_DESCRIPTOR default_keyboard_hid_desc = {
    .bLength = USB_DT_HID_SIZE,
    .bDescriptorType = USB_DT_HID,
    .wHIDClassSpecComp = __constant_cpu_to_le16(0x0101),
    .bCountry = 0,
    .bNumDescriptors = 1,
    .b1stDescType = 0x22,
    .w1stDescLength = __constant_cpu_to_le16(sizeof(KeyboardReportDescriptor))
};

static const struct usb_endpoint_descriptor default_keyboard_ep_desc_fs = {
    .bLength =		USB_DT_ENDPOINT_SIZE,
    .bDescriptorType =	USB_DT_ENDPOINT,
    
    .bEndpointAddress =	USB_DIR_IN | (2 * PP_FEAT_USB_MASS_STORAGE_NO + 2),
    .bmAttributes =	USB_ENDPOINT_XFER_INT,
    .wMaxPacketSize =	__constant_cpu_to_le16(FS_INTR_IN_MAXPACKET),
    .bInterval =	10 /* 10 ms */
};

static const struct usb_endpoint_descriptor default_keyboard_ep_desc_hs = {
    .bLength =		USB_DT_ENDPOINT_SIZE,
    .bDescriptorType =	USB_DT_ENDPOINT,
    
    .bEndpointAddress =	USB_DIR_IN | (2 * PP_FEAT_USB_MASS_STORAGE_NO + 2),
    .bmAttributes =	USB_ENDPOINT_XFER_INT,
    .wMaxPacketSize =	__constant_cpu_to_le16(FS_INTR_IN_MAXPACKET),
    .bInterval =	4 /* 1 ms, misinterpreted as 16 ms by Linux kernels < 2.6.14 */
};

static int peppercon_send_data(struct FTC_zero_dev *ZeroDev, const uint8_t type, const char* data, const int length)
{
    if (type == 1 && ZeroDev->keyboard_enabled) {
	pphid_keyboard_set_report(ZeroDev->hid, ZeroDev->keyboard_ep, data, length);
    } else if (type == 0 && ZeroDev->mouse_enabled) {
	pphid_mouse_set_pointer(ZeroDev->hid, ZeroDev->mouse_ep, data, length);
    } else {
	return -1;
    }
    return 0;
}

/* enables the endpoint ep, allocs the ep_driver data struct */
static int enable_endpoint(struct FTC_zero_dev*ZeroDev, struct usb_ep *ep,
			   const struct usb_endpoint_descriptor *d)
{
    int	rc;
    ep_driver_t *ep_driver;
    D(D_VERBOSE, "enable_endpoint(%s)\n", ep->name);
    
    ep_driver = kmalloc(sizeof(ep_driver_t), GFP_KERNEL);
    ep_driver->ep = ep;
    ep_driver->bufhd = NULL;
    ep_driver->dev = ZeroDev;
    ep_driver->complete = NULL;
    ep_driver->class_ep_data = NULL;
    
    ep->driver_data = ep_driver;
    D(D_BLABLA, "set driver_data of ep: %s (%p) to %p, addr: %x\n", ep->name, ep, ep->driver_data, d->bEndpointAddress);
    rc = usb_ep_enable(ep, d);
    if (rc)
	D(D_ERROR, "can't enable endpoint %s (rc = %d).\n", ep->name, rc);
    return rc;
}

/* disables the endpoint and frees associated data structures */
static int disable_endpoint(struct FTC_zero_dev *ZeroDev, struct usb_ep* ep) {
    int rc;
    ep_driver_t* ep_driver = (ep_driver_t*)ep->driver_data;
    D(D_VERBOSE, "disable_endpoint(%s)\n", ep->name);

    rc = usb_ep_disable(ep);
    if (rc)
	D(D_ERROR, "can't disable endpoint %s (rc = %d)\n", ep->name, rc);
    if (ep_driver) {
	ZeroDev= ep_driver->dev;
	if (ep_driver->class_ep_data) {
	    kfree(ep_driver->class_ep_data);
	}
	kfree(ep_driver);
	ep->driver_data = NULL;
    }
    return rc;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)

static void peppercon_generate_serial_number(struct FTC_zero_dev *ZeroDev)
{
    /* we calculate the serial number as an MD5 hash over all our descriptors */
    int i;
    char md5sum[16];
    char md5sum_text[33];

    //struct crypto_tfm *tfm;
    struct digest_context *dx = NULL;
    struct digest_implementation *di = NULL;
    di = find_digest_by_name("md5", 1);
    if (!di) return;

    di->lock();
    dx = di->realloc_context(NULL, di);
    if (!dx){
	di->unlock();
	return;
    }
    int serial_length = 32;

    memset(ZeroDev->pp_usb_serial, 0, sizeof(ZeroDev->pp_usb_serial));

    	/* initialize it */
    	memset(md5sum, 0, sizeof(md5sum));
    	//crypto_digest_init(tfm);
	int error = 0;
	
	if ((error = dx->di->open(dx)) < 0) return;
    	
    	/* add all descriptors */
	if ((error = dx->di->update(dx, (char*)&ZeroDev->device_desc, le16_to_cpu((ZeroDev->device_desc.bLength))))) return;
	if ((error = dx->di->update(dx, ZeroDev->config_desc_hs, le16_to_cpu(((struct usb_config_descriptor*)&ZeroDev->config_desc_hs)->wTotalLength)))) return;
	for (i = 0; ZeroDev->stringtab.strings[i].s != NULL; i++) {
	    if ((error = dx->di->update(dx, ZeroDev->stringtab.strings[i].s, strlen(ZeroDev->stringtab.strings[i].s))) < 0) return;
    	}
    	/* finalize it */
	if ((error = dx->di->close(dx, md5sum)) < 0) return;

	di->free_context(dx);
	di->unlock();
    	/* make string out of md5 sum */
    	for (i = 0; i < sizeof(md5sum); i++) {
    	    sprintf(&md5sum_text[i*2], "%02X", md5sum[i] & 0xff);
    	}

    	/* now write the md5 sum as new serial number */
    	for (i = 0 ; i < serial_length; i++) {
    	    ZeroDev->pp_usb_serial[i] = md5sum_text[i];
    	}
	ZeroDev->pp_usb_serial[i] = '\0';
    	UD("MD5 sum as USB serial number: %s\n", md5sum_text);

	for (i = 0; ZeroDev->stringtab.strings[i].s != NULL; i++) {
	    if (ZeroDev->stringtab.strings[i].id == Dev_iSerialNumber) {
		ZeroDev->stringtab.strings[i].s = (char*)&ZeroDev->pp_usb_serial;
	    }
    	}

	// in case of a problem, the old (default) serial descriptor is not overriden
}

#else

static inline struct scatterlist sgent(const void *pointer, int len)
{
    struct scatterlist result;
    result.page = virt_to_page(pointer);
    result.offset = offset_in_page(pointer);
    result.length = len;
    return result;
}

static void peppercon_generate_serial_number(struct FTC_zero_dev *ZeroDev)
{
    /* we calculate the serial number as an MD5 hash over all our descriptors */
    struct crypto_tfm *tfm;
    struct scatterlist sg[32];
    int i, n;
    char md5sum[16];
    char md5sum_text[33];
    int serial_length = 32;

    tfm = crypto_alloc_tfm("md5", 0);
    if (!tfm) return;
    crypto_digest_init(tfm);

    memset(ZeroDev->pp_usb_serial, 0, sizeof(ZeroDev->pp_usb_serial));

    /* initialize it */
    memset(md5sum, 0, sizeof(md5sum));

    /* add all descriptors */
    n = 0;
    sg[n++] = sgent(&ZeroDev->device_desc, le16_to_cpu(ZeroDev->device_desc.bLength));
    sg[n++] = sgent(ZeroDev->config_desc_hs, le16_to_cpu(((struct usb_config_descriptor*)&ZeroDev->config_desc_hs)->wTotalLength));
    for (i = 0; ZeroDev->stringtab.strings[i].s != NULL; i++) {
	if (n >= 32) break;
	sg[n++] = sgent(ZeroDev->stringtab.strings[i].s, strlen(ZeroDev->stringtab.strings[i].s));
    }

    /* finalize it */
    crypto_digest_update(tfm, sg, n);
    crypto_digest_final(tfm, md5sum);

    /* make string out of md5 sum */
    for (i = 0; i < sizeof(md5sum); i++) {
	sprintf(&md5sum_text[i*2], "%02X", md5sum[i] & 0xff);
    }

    /* now write the md5 sum as new serial number */
    for (i = 0 ; i < serial_length; i++) {
	ZeroDev->pp_usb_serial[i] = md5sum_text[i];
    }
    ZeroDev->pp_usb_serial[i] = '\0';
    UD("MD5 sum as USB serial number: %s\n", md5sum_text);

    for (i = 0; ZeroDev->stringtab.strings[i].s != NULL; i++) {
	if (ZeroDev->stringtab.strings[i].id == Dev_iSerialNumber) {
	    ZeroDev->stringtab.strings[i].s = (char*)&ZeroDev->pp_usb_serial;
	}
    }

    // in case of a problem, the old (default) serial descriptor is not overriden
}

#endif

/*
 * Config descriptors are handcrafted.  They must agree with the code
 * that sets configurations and with code managing interfaces and their
 * altsettings. They must also handle different speeds and other-speed
 * requests.
 */


static int add_to_config_descriptor(struct usb_config_descriptor *config_desc, char* data, int size) {
    int curr_pos = le16_to_cpu(config_desc->wTotalLength);

    /* size check */
    if ((curr_pos + size) > EP0_BUFSIZE) {
	return -1;
    }

    /* add new descriptor */
    memcpy(&((char*)config_desc)[curr_pos], data, size);

    /* adjust size */
    config_desc->wTotalLength = cpu_to_le16(curr_pos + size);

    return 0;
}

static int add_to_config_descriptor_fshs(struct usb_config_descriptor *config_desc_fs,
					 struct usb_config_descriptor *config_desc_hs,
					 char* data, int size)
{
    int rc1, rc2;
    rc1 = add_to_config_descriptor(config_desc_fs,  data, size);
    rc2 = add_to_config_descriptor(config_desc_hs,  data, size);
    return rc1<0 || rc2<0? -1 : 0;
}

int populate_config_buf(struct FTC_zero_dev *ZeroDev, enum usb_device_speed speed,
			u8 *buf0, u8 type, unsigned index)
{
    int	hs;
    D(D_BLABLA,"+ (FTC_zero)populate_config_buf()\n");
    if (type != USB_DT_OTHER_SPEED_CONFIG && type != USB_DT_CONFIG) {
	return -EINVAL;
    }

    /* don't support other config descriptors  */
    if (index > 0)
	return -EINVAL;
    

    hs = (speed == USB_SPEED_HIGH);
    if (type == USB_DT_OTHER_SPEED_CONFIG)
	hs = !hs;

    struct usb_config_descriptor *real_config_desc = (struct usb_config_descriptor *)(hs? ZeroDev->config_desc_hs : ZeroDev->config_desc_fs);
    // set some stuff

    real_config_desc->bDescriptorType = type;

    // return high speed or full speed config descriptor

    if ((le16_to_cpu(real_config_desc->wTotalLength))  > EP0_BUFSIZE)
	return -EDOM;
    
    memcpy(buf0, real_config_desc, le16_to_cpu(real_config_desc->wTotalLength));
    
    return le16_to_cpu(real_config_desc->wTotalLength);
}


static void pp_prepare_descriptors(struct FTC_zero_dev *ZeroDev)
{
    int ms;

    int curr_intf = 0;
    int curr_ep = 1;
    struct usb_config_descriptor *real_config_desc_fs = (struct usb_config_descriptor*) ZeroDev->config_desc_fs;
    struct usb_config_descriptor *real_config_desc_hs = (struct usb_config_descriptor*) ZeroDev->config_desc_hs;

    struct usb_config_descriptor config_desc;
    struct usb_interface_descriptor intf_mass_storage;
    struct usb_interface_descriptor intf_desc_mouse;
    struct usb_interface_descriptor intf_desc_keyboard;

    // prepare one highspeed and one full speed descriptor
    memset(real_config_desc_hs, 0, MAX_CONFIG_DESC_LEN);
    memset(real_config_desc_fs, 0, MAX_CONFIG_DESC_LEN);

    // Copy default descriptors
    memcpy(&config_desc, &default_config_desc, sizeof(default_config_desc));
    memcpy(&intf_mass_storage, &default_intf_mass_storage, sizeof(default_intf_mass_storage));
    memcpy(&intf_desc_mouse, &default_intf_desc_mouse, sizeof(default_intf_desc_mouse));
    memcpy(&intf_desc_keyboard, &default_intf_desc_keyboard, sizeof(default_intf_desc_keyboard));

    ZeroDev->device_desc.bMaxPacketSize0 = ZeroDev->ep0->maxpacket;
    ZeroDev->dev_qualifier.bMaxPacketSize0 = ZeroDev->ep0->maxpacket;
    //config_desc.wTotalLength = pp_get_config_desc_len();

    int i;
    for (i = 0; i < 16; i++) {
	ZeroDev->eps[i].used = 0;
	ZeroDev->eps[i].ep_ptr = NULL;
    }
    // TODO: check return value
    add_to_config_descriptor_fshs(real_config_desc_fs, real_config_desc_hs, (uint8_t*)&config_desc, USB_DT_CONFIG_SIZE);

    real_config_desc_fs->bNumInterfaces = 0;
    real_config_desc_hs->bNumInterfaces = 0;

    /* mass storage devices */
    ZeroDev->ms_enabled = 0;
    for (ms = 0; ms < PP_FEAT_USB_MASS_STORAGE_NO; ms++) {
	if (ZeroDev->mass_storage_enabled[ms]) {
	    ZeroDev->ms_enabled |= 1;
	    intf_mass_storage.bInterfaceNumber = curr_intf++;
	    ZeroDev->mass_storage_bulk_in_desc[ms].bEndpointAddress = USB_DIR_IN | curr_ep;
	    ZeroDev->eps[curr_ep].used = 1;
	    ZeroDev->eps[curr_ep].ep_ptr = &ZeroDev->storage[ms]->Bin_ep;
	    snprintf(ZeroDev->eps[curr_ep].name, sizeof(ZeroDev->eps[curr_ep].name), "ep%d", curr_ep);
	    curr_ep++;

	    ZeroDev->mass_storage_bulk_out_desc[ms].bEndpointAddress = USB_DIR_OUT | curr_ep;
	    ZeroDev->eps[curr_ep].used = 1;
	    ZeroDev->eps[curr_ep].ep_ptr = &ZeroDev->storage[ms]->Bout_ep;
	    snprintf(ZeroDev->eps[curr_ep].name, sizeof(ZeroDev->eps[curr_ep].name), "ep%d", curr_ep);
	    curr_ep++;

	    if (ZeroDev->mass_storage_type[ms] == PP_USB_IMAGE_TYPE_FLOPPY) {
		intf_mass_storage.bInterfaceSubClass = USB_SC_8070;
	    } else {
		intf_mass_storage.bInterfaceSubClass = USB_SC_SCSI;
	    }
	    add_to_config_descriptor_fshs(real_config_desc_fs, real_config_desc_hs, (uint8_t*)&intf_mass_storage, intf_mass_storage.bLength);

	    ZeroDev->mass_storage_bulk_in_desc[ms].wMaxPacketSize = cpu_to_le16(FS_BULK_IN_MAXPACKET);
	    add_to_config_descriptor(real_config_desc_fs, (uint8_t*)&ZeroDev->mass_storage_bulk_in_desc[ms], ZeroDev->mass_storage_bulk_in_desc[ms].bLength);

	    ZeroDev->mass_storage_bulk_in_desc[ms].wMaxPacketSize = cpu_to_le16(HS_BULK_IN_MAXPACKET);
	    add_to_config_descriptor(real_config_desc_hs, (uint8_t*)&ZeroDev->mass_storage_bulk_in_desc[ms], ZeroDev->mass_storage_bulk_in_desc[ms].bLength);

	    ZeroDev->mass_storage_bulk_out_desc[ms].wMaxPacketSize = cpu_to_le16(FS_BULK_OUT_MAXPACKET);
	    add_to_config_descriptor(real_config_desc_fs, (uint8_t*)&ZeroDev->mass_storage_bulk_out_desc[ms], ZeroDev->mass_storage_bulk_out_desc[ms].bLength);

	    ZeroDev->mass_storage_bulk_out_desc[ms].wMaxPacketSize = cpu_to_le16(HS_BULK_OUT_MAXPACKET);
	    add_to_config_descriptor(real_config_desc_hs, (uint8_t*)&ZeroDev->mass_storage_bulk_out_desc[ms], ZeroDev->mass_storage_bulk_out_desc[ms].bLength);

	    ZeroDev->storage[ms]->interface_number = intf_mass_storage.bInterfaceNumber;
	    ZeroDev->g_ud.class_interface_request[intf_mass_storage.bInterfaceNumber] = ppstorage_handle_class_req;

	    real_config_desc_fs->bNumInterfaces++;
	    real_config_desc_hs->bNumInterfaces++;
	} else {
	    ZeroDev->storage[ms]->interface_number = -1;
	}
    }

    /* peppercon mouse */
    if (ZeroDev->usb_device_mouse_type != peppercon_mouse_type_none) {
	if (ZeroDev->usb_device_mouse_type == peppercon_mouse_type_absolute) {
	    ZeroDev->mouse_hid_desc.w1stDescLength = cpu_to_le16(sizeof(MouseReportDescriptorAbsolute));
	    ZeroDev->mouse_report_descriptor = (char*)&MouseReportDescriptorAbsolute;
	    pphid_set_mouse_report_length(ZeroDev->hid, MouseAbsReportLength);
	    intf_desc_mouse.bInterfaceSubClass = 0; // no subclass
	} else {
	    if (ZeroDev->usb_device_mouse_type != peppercon_mouse_type_relative) {
		printk("unknown mouse type: %d, set to default (relative mouse type)\n", ZeroDev->usb_device_mouse_type);
	    }
	    ZeroDev->mouse_hid_desc.w1stDescLength = cpu_to_le16(sizeof(MouseReportDescriptorRelative));
	    ZeroDev->mouse_report_descriptor = (char*)&MouseReportDescriptorRelative;
	    pphid_set_mouse_report_length(ZeroDev->hid, MouseRelReportLength);
	    intf_desc_mouse.bInterfaceSubClass = 1; // boot interface subclass
	}
	pphid_set_mouse_type(ZeroDev->hid, ZeroDev->usb_device_mouse_type);

	ZeroDev->eps[curr_ep].used = 1;
	ZeroDev->eps[curr_ep].ep_ptr = &ZeroDev->mouse_ep;
	snprintf(ZeroDev->eps[curr_ep].name, sizeof(ZeroDev->eps[curr_ep].name), "ep%d", curr_ep);

	intf_desc_mouse.bInterfaceNumber = curr_intf++;
	ZeroDev->mouse_ep_desc_fs.bEndpointAddress = USB_DIR_IN | curr_ep;
	ZeroDev->mouse_ep_desc_hs.bEndpointAddress = USB_DIR_IN | curr_ep;
	add_to_config_descriptor_fshs(real_config_desc_fs, real_config_desc_hs, (uint8_t*)&intf_desc_mouse, intf_desc_mouse.bLength);
	add_to_config_descriptor_fshs(real_config_desc_fs, real_config_desc_hs, (uint8_t*)&ZeroDev->mouse_hid_desc, ZeroDev->mouse_hid_desc.bLength);
	add_to_config_descriptor(real_config_desc_fs, (uint8_t*)&ZeroDev->mouse_ep_desc_fs, ZeroDev->mouse_ep_desc_fs.bLength);
	add_to_config_descriptor(real_config_desc_hs, (uint8_t*)&ZeroDev->mouse_ep_desc_hs, ZeroDev->mouse_ep_desc_hs.bLength);
	ZeroDev->HID_MouseInterfaceNumber = intf_desc_mouse.bInterfaceNumber;
	real_config_desc_fs->bNumInterfaces++;
	real_config_desc_hs->bNumInterfaces++;
	ZeroDev->g_ud.class_interface_request[intf_desc_mouse.bInterfaceNumber] = pphid_mouse_handle_classrequest;
	curr_ep++;
    } else {
	ZeroDev->HID_MouseInterfaceNumber = -1;
    }

    /* peppercon keyboard */
    if (ZeroDev->usb_device_kbd_type != usb_device_kbd_none) {
	ZeroDev->eps[curr_ep].used = 1;
	ZeroDev->eps[curr_ep].ep_ptr = &ZeroDev->keyboard_ep;
	snprintf(ZeroDev->eps[curr_ep].name, sizeof(ZeroDev->eps[curr_ep].name), "ep%d", curr_ep);

	intf_desc_keyboard.bInterfaceNumber = curr_intf++;
	ZeroDev->keyboard_ep_desc_fs.bEndpointAddress = USB_DIR_IN | curr_ep;
	ZeroDev->keyboard_ep_desc_hs.bEndpointAddress = USB_DIR_IN | curr_ep;
	add_to_config_descriptor_fshs(real_config_desc_fs, real_config_desc_hs, (uint8_t*)&intf_desc_keyboard, intf_desc_keyboard.bLength);
	add_to_config_descriptor_fshs(real_config_desc_fs, real_config_desc_hs, (uint8_t*)&ZeroDev->keyboard_hid_desc, ZeroDev->keyboard_hid_desc.bLength);
	add_to_config_descriptor(real_config_desc_fs, (uint8_t*)&ZeroDev->keyboard_ep_desc_fs, ZeroDev->keyboard_ep_desc_fs.bLength);
	add_to_config_descriptor(real_config_desc_hs, (uint8_t*)&ZeroDev->keyboard_ep_desc_hs, ZeroDev->keyboard_ep_desc_hs.bLength);
	ZeroDev->HID_KeyboardInterfaceNumber = intf_desc_keyboard.bInterfaceNumber;
	real_config_desc_fs->bNumInterfaces++;
	real_config_desc_hs->bNumInterfaces++;
	ZeroDev->g_ud.class_interface_request[intf_desc_keyboard.bInterfaceNumber] = pphid_keyboard_handle_classrequest;
    } else {
	ZeroDev->HID_KeyboardInterfaceNumber = -1;
    }

    for (i = 0; i < 16; i++) {
	if (ZeroDev->eps[i].used) D(D_BLABLA, "ep: %s used: %d\n", ZeroDev->eps[i].name, ZeroDev->eps[i].used);
    }
}

static int pp_usb_open(struct inode * inode, struct file * file)
{
    uint8_t minor = MINOR(inode->i_rdev);
    if (minor >= MAX_USB_DEVICES) return -ENODEV;
    if (!device_table[minor]) return -ENODEV;
    return 0;
}

#if defined(PP_FEAT_USB_FORCE_FS)
static int has_force_fs(void)
{
    /* FIXME: this check should take low level driver into account, since OTG
     *        does not support force FS; for now, just assume OTG is not used */
    return (KIRA_MAJOR(pp_kira_get_revision()) >= 2);
}
#endif

static int pp_usb_ioctl(struct inode * inode, struct file * file, uint cmd, ulong arg)
{
    uint8_t data[64];
    struct FTC_zero_dev *ZeroDev;
    uint8_t minor = MINOR(inode->i_rdev);

    if (minor >= MAX_USB_DEVICES) return -ENODEV;
    ZeroDev = device_table[minor];
    if (!ZeroDev) return -ENODEV;

    switch (cmd) {
      case LARAUSBENABLESETUPPROTO:
	  {
	      uint8_t enable_sp;
	      if (copy_from_user(&enable_sp, (uint8_t*)arg, sizeof(uint8_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	      scsi_setup_proto_enabled = enable_sp;
	      return SUCCESS;
	      break;
	  }
      case LARAUSBSETFILE:
	  {
	      file_info_t file_info;
	      
	      if (copy_from_user(&file_info, (file_info_t*)arg, sizeof(file_info_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	      // check parameters

	      if (file_info.dev_no >= PP_FEAT_USB_MASS_STORAGE_NO) {
		  printk("mass storage device number (dev_no) out of bounds (%d, max: %d)\n",
			 file_info.dev_no, PP_FEAT_USB_MASS_STORAGE_NO);
		  return -EINVAL;
	      }
	      
	      D(D_BLABLA, " set new backing file: %s, imagetype: %d\n", file_info.filename, file_info.image_type);
	      ppstorage_set_file(ZeroDev, file_info.dev_no, file_info.filename, file_info.readonly, file_info.image_type, file_info.blocklen);
	      if (file_info.image_type != PP_USB_IMAGE_TYPE_NONE) {
		  ZeroDev->mass_storage_types[file_info.dev_no] = file_info.image_type;
	      }
	      
	      return SUCCESS;
	      break;
	  }
      case LARAUSBGETDEVICESTATE:
	  {
	      usb_device_state_t d;
	      d = ZeroDev->g_DeviceSuspended ? (PP_USB_DEVSTAT_SUSPENDED) : PP_USB_DEVSTAT_CONFIGURED;
	      if (copy_to_user((char *)arg, (uint8_t*)&d, sizeof(usb_device_state_t))) {
		  printk("cannot copy to user\n");
		  return -EFAULT;
	      }
	  }
	  return SUCCESS;
	  break;
      case LARAUSBSETDEVICE:
	  {
	      /* set device */
	      lara_usb_device_t d;
	      int do_change_device = 0;
	      int i;
	      
	      if (copy_from_user(&d, (lara_usb_device_t*) arg, sizeof(lara_usb_device_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	      if (d.dev != usb_device_unchanged && d.dev != usb_device_peppercon) {
		  return -EINVAL;
	      }
	      if (d.dev != usb_device_unchanged) {
		  if (ZeroDev->usb_device_new != d.dev) {
		      ZeroDev->usb_device_new = d.dev;
		      do_change_device = 1;
		  }
	      }

	      if (d.keyboard_type != usb_device_kbd_unchanged) {
		  if (ZeroDev->usb_device_kbd_type_new != d.keyboard_type) {
		      ZeroDev->usb_device_kbd_type_new = d.keyboard_type;
		      do_change_device = 1;
		  }
	      }

	      if (d.mouse_type != peppercon_mouse_type_unchanged) {
		  if (ZeroDev->usb_device_mouse_type_new != d.mouse_type) {
		      ZeroDev->usb_device_mouse_type_new = d.mouse_type;
		      do_change_device = 1;
		  }
	      }

	      for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	          if (d.mass_storage[i] != usb_device_ms_unchanged) {
		      ZeroDev->enable_mass_storage_new[i] = d.mass_storage[i] == usb_device_ms_enabled;
		      if (ZeroDev->enable_mass_storage_new[i] != ZeroDev->enable_mass_storage[i]) {
			  do_change_device = 1;
		      }
	          }
	      }

	      if (do_change_device) {
		  D(D_BLABLA, "call change_usb_device\n");
		  change_usb_device(ZeroDev);
	      }
	  }
	  return SUCCESS;
	  break;
      case LARAUSBRESET:
	  change_usb_device(ZeroDev);
	  return SUCCESS;
	  break;
      case LARAUSBSENDDATA:
	  {
	      lara_usb_command_t command;
	      if (copy_from_user(&command, (lara_usb_command_t*) arg, sizeof(lara_usb_command_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	  
	      /* this is only a sanity check, so that we support up to 64 bytes transfers,
		 but the user space has to beware not to send more than 32+2 bytes for an
		 interrupt or bulk endpoint */
	      
	      if (command.length > (64+2)) {
		  printk("command_length too long: %d\n", command.length);
		  return -EFAULT;
	      }
	      
	      if (copy_from_user((char*) data, &((char*)arg)[sizeof(command)], command.length)) {
		  printk("cannot copy from user cmd\n");
		  return -EFAULT;
	      }
	      //      spin_lock_irqsave(&dev.chip_access_lock, dev.flags);
	      peppercon_send_data(ZeroDev, command.type, data, command.length);
	      //spin_unlock_irqrestore(&dev.chip_access_lock, dev.flags);
	      return SUCCESS;
	  }
	  break;

#if defined(PP_FEAT_USB_FORCE_FS)
      case LARAUSBSETFORCEFS:  /* set force full speed */
	  {
	      uint8_t force_fs;

	      if (copy_from_user(&force_fs, (uint8_t*)arg, sizeof(uint8_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	      ZeroDev->force_fs = force_fs;
	      
	      return 0;
	  }

      case LARAUSBHASFORCEFS:  /* force full speed supported? */
	  {
	      uint8_t u = has_force_fs();

	      if (copy_to_user((char *)arg, (uint8_t*)&u, sizeof(uint8_t))) {
		  printk("cannot copy to user\n");
		  return -EFAULT;
	      }
	      return 0;
	  }
#endif

      default:
	  D(D_ERROR, "Invalid ioctl(): cmd = %d\n", cmd);
	  return -EINVAL;
    }
}

static void change_usb_device(struct FTC_zero_dev *ZeroDev)
{
    int i;
    ZeroDev->usb_device = ZeroDev->usb_device_new;

    ZeroDev->usb_device_kbd_type = ZeroDev->usb_device_kbd_type_new;
    ZeroDev->usb_device_mouse_type = ZeroDev->usb_device_mouse_type_new;

    memcpy(ZeroDev->enable_mass_storage, ZeroDev->enable_mass_storage_new, sizeof(ZeroDev->enable_mass_storage));
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	ZeroDev->mass_storage_enabled[i] = ZeroDev->enable_mass_storage[i];
	ZeroDev->ms_enabled |= ZeroDev->mass_storage_enabled[i];
	ZeroDev->mass_storage_type[i] = ZeroDev->mass_storage_types[i];
    }

    if (ZeroDev->usb_device_mouse_type == peppercon_mouse_type_none
	    && ZeroDev->usb_device_kbd_type == usb_device_kbd_none
	    && !ZeroDev->ms_enabled) {
	ZeroDev->usb_device = usb_device_none;
	ZeroDev->usb_device_new = usb_device_none;
	return;
    }

    D(D_BLABLA, "will call restart\n");
    restart(ZeroDev);
}

/* These routines may be called in process context or in_irq */
static void wakeup_thread(struct FTC_zero_dev*ZeroDev)
{
	/* Tell the main thread that something has happened */
	ZeroDev->thread_wakeup_needed = 1;
	wake_up_all(&ZeroDev->thread_wqh);
}

/* Completion callbacks for usb_ep_queue */
static void ep0_complete(struct usb_ep *ep, struct usb_request *req)
{
    D(D_BLABLA, "ep0_complete()\n");
    if (req->status || req->actual != req->length) {
	D(D_NOTICE, "%s --> %d, %u/%u\n", __FUNCTION__, req->status, req->actual, req->length);
    }
    if (req->status == -ECONNRESET) {
	// Request was cancelled
	usb_ep_fifo_flush(ep);
    }
}

static void interrupt_in_complete(struct usb_ep *ep, struct usb_request *req)
{
    ep_driver_t * ep_driver = (ep_driver_t*) ep->driver_data;
    struct FTC_zero_dev *ZeroDev = ep_driver->dev;
    struct zero_buffhd	*bh = (struct zero_buffhd *) req->context;
    D(D_BLABLA, "interrupt_in_complete(ep = %s): status = %d, actual = %d, length = %d",
	    ep->name, req->status, req->actual, req->length);
    if (req->status == -ECONNRESET)		// Request was cancelled
	usb_ep_fifo_flush(ep);

    /* Hold the lock while we update the request and buffer states */
    spin_lock(&ZeroDev->lock);
    bh->inreq_busy = 0;
    bh->state = BUF_STATE_EMPTY;

    if (ep_driver->complete) {
	ep_driver->complete(ep);
    }

    spin_unlock(&ZeroDev->lock);
    wakeup_thread(ZeroDev);
}

static void bulk_in_complete(struct usb_ep *ep, struct usb_request *req)
{
    ep_driver_t * ep_driver = (ep_driver_t*) ep->driver_data;
    struct FTC_zero_dev *ZeroDev = ep_driver->dev;
	struct zero_buffhd	*bh = (struct zero_buffhd *) req->context;
	D(D_BLABLA, "bulk_in_complete(ep = %s): status = %d, actual = %d, length = %d\n",
		ep->name, req->status, req->actual, req->length);
	if (req->status == -ECONNRESET)		// Request was cancelled
		usb_ep_fifo_flush(ep);

	/* Hold the lock while we update the request and buffer states */
	spin_lock(&ZeroDev->lock);
	bh->inreq_busy = 0;
	bh->state = BUF_STATE_EMPTY;
	spin_unlock(&ZeroDev->lock);
	wakeup_thread(ZeroDev);
}

static void bulk_out_complete(struct usb_ep *ep, struct usb_request *req)
{
    ep_driver_t * ep_driver = (ep_driver_t*) ep->driver_data;
    struct FTC_zero_dev *ZeroDev = ep_driver->dev;
	struct zero_buffhd	*bh = (struct zero_buffhd *) req->context;
	D(D_BLABLA, "bulk_out_complete(ep = %s): status = %d, actual = %d, length = %d\n",
		ep->name, req->status, req->actual, req->length);
	if (req->status == -ECONNRESET)		// Request was cancelled
		usb_ep_fifo_flush(ep);

	/* Hold the lock while we update the request and buffer states */
	spin_lock(&ZeroDev->lock);
	bh->outreq_busy = 0;
	bh->state = BUF_STATE_FULL;
	spin_unlock(&ZeroDev->lock);
	wakeup_thread(ZeroDev);
}

/*-------------------------------------------------------------------------*/
static void reset_config (struct FTC_zero_dev *dev)
{
    int i;
    
	dev->u8UsbConfigValue = 0;
	dev->u8UsbInterfaceAlternateSetting = 0;

	/* just disable endpoints, forcing completion of pending i/o.
	 * all our completion handlers free their requests in this case.
	 */

	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    if (dev->storage[i]->Bin_ep) {
		disable_endpoint(dev, dev->storage[i]->Bin_ep);
		dev->storage[i]->Bin_ep = 0;
	    }
	    if (dev->storage[i]->Bout_ep) {
		disable_endpoint(dev, dev->storage[i]->Bout_ep);
		dev->storage[i]->Bout_ep = 0;
	    }
	}

	if (dev->mouse_ep) {
	    disable_endpoint(dev, dev->mouse_ep);
	    dev->mouse_ep = 0;
	}

	if (dev->keyboard_ep) {
	    disable_endpoint(dev, dev->mouse_ep);
	    dev->keyboard_ep = 0;
	}
}

/*-------------------------------------------------------------------------*/

int ep0_queue(struct FTC_zero_dev *ZeroDev)
{
	int	rc;

	rc = usb_ep_queue(ZeroDev->ep0, ZeroDev->ep0req, GFP_ATOMIC);
	if (rc != 0 && rc != -ESHUTDOWN)
	{
	    D(D_VERBOSE, "error in submission: %s --> %d\n",
				ZeroDev->ep0->name, rc);

	}
	return rc;
}

void raise_exception(struct FTC_zero_dev *ZeroDev, enum zero_state new_state)
{
	unsigned long		flags;
	struct task_struct	*thread_task;
	D(D_NOTICE, "raise_exception(new_state = %d)\n", new_state);

	/* Do nothing if a higher-priority exception is already in progress.
	* If a lower-or-equal priority exception is in progress, preempt it
	* and notify the main thread by sending it a signal. */
	spin_lock_irqsave(&ZeroDev->lock, flags);
	if (ZeroDev->state <= new_state) 
	{
		ZeroDev->exception_req_tag = ZeroDev->ep0_req_tag;
		ZeroDev->state = new_state;
		thread_task = ZeroDev->thread_task;
		if (thread_task) {
			send_sig_info(SIGUSR1, (void *) 1L, thread_task);
		}
	} else {
	    printk("higher exception in progress, ZeroDev->state: %d, new_state: %d\n", ZeroDev->state, new_state);
	}
	spin_unlock_irqrestore(&ZeroDev->lock, flags);
}


static int class_setup_req(struct FTC_zero_dev *ZeroDev, const struct usb_ctrlrequest *ctrl) {
    int rc = -ENOTSUPP;
    if (ctrl->wIndex >= MAX_INTERFACES) {
        D(D_NOTICE, "class interface request out of range: %d (max: %d)\n", ctrl->wIndex, MAX_INTERFACES);
    } else {
	    if (ZeroDev->g_ud.class_interface_request[ctrl->wIndex]) {
		D(D_VERBOSE, "class interface request for index: %x\n", ctrl->wIndex);
		rc = ZeroDev->g_ud.class_interface_request[ctrl->wIndex](ZeroDev, ctrl);
	    } else {
		D(D_ERROR, "handler not found\n");
	    }
    }
    return rc;
}

static int vendor_setup_req(struct FTC_zero_dev *ZeroDev, const struct usb_ctrlrequest *ctrl) {
    int rc = -ENOTSUPP;

    if (ZeroDev->g_ud.vendor_device_request) {
	rc = ZeroDev->g_ud.vendor_device_request(ZeroDev, ctrl);
    } else {
	D(D_ERROR, "unknown vendor device request\n");
    }
    return rc;
}


/*
 * The setup() callback implements all the ep0 functionality that's
 * not handled lower down, in hardware or the hardware driver (like
 * device and endpoint feature flags, and their status).  It's all
 * housekeeping for the gadget function we're implementing.  Most of
 * the work is in config-specific setup.
 */

static int
zero_setup (struct usb_gadget *gadget, const struct usb_ctrlrequest *ctrl)
{
	struct FTC_zero_dev *ZeroDev = get_gadget_data (gadget);
	int rc = -EOPNOTSUPP;
	u8 *Ep0ReqBuf;
	int queued = 0;
	uint8_t type;
	uint8_t bRecipient;

	Ep0ReqBuf=ZeroDev->ep0req->buf;
	type = ctrl->bRequestType & USB_TYPE_MASK;
	bRecipient = ctrl->bRequestType & USB_RECIPIENT_MASK;
	++ZeroDev->ep0_req_tag;		// Record arrival of a new request
	ZeroDev->ep0req->context = NULL;
	ZeroDev->ep0req->length = 0;
	ZeroDev->ep0req->zero = 0;
	if (!(ctrl->bRequestType & USB_DIR_IN) && ctrl->wLength > 0) {
	    /* data out setup request, just assume we consume all data */
	    ZeroDev->ep0req->length = ctrl->wLength;
	    ZeroDev->ep0req_name = (ctrl->bRequestType & USB_DIR_IN ? "ep0-in" : "ep0-out");
	    rc = ep0_queue(ZeroDev);
	    queued = 1;
	}
	D(D_VERBOSE, "ctrl-req: Recip: %x, Type: %x, bReq: %x, wIndex: %x, wLength: %x, wValue: %x\n",
	  bRecipient, type, ctrl->bRequest, ctrl->wIndex, ctrl->wLength, ctrl->wValue);
    if ((type == USB_STANDARD_REQUEST) && (ctrl->bRequest < MAX_STANDARD_REQUEST)) {
	rc = standard_setup_req(ZeroDev, ctrl);
    } else if ((type == USB_CLASS_REQUEST) && (bRecipient == USB_RECIPIENT_INTERFACE)) {
	/* all of our supported class specific requests have recipient == interface */
	rc = class_setup_req(ZeroDev, ctrl);
    } else if (type == USB_VENDOR_REQUEST) {
	rc = vendor_setup_req(ZeroDev, ctrl);
    }
	if (!queued) {
	    if (rc >= 0 && rc != DELAYED_STATUS) 
		/* Respond with data/status or defer until later? */
		
		{
		    ZeroDev->ep0req->length = rc;
		    if (ctrl->wLength > ZeroDev->ep0req->length) {
			// we want to send less than expected
			// so we must end possibly with a zero length packet
			ZeroDev->ep0req->zero = 1;
		    }
		    ZeroDev->ep0req_name = (ctrl->bRequestType & USB_DIR_IN ? "ep0-in" : "ep0-out");
		    rc = ep0_queue(ZeroDev);
		    
		    if(rc>=0)
			{
			    // remove vendor stuff
			}
		}
	}
	    
	/* Device either stalls (rc < 0) or reports success */
	return rc;
}

static int inline exception_in_progress(struct FTC_zero_dev*ZeroDev)
{
	return (ZeroDev->state > ZERO_STATE_IDLE);
}


static int sleep_thread(struct FTC_zero_dev *ZeroDev)
{
	int	rc;
	/* Wait until a signal arrives or we are woken up */
	D(D_BLABLA, "sleep_thread (ZeroDev = %p)\n", ZeroDev);
	rc = wait_event_interruptible(ZeroDev->thread_wqh,
			ZeroDev->thread_wakeup_needed);
	ZeroDev->thread_wakeup_needed = 0;
	return (rc ? -EINTR : 0);
}


int zero_set_halt(struct FTC_zero_dev*ZeroDev, struct usb_ep *ep)
{
    D(D_VERBOSE, "zero_set_halt(ep = %s)\n", ep->name);
    return usb_ep_set_halt(ep);
}


static int alloc_request(struct FTC_zero_dev*ZeroDev, struct usb_ep *ep,
			 struct usb_request **preq) {	
    int ret = -ENOMEM;
    D(D_BLABLA, "alloc_request(ep = %s)\n", ep->name);

    *preq = usb_ep_alloc_request(ep, GFP_ATOMIC);
    if (*preq) 
	ret = 0;
    else 
	D(D_ERROR, "can't allocate request for %s\n", ep->name);

    return ret;
}


/*
 * Reset interface setting and re-init endpoint state (toggle etc).
 * Call with altsetting < 0 to disable the interface.  The only other
 * available altsetting is 0, which enables the interface.
 */
static int do_set_interface(struct FTC_zero_dev*ZeroDev, int altsetting)
{
	int	rc = 0;
	int	i, j;
	struct usb_endpoint_descriptor	*d;
	
	D(D_VERBOSE, "do_set_interface(altsetting = %d)\n", altsetting);

reset:
	D(D_BLABLA, "do_set_interface: deallocating requests\n");
	/* Deallocate the requests */

	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    for (i = 0; i < NUM_BUFFERS; ++i) 
		{
		    struct zero_buffhd *bh = &ZeroDev->storage[j]->buffhds[i];
		    if (bh->inreq)
			{
			    usb_ep_free_request(ZeroDev->storage[j]->Bin_ep, bh->inreq);
			    bh->inreq = NULL;
			}
		    if (bh->outreq) 
			{
			    usb_ep_free_request(ZeroDev->storage[j]->Bout_ep, bh->outreq);
			    bh->outreq = NULL;
			}
		}
	}

	#if 0
	if (ZeroDev->intreq) {
		usb_ep_free_request(ZeroDev->intr_in, ZeroDev->intreq);
		ZeroDev->intreq = NULL;
	}
	#endif
	

	if(ZeroDev->buffhds_mouse_intr_in.inreq)
	{
		usb_ep_free_request(ZeroDev->mouse_ep, ZeroDev->buffhds_mouse_intr_in.inreq);
		ZeroDev->buffhds_mouse_intr_in.inreq = NULL;
	}
	if(ZeroDev->buffhds_keyboard_intr_in.inreq)
	{
		usb_ep_free_request(ZeroDev->keyboard_ep, ZeroDev->buffhds_keyboard_intr_in.inreq);
		ZeroDev->buffhds_keyboard_intr_in.inreq = NULL;
	}
	D(D_BLABLA, "do_set_interface(): disabling endpoints\n");

	/* Disable the endpoints */
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    if (ZeroDev->storage[i]->bulk_in_enabled) {
		disable_endpoint(ZeroDev, ZeroDev->storage[i]->Bin_ep);
		ZeroDev->storage[i]->bulk_in_enabled = 0;
	    }
	    if (ZeroDev->storage[i]->bulk_out_enabled) {
		disable_endpoint(ZeroDev, ZeroDev->storage[i]->Bout_ep);
		ZeroDev->storage[i]->bulk_out_enabled = 0;
	    }
	}

	if (ZeroDev->mouse_enabled) {
	    disable_endpoint(ZeroDev, ZeroDev->mouse_ep);
	    ZeroDev->mouse_enabled = 0;
	}

	if (ZeroDev->keyboard_enabled) {
	    disable_endpoint(ZeroDev, ZeroDev->keyboard_ep);
	    ZeroDev->keyboard_enabled = 0;
	}

	if (altsetting < 0 || rc != 0)
		return rc;

	D(D_BLABLA, "do_set_interface(): enabling endpoints\n");

	/* Enable the endpoints */
	if (ZeroDev->ms_enabled) {
	    //printk("***** ms enabled 1\n");
	    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
		if (!ZeroDev->mass_storage_enabled[i]) continue;
		d = &ZeroDev->mass_storage_bulk_in_desc[i];
		ep_set_speed(ZeroDev->gadget, d, FS_BULK_OUT_MAXPACKET, HS_BULK_OUT_MAXPACKET);
		
		//ep_desc(ZeroDev->gadget, &fs_bulk_in_desc, &hs_bulk_in_desc);
		if ((rc = enable_endpoint(ZeroDev, ZeroDev->storage[i]->Bin_ep, d)) != 0)
/* FIXME: goto */
		    goto reset;
		ZeroDev->storage[i]->bulk_in_enabled = 1;
		
		
		d = &ZeroDev->mass_storage_bulk_out_desc[i];
		ep_set_speed(ZeroDev->gadget, d, FS_BULK_IN_MAXPACKET, HS_BULK_IN_MAXPACKET);
		//d = ep_desc(ZeroDev->gadget, &fs_bulk_out_desc, &hs_bulk_out_desc);
		if ((rc = enable_endpoint(ZeroDev, ZeroDev->storage[i]->Bout_ep, d)) != 0)
/* FIXME: goto */
		    goto reset;
		//printk("enable bulk out: <%d>,\n", i);
		ZeroDev->storage[i]->bulk_out_enabled = 1;
	    }

	}

	if (ZeroDev->usb_device_mouse_type != peppercon_mouse_type_none) {
	    d = ep_desc(ZeroDev->gadget, &ZeroDev->mouse_ep_desc_fs, &ZeroDev->mouse_ep_desc_hs);
	    if ((rc = enable_endpoint(ZeroDev, ZeroDev->mouse_ep, d)) != 0)
		goto reset;
	    {
		ep_driver_t * ep_driver = (ep_driver_t*) ZeroDev->mouse_ep->driver_data;
		ep_driver->complete = pphid_ep_cb;
		ep_driver->class_ep_data = kmalloc(sizeof(hid_class_ep_data_t), GFP_KERNEL);
		memset(ep_driver->class_ep_data, 0, sizeof(hid_class_ep_data_t));
	    }
	    ZeroDev->mouse_enabled = 1;
	}
	
	if (ZeroDev->usb_device_kbd_type != usb_device_kbd_none) {
	    d = ep_desc(ZeroDev->gadget, &ZeroDev->keyboard_ep_desc_fs, &ZeroDev->keyboard_ep_desc_hs);
	    if ((rc = enable_endpoint(ZeroDev, ZeroDev->keyboard_ep, d)) != 0)
		goto reset;
	    {
		ep_driver_t * ep_driver = (ep_driver_t*) ZeroDev->keyboard_ep->driver_data;
		ep_driver->complete = pphid_ep_cb;
		ep_driver->class_ep_data = kmalloc(sizeof(hid_class_ep_data_t), GFP_KERNEL);
		memset(ep_driver->class_ep_data, 0, sizeof(hid_class_ep_data_t));
	    }
	    ZeroDev->keyboard_enabled = 1;
	}

	D(D_BLABLA, "do_set_interface(): allocating requests");
	/* Allocate the requests */
	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    if (ZeroDev->mass_storage_enabled[j]) {
		for (i = 0; i < NUM_BUFFERS; ++i) {
		    struct zero_buffhd *bh = &ZeroDev->storage[j]->buffhds[i];
		    if ((rc = alloc_request(ZeroDev, ZeroDev->storage[j]->Bin_ep, &bh->inreq)) != 0)
			goto reset;
		    if ((rc = alloc_request(ZeroDev, ZeroDev->storage[j]->Bout_ep, &bh->outreq)) != 0)
			    goto reset;
		    bh->inreq->buf = bh->outreq->buf = bh->buf;
		    bh->inreq->dma = bh->outreq->dma = bh->dma;
		    bh->inreq->context = bh->outreq->context = bh;
		    bh->inreq->complete = bulk_in_complete;
		    bh->outreq->complete = bulk_out_complete;
		}
	    }
	}

	if (ZeroDev->mouse_enabled) {
	    if ((rc = alloc_request(ZeroDev, ZeroDev->mouse_ep, &ZeroDev->buffhds_mouse_intr_in.inreq)) != 0)
		goto reset;
	    {
		ep_driver_t* epd = (ep_driver_t*)ZeroDev->mouse_ep->driver_data;
		epd->bufhd = &ZeroDev->buffhds_mouse_intr_in;
	    }
	    ZeroDev->buffhds_mouse_intr_in.inreq->buf =  ZeroDev->buffhds_mouse_intr_in.buf;
	    ZeroDev->buffhds_mouse_intr_in.inreq->dma =  ZeroDev->buffhds_mouse_intr_in.dma;
	    ZeroDev->buffhds_mouse_intr_in.inreq->context =  &ZeroDev->buffhds_mouse_intr_in;
	    ZeroDev->buffhds_mouse_intr_in.inreq->complete = interrupt_in_complete;
	}
	
	if (ZeroDev->keyboard_enabled) {
	    if ((rc = alloc_request(ZeroDev, ZeroDev->keyboard_ep, &ZeroDev->buffhds_keyboard_intr_in.inreq)) != 0)
		goto reset;
	    {
		ep_driver_t* epd = (ep_driver_t*)ZeroDev->keyboard_ep->driver_data;
		epd->bufhd = &ZeroDev->buffhds_keyboard_intr_in;
	    }
	    ZeroDev->buffhds_keyboard_intr_in.inreq->buf = ZeroDev->buffhds_keyboard_intr_in.buf;
	    ZeroDev->buffhds_keyboard_intr_in.inreq->dma = ZeroDev->buffhds_keyboard_intr_in.dma;
	    ZeroDev->buffhds_keyboard_intr_in.inreq->context = &ZeroDev->buffhds_keyboard_intr_in;
	    ZeroDev->buffhds_keyboard_intr_in.inreq->complete = interrupt_in_complete;
	}

	return rc;
}


/*
 * Change our operational configuration.  This code must agree with the code
 * that returns config descriptors, and with interface altsetting code.
 *
 * It's also responsible for power management interactions.  Some
 * configurations might not work with our current power sources.
 * For now we just assume the gadget is always self-powered.
 */
static int do_set_config(struct FTC_zero_dev*ZeroDev, u8 new_config)
{
	int	rc = 0;
	D(D_VERBOSE, "do_set_config()\n");
	
	/* Disable the single interface */
	if (ZeroDev->config != 0) 
	{
		ZeroDev->config = 0;
		rc = do_set_interface(ZeroDev, -1);
	}
	/* Enable the interface */
	if (new_config != 0) 
	{
		ZeroDev->config = new_config;
		if ((rc = do_set_interface(ZeroDev, 0)) != 0)
			ZeroDev->config = 0;	// Reset on errors
	}
	ZeroDev->g_DeviceSuspended = 0;
	propchange_fire(PP_PROP_USB_DEV_STATE_CHANGED, 0);
	return rc;
}

static void handle_exception(struct FTC_zero_dev*ZeroDev)
{
	siginfo_t		info;
	int			sig;
	int			i;
	int			num_active = 0;
	struct zero_buffhd	*bh;
	enum zero_state	old_state;
	u8			new_config;
	unsigned int	exception_req_tag;
	int			rc;

	D(D_NOTICE, "handle_exception(state = %d)\n", ZeroDev->state);
	/* Clear the existing signals.  Anything but SIGUSR1 is converted
	 * into a high-priority EXIT exception. */
	for (;;)
	{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		spin_lock_irq(&current->sigmask_lock);
		sig = dequeue_signal(&ZeroDev->thread_signal_mask, &info);
		spin_unlock_irq(&current->sigmask_lock);
#else
		sig = dequeue_signal(current, &ZeroDev->thread_signal_mask, &info);
#endif
		if (!sig)
			break;
		if (sig != SIGUSR1) 
		{
			raise_exception(ZeroDev, ZERO_STATE_EXIT);
		}
	}
#if 0
	/* Cancel all the pending transfers */
	if (ZeroDev->intreq_busy)
		usb_ep_dequeue(ZeroDev->intr_in, ZeroDev->intreq);
#endif
	// dequeue bulk endpoint request
	int j;
	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    for (i = 0; i < NUM_BUFFERS; i++) {
		bh = &ZeroDev->storage[j]->buffhds[i];
		
		if (bh->inreq_busy)			
		    usb_ep_dequeue(ZeroDev->storage[j]->Bin_ep, bh->inreq);
		if (bh->outreq_busy)
		    usb_ep_dequeue(ZeroDev->storage[j]->Bout_ep, bh->outreq);
		bh->inreq_busy=bh->outreq_busy=0;
	    }
	}

#if 0
	/* dequeue mouse and keyboard instead of other stuff */
	// dequeue interrupt endpoint request
	if(ZeroDev->buffhds_intr_in.inreq_busy)
		usb_ep_dequeue(ZeroDev->Iin_ep, ZeroDev->buffhds_intr_in.inreq);
	
	if(ZeroDev->buffhds_intr_out.outreq_busy)
		usb_ep_dequeue(ZeroDev->Iout_ep, ZeroDev->buffhds_intr_out.outreq);
	ZeroDev->buffhds_intr_in.inreq_busy=ZeroDev->buffhds_intr_out.outreq_busy=0;
#endif
	/* Wait until everything is idle */
	for (;;) 
	{
	    int j;
	    for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
				
		for (i = 0; i < NUM_BUFFERS; ++i) 
		{
			bh = &ZeroDev->storage[j]->buffhds[i];
			num_active += 0;
		}
	    }
		if ((num_active == 0) ||
		     (ZeroDev->state == ZERO_STATE_EXIT) ||
		     (ZeroDev->state == ZERO_STATE_TERMINATED))
			break;
		if (sleep_thread(ZeroDev))
			return;
	}
	/* Clear out the controller's fifos */
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    if (ZeroDev->storage[i]->bulk_in_enabled)
		usb_ep_fifo_flush(ZeroDev->storage[i]->Bin_ep);
	    if (ZeroDev->storage[i]->bulk_out_enabled)
		usb_ep_fifo_flush(ZeroDev->storage[i]->Bout_ep);
	}
	if (ZeroDev->mouse_enabled)
	    usb_ep_fifo_flush(ZeroDev->mouse_ep);
	if (ZeroDev->keyboard_enabled)
	    usb_ep_fifo_flush(ZeroDev->keyboard_ep);
	/* Reset the I/O buffer states and pointers, the SCSI
	 * state, and the exception.  Then invoke the handler. */
	spin_lock_irq(&ZeroDev->lock);

	// Set Bulk buffer structure state
	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    for (i = 0; i < NUM_BUFFERS; ++i) 
		{
		    bh = &ZeroDev->storage[j]->buffhds[i];
		    bh->state = BUF_STATE_EMPTY;
		}
	}
	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    ZeroDev->storage[j]->next_buffhd_to_fill
		= ZeroDev->storage[j]->next_buffhd_to_drain
		= &ZeroDev->storage[j]->buffhds[0];
	}

	exception_req_tag = ZeroDev->exception_req_tag;
	new_config = ZeroDev->new_config;
	old_state = ZeroDev->state;

	ZeroDev->state = ZERO_STATE_IDLE;
	
	spin_unlock_irq(&ZeroDev->lock);
	/* Carry out any extra actions required for the exception */
	switch (old_state) 
	{
		default:
			break;

		case ZERO_STATE_INTERFACE_CHANGE:
			for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
			    if (ZeroDev->ms_enabled && ZeroDev->storage[i]) ppstorage_suspend_thread(ZeroDev->storage[i], 1);
			}
			rc = do_set_interface(ZeroDev, 0);
			for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
			    if (ZeroDev->mass_storage_enabled[i]) {
				usb_ep_fifo_flush(ZeroDev->storage[i]->Bin_ep);
				usb_ep_clear_halt(ZeroDev->storage[i]->Bin_ep);
				usb_ep_fifo_flush(ZeroDev->storage[i]->Bout_ep);
				usb_ep_clear_halt(ZeroDev->storage[i]->Bout_ep);
				if (ZeroDev->ms_enabled && ZeroDev->storage[i]) ppstorage_suspend_thread(ZeroDev->storage[i], 0);
			    }
			}
			if (ZeroDev->ep0_req_tag != exception_req_tag)
				break;
			if (rc != 0)			// STALL on errors
				zero_set_halt(ZeroDev, ZeroDev->ep0);
			else				// Complete the status stage
				ep0_queue(ZeroDev);
			break;

		case ZERO_STATE_CONFIG_CHANGE:
			for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
			    if (ZeroDev->ms_enabled && ZeroDev->storage[i]) ppstorage_suspend_thread(ZeroDev->storage[i], 1);
			}
			rc = do_set_config(ZeroDev, new_config);
			for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
			    if (ZeroDev->mass_storage_enabled[i]) {
				usb_ep_fifo_flush(ZeroDev->storage[i]->Bin_ep);
				usb_ep_clear_halt(ZeroDev->storage[i]->Bin_ep);
				usb_ep_fifo_flush(ZeroDev->storage[i]->Bout_ep);
				usb_ep_clear_halt(ZeroDev->storage[i]->Bout_ep);
				if (new_config != 0 && ZeroDev->ms_enabled && ZeroDev->storage[i]) ppstorage_suspend_thread(ZeroDev->storage[i], 0);
			    }
			}
			if (ZeroDev->ep0_req_tag != exception_req_tag)
				break;
			if (rc != 0)			// STALL on errors
				zero_set_halt(ZeroDev, ZeroDev->ep0);
			else				// Complete the status stage
				ep0_queue(ZeroDev);
			break;

		case ZERO_STATE_DISCONNECT:
			for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
			    ppstorage_sync_file(ZeroDev->storage[i]);
			}
			do_set_config(ZeroDev, 0);		// Unconfigured state
			break;

		case ZERO_STATE_EXIT:
		case ZERO_STATE_TERMINATED:
			D(D_VERBOSE, "handle_exception(): terminating thread\n");
			do_set_config(ZeroDev, 0);			// Free resources
			spin_lock_irq(&ZeroDev->lock);
			ZeroDev->state = ZERO_STATE_TERMINATED;	// Stop the thread
			spin_unlock_irq(&ZeroDev->lock);
			break;
	}
}

////////////////////// Bulk test function start ///////////////////////

/* Use this for bulk or interrupt transfers, not ep0 */
void start_transfer(struct FTC_zero_dev*ZeroDev, struct usb_ep *ep,
		struct usb_request *req, volatile int *pbusy, volatile enum fsg_buffer_state *state)
{
	int	rc;
	unsigned long flags;
	
	spin_lock_irqsave(&ZeroDev->lock, flags);
	*pbusy = 1;
	*state = BUF_STATE_BUSY;
	
	rc = usb_ep_queue(ep, req, GFP_KERNEL);
	D(D_BLABLA, "enqueue on ep: %s, queue ret: %d\n", ep->name, rc);
	if (rc != 0) 
	{
		*pbusy = 0;
		*state = BUF_STATE_EMPTY;

		/* We can't do much more than wait for a reset */
		/* Note: currently the net2280 driver fails zero-length
		 * submissions if DMA is enabled. */
		if (rc != -ESHUTDOWN && 
		   !(rc == -EOPNOTSUPP && req->length == 0))
		{
		    D(D_VERBOSE, "start_transfer()-> error in submission: %s --> %d\n", ep->name, rc);
		}
	}
	spin_unlock_irqrestore(&ZeroDev->lock, flags);
}

int zero_wait_for_buf_free(struct FTC_zero_dev *zero_dev, struct zero_buffhd *bh)
{
    return wait_event_interruptible(zero_dev->thread_wqh, (bh->state == BUF_STATE_EMPTY));
}

int zero_wait_for_buf_full(struct FTC_zero_dev *zero_dev, struct zero_buffhd *bh)
{
    return wait_event_interruptible(zero_dev->thread_wqh, (bh->state == BUF_STATE_FULL));
}


////////////////////// Interrupt test function finish ///////////////////

static int zero_main_thread(void *ZeroDev_)
{
	struct FTC_zero_dev *ZeroDev = (struct FTC_zero_dev *) ZeroDev_;

	D(D_VERBOSE, "main thread starting (ZeroDev = %p)\n", ZeroDev);

	
	ZeroDev->thread_task = current;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	/* Release all our userspace resources */
	daemonize();
	reparent_to_init();
	strncpy(current->comm, "FTC-zero-gadget",
			sizeof(current->comm) - 1);
#else
	daemonize("FTC-zero-gadget");
#endif

	/* Allow the thread to be killed by a signal, but set the signal mask
	 * to block everything but INT, TERM, KILL, and USR1. */
	siginitsetinv(&ZeroDev->thread_signal_mask, sigmask(SIGINT) |
			sigmask(SIGTERM) | sigmask(SIGKILL) |
			sigmask(SIGUSR1));
	current->blocked = ZeroDev->thread_signal_mask;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	spin_lock_irq(&current->sigmask_lock);
	flush_signals(current);
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);
#else
	flush_signals(current);
#endif

	/* Wait for the gadget registration to finish up */
	wait_for_completion(&ZeroDev->thread_notifier_start);

	/* The main loop */
	while (ZeroDev->state != ZERO_STATE_TERMINATED) {
	    D(D_BLABLA, "**** config: %x\n", ZeroDev->config);
	    if (exception_in_progress(ZeroDev) || signal_pending(current)) {
		D(D_VERBOSE, "ex in progress: %d, sig pending: %d\n", exception_in_progress(ZeroDev), signal_pending(current));
		handle_exception(ZeroDev);
		continue;
	    }
	    /* Wait for the next signal to handle */
	    sleep_thread(ZeroDev);
	}

	ZeroDev->thread_task = NULL;
	flush_signals(current);

	/* Let the unbind and cleanup routines know the thread has exited */
	complete_and_exit(&ZeroDev->thread_notifier_exit, 0);
}

////////////////////////////////////////////////////////////////////////////

static void
zero_suspend (struct usb_gadget *gadget)
{
    struct FTC_zero_dev *dev = get_gadget_data (gadget);
    dev->g_DeviceSuspended = 1;
    D(D_VERBOSE, "Suspended...\n");
    propchange_fire(PP_PROP_USB_DEV_STATE_CHANGED, 0);
}


static void
zero_disconnect (struct usb_gadget *gadget)
{
	struct FTC_zero_dev	*dev = get_gadget_data (gadget);
	unsigned long		flags;

	D(D_VERBOSE, "now Disconnect device\n");

	spin_lock_irqsave (&dev->lock, flags);
	reset_config (dev);

	/* a more significant application might have some non-usb
	 * activities to quiesce here, saving resources like power
	 * or pushing the notification up a network stack.
	 */
	spin_unlock_irqrestore (&dev->lock, flags);


	raise_exception(dev, ZERO_STATE_DISCONNECT);

	/* next we may get setup() calls to enumerate new connections;
	 * or an unbind() during shutdown (including removing module).
	 */
}

/*-------------------------------------------------------------------------*/
static void
zero_unbind (struct usb_gadget *gadget)
{
	struct FTC_zero_dev *ZeroDev = get_gadget_data (gadget);
	int	i;
	struct usb_request *req = ZeroDev->ep0req;

	D(D_VERBOSE, "zero_unbind(ZeroDev = %p)\n", ZeroDev);

	clear_bit(REGISTERED, &ZeroDev->atomic_bitflags);

	ppstorage_unbind(gadget);

	/* If the thread isn't already dead, tell it to exit now */
	if (ZeroDev->state != ZERO_STATE_TERMINATED) 
	{
		init_completion(&ZeroDev->thread_notifier_exit);
		raise_exception(ZeroDev, ZERO_STATE_EXIT);
		wait_for_completion(&ZeroDev->thread_notifier_exit);
	}
	
	/* Free the data buffers */
	int j;
	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    for (i = 0; i < NUM_BUFFERS; ++i)  {
		struct zero_buffhd *bh = &ZeroDev->storage[j]->buffhds[i];
		
		if (bh->buf)
		    usb_ep_free_buffer(ZeroDev->storage[j]->Bin_ep, bh->buf, bh->dma, BULK_EP_BUFLEN);
	    }
	}

#if 0
	/* fixme free endpoints for kbd and mouse here */
	if(ZeroDev->buffhds_intr_in.buf)
		usb_ep_free_buffer(ZeroDev->Iin_ep, ZeroDev->buffhds_intr_in.buf, 
							ZeroDev->buffhds_intr_in.dma, USB_ZERO_BUFSIZ);
	
	if(ZeroDev->buffhds_intr_out.buf)
		usb_ep_free_buffer(ZeroDev->Iout_ep, ZeroDev->buffhds_intr_out.buf,
							ZeroDev->buffhds_intr_out.dma, USB_ZERO_BUFSIZ);
#endif
	/* Free the request and buffer for endpoint 0 */
	if (req) 
	{
		if (req->buf)
			usb_ep_free_buffer(ZeroDev->ep0, req->buf,
					req->dma, EP0_BUFSIZE);
		usb_ep_free_request(ZeroDev->ep0, req);
	}
	
	/* we've already been disconnected ... no i/o is active */
	// kfree (ZeroDev);
	set_gadget_data (gadget, 0);

}

// don't use this anywhere outside restart() or zero_bind()!
static struct FTC_zero_dev *zero_dev_tmp;
static spinlock_t zero_dev_tmp_lock;

static int
zero_bind (struct usb_gadget *gadget)
{
	struct FTC_zero_dev *ZeroDev = zero_dev_tmp;
	int i, rc;
	struct usb_ep		*ep;
	struct usb_request	*req;	
	D(D_VERBOSE, "zero_bind(ZeroDev = %p)\n", ZeroDev);

	ZeroDev->gadget = gadget;
	set_gadget_data(gadget, ZeroDev);
	ZeroDev->ep0 = gadget->ep0;
	ZeroDev->ep0->driver_data = ZeroDev;

	for (i = 0; i < MAX_INTERFACES; i++) {
	    ZeroDev->g_ud.class_interface_request[i] = NULL;
	}
	
	D(D_BLABLA, "start pp_prepare_descriptors\n");
	pp_prepare_descriptors(ZeroDev);

	D(D_BLABLA, "start pp_generate_serial_number\n");
	peppercon_generate_serial_number(ZeroDev);
	D(D_BLABLA, "serial number successfully calculated\n");
	
	/* Find all the endpoints we will use */
	// these are all endpoints the chip has
	gadget_for_each_ep(ep, gadget) {
		// will we use this endpoint?
		for (i = 0; i < 16; i++) {
		    if (ZeroDev->eps[i].used && (!strcmp(ep->name, ZeroDev->eps[i].name))) {
			//printk("found and use ep: %s\n", ep->name);
			if (ZeroDev->eps[i].ep_ptr) *ZeroDev->eps[i].ep_ptr = ep;
		    }
		}
	}
	ZeroDev->bulk_out_maxpacket = (gadget->speed == USB_SPEED_HIGH ? 512 :
			FS_BULK_OUT_MAXPACKET);
			
	/* Allocate the request and buffer for endpoint 0 */
	ZeroDev->ep0req = req = usb_ep_alloc_request(ZeroDev->ep0, GFP_KERNEL);

	if (!req)
		goto out;
	req->buf = usb_ep_alloc_buffer(ZeroDev->ep0, EP0_BUFSIZE,
			&req->dma, GFP_KERNEL);
	if (!req->buf)
		goto out;
	req->complete = ep0_complete;

	/* ring buffer for bulk in */
	int j;
	for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
	    if (ZeroDev->storage[j]->Bin_ep) {
		//printk("reserve bulk_in ring buffer\n");
		for (i = 0; i < NUM_BUFFERS; ++i) 
		    {
			struct zero_buffhd *bh = &ZeroDev->storage[j]->buffhds[i];
			bh->buf = usb_ep_alloc_buffer(ZeroDev->storage[j]->Bin_ep, BULK_EP_BUFLEN,
						      &bh->dma, GFP_KERNEL);
			if (!bh->buf)
			    goto out;
			bh->next = bh + 1;
		    }
		ZeroDev->storage[j]->buffhds[NUM_BUFFERS - 1].next = &ZeroDev->storage[j]->buffhds[0];
	    }
	}

	if (ZeroDev->usb_device_mouse_type != peppercon_mouse_type_none) {
	    ZeroDev->buffhds_mouse_intr_in.buf=usb_ep_alloc_buffer(ZeroDev->mouse_ep, USB_ZERO_BUFSIZ,
								   &ZeroDev->buffhds_mouse_intr_in.dma, GFP_KERNEL);
	    ZeroDev->buffhds_mouse_intr_in.next=&ZeroDev->buffhds_mouse_intr_in;
	}

	if (ZeroDev->usb_device_kbd_type != usb_device_kbd_none) {
	    ZeroDev->buffhds_keyboard_intr_in.buf=usb_ep_alloc_buffer(ZeroDev->keyboard_ep, USB_ZERO_BUFSIZ,
								      &ZeroDev->buffhds_keyboard_intr_in.dma, GFP_KERNEL);
	    ZeroDev->buffhds_keyboard_intr_in.next=&ZeroDev->buffhds_keyboard_intr_in;
	}

	/* This should reflect the actual gadget power source */
	usb_gadget_set_selfpowered(gadget);

	if (ZeroDev->ms_enabled) {
	    //printk("***** ms enabled 3\n");
	    if (ppstorage_bind(gadget)) {
		printk("********** arg problem\n");
		goto out;
	    }
	}
	
	init_completion(&ZeroDev->thread_notifier_start);
	ZeroDev->state = ZERO_STATE_IDLE;
	if ((rc = kernel_thread(zero_main_thread, ZeroDev, (CLONE_VM | CLONE_FS | CLONE_FILES))) < 0)
		goto out;
	ZeroDev->thread_pid = rc;

	D(D_NOTICE, "I/O thread pid: %d\n", ZeroDev->thread_pid);
	return 0;

out:
	ZeroDev->state = ZERO_STATE_TERMINATED;	// The thread is dead
	zero_unbind(gadget);
	return -ENOMEM;
}

/*-------------------------------------------------------------------------*/

static struct usb_gadget_driver zero_driver = 
{
	.speed		= USB_SPEED_HIGH,    /* max req speed (may change) */
	.function	= (char *) longname,
	.bind		= zero_bind,
	.unbind		= zero_unbind,

	.setup		= zero_setup,
	.disconnect	= zero_disconnect,
	.suspend        = zero_suspend,
	.driver 	= 
	{
		.name		= (char *) shortname,
		// .shutdown = ...
		// .suspend = ...
		// .resume = ...
	},
};

MODULE_AUTHOR ("Peppercon AG");
MODULE_LICENSE ("GPL");

static struct FTC_zero_dev *zero_alloc(void)
{
	struct FTC_zero_dev *ZeroDev;
	int i;

	spin_lock_init(&zero_dev_tmp_lock);

	ZeroDev = kmalloc(sizeof *ZeroDev, GFP_KERNEL);

	if (!ZeroDev) return NULL;
	memset(ZeroDev, 0, sizeof *ZeroDev);

	ZeroDev->stringtab.language = 0x0409; // en-us
	ZeroDev->stringtab.strings = pp_usb_strings_new();

	ZeroDev->hid = pphid_init();
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    ZeroDev->storage[i] = ppstorage_intf_new(ZeroDev, i);
	}

	spin_lock_init(&ZeroDev->lock);
	init_waitqueue_head(&ZeroDev->thread_wqh);
	init_completion(&ZeroDev->thread_notifier_start);
	init_completion(&ZeroDev->thread_notifier_exit);

	memset(ZeroDev->enable_mass_storage, 0, sizeof(ZeroDev->enable_mass_storage));;
	memset(ZeroDev->enable_mass_storage_new, 0, sizeof(ZeroDev->enable_mass_storage_new));;
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    ZeroDev->mass_storage_types[i] = PP_USB_IMAGE_TYPE_NONE;
	}

	return ZeroDev;
}

static void zero_free(struct FTC_zero_dev *ZeroDev)
{
    int i;

    pp_usb_strings_delete(ZeroDev->stringtab.strings);

    /* Free pphid and ppstorage interface structures */
    pphid_deinit(ZeroDev->hid);
    ZeroDev->hid = NULL;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	ppstorage_intf_free(ZeroDev->storage[i]);
	ZeroDev->storage[i] = NULL;
    }

    kfree(ZeroDev);
}

static void restart(struct FTC_zero_dev *ZeroDev)
{
    int rc;
    D(D_VERBOSE, "restart()\n");

    /* stop the driver */
    if (test_and_clear_bit(REGISTERED, &ZeroDev->atomic_bitflags)) {
	ZeroDev->unregister_driver(&zero_driver);
    }

    /* restart the driver */
    spin_lock(&zero_dev_tmp_lock);
#if defined(PP_FEAT_USB_FORCE_FS)
    // pass "force full speed" down to low level driver
    zero_driver.speed = (ZeroDev->force_fs) ? USB_SPEED_FULL : USB_SPEED_HIGH;
#endif
    // This is ugly! There must be a better way to make the ZeroDev pointer
    // known in the zero_bind() function called by register_driver().
    // Is this what usb_gadget_driver.__gadget_driver.driver_data is for?
    zero_dev_tmp = ZeroDev;
    rc = ZeroDev->register_driver(&zero_driver);
    spin_unlock(&zero_dev_tmp_lock);
    if (rc != 0) {
	printk("problem\n");
	return;
    }

    set_bit(REGISTERED, &ZeroDev->atomic_bitflags);
    ZeroDev->g_ud.vendor_device_request = NULL;
    complete(&ZeroDev->thread_notifier_start);
}

/*
 * Note: The original Linux USB gadget interface only supports a single device
 *       controller, and the higher-level drivers called its register_driver()
 *       function. Since we want to support a dynamic number of DC drivers we
 *       needed to swap roles: This driver has to be loaded first and exports
 *       a register_lowlevel_driver function. After that low-level drivers
 *       can sign up for any valid minor device number.
 */
int FTC_zero_register_lowlevel_driver(int minor, struct usb_lowlevel_driver driver)
{
    int i;
    struct FTC_zero_dev *ZeroDev;

    if (minor >= MAX_USB_DEVICES) return -EINVAL;
    if (device_table[minor]) {
	D(D_ERROR, "USB device #%d is already in use\n", minor);
	return -EBUSY;
    }

    ZeroDev = device_table[minor] = zero_alloc();
    if (!ZeroDev) return -ENOMEM;
    ZeroDev->minor = minor;
    ZeroDev->register_driver = driver.register_driver;
    ZeroDev->unregister_driver = driver.unregister_driver;

    memset(ZeroDev->eps, 0, sizeof(ZeroDev->eps));

    for (i = 0; i < MAX_INTERFACES; i++) {
	ZeroDev->g_ud.class_interface_request[i] = NULL;
    }

    /* set some defaults */
    ZeroDev->ms_enabled = 0;
    ZeroDev->keyboard_enabled = 0;
    ZeroDev->mouse_enabled = 0;
    ZeroDev->HID_MouseInterfaceNumber = -1;
    ZeroDev->HID_KeyboardInterfaceNumber = -1;
    ZeroDev->g_DeviceSuspended = 1;
    ZeroDev->usb_device_new = usb_device_none;
    ZeroDev->usb_device = usb_device_none;
    ZeroDev->usb_device_kbd_type_new = usb_device_kbd_none;
    ZeroDev->usb_device_kbd_type = usb_device_kbd_none;
    ZeroDev->usb_device_mouse_type_new = peppercon_mouse_type_none;
    ZeroDev->usb_device_mouse_type = peppercon_mouse_type_none;

    /* Copy default descriptors */
    memcpy(&ZeroDev->device_desc, &default_device_desc, sizeof(default_device_desc));
    memcpy(&ZeroDev->dev_qualifier, &default_dev_qualifier, sizeof(default_dev_qualifier));
    memcpy(&ZeroDev->mouse_hid_desc, &default_mouse_hid_desc, sizeof(default_mouse_hid_desc));
    memcpy(&ZeroDev->mouse_ep_desc_fs, &default_mouse_ep_desc_fs, sizeof(default_mouse_ep_desc_fs));
    memcpy(&ZeroDev->mouse_ep_desc_hs, &default_mouse_ep_desc_hs, sizeof(default_mouse_ep_desc_hs));
    memcpy(&ZeroDev->keyboard_hid_desc, &default_keyboard_hid_desc, sizeof(default_keyboard_hid_desc));
    memcpy(&ZeroDev->keyboard_ep_desc_fs, &default_keyboard_ep_desc_fs, sizeof(default_keyboard_ep_desc_fs));
    memcpy(&ZeroDev->keyboard_ep_desc_hs, &default_keyboard_ep_desc_hs, sizeof(default_keyboard_ep_desc_hs));

    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	ZeroDev->mass_storage_bulk_in_desc[i].bLength = USB_DT_ENDPOINT_SIZE;
	ZeroDev->mass_storage_bulk_in_desc[i].bDescriptorType = USB_DT_ENDPOINT;
	ZeroDev->mass_storage_bulk_in_desc[i].bEndpointAddress = 0 | USB_DIR_IN;
	ZeroDev->mass_storage_bulk_in_desc[i].bmAttributes = USB_ENDPOINT_XFER_BULK;
	ZeroDev->mass_storage_bulk_in_desc[i].wMaxPacketSize = cpu_to_le16(FS_BULK_OUT_MAXPACKET);

	ZeroDev->mass_storage_bulk_out_desc[i].bLength = USB_DT_ENDPOINT_SIZE;
	ZeroDev->mass_storage_bulk_out_desc[i].bDescriptorType = USB_DT_ENDPOINT;
	ZeroDev->mass_storage_bulk_out_desc[i].bEndpointAddress = 0;
	ZeroDev->mass_storage_bulk_out_desc[i].bmAttributes = USB_ENDPOINT_XFER_BULK;
	ZeroDev->mass_storage_bulk_out_desc[i].wMaxPacketSize = cpu_to_le16(FS_BULK_OUT_MAXPACKET);
    }

    return minor;
}
EXPORT_SYMBOL(FTC_zero_register_lowlevel_driver);

int FTC_zero_unregister_lowlevel_driver(int minor)
{
    int i;
    struct FTC_zero_dev *ZeroDev;
    if (!device_table[minor]) return -EINVAL;
    ZeroDev = device_table[minor];

    if (test_and_clear_bit(REGISTERED, &ZeroDev->atomic_bitflags)) {
	/* Shut down all ppstorage interfaces */
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    if (ZeroDev->mass_storage_enabled[i]) {
		ppstorage_suspend_thread(ZeroDev->storage[i], 1);
		ppstorage_close_file(ZeroDev->storage[i]);
	    }
	}
	/* Unregister the driver in case the thread hasn't already done so */
	ZeroDev->unregister_driver(&zero_driver);
    }
    zero_free(ZeroDev);
    device_table[minor] = NULL;

    return 0;
}
EXPORT_SYMBOL(FTC_zero_unregister_lowlevel_driver);

static int __init init (void)
{
    memset(device_table, 0, sizeof(device_table));
    if (register_chrdev(LARA_USB_MAJOR, "ppusbdrv", &lara_usb_ops) < 0) {
	D(D_ERROR, "%s: failed to register driver (register_chrdev)\n", "pp_usbdrv");
	return -EBUSY;
    }
#ifdef PP_FEAT_IPMI_SERVER_SCSI_CHAN
    scsi_ipmi_init();
#endif
    return 0;
}
module_init (init);

static void __exit cleanup (void)
{
    int i;
    for (i = 0; i < MAX_USB_DEVICES; i++) {
	if (device_table[i]) FTC_zero_unregister_lowlevel_driver(i);
    }

    if ((i = unregister_chrdev(LARA_USB_MAJOR, "ppusbdrv")) < 0) {
	printk("%s: failed to unregister driver (%d)\n", "ppusbdrv", i);
    } else {
	printk("%s: driver unregistered\n", "ppusbdrv");
    }
}
module_exit (cleanup);

