/*
 * Interruptible background sync
 * (C) 2007 Raritan Computer, Ingo van Lil <ingo.lil@raritan.com>
 *
 * Purpose: SCSI commands must be completed within a definite time, e.g. a
 *   Windows host will time out and reset the SCSI device if it doesn't
 *   receive a response within ten seconds. For this reason the read() and
 *   write() calls in our SCSI implementation are cancelled by a signal if
 *   they block too long.
 *   A SCSI Write should ideally block until all data has been successfully
 *   written to the backing medium. Therefore the SCSI_Write function
 *   invokes fdatasync() call after performing all writes. Unfortunately
 *   fdatasync() may indefinitely remain in uninterruptible wait.
 *   This implementation delegates the fdatasync() to a separate thread
 *   which will trigger a completion after all data has been written.
 *   Waiting for this completion can be interrupted by a signal.
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/wait.h>
#include <linux/file.h>
#include <linux/version.h>

#include "bg_sync.h"

struct bg_sync_s {
    spinlock_t lock;
    struct file *filp;

    wait_queue_head_t request_wq;
    wait_queue_head_t done_wq;

    volatile int req_count; // increased for each new sync request
    volatile int req_done;  // counter value at start of last successful sync
    volatile int terminate;
};

static int requests_pending(bg_sync_t *bg_sync)
{
    spin_lock(&bg_sync->lock);
    int pending = bg_sync->req_count - bg_sync->req_done;
    spin_unlock(&bg_sync->lock);
    return pending;
}

static int request_done(bg_sync_t *bg_sync, int request)
{
    spin_lock(&bg_sync->lock);
    int ret = ((bg_sync->req_done - request) >= 0);
    spin_unlock(&bg_sync->lock);
    return ret;
}

/*
 * Sync the file data, don't bother with the metadata.
 * This code was copied from fs/buffer.c:sys_fdatasync().
 *
 * For a Linux 2.4.x compatible version see the history
 * of ppstorage.c
 */
static int sync_file(struct file *filp)
{
    int rc, err;

    if (!filp) return 0;
    if (!filp->f_op || !filp->f_op->fsync) return -EINVAL;

    struct address_space *mapping = filp->f_mapping;
    current->flags |= PF_SYNCWRITE;
    rc = filemap_fdatawrite(mapping);
    down(&mapping->host->i_sem);
    err = filp->f_op->fsync(filp, filp->f_dentry, 1);
    if (!rc) rc = err;
    up(&mapping->host->i_sem);
    err = filemap_fdatawait(mapping);
    if (!rc) rc = err;
    current->flags &= ~PF_SYNCWRITE;

    return rc;
}

static int bg_sync_thread(void *_bg_sync)
{
    daemonize("bg_sync");

    bg_sync_t *bg_sync = _bg_sync;

    while (!bg_sync->terminate) {
	wait_event_interruptible(bg_sync->request_wq,
		bg_sync->terminate || requests_pending(bg_sync));
	if (bg_sync->terminate) break;

	int pending = requests_pending(bg_sync);
	if (pending) {
	    if (sync_file(bg_sync->filp) == 0) {
		spin_lock(&bg_sync->lock);
		bg_sync->req_done += pending;
		spin_unlock(&bg_sync->lock);
		wake_up_all(&bg_sync->done_wq);
	    }
	}
    }

    fput(bg_sync->filp);
    kfree(bg_sync);
    return 0;
}

bg_sync_t *bg_sync_new(struct file *filp)
{
    get_file(filp);

    bg_sync_t *bg_sync = kmalloc(sizeof(bg_sync_t), GFP_KERNEL);
    memset(bg_sync, 0, sizeof(*bg_sync));
    spin_lock_init(&bg_sync->lock);
    init_waitqueue_head(&bg_sync->request_wq);
    init_waitqueue_head(&bg_sync->done_wq);
    bg_sync->filp = filp;

    kernel_thread(bg_sync_thread, bg_sync, (CLONE_VM | CLONE_FS | CLONE_FILES));
    return bg_sync;
}

void bg_sync_delete(bg_sync_t *bg_sync)
{
    if (bg_sync) {
	bg_sync->terminate = 1;
	wake_up_interruptible(&bg_sync->request_wq);
    }
}

int bg_sync_do_sync(bg_sync_t *bg_sync, int timeout)
{
    int rc;

    if (!bg_sync) return -EINVAL;

    spin_lock(&bg_sync->lock);
    int this_req = (++bg_sync->req_count);
    spin_unlock(&bg_sync->lock);
    wake_up_interruptible(&bg_sync->request_wq);

    if (timeout > 0) {
	rc = wait_event_interruptible_timeout(bg_sync->done_wq, request_done(bg_sync, this_req), timeout);
    } else {
	rc = wait_event_interruptible(bg_sync->done_wq, request_done(bg_sync, this_req));
    }

    if (request_done(bg_sync, this_req)) {
	return 0;
    } else if (rc == 0) {
	return -ETIMEDOUT;
    }
    return rc;
}

