#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/rtc.h>
#include <linux/interrupt.h>
#include <asm/time.h>
#include <asm/io.h>
#include <asm/irq.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/ppc405_pcimcs.h>
#else
# include <syslib/ppc405_pcimcs.h>
#endif

#include "pp_usb_ISP1181B.h"
#include "hal_isp1181.h"
#include "pp_usb.h"

#undef INTERRUPT_DEBUG

#if defined(INTERRUPT_DEBUG)
#define PPC405_GPIO_REGS_ADDR           ((u32)0xEF600700)
#define PPC405_GPIO_REGS_SIZE           ((size_t)0x20)
#define LARA_HOST_RST                   LARA_PPC405_GPIO6_BIT
#define LARA_PPC405_GPIO6_BIT           25
#endif

#define LARA_USB_ADDR			((u32)0xF0000000)
#define LARA_USB_SIZE			((size_t)0x4)
#define LARA_USB_IRQ			25

#if (PP_HWID_INT >= 0xa)
# define BUS16
# define TIME_MEASURE
# include "../timer.h"
#else
# undef BUS16
#endif


/* this chip delay should give as a delay in nanoseconds (ca. 200ns) (look at
   lara_usb_ISP1181B.h)

   it seems, that our actual processor with 200 Mhz is slow enough to fullfill
   the constrains without any chip delay

   this assumption is based on analyzer measurment
*/


/* for timediff */
#ifdef BUS16
# define CHIPDELAY() { static unsigned long long t2start; t2start = ppc_timebase();while (ppc_timediff_ns(t2start, ppc_timebase()) < (unsigned long long) ISP1181B_DELAY ) {}; }
#else 
# define CHIPDELAY() { {}; }
#endif

#if 0
/* this should be slightly better, but untested */
inline void CHIPDELAY(void) {
    register uint32_t w = 2; /* 100ns */
    asm volatile("   mtctr %0\n" \
		 "1: bdnz 1b\n"   \
		 ::"r"(w)	  \
		 );
}
#endif

static void write_command(uint8_t cmd);
static void read_data(uint8_t* data, uint8_t size);
static void write_data(const uint8_t* data, uint8_t size);
extern irqreturn_t lara_usb_isr(int irq, void * dev_id, struct pt_regs * regs);
extern lara_usb_dev_t dev;

#ifdef BUS16
static u16 *		base_addr;
#else
static u8 *		base_addr;
#endif


#ifdef INTERRUPT_DEBUG
static volatile u32* gpio_regs;
#endif

int __init
hal_init()
{
    
#ifdef INTERRUPT_DEBUG
    /* ---- map GPIO registers ----------------------------------------- */
    gpio_regs = ioremap_nocache(PPC405_GPIO_REGS_ADDR,
				PPC405_GPIO_REGS_SIZE);
    if (gpio_regs == NULL) {    
        printk("failed to map GPIO registers of PPC405\n");
    }   
#endif
    
    /* ---- map the USB chip ----------------------------------------------- */
    base_addr = ioremap_nocache(LARA_USB_ADDR, LARA_USB_SIZE);
    if (base_addr == NULL) {
	printk("%s: failed to map USB chip\n", dev.name);
	goto err_out;
    }
    /* ---- check for existance of USB chip -------------------------------- */

    {
	uint8_t id_low, id_high;
	int error;
	
	/* because it is possible, that the bank1 is not mapped, use
	 * the machine check aware read/write functions
	 */
#ifdef BUS16
	mcs_out_le16(ISP1181B_CMD_READ_CHIP_ID << 8, (uint16_t*)((uint32_t)base_addr | 2), &error);
#else
	mcs_out_8(ISP1181B_CMD_READ_CHIP_ID, (uint8_t*)((uint32_t)base_addr | 1), &error);
#endif
        CHIPDELAY();
	if (error) {
	    printk("%s: chip not accessible\n", dev.name);
	    goto err_out;
	}
#ifdef BUS16
	{
	    uint16_t id;
	    id = mcs_in_le16(base_addr, &error);
	    id = le16_to_cpu(id);
            CHIPDELAY();
	    if (error) goto err_out;
	    id_low = (uint8_t) (id & 0xff);
	    id_high = (uint8_t) ((id & 0xff00) >> 8);
	}
#else
	id_low = mcs_in_8(base_addr, &error);
        CHIPDELAY();
	if (error) goto err_out;
	id_high = mcs_in_8(base_addr, &error);
	if (error) goto err_out;
#endif
	printk("%s: chip ID low: %x, high: %x\n", dev.name, id_low, id_high);
	
	if (id_high != ISP1181B_CHIP_ID) {
	    goto err_out;
	}
    }
    
    hal_ResetDevice();
    
    /* ---- register ISR --------------------------------------------------- */

    if (request_irq(LARA_USB_IRQ, lara_usb_isr, SA_SHIRQ | SA_SAMPLE_RANDOM,
		    "LARA USB", &dev)) {
	printk("%s: couldn't register USB IRQ handler\n", dev.name);
	goto err_out;
    }
    dev.isr_registered = 1;
    
    return SUCCESS;

 err_out:
    hal_cleanup();
    return -ENODEV;
}


void hal_cleanup(void)
{
    if (dev.isr_registered) {
	free_irq(LARA_USB_IRQ, &dev);
	dev.isr_registered = 0;
    }
    
    if (base_addr != NULL) {
	iounmap((void *)base_addr);
	base_addr = NULL;
    }
}

static void set_int_polarity(void) {
    /* protected by calling reset function with mutex */
    union {
    	struct 	{
	    uint8_t hw1;
	    uint8_t hw2;
	} hw8;
	uint16_t hw;
    } hw;

    write_command(ISP1181B_CMD_READ_HW_CONFIG);
    read_data((uint8_t*)&hw.hw, 2);
    hw.hw8.hw1 &= ~ISP1181B_HW1_INTPOL;
    write_command(ISP1181B_CMD_WRITE_HW_CONFIG);
    write_data((uint8_t*)&hw.hw, 2);
}

void hal_ResetDevice(void) {
    ulong flags;
    
    spin_lock_irqsave(&dev.chip_access_lock, flags);
    disable_irq(LARA_USB_IRQ); /* to avoid conflicts with wrong polarity */
    write_command(ISP1181B_CMD_RESET);
    set_int_polarity();
    enable_irq(LARA_USB_IRQ);
    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
    udelay(3);
}


void hal_SetEndpointConfig(uint8_t ep, uint8_t config) {
    write_command(ISP1181B_CMD_EP_WRITE_CONFIG_BASE + ep);
    write_data(&config, 1);
}

uint8_t hal_GetEndpointConfig(uint8_t ep) {
    uint8_t config;
    write_command(ISP1181B_CMD_EP_READ_CONFIG_BASE + ep);
    read_data(&config, 1);
    return config;
}

void hal_SetAddressEnable(uint8_t address, uint8_t enable) {
    if (enable) {
	address |= 0x80;
    } else {
	address &= ~0x80;
    }
    write_command(ISP1181B_CMD_WRITE_ADDRESS);
    write_data(&address, 1);
}

uint8_t hal_GetAddress(void) {
    uint8_t address;
    write_command(ISP1181B_CMD_READ_ADDRESS);
    read_data(&address, 1);
    return address;
}
    

void hal_SetMode(uint8_t mode) {
    write_command(ISP1181B_CMD_WRITE_MODE);
    write_data(&mode, 1);
}

uint8_t hal_GetMode(void) {
    uint8_t mode;
    write_command(ISP1181B_CMD_READ_MODE);
    read_data(&mode, 1);
    return mode;
}

void hal_SetDevConfig(uint16_t config) {
    config = cpu_to_le16(config);
    write_command(ISP1181B_CMD_WRITE_HW_CONFIG);
    write_data((uint8_t*)&config, 2);
}

uint16_t hal_GetDevConfig(void) {
    uint16_t config;
    write_command(ISP1181B_CMD_READ_HW_CONFIG);
    read_data((uint8_t*)&config, 2);
    config = le16_to_cpu(config);
    return config;
}

void hal_SetIntEnable(uint32_t interrupts) {
    interrupts = cpu_to_le32(interrupts);
    write_command(ISP1181B_CMD_WRITE_INT_ENABLE);
    write_data((uint8_t*)&interrupts, 4);
}

uint32_t hal_GetIntEnable(void) {
    uint32_t interrupts;
    write_command(ISP1181B_CMD_READ_INT_ENABLE);
    read_data((uint8_t*)&interrupts, 4);
    interrupts = le32_to_cpu(interrupts);
    return interrupts;
}

void hal_WriteEndpoint(uint8_t ep, const uint8_t * buf, uint16_t len) {
    uint16_t buflen;
    buflen = cpu_to_le16(len);
    write_command(ISP1181B_CMD_EP_WRITE_BASE + ep);
    write_data((uint8_t*)&buflen, 2);
    write_data(buf, len);
    hal_ValidateEndpoint(ep);
}

uint16_t hal_ReadEndpoint(uint8_t ep, uint8_t * buf, uint16_t len) {
    uint16_t bufsize;
    
    bufsize = hal_ReadEndpointWOClear(ep, buf, len);
    hal_ClearEndpoint(ep);
    
    return bufsize;
}
    
uint16_t hal_ReadEndpointWOClear(uint8_t ep, uint8_t * buf, uint16_t len) {
    uint16_t bufsize;

    write_command(ISP1181B_CMD_EP_READ_BASE + ep);

    read_data((uint8_t*)&bufsize, 2);
    bufsize = le16_to_cpu(bufsize);

    if (bufsize > len) {
	bufsize = len;
    }

    read_data(buf, bufsize);

    return bufsize;
}

void hal_SetEndpointStatus(uint8_t ep, uint8_t stalled) {
    if (stalled) {
	// stall EP
	write_command(ISP1181B_CMD_EP_EP_STALL_BASE + ep);
    } else {
	write_command(ISP1181B_CMD_EP_EP_UNSTALL_BASE + ep);
    }
    UD("stall ep (%x): %x\n", ep, stalled); 
}
   
uint8_t hal_GetEndpointStatus(uint8_t ep) {
    uint8_t status;
    write_command(ISP1181B_CMD_EP_STATUS_BASE + ep);
    read_data(&status, 1);
    return status;
}

uint8_t hal_GetEndpointStatusWOClear(uint8_t ep) {
    uint8_t status;
    write_command(ISP1181B_CMD_EP_CHECK_STATUS_BASE + ep);
    read_data(&status, 1);
    return status;
}

uint8_t hal_GetEndpointError(uint8_t ep) {
    uint8_t status;
    write_command(ISP1181B_CMD_EP_ERROR_CODE_BASE + ep);
    read_data(&status, 1);
    return status;
}

void hal_ValidateEndpoint(uint8_t ep) {
    write_command(ISP1181B_CMD_EP_VALIDATE_BASE + ep);
}

void hal_ClearEndpoint(uint8_t ep) {
    write_command(ISP1181B_CMD_EP_CLEAR_BASE + ep);
}

void hal_AcknowledgeSetup(void) {
    write_command(ISP1181B_CMD_EP_ACK_SETUP);
}

void hal_UnlockDevice(uint16_t code) {
    write_command(ISP1181B_CMD_UNLOCK_DEVICE);
    code = cpu_to_le16(code);
    write_data((uint8_t*)&code, 2);
}

uint16_t hal_ReadChipID(void) {
    uint16_t id;
    write_command(ISP1181B_CMD_READ_CHIP_ID);
    read_data((uint8_t*)&id, 2);
    id = le16_to_cpu(id);
    return id;
}

uint32_t hal_ReadInterruptRegister(void) {
    uint32_t interrupts;

    const u_int32_t wanted_bits  =
	ISP1181B_INT_BUSSTATE |
	ISP1181B_INT_IERESM |
	ISP1181B_INT_IESUSP |
	ISP1181B_INT_IERST
	;

    write_command(ISP1181B_CMD_READ_INT_STATE);
    read_data((uint8_t*)&interrupts, 4);
    interrupts = le32_to_cpu(interrupts);
    
    if ((interrupts & 0xff) & ~wanted_bits) {
	/* throw away garbage */
#ifdef INTERRUPT_DEBUG
	set_bit(LARA_HOST_RST, &gpio_regs[0]);
	udelay(1);
	clear_bit(LARA_HOST_RST, &gpio_regs[0]);
#endif
	printk("broken USB irq: %08lx, mask: %08lx\n", (long unsigned int)(le32_to_cpu(interrupts)), (long unsigned int)~wanted_bits);
	interrupts &= 0xffffff00;
    }
    
    return interrupts;
}

void hal_StallEP0InControlWrite(void)
{
        hal_SetEndpointStatus(EP0IN, 1);
	// it seems, that the chip doesn't like it, if we stall an endpoint
	// which has already got some data
	// so just stall the DATAIN
	// hal_SetEndpointStatus(EP0OUT, 1);
}

void hal_StallEP0InControlRead(void)
{
    //hal_SetEndpointStatus(EP0OUT, 1);
    /* this is very strange, but let explain it:
       we ALWAYS STALL control request during the IN transfer:
       1. control request with data IN: read request, STALL DATA IN phase
       2. control request with data OUT: read request, read data, STALL HANDSHAKE IN phase
       3. control request with no data: read request, STALL HANDSHAKE IN phase
       That's why we always stall IN 
    */
    hal_SetEndpointStatus(EP0IN, 1);
}

void hal_SingleTransmitEP0(const uint8_t* buf, const uint16_t len)
{
    if( len <= ISP1181B_EP0IN_FIFO_SIZE) {
	hal_WriteEndpoint(EP0IN, buf, len);
    }
}


/* ------------------------------------------------------------------------- *
 * interrupt service routines
 * ------------------------------------------------------------------------- */
static void write_command(uint8_t cmd) {
    //printk("command: %02x\n", cmd);
#ifdef BUS16
    writew(cmd << 8, (uint16_t*)((uint32_t)base_addr | 2));
#else
    writeb(cmd, (uint8_t*)((uint32_t)base_addr | 1));
#endif
    CHIPDELAY();
}

static void read_data(uint8_t* data, uint8_t size) {
    int i;
    
#ifdef BUS16
    {
	uint16_t* data16 = (uint16_t*) data;
	for (i = 0; i < size / 2; i++) {
	    data16[i] = (readw(base_addr));
	    CHIPDELAY();
	}
	/* check for odd byte */
	if (i*2 < size) {
	    uint16_t last_byte;
	    last_byte = (readw(base_addr));
	    data[size-1] = (uint8_t)((last_byte & 0xff00) >> 8) ;
	}

    }
#else
    for (i = 0; i < size; i++) {
	data[i] = readb(base_addr);
	CHIPDELAY();
    }
#endif
}

static void write_data(const uint8_t* data, uint8_t size) {
    int i;
#ifdef BUS16
    { 
	uint16_t* data16 = (uint16_t*) data;
	for (i = 0; i < size / 2; i ++) {
	    writew((data16[i]), base_addr);
	    CHIPDELAY();
	}
	if (i*2 < size) {
	    uint16_t last_byte;
	    last_byte = data[size-1] << 8;
	    writew(last_byte, base_addr);
	    CHIPDELAY();
	}


    }
#else
    for (i = 0; i < size; i++) {
	writeb(data[i], base_addr);
	CHIPDELAY();
    }
#endif
}

#undef BUS16
