#include <linux/kernel.h>
#include <linux/fs.h>
#include <asm/current.h>
#include <linux/sched.h>
#include <linux/file.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include <linux/fsnotify.h>
#endif
#include "usb-mass-storage-bot.h"
#include "ppstorage.h"
#include "scsi.h"
#include "bg_sync.h"

#include <linux/kernel.h>
static const char shortname [] = 	"FTC_STORAGE";
#define DRV_NAME "ftc_storage"


#define backing_file_is_open(intf)     ((intf)->filp != NULL)

static void set_state(struct ppstorage_intf_t *intf, ppstorage_state_t state);

/* Make bulk-out requests be divisible by the maxpacket size */
static void inline set_bulk_out_req_length(struct FTC_zero_dev *fsg,
	struct zero_buffhd *bh, unsigned int length)
{
    unsigned int rem;
    bh->out_intended_length = length;
    rem = length % fsg->bulk_out_maxpacket;
    if (rem > 0) length += fsg->bulk_out_maxpacket - rem;
    bh->outreq->length = length;
}

int ppstorage_handle_class_req(struct FTC_zero_dev *ZeroDev, const struct usb_ctrlrequest *ctrl)
{
    struct usb_request *req = ZeroDev->ep0req;
    int value = -EOPNOTSUPP;
    int i;
    switch (ctrl->bRequest) {
	case BO_MASS_STORAGE_RESET:
	    if (ctrl->bRequestType !=
		    (USB_DIR_OUT | USB_TYPE_CLASS | USB_RECIP_INTERFACE)
		|| ctrl->wValue != 0
		|| ctrl->wLength != 0) {
		D(D_ERROR, "Invalid Reset request: bRequestType = 0x%02x, wLength = %d, wValue = %d\n",
			ctrl->bRequestType, ctrl->wLength, ctrl->wValue);
		break;
	    }
	    D(D_BLABLA, "mass storage reset, if: %d\n", ctrl->wIndex);
	    for (i=0; i<PP_FEAT_USB_MASS_STORAGE_NO; i++) {
		struct ppstorage_intf_t *intf = ZeroDev->storage[i];
		if (intf->interface_number == ctrl->wIndex) {
		    usb_ep_fifo_flush(intf->Bin_ep);
		    usb_ep_fifo_flush(intf->Bout_ep);
		    set_state(intf, PPSTORAGE_STATE_RESET);
		    /* The ppstorage_thread will transmit the status as soon as
		       the mass-storage reset has been completed */
		    intf->response_pending = 1;
		    return DELAYED_STATUS;
		}
	    }
	    D(D_ERROR, "Invalid reset request: wIndex = %d\n", ctrl->wIndex);
	    break;

	case BO_MASS_STORAGE_GET_MAX_LUN:
	    if (ctrl->bRequestType !=
		    (USB_DIR_IN | USB_TYPE_CLASS | USB_RECIP_INTERFACE)
		|| ctrl->wValue != 0
		|| ctrl->wLength != 1) {
		D(D_ERROR, "Invalid Get Max LUN request: bRequestType = 0x%02x, wLength = %d, wValue = %d\n",
			ctrl->bRequestType, ctrl->wLength, ctrl->wValue);
		break;
	    }
	    *(u8 *) req->buf = 0; /* No multiple LUN support */
	    value = min(ctrl->wLength, (u16) 1);
	    break;

	default:
	    D(D_ERROR, "Unsupported class request: bRequestType = 0x%02x\n", ctrl->bRequestType);
    }
    return value;
}

static int halt_bulk_in_endpoint(struct ppstorage_intf_t *intf)
{
    int rc;

    D(D_VERBOSE, "stalling bulk-in endpoint\n");
    rc = zero_set_halt(intf->zero_dev, intf->Bin_ep);
    while (rc != 0) {
	if (rc != -EAGAIN) {
	    rc = 0;
	    break;
	}
	/* Wait for a short time and then try again */
	set_current_state(TASK_INTERRUPTIBLE);
	if (schedule_timeout(HZ / 10) != 0) return -EINTR;
	rc = zero_set_halt(intf->zero_dev, intf->Bin_ep);
    }
    return rc;
}

static int pad_with_zeros(struct ppstorage_intf_t *intf)
{
    struct zero_buffhd *bh = NULL;
    while (intf->usb_amount_left) {
	int amount = min(intf->usb_amount_left, (u32)BULK_EP_BUFLEN);
	bh = intf->next_buffhd_to_fill;
	if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return -EINTR;
	bh->inreq->length = amount;
	bh->inreq->zero = 0;
	start_transfer(intf->zero_dev, intf->Bin_ep, bh->inreq,
		&bh->inreq_busy, &bh->state);
	intf->usb_amount_left -= amount;
    }
    if (bh) {
	intf->next_buffhd_to_fill = bh->next;
    }
    return 0;
}

static int throw_away_data(struct ppstorage_intf_t *intf)
{
    struct zero_buffhd *bh = intf->next_buffhd_to_drain;
    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return -EINTR;
    while (intf->usb_amount_left) {
	int amount = min(intf->usb_amount_left, (u32)BULK_EP_BUFLEN);
	set_bulk_out_req_length(intf->zero_dev, bh, amount);
	start_transfer(intf->zero_dev, intf->Bout_ep, bh->outreq,
		&bh->outreq_busy, &bh->state);
	if (zero_wait_for_buf_full(intf->zero_dev, bh) != 0) {
	    return -EINTR;
	}
	bh->state = BUF_STATE_EMPTY; /* Ignore received data */
	/* A short packet or an error ends everything */
	if (bh->outreq->actual != bh->outreq->length ||
		bh->outreq->status != 0) {
	    return -EINTR;
	}
	intf->usb_amount_left -= amount;
    }
    return 0;
}

static int send_csw(struct ppstorage_intf_t *intf)
{
    struct zero_buffhd *bh;
    struct bulk_cs_wrap *csw;
    u8 status = USB_STATUS_PASS;
    D(D_BLABLA, "ppstorage_send_csw()\n");

    /* Wait for the next buffer to become available */
    bh = intf->next_buffhd_to_fill;
    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return -EINTR;
    intf->next_buffhd_to_fill = bh->next;

    if (intf->phase_error) {
	D(D_VERBOSE, "sending phase-error status\n");
	status = USB_STATUS_PHASE_ERROR;
    } else if (intf->error) {
	D(D_BLABLA, "sending command-failure status\n");
	status = USB_STATUS_FAIL;
	D(D_VERBOSE, "status: %d (fail=%d)\n", status, USB_STATUS_FAIL);
    }

    /* Store and send the Bulk-only CSW */
    csw = (struct bulk_cs_wrap *) bh->buf;
    csw->Signature = __constant_cpu_to_le32(USB_BULK_CS_SIG);
    csw->Tag = intf->tag;
    csw->Residue = cpu_to_le32(intf->residue);
    csw->Status = status;
    D(D_BLABLA, "status2: %d (fail=%d)\n", csw->Status, USB_STATUS_FAIL);
    bh->inreq->length = USB_BULK_CS_WRAP_LEN;
    bh->inreq->zero = 0;
    D(D_BLABLA, "devno: %d, ep: %s\n", intf->dev_no, intf->Bin_ep->name);
    start_transfer(intf->zero_dev, intf->Bin_ep, bh->inreq,
	    &bh->inreq_busy, &bh->state);

    D(D_BLABLA, "ppstorage_send_csw() finished\n");
    return 0;
}

static int call_pp_scsi(struct ppstorage_intf_t *intf)
{
    transfer_phase_t data_phase;
    uint32_t data_len;
    uint8_t error;
    uint8_t* buf;

    // the residue is set to the data size from the CBW
    intf->residue = intf->data_size;
    // theoretically the amount of data which must be transfered is the data size from CBW
    intf->usb_amount_left = intf->data_size;

    error = scsi_handler(&intf->bot, intf->cmnd, intf->cmnd_size,
	    intf->data_size, intf->data_dir, &buf, &data_phase, &data_len);
    intf->data_size_from_scsi = data_len;
    intf->data_dir_from_scsi = data_phase;

    if (intf->usb_amount_left > 0) {
	/* scsi_handler() didn't transfer as much data as the host requested
	   in the CBW, so we have to get read some more data and throw it
	   away (out phase) or pad the data with zeros (in phase) */
	if (intf->data_dir == DATA_DIR_FROM_HOST) {
	    throw_away_data(intf);
	} else {
	    pad_with_zeros(intf);
	}
    }

    intf->error = (!error)? 1 : 0;

    /* Check for phase error condition -- TODO: Move into CSW function */
    if ( (intf->data_size < intf->data_size_from_scsi) ||
	    (intf->data_dir_from_scsi != intf->data_dir
	     && intf->data_size_from_scsi > 0) ) {
	D(D_VERBOSE, "Phase error (data_size=%d, data_dir=%d, data_size_from_scsi=%d, data_dir_from_scsi=%d\n",
		intf->data_size, intf->data_dir, intf->data_size_from_scsi, intf->data_dir_from_scsi);
	intf->phase_error = 1;
	return -EINVAL;
    }
    return intf->data_size_from_scsi;
}

static int do_scsi_command(struct ppstorage_intf_t *intf)
{
    int reply = -EINVAL;

    intf->phase_error = 0;
    intf->short_packet_received = 0;

    reply = call_pp_scsi(intf);
    if (reply == -EINTR || signal_pending(current)) {
	return -EINTR;
    }
    return 0;
}

/*-------------------------------------------------------------------------*/

static int received_cbw(struct ppstorage_intf_t *intf, struct zero_buffhd *bh)
{
    struct usb_request *req = bh->outreq;
    struct bulk_cb_wrap *cbw = (struct bulk_cb_wrap *) req->buf;

    /* Was this a real packet? */
    if (req->status) return -EINVAL;

    /* Is the CBW valid? Stall endpoints otherwise */
    if (req->actual != USB_BULK_CB_WRAP_LEN ||
	    cbw->Signature != __constant_cpu_to_le32(USB_BULK_CB_SIG)) {
	D(D_ERROR, "invalid CBW: len %u sig 0x%x\n",
		req->actual, le32_to_cpu(cbw->Signature));
	zero_set_halt(intf->zero_dev, intf->Bout_ep);
	halt_bulk_in_endpoint(intf);
	return -EINVAL;
    }

    /* Is the CBW meaningful? */
    if (cbw->Lun > 0 || cbw->Flags & ~USB_BULK_IN_FLAG ||
	    cbw->Length < 6 || cbw->Length > MAX_COMMAND_SIZE) {
	D(D_ERROR, "non-meaningful CBW: lun = %u, flags = 0x%x, cmdlen %u\n",
		cbw->Lun, cbw->Flags, cbw->Length);
	return -EINVAL;
    }

    /* Save the command for later */
    intf->cmnd_size = cbw->Length;
    memcpy(intf->cmnd, cbw->CDB, intf->cmnd_size);
    if (cbw->Flags & USB_BULK_IN_FLAG) {
	intf->data_dir = DATA_DIR_TO_HOST;
    } else {
	intf->data_dir = DATA_DIR_FROM_HOST;
    }
    intf->data_size = le32_to_cpu(cbw->DataTransferLength);
    if (intf->data_size == 0) intf->data_dir = DATA_DIR_NONE;
    intf->lun = cbw->Lun;
    intf->tag = cbw->Tag;
    D(D_BLABLA, "Valid CBW received: lun=%d, tag=%d, dir=%d, len=%d\n",
	    intf->lun, intf->tag, intf->data_dir, intf->data_size);
    return 0;
}

static int wait_for_cbw(struct ppstorage_intf_t *intf)
{
    struct zero_buffhd *bh = intf->next_buffhd_to_drain;
    intf->next_buffhd_to_drain = bh->next;
    int rc;
    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return -EINTR;
    set_bulk_out_req_length(intf->zero_dev, bh, USB_BULK_CB_WRAP_LEN);
    start_transfer(intf->zero_dev, intf->Bout_ep, bh->outreq,
	    &bh->outreq_busy, &bh->state);
    D(D_BLABLA, "Waiting for CBW.\n");
    if (zero_wait_for_buf_full(intf->zero_dev, bh) != 0) return -EINTR;
    rc = received_cbw(intf, bh);
    bh->state = BUF_STATE_EMPTY;
    return rc;
}

static int ppstorage_thread(void *intf_)
{
    int rc;
    struct ppstorage_intf_t *intf = intf_;
    intf->thread_task = current;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    daemonize();
    reparent_to_init();
    strncpy(current->comm, "usb-mass-storage", sizeof(current->comm)-1);
#else
    daemonize("usb-mass-storage-%d", intf->dev_no);
#endif
    /* Arrange for userspace references to be interpreted as kernel
     * pointers.  That way we can pass a kernel pointer to a routine
     * that expects a __user pointer and it will work okay. */
    set_fs(get_ds());

    sigset_t mask;
    siginitsetinv(&mask, sigmask(SIGINT) |
	    sigmask(SIGTERM) | sigmask(SIGKILL) |
	    sigmask(SIGUSR1) | sigmask(SIGUSR2));
    current->blocked = mask;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    spin_lock_irq(&current->sigmask_lock);
    flush_signals(current);
    recalc_sigpending(current);
    spin_unlock_irq(&current->sigmask_lock);
#else
    flush_signals(current);
#endif

    intf->thread_suspend = 1;
    intf->thread_state = PPSTORAGE_STATE_RUNNING;
    complete(&intf->thread_notifier_start);
    D(D_NOTICE, "Thread for ppstorage interface #%d started\n", intf->dev_no);

    while (intf->thread_state != PPSTORAGE_STATE_EXIT) {
	flush_signals(current);

	if (intf->thread_state == PPSTORAGE_STATE_RESET) {
	    int i;
	    unsigned long flags;
	    spin_lock_irqsave(&intf->thread_lock, flags);
	    D(D_VERBOSE, "ppstorage reset (dev #%d)\n", intf->dev_no);
	    for (i=0; i<NUM_BUFFERS; i++) {
		struct zero_buffhd *bh = &intf->buffhds[i];
		if (bh->inreq_busy) usb_ep_dequeue(intf->Bin_ep, bh->inreq);
		if (bh->outreq_busy) usb_ep_dequeue(intf->Bout_ep, bh->outreq);
		bh->inreq_busy = bh->outreq_busy = 0;
		bh->state = BUF_STATE_EMPTY;
	    }
	    if (intf->bulk_in_enabled) {
		usb_ep_fifo_flush(intf->Bin_ep);
	    }
	    if (intf->bulk_out_enabled) {
		usb_ep_fifo_flush(intf->Bout_ep);
	    }
	    intf->next_buffhd_to_fill = &intf->buffhds[0];
	    intf->next_buffhd_to_drain = &intf->buffhds[0];
	    scsi_reset(intf->bot.scsi);
	    if (intf->response_pending) {
		/* Complete status stage */
		intf->zero_dev->ep0req->length = 0;
		ep0_queue(intf->zero_dev);
		intf->response_pending = 0;
	    }
	    intf->thread_state = PPSTORAGE_STATE_RUNNING;
	    spin_unlock_irqrestore(&intf->thread_lock, flags);
	}

	complete(&intf->thread_notifier_suspend);
	if (intf->thread_suspend) {
	    wait_event_interruptible(intf->thread_wqh, 0);
	    continue;
	}

	if (intf->thread_state == PPSTORAGE_STATE_RUNNING) {
	    rc = wait_for_cbw(intf);
	    if (rc) continue;
	}

	if (intf->thread_state == PPSTORAGE_STATE_RUNNING) {
	    rc = do_scsi_command(intf);
	    if (rc) continue;
	}

	if (intf->thread_state == PPSTORAGE_STATE_RUNNING) {
	    rc = send_csw(intf);
	}
    }

    D(D_NOTICE, "Thread for ppstorage interface #%d terminated\n", intf->dev_no);
    complete_and_exit(&intf->thread_notifier_exit, 0);
}

static void start_bot_thread(struct ppstorage_intf_t *intf)
{
    init_completion(&intf->thread_notifier_start);
    int pid = kernel_thread(ppstorage_thread, intf, (CLONE_VM | CLONE_FS | CLONE_FILES));
    D(D_NOTICE, "ppstorage #%d thread pid: %d\n", intf->dev_no, pid);
    wait_for_completion(&intf->thread_notifier_start);
}

static void stop_bot_thread(struct ppstorage_intf_t *intf)
{
    init_completion(&intf->thread_notifier_exit);
    if (intf && intf->thread_state != PPSTORAGE_STATE_EXIT) {
	D(D_NOTICE, "Terminating ppstorage #%d thread\n", intf->dev_no);
	set_state(intf, PPSTORAGE_STATE_EXIT);
	wait_for_completion(&intf->thread_notifier_exit);
    }
    intf->thread_task = NULL;
}

static void set_state(struct ppstorage_intf_t *intf, ppstorage_state_t state)
{
    unsigned long flags;
    spin_lock_irqsave(&intf->thread_lock, flags);
    if (intf->thread_state < state) {
	intf->thread_state = state;
	if (intf->thread_task) {
	    force_sig(SIGUSR1, intf->thread_task);
	}
    }
    spin_unlock_irqrestore(&intf->thread_lock, flags);
}

static int open_file(struct ppstorage_intf_t *intf,
	const char *filename, int ro, usb_device_type_t type, uint32_t blocklen)
{
    int rc = -EINVAL;
    uint32_t blocklen_tmp;
    struct file *filp = NULL;
    struct inode *inode = NULL;
    loff_t size;
    loff_t num_sectors;

    /* R/W if we can, R/O if we must */
    if (!ro) {
	filp = filp_open(filename, O_RDWR | O_LARGEFILE, 0);
	if (-EROFS == PTR_ERR(filp)) ro = 1;
    }
    if (ro) filp = filp_open(filename, O_RDONLY | O_LARGEFILE, 0);

    if (IS_ERR(filp)) {
	D(D_ERROR, "unable to open backing file: %s\n", filename);
	return PTR_ERR(filp);
    }

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    fsnotify_open(filp->f_dentry);
#endif
    if (!(filp->f_mode & FMODE_WRITE)) ro = 1;

    if (filp->f_dentry) inode = filp->f_dentry->d_inode;
    if (inode && S_ISBLK(inode->i_mode)) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
       kdev_t dev = inode->i_rdev;

       if (blk_size[MAJOR(dev)]) {
           size = (loff_t) blk_size[MAJOR(dev)][MINOR(dev)] << BLOCK_SIZE_BITS;
       } else {
           D(D_ERROR, "unable to find file size: %s\n", filename);
           goto out;
       }
       D(D_BLABLA, "blk_size: %d, size: %llu, blocklen: %d\n", blk_size[MAJOR(dev)][MINOR(dev)], size, blocklen);
#else	
	size = (loff_t)get_capacity(inode->i_bdev->bd_disk) << 9;
#endif
    } else if (inode && S_ISREG(inode->i_mode)) {
	size = inode->i_size;
    } else {
	D(D_ERROR, "invalid file type: %s\n", filename);
	goto out;
    }

    /* If we can't read the file, it's no good.
     * If we can't write the file, use it read-only. */
    if (!filp->f_op || !filp->f_op->read) {
	D(D_ERROR, "file not readable: %s\n", filename);
	goto out;
    }
    if (IS_RDONLY(inode) || !filp->f_op->write) ro = 1;

    /* 64-bit division -- blocklen must be a power of 2 */
    blocklen_tmp = blocklen;
    num_sectors = size;
    while (blocklen_tmp > 1) {
	if ((blocklen_tmp & 1)) {
	    D(D_ERROR, "invalid blocklen: %d\n", blocklen);
	    goto out;
	}
	if ((num_sectors & 1)) {
	    D(D_ERROR, "file size not a multiple of block size: %s (size=%llu, blocklen=%d)",
		    filename, size, blocklen);
	    goto out;
	}
	blocklen_tmp >>= 1;
	num_sectors >>= 1;
    }
    if (num_sectors == 0) {
	D(D_ERROR, "file too small: %s (size=%llu)\n", filename, size);
	rc = -ETOOSMALL;
	goto out;
    }

    D(D_ERROR, "get_file...\n");
    get_file(filp);
    D(D_ERROR, "get_file Finish...\n");

    intf->filp = filp;
    intf->file_length = size;

    if (!ro) {
	intf->bg_sync = bg_sync_new(filp);
    }

    //unit_att = SCSI_UNIT_ATT_NONE;
    //curlun->unit_attention_data = SS_NO_SENSE;
    D(D_VERBOSE, "open backing file: %s\n", filename);
    rc = 0;
    scsi_set_file(intf->bot.scsi, num_sectors-1, blocklen, type, ro);
    filp_close(filp, current->files);
    return rc;

out:
    scsi_set_file(intf->bot.scsi, 0, 0, PP_USB_IMAGE_TYPE_NONE, ro);
    filp_close(filp, current->files);
    return rc;
}

void ppstorage_close_file(struct ppstorage_intf_t *intf)
{
    if (intf->filp) {
	D(D_VERBOSE, "close backing file\n");
	fput(intf->filp);
	intf->filp = NULL;
	bg_sync_delete(intf->bg_sync);
	intf->bg_sync = NULL;
    }
}

void ppstorage_set_file(struct FTC_zero_dev *fsg, int dev_no, char* filename,
	int readonly, usb_device_type_t type, uint32_t blocklen)
{
    D(D_BLABLA, "ppstorage_set_file - file: %s, readonly: %d, devno: %d\n",
	    type != PP_USB_IMAGE_TYPE_NONE? filename : "<no image>", readonly, dev_no);
    ppstorage_close_file(fsg->storage[dev_no]);

    if (type == PP_USB_IMAGE_TYPE_NONE) {
	scsi_set_file(fsg->storage[dev_no]->bot.scsi, 0, 0, PP_USB_IMAGE_TYPE_NONE, 1);
    } else {
	open_file(fsg->storage[dev_no], filename, readonly, type, blocklen);
    }
}

int ppstorage_sync_file(struct ppstorage_intf_t *intf)
{
    if (intf->bg_sync) {
	return bg_sync_do_sync(intf->bg_sync, 10 * HZ);
    }
    return 0;
}

int ppstorage_bind(struct usb_gadget *gadget)
{
    struct FTC_zero_dev *fsg = gadget->dev.driver_data;
    int i;
    // struct usb_ep *ep;
    // struct usb_request *req;
    char *pathbuf, *p;

    fsg->bulk_out_maxpacket =
	(gadget->speed == USB_SPEED_HIGH ? 512 : FS_BULK_OUT_MAXPACKET);

    pathbuf = kmalloc(PATH_MAX, GFP_KERNEL);
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	if (backing_file_is_open(fsg->storage[i])) {
	    p = NULL;
	    if (pathbuf) {
		p = d_path(fsg->storage[i]->filp->f_dentry,
			fsg->storage[i]->filp->f_vfsmnt, pathbuf, PATH_MAX); // When driver re-insert, linux will show memory access
		// error in this function, i think this is a bug of file storage driver.
		if (IS_ERR(p)) p = NULL;
	    }
	    D(D_NOTICE, "file: %s\n", (p ? p : "(error)"));
	}
    }
    kfree(pathbuf);

    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	start_bot_thread(fsg->storage[i]);
    }

    return 0;
}

int ppstorage_unbind(struct usb_gadget *gadget)
{
    int i;
    struct FTC_zero_dev *fsg = gadget->dev.driver_data;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	stop_bot_thread(fsg->storage[i]);
    }
    return 0;
}

int ppstorage_suspend_thread(struct ppstorage_intf_t *intf, int suspend)
{
    init_completion(&intf->thread_notifier_suspend);
    D(D_VERBOSE, "%s ppstorage #%d thread\n",
	    suspend? "Suspending" : "Resuming", intf->dev_no);
    intf->thread_suspend = suspend;
    if (suspend) intf->thread_state = PPSTORAGE_STATE_RESET;
    force_sig(SIGUSR1, intf->thread_task);
    wait_for_completion(&intf->thread_notifier_suspend);
    return 0;
}

struct ppstorage_intf_t *ppstorage_intf_new(struct FTC_zero_dev *zero_dev, int index)
{
    struct ppstorage_intf_t *intf;
    intf = kmalloc(sizeof(*intf), GFP_KERNEL);
    memset(intf, 0, sizeof(*intf));
    intf->zero_dev = zero_dev;
    intf->dev_no = index;
    intf->interface_number = -1;
    intf->bot.scsi = scsi_new(index);
    intf->bot.index = index;
    intf->bot.zero_dev = zero_dev;
    intf->protocol = USB_SC_SCSI;
    intf->next_buffhd_to_fill = &intf->buffhds[0];
    intf->next_buffhd_to_drain = &intf->buffhds[0];
    intf->thread_task = NULL;
    intf->thread_state = PPSTORAGE_STATE_EXIT;
    intf->response_pending = 0;
    init_completion(&intf->thread_notifier_start);
    init_completion(&intf->thread_notifier_exit);
    init_completion(&intf->thread_notifier_reset);
    init_completion(&intf->thread_notifier_suspend);
    spin_lock_init(&intf->thread_lock);
    init_waitqueue_head(&intf->thread_wqh);
    intf->thread_wakeup = 0;
    return intf;
}

void ppstorage_intf_free(struct ppstorage_intf_t *intf)
{
    scsi_free(intf->bot.scsi);
    kfree(intf);
}

