/* global linux includes */
#include <linux/types.h>
#include <asm/atomic.h>
#include <linux/socket.h>
#include <linux/net.h>
#include <linux/un.h>
#include <net/sock.h>

/* local includes */
#include "pp_usb.h"
#include "scsi_intern.h"
#include "scsi.h"
#include "scsi_ipmi.h"
#ifndef __arm__
# include "mass-storage-bo.h"
#else
# include "ppstorage.h"
/* user space include for eric control socket communication */
# include "pp_ipc_socket.h"
#endif


#ifndef __arm__
extern usb_driver_t g_ud;
#endif

/* setup proto enabled */
volatile uint8_t scsi_setup_proto_enabled = 0;

/******************************************************************************
* local SCSI structures                                                       *
******************************************************************************/

TOC_DATA_TRACK1 tocdata_track1_template = {
    Head: {
	DataLength_be16 : __constant_cpu_to_be16(0x0a),
	FirstTrack: 0x01,
	LastTrack: 0x01
    },
    Track1: {
	Reserved0: 0x00,
    /* according to the USB Mass Storage Specification For Bootability ADR and CONTROL are
       reserved and should be zero */
	ADR: 0x01,
	CONTROL: 0x04,
	TrackNumber: 0x01,
	Reserved1: 0x00,
	Adress: {
	    LBA_be32: __constant_cpu_to_be32(0x00)
	}
    }
};

TOC_DATA_TRACK1_AND_LO tocdata_track1_and_lo_template = {
    Head: {
	DataLength_be16 : __constant_cpu_to_be16(0x12),
	FirstTrack: 0x01,
	LastTrack: 0x01
    },
    Track1: {
	Reserved0: 0x00,
	ADR: 0x01,
	CONTROL: 0x04,
	TrackNumber: 0x01,
	Reserved1: 0x00,
	Adress: {
	    LBA_be32: __constant_cpu_to_be32(0x00)
	}
    },
    LeadOut: {
	Reserved0: 0x00,
	ADR: 0x01,
	CONTROL: 0x04,
	TrackNumber: 0xaa,
	Reserved1: 0x00,
	Adress: {
	    LBA_be32: __constant_cpu_to_be32(0x00)
	}
    }
};

READ_FORMAT_CAP readformatcap_template = {
    CapacityListHeader: {
	Reserved0: 0,
	Reserved1: 0,
	Reserved2: 0,
    },
    MaxCapacityDescriptor: {
	NumberOfBlocks_be32: __constant_cpu_to_be32(0),
	Reserved0: 0,
	DescriptorType: 3, /* no media present */
	BlockLength_MSB: 0,
	BlockLength_LSB_be16: __constant_cpu_to_be16(0)
    }
};

EVENT_STATUS_NOTIFICATION_MMC_MEDIA_CHANGE eventStatusNotificationMediaChange_template = {
    Header: {
        EventDataLength_be16: __constant_cpu_to_be16(0),
        NotificationClass: 0,       /* 0 means requested event not supported */
        NEA: 1,                     /* no event available */
        SupportedEventClasses: 0    /* none of the events is available */
    },
    MediaEvent: 0,      /* filled out later */
    DoorOpen: 0,        /* filled out later */
    MediaPresent: 0,    /* filled out later */
    StartSlot: 0,
    EndSlot: 0
};

FEATURE_DESCRIPTOR featureDescriptor_template = {
    Header: {
        DataLength_be32: __constant_cpu_to_be32(sizeof(featureDescriptor_template) - sizeof(featureDescriptor_template.Header.DataLength_be32)),
        CurrentProfile_be16: 0  /* filled out later */
    },
    Features: {
        ProfileList: {
            Header: {
                FeatureCode_be16: __constant_cpu_to_be16(0),    /* Feature 0: Profile List */
                Current: 1,
                Persistent: 1,
                Version: 0,
                Reserved: 0,
                AdditionalLength: sizeof(featureDescriptor_template.Features.ProfileList) - sizeof(featureDescriptor_template.Features.ProfileList.Header)
            },
            Profiles: {
                {
                    ProfileNumber_be16: __constant_cpu_to_be16(PROFILE_NUMBER_DVDROM),
                    CurrentP: 0
                },
                {
                    ProfileNumber_be16: __constant_cpu_to_be16(PROFILE_NUMBER_CDROM),
                    CurrentP: 0
                }
            }
        }
    }
};

DISC_INFORMATION_MMC discInformation_template = {
    DiscInformationLength_be16: __constant_cpu_to_be16(sizeof(discInformation_template)),
    DiscStatus: 0x02,   /* complete disc */
    StateOfLastSession: 0x03,   /* complete session */
    Erasable: 0,
    NumberOfFirstTrack: 1,
    NumberOfSessionsLSB: 1,
    FirstTrackInLastSessionLSB: 1,
    LastTrackInLastSessionLSB: 1,
    URU: 0,
    DBC_V: 0,
    DID_V: 0,
    DiscType: 0,    /* CD-ROM */
    NumberOfSessionsMSB: 0,
    FirstTrackInLastSessionMSB: 0,
    LastTrackInLastSessionMSB: 0,
    DiscIdentification_be32: __constant_cpu_to_be32(0),
    LastSessionLeadInStartTime_be32: __constant_cpu_to_be32(0xffffffff),
    LastPossibleStartTimeForLeadOut_be32: __constant_cpu_to_be32(0xffffffff),
    NumberOfOPCTableEntries: 0
};

/* thre:
   commented the error recovery page out
   reason: Windows ignored the WriteProtected flag of the
   device when this page is sent. Without it, it works.
   Don't know why. Maybe it misinterprets some bits in
   the page, but I can't imagine which. */
#if 0
        error_recovery_page: {
        PageSavable: 1,
	Reserved: 0,
	PageCode: 0x01, /* Error Recovery Page Code SBC 7.1.3.7 */
	PageLength: 0x0a, /* ... */
	AWRE: 0,
	ARRE: 0,
	TB: 0,
	RC: 0,
	EER: 0,
	PER: 0,
	DTe: 0,
	DCR: 0,
	ReadRetryCount: 2,
	CorrectionSpan: 0,
	HeadOffsetCount: 0,
	DataStrobeOffsetCount: 0,
	Reserved0: 0,
	WriteRetryCount: 2,
	Reserved1: 0,
	RecoveryTimeLimit: 0
    }
};
#endif

MODE_PARAMETER_HEAD10 mode_parameter_head10_template = {
/* FIXME: adjust len */
    DataLen_be16: 0, /* must be adjusted */
    MediumType: 0x94, /* as specified in the UFI spec, 4.5.3 */
    { 0, 0, 0, 0},
    { 0, 0},
    0
};

MODE_PARAMETER_HEAD6 mode_parameter_head6_template = {
/* FIXME: adjust len */
    DataLen: 0, /* must be adjusted */
    MediumType: 0x94, /* as specified in the UFI spec, 4.5.3 */
    { 0, 0, 0, 0},
    0
};

FLEXIBLE_DISK_PAGE flexible_disk_page_template = {
	PageSavable: 0,
	Reserved: 0,
	PageCode: 0x05,
	
	PageLength: 0x1e,
	
	TransferRate_be16: __constant_cpu_to_be16(0x1f4),
	NumberOfHeads: 2,
	SectorsPerTrack: 18,
	DataBytesPerSector_be16: __constant_cpu_to_be16(512),
	NumberOfCylinders_be16: __constant_cpu_to_be16(80),
	StartWritePreComp_be16: __constant_cpu_to_be16(0),
	StartWriteCurr_be16: __constant_cpu_to_be16(0),
	StepRate_be16: __constant_cpu_to_be16(0),
	PulseWidth: 0,
	HeadSettleDelay_be16: __constant_cpu_to_be16(0),
	MotorOnDelay: 0x05,
	MotorOffDelay: 0x28,

	TRDY:0,
	SSN:0,
	MO:0,
	Reserved2:0,
	
	Reserved3:0,
	SPC: 0,
	
	WriteCompensation: 0,
	HeadLoadDelay: 0,
	HeadUnloadDelay: 0,
	
	Pin34:0,
	Pin2:0,

	Pin4:0,
	Pin1:0,
	
	MediumRotationRate_be16: __constant_cpu_to_be16(0x12c),
	Reserved4: 0,
	Reserved5: 0
};


union {
    MODE_PAGE10 disk_page_10;
    MODE_PAGE6 disk_page_6;
} __attribute__((__packed__)) mode_page_data;

VPD_SERIAL_PAGE vpd_serial_template = {
    DeviceTypeQualifier: 0,
    DeviceType: 0,
    PageCode: VPDPAGE_SERIAL_NUMBER,
    Reserved0: 0,
    PageLength: SERIALNUMBER_LENGTH,
    SerialNumber: { '0', 0,
		    '0', 0,
		    '0', 0,
		    '0', 0,
		    
		    '0', 0,
		    '0', 0,
		    '0', 0,
		    '0', 0,
		    
		    '0', 0,
		    '0', 0,
		    '0', 0,
		    '1', 0,
    }
};

VPD_SUPPORTED_PAGE vpd_supported_template = {
    DeviceTypeQualifier: 0,
    DeviceType: 0,
    PageCode: VPDPAGE_SUPPORTED,
    Reserved0: 0,
    PageLength: NUMBER_VPD_PAGES,
    SupportedPages: {
	VPDPAGE_SUPPORTED,
	VPDPAGE_SERIAL_NUMBER,
	VPDPAGE_DEVICE_IDENTITY
    }
};

VPD_DEVICE_ID_PAGE vpd_device_id_template = {
    DeviceTypeQualifier: 0,
    DeviceType: 0,
    PageCode: VPDPAGE_DEVICE_IDENTITY,
    Reserved0: 0,
    PageLength: sizeof(ASCII_ID_DESCRIPTOR),
    AsciiIdDescriptor: {
	Reserved0: 0,
	CodeSet: 2, /* SPC 8.4.4, ASCII code set */
	Reserved1: 1,
	Association: 0,
	IDType: 0,
	Reserved2: 0,
	IDLength: ASCII_ID_STRING, // max. 25 characters!
#if defined(OEM_BELKIN)
	AsciiID: { 'B', 'e', 'l', 'k', 'i', 'n', ',', ' ',
		   'M', 'a', 's', 's', ' ', 'S', 't', 'o', 'r', 'a', 'g', 'e'
		    , ' ', ' ', ' ', ' ', ' ' },
#elif defined(OEM_INTEL)
	AsciiID: "Intel(R) RMM2 VDrive",
#elif defined(OEM_LENOVO)
        AsciiID: "Lenovo Mass Storage",
#else
	AsciiID: { 'P', 'e', 'p', 'p', 'e', 'r', 'c', 'o', 'n', ' ', 'A', 'G', ',',
		   'M', 'a', 's', 's', ' ', 'S', 't', 'o', 'r', 'a', 'g', 'e' },
#endif
    }
};

REPORT_LUNS_DATA lunsData = {
    LunListLength_be32 : __constant_cpu_to_be32(8), // 1 lun
    Reserved1 : 0,
    Lun0_be64 : __constant_cpu_to_be64(0x0)
};

REQUEST_SENSE_DATA RBC_SenseData_template = {
	0,     //   INT8 Valid : 1;
	SCSI_RESPONSECODE_CURRENT_ERROR, // Response Code
	
	0,//   INT8 SegmentNum;
	
	0,//   INT8 FileMark : 1;
	0,//   INT8 EndofMedium : 1;
	0,//   INT8 WrongLenIndicator : 1;
	0,  // INT8 Reserved0 : 1;
	0x0,// INT8 SenseKey : 4; 5= illegal request

	0,//   INT8 Info_0;
	0,//   INT8 Info_1;
	0,//   INT8 Info_2;
	0,//   INT8 Info_3;
	0x0,// INT8 AdditionalSenseLen;
	0,//   INT8 CommandSpecInfo_0;
	0,//   INT8 CommandSpecInfo_1;
	0,//   INT8 CommandSpecInfo_2;
	0,//   INT8 CommandSpecInfo_3;
	0x0,// INT8 ASC;
	0,//   INT8 ASCQ;
	0,//   INT8 FieldReplacableUnitCode;

	0,//   INT8 SenseKeySpecValid : 1;
	0,//   INT8 SenseKeySpec_0 : 7;
	0,//   INT8 SenseKeySpec_1;
	0 //   INT8 SenseKeySpec_2;
};

STD_INQUIRYDATA inquiryData_template = {
    PeripheralQualifier: 0x00,
    DeviceType: READ_ONLY_DIRECT_ACCESS_DEVICE,
    
    RemovableMedia: 0x01,
    Reserved1: 0x00,
	
    Version: 0x03, /* X3.301:1997 */
    
    AERC: 0x00,
    Obsolete0: 0x00,
    NormACA: 0x00,
    HiSup: 0x0,
    ResponseDataFormat: 2,
    
    AdditionalLength: 0x1f, /* up to byte 35 valid */
    Reserved4: { 0x00, 0x00 },
    
    RelativeAddressing: 0x00,
    Wide32Bit: 0x00,
    Wide16Bit: 0x00,
    Synchronous: 0x00,
    LinkedCommands: 0x00,
    Reserved5: 0x00,
    CommandQueue: 0x00,
    SoftReset: 0x00,
    
     // max. 8 characters!
#ifdef PRODUCT_MSIDC
    VendorId: { 'M','S','I',' ','K','V','M',' ' },
#elif defined(OEM_BELKIN)
    VendorId: { 'B','e','l','k','i','n',' ',' ' },
#elif defined(OEM_INTEL)
    VendorId: { 'I','n','t','e','l','(','R',')' },
#elif defined(OEM_LENOVO)
    VendorId: { 'L','e','n','o','v','o',' ',' ' },
#else
    VendorId: { 'P','e','p','p','e','r','C',' ' },
#endif
    
    // max. 16 characters!
    // use '#' as placeholder for one-digit drive instance number
#if defined(OEM_INTEL)
    ProductId: "RMM2 VDrive #",
#else
    ProductId: { 'V','i','r','t','u','a','l',' ','D','i','s','c', ' ','#',' ',' '       },
#endif
    
    ProductRevisionLevel: { '0','.','0','1' },
    
    VendorSpecific: {
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0
    },
    
    Reserved6: 0x00,
    Clocking: 0x00,
    QuickArbitSupport: 0x00,
    InfoUnitSupport: 0x00,
    
    Reserved7: 0x00,
    
    VersionDescriptor_be16: { __constant_cpu_to_be16(0x013c), /* SPC ANSI X3.301:1997 */
			   __constant_cpu_to_be16(0x019c), /* SBC ANSI NCITS.306:1998 */
			   0, 0, 0, 0, 0, 0 },
    
    Reserved8:  {
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0
    }
};


struct scsi_s *scsi_new(int index)
{
    struct scsi_s *scsi = kmalloc(sizeof(struct scsi_s), GFP_KERNEL);
    memset(scsi, 0, sizeof(struct scsi_s));

    scsi->index = index;
    scsi->unit_ready = 0;
    scsi->unit_att = SCSI_UNIT_ATT_RESET;
    scsi->event_notf = SCSI_EVENT_NONE;
    scsi->capacity_data.lba_be32 = 0;
    scsi->capacity_data.blocklen_be32 = 0;
    scsi->write_protect = 1;
    scsi->device_type = PP_USB_IMAGE_TYPE_NONE;

    memcpy(&scsi->tocdata_track1, &tocdata_track1_template, sizeof(tocdata_track1_template));
    memcpy(&scsi->tocdata_track1_and_lo, &tocdata_track1_and_lo_template, sizeof(tocdata_track1_and_lo_template));
    memcpy(&scsi->readformatcap, &readformatcap_template, sizeof(readformatcap_template));
    memcpy(&scsi->flexible_disk_page, &flexible_disk_page_template, sizeof(flexible_disk_page_template));
    memcpy(&scsi->vpd_serial, &vpd_serial_template, sizeof(vpd_serial_template));
    memcpy(&scsi->vpd_supported, &vpd_supported_template, sizeof(vpd_supported_template));
    memcpy(&scsi->vpd_device_id, &vpd_device_id_template, sizeof(vpd_device_id_template));
    memcpy(&scsi->RBC_SenseData, &RBC_SenseData_template, sizeof(RBC_SenseData_template));
    memcpy(&scsi->inquiryData, &inquiryData_template, sizeof(inquiryData_template));
    memcpy(&scsi->eventStatusNotificationMediaChange, &eventStatusNotificationMediaChange_template, sizeof(eventStatusNotificationMediaChange_template));
    memcpy(&scsi->featureDescriptor, &featureDescriptor_template, sizeof(featureDescriptor_template));
    memcpy(&scsi->discInformation, &discInformation_template, sizeof(discInformation_template));

    {
        /* set the device number in the inquiry data
           adjust this if more than 9 mass storage devices are used */
        int i;
        for (i = 0; i < sizeof(scsi->inquiryData.ProductId); i++)
            if (scsi->inquiryData.ProductId[i] == '#') {
                scsi->inquiryData.ProductId[i] = '1' + index % 10;
                break;
            }
    }

#ifdef __arm__
    scsi->setup_proto_socket = NULL;
#endif

    return scsi;
}

#ifdef __arm__
static void release_socket(struct scsi_s *scsi);
#endif

void scsi_free(struct scsi_s *scsi)
{
#ifdef __arm__
    release_socket(scsi);
#endif
    kfree(scsi);
}

void scsi_reset(struct scsi_s *scsi)
{
    scsi->unit_att = SCSI_UNIT_ATT_RESET;
}

void scsi_set_file(struct scsi_s *scsi, uint32_t lba, uint32_t blocklen, usb_device_type_t type, uint8_t readonly)
{
    SD("%s: lba %x, blocklen %x, enable, readonly: %d %d\n", __FUNCTION__, lba, blocklen, type, readonly);
    SD("inquiry data@ %p\n", &scsi->inquiryData);
    //printk("set device_type to: %d\n", type);
    scsi->device_type = type;
    scsi->write_protect = readonly;
    switch (scsi->device_type) {
      case PP_USB_IMAGE_TYPE_ISO:
	  scsi->inquiryData.DeviceType = READ_ONLY_DIRECT_ACCESS_DEVICE;
	  scsi->inquiryData.RemovableMedia = 1;
	  break;
      case PP_USB_IMAGE_TYPE_REMOVABLE:
	  scsi->inquiryData.DeviceType = DIRECT_ACCESS_DEVICE;
	  scsi->inquiryData.RemovableMedia = 1;
	  break;
      case PP_USB_IMAGE_TYPE_FLOPPY:
	  scsi->inquiryData.DeviceType = DIRECT_ACCESS_DEVICE;
	  scsi->inquiryData.RemovableMedia = 1;
	  break;
      case PP_USB_IMAGE_TYPE_FIXED:
	  scsi->inquiryData.DeviceType = DIRECT_ACCESS_DEVICE;
	  scsi->inquiryData.RemovableMedia = 0;
	  break;
      default:
//	  scsi->inquiryData.DeviceType = READ_ONLY_DIRECT_ACCESS_DEVICE;
	  scsi->inquiryData.RemovableMedia = 1;
	  break;
    }
    scsi->capacity_data.lba_be32 = cpu_to_be32(lba);
    scsi->capacity_data.blocklen_be32 = cpu_to_be32(blocklen);
    scsi->unit_ready = (scsi->device_type != PP_USB_IMAGE_TYPE_NONE) ? 1 : 0;
    scsi->unit_att = SCSI_UNIT_ATT_CHANGED;
    scsi->event_notf = scsi->device_type == PP_USB_IMAGE_TYPE_NONE ? SCSI_EVENT_REMOVED : SCSI_EVENT_INSERTED;
printk("scsi_set_file - event_notf = %d\n", scsi->event_notf);
}

#ifdef __arm__
static int SCSI_send_reply(bot_t *bot, const char *buf, int len)
{
    /* Send the SCSI reply, if any */
    struct FTC_zero_dev *fsg = bot->zero_dev;
    struct ppstorage_intf_t *intf = fsg->storage[bot->index];
    int amount;

    if (!buf) return 0;
    struct zero_buffhd *bh = intf->next_buffhd_to_fill;
    if (zero_wait_for_buf_free(fsg, bh) != 0) return -EINTR;
    intf->next_buffhd_to_fill = bh->next;

    if (len > BULK_EP_BUFLEN) len = BULK_EP_BUFLEN;
    memcpy(bh->buf, buf, len);

    if (intf->usb_amount_left <= BULK_EP_BUFLEN) {
	/* last and only packet, no padding required */
	amount = intf->usb_amount_left;
    } else {
	/* Host requested more than on buffer size, shouldn't really happen */
	SD("%s: host requested very long SCSI reply: %d bytes", __FUNCTION__, intf->usb_amount_left);
	amount = BULK_EP_BUFLEN;
    }
    bh->inreq->zero = 0;
    bh->inreq->length = amount;
    bh->state = BUF_STATE_FULL;
    start_transfer(fsg, intf->Bin_ep, bh->inreq,
	    &bh->inreq_busy, &bh->state);
    intf->next_buffhd_to_fill = bh->next;
    intf->residue -= len;
    intf->usb_amount_left -= amount;

    return len;
}
#endif

uint8_t
scsi_handler(bot_t *bot,
	     uint8_t *cbwcb, uint8_t cbwcb_len, uint32_t data_len_req, transfer_phase_t trans_dir_in, uint8_t **buf,
	     transfer_phase_t *data_phase, uint32_t *data_len)
{
    CDB_RBC *cdb;
    uint8_t retStatus = 0;
    uint32_t data_len_ret;
    transfer_phase_t data_phase_ret;

    struct scsi_s * scsi = bot->scsi;

    // we are a scsi block device, so convert the abstract
    // command block to our cdb
    cdb = (CDB_RBC*) cbwcb;
    
    SD("%s: operation code: %x\n", __FUNCTION__, cdb->Cdb_Generic.OperationCode);

    if (scsi->unit_att != SCSI_UNIT_ATT_NONE) {
	SD("attention pending: %d\n", scsi->unit_att);
	// we have a unit attention pending
	if (cdb->Cdb_Generic.OperationCode == SPC_CMD_REQUESTSENSE) {
	    // SCSI-3 SAM, 5.6.5, report unit attention condition
	    switch(scsi->unit_att) {
	      case SCSI_UNIT_ATT_RESET:
		  RBC_BuildSenseData(bot, scsi, SCSI_SENSE_UNIT_ATTENTION, POWER_ON__RESET__OR_BUS_DEVICE_RESET_OCCURRED);
		  break;
	      case SCSI_UNIT_ATT_CHANGED:
		  RBC_BuildSenseData(bot, scsi, SCSI_SENSE_UNIT_ATTENTION, NOT_READY_TO_READY_CHANGE__MEDIUM_MAY_HAVE_CHANGED);
		  break;
	      case SCSI_UNIT_ATT_INQUIRY:
		  RBC_BuildSenseData(bot, scsi, SCSI_SENSE_UNIT_ATTENTION, INQUIRY_DATA_HAS_CHANGED);
		  break;

	      default:
		  // should never happen
		  printk("wrong unit_attention: %d\n", scsi->unit_att);
		  RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
	    }
	    // clear unit attention
	    SD("clear unit attention, because REQUEST_SENSE was called\n");
	    scsi->unit_att = SCSI_UNIT_ATT_NONE;
	} else if (cdb->Cdb_Generic.OperationCode == SPC_CMD_INQUIRY) {
	    // do nothing here, the command is executed some lines below
	} else {
	    SD("%s: don't execute command\n", __FUNCTION__);
	    // any other command -> don't execute command - just return check_condition

	    //RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, INVALID_COMMAND_OPERATION_CODE);
	    
	    *data_len = 0;
	    *data_phase = PHASE_DEVICE_NONE;
	    return 0;
	}
    }

    switch(cdb->Cdb_Generic.OperationCode) {
	/* required commands */
      case SPC_CMD_INQUIRY: /* SBC: mandatory, SPC, SPC spec, 7.3 */
	  retStatus = SPC_Inquiry(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_READCAPACITY: /* SBC: mandatory, SBC spec, 6.1.6 */
	  retStatus = RBC_ReadCapacity(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_TESTUNITREADY: /* SPC/SBC: mandatory, SPC spec, 7.25 */
	  retStatus = SPC_TestUnit(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_REQUESTSENSE: /* SBC: mandatory, SPC spec, 7.20 */
	  retStatus = SPC_RequestSense(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_READ6: /* SBC: mandatory, SBC spec, 6.1.4 */
	  retStatus = RBC_Read6(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_READ10: /* SBC: mandatory, SBC spec, 6.1.5 */
	  retStatus = RBC_Read10(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_WRITE6: /* SBC spec, 6.1.17 */
	  retStatus = RBC_Write6(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_WRITE10: /* SBC spec, 6.1.18 */
	  retStatus = RBC_Write10(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case UFI_CMD_WRITE12: /* UFI spec, 4.19 */
	  retStatus = UFI_Write12(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case UFI_CMD_WRITEVERIFY: /* UFI spec, 4.20 */
	  retStatus = UFI_WriteVerify(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case UFI_CMD_READ12: /* UFI: 4.8 */
	  retStatus = UFI_Read12(bot, scsi, cdb, data_len_req, trans_dir_in, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_STARTSTOPUNIT: /* SBC spec, 6.1.14 */
	  retStatus = RBC_OnOffUnit(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_PRVENTALLOWMEDIUMREMOVAL: /* SPC spec, 7.12 */
	  retStatus = SPC_LockMedia(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_MODESENSE6: /* SPC spec, 7.8 */
	  retStatus = SPC_ModeSense6(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_MODESENSE10: /* SPC spec, 7.9 */
	  retStatus = SPC_ModeSense10(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_RELEASE: /* SBC: mandatory, SPC spec, 7.16 */
	  retStatus = SPC_Release(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_RESERVE: /* SBC: mandatory, SPC spec, 7.12 */
	  retStatus = SPC_Reserve(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_SEND_DIAGNOSTIC: /* SBC: mandatory, SPC spec, 7.23 */
	  retStatus = SPC_SendDiagnostic(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case MMC_CMD_READ_FORMAT_CAPACITIES: /* MMC-2: 6.1.20 */
	  retStatus = MMC_ReadFormatCapacities(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case MMC_CMD_READ_TOC: /* MMC: 5.1.13 */
	  retStatus = MMC_ReadTOC(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case MMC_GET_CONFIGURATION: /* MMC-2: 6.1.4 */
          retStatus = MMC_GetConfiguration(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
          break;
      case MMC_GET_EVENT_STATUS_NOTIFICATION: /* MMC-2: 6.1.5 */
          retStatus = MMC_EventStatusNotification(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
          break;
      case MMC_CMD_DISC_INFORMATION: /* MMC-2 6.1.18 */
          retStatus = MMC_DiscInformation(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
          break;
      case SCSI_VENDOR_CMD_SETUP_PROTO_REQ: /* vendor specific */
	  retStatus = SetupProtoReq(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SCSI_VENDOR_CMD_SETUP_PROTO_RSP: /* vendor specific */
	  retStatus = SetupProtoRsp(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
#ifdef PP_FEAT_IPMI_SERVER_SCSI_CHAN
      case SCSI_VENDOR_CMD_IPMI: /* vendor specific */
	  retStatus = IpmiOverScsi(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
#endif
      case SCSI_VENDOR_CMD_GET_PP_ID:
          retStatus = GetPpId(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
          break;
      case UFI_CMD_REZEROUNIT: /* UFI 4.12 */
	  retStatus = UFI_ReZeroUnit(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case UFI_CMD_SEEK10: /* UFI 4.13 */
	  retStatus = UFI_Seek(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
      case SPC_CMD_REPORTLUNS: /* SPC: mandatory? because of SBC?, SPC spec, 7.19 */
	  retStatus = SPC_ReportLuns(bot, scsi, cdb, data_len_req, &data_phase_ret, &data_len_ret);
	  break;
#if 0
      case SBC_CMD_FORMAT: /* SBC: mandatory, SBC spec, 6.1.1 */
          /* according to USB Mass Storage Specification for Bootability Format Unit should be supported */
	  retStatus = RBC_Format(bot, scsi, cdb, &data_phase_ret, &data_len_ret);
	  break;
      case SBC_CMD_SYNCCACHE:
	  retStatus = RBC_SyncCache(cdb);
	  break;
      case SBC_CMD_VERIFY10:
          /* according to USB Mass Storage Specification for Bootability VERIFY should be supported */
	  retStatus = RBC_Verify(cdb);
	  break;
      case SPC_CMD_MODESELECT6:
	  retStatus = SPC_ModeSelect6(bot, scsi, cdb);
	  break;
      case SPC_CMD_MODESELECT10:
	  retStatus = SPC_ModeSelect10(bot, scsi, cdb);
	  break;
      case SPC_CMD_RESERVE6:
	  retStatus = SPC_Reserve6(bot, scsi, cdb);
	  break;
      case SPC_CMD_RELEASE6:
	  retStatus = SPC_Release6(bot, scsi, cdb);
	  break;
      case SPC_CMD_PERSISTANTRESERVIN:
	  retStatus = SPC_PersisReserveIn(cdb);
	  break;
      case SPC_CMD_PERSISTANTRESERVOUT:
	  retStatus = SPC_PersisReserveOut(cdb);
	  break;
      case SPC_CMD_WRITEBUFFER:
	  retStatus = SPC_WriteBuff(cdb);
	  break;
#endif
      default:
	  RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
	  data_len_ret = 0;
	  retStatus = 0;
	  data_phase_ret = PHASE_DEVICE_NONE;
	  SD("cmd not supported: %02x, send len: %x\n", cdb->Cdb_Generic.OperationCode, data_len_ret);
	  break;
    }

    *data_len = data_len_ret;
    *data_phase = (*data_len == 0) ? PHASE_DEVICE_NONE : data_phase_ret;
    if (buf) *buf = bot->BOTXfer_pData;
#ifdef __arm__
    /* Send SCSI reply (will be done by usb_ms_bo.c on PPC) */
    if (bot->BOTXfer_pData && *data_phase == PHASE_DEVICE_IN) {
	SCSI_send_reply(bot, bot->BOTXfer_pData, *data_len);
    }
    *buf = bot->BOTXfer_pData = NULL;
#endif

    return retStatus;
}

char SCSI_cmd_nop_no_error(bot_t *bot, struct scsi_s *scsi, transfer_phase_t *data_phase, uint32_t *data_len)
{
    /* self test command always successful :-) */
    *data_len = 0;
    *data_phase = PHASE_DEVICE_NONE;
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    return 1;
}



static char SCSI_Read(bot_t *bot, struct scsi_s *scsi, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len, uint32_t xferlen, uint32_t lba);

char RBC_Read6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbRead6 cdb->RbcCdb_Read6
    uint32_t lba;
    
    lba = be16_to_cpu(cdbRead6.LBA_W_be16) | (((uint32_t)cdbRead6.LBA_MSB) << 16);

    return SCSI_Read(bot, scsi, data_len_req, trans_dir_in, data_phase, data_len, cdbRead6.XferLength, lba);
#undef cdbRead6
}

char RBC_Read10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbRead cdb->RbcCdb_Read10
    return SCSI_Read(bot, scsi, data_len_req, trans_dir_in,
		     data_phase, data_len,
		     be16_to_cpu(cdbRead.XferLength_be16),
		     be32_to_cpu(cdbRead.LBA_W_be32));
#undef cdbRead
}

char UFI_Read12(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbRead12 cdb->UfiCdb_Read12
    return SCSI_Read(bot, scsi, data_len_req, trans_dir_in,
		     data_phase, data_len,
		     be32_to_cpu(cdbRead12.XferLength_be32),
		     be32_to_cpu(cdbRead12.LBA_W_be32));
#undef cdbRead12
}


static char SCSI_Write(bot_t *bot, struct scsi_s *scsi, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len, uint32_t xferlen, uint32_t lba);

char RBC_Write6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbWrite    cdb->RbcCdb_Write6
    uint32_t lba;

    lba = be16_to_cpu(cdbWrite.LBA_W_be16) | (((uint32_t)cdbWrite.LBA_MSB) << 16);
    
    return SCSI_Write(bot, scsi, data_len_req, trans_dir_in, data_phase, data_len, cdbWrite.XferLength, lba);
#undef cdbWrite
}

char RBC_Write10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbWrite    cdb->RbcCdb_Write10
    return SCSI_Write(bot, scsi, data_len_req, trans_dir_in, data_phase, data_len, be16_to_cpu(cdbWrite.XferLength_be16), be32_to_cpu(cdbWrite.LBA_W_be32));
#undef cdbWrite
}

char UFI_Write12(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbWrite    cdb->UfiCdb_Write12
    return SCSI_Write(bot, scsi, data_len_req, trans_dir_in, data_phase, data_len, be32_to_cpu(cdbWrite.XferLength_be32), be32_to_cpu(cdbWrite.LBA_W_be32));
#undef cdbWrite
}


char UFI_WriteVerify(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbWrite    cdb->UfiCdb_WriteVerify
    return SCSI_Write(bot, scsi, data_len_req, trans_dir_in, data_phase, data_len, be16_to_cpu(cdbWrite.XferLength_be16), be32_to_cpu(cdbWrite.LBA_W_be32));
#undef cdbWrite
}

#ifdef __arm__

static void timeout_handler(unsigned long arg)
{
    force_sig(SIGUSR2, (struct task_struct *)arg);
}

static char SCSI_Write(bot_t *bot, struct scsi_s *scsi, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len, uint32_t xferlen, uint32_t lba)
{
    int ret = 0;
    struct FTC_zero_dev *fsg = bot->zero_dev;
    struct ppstorage_intf_t *intf = fsg->storage[bot->index];
    struct zero_buffhd *bh;
    struct timer_list tl;
    u32 amount_left;
    loff_t file_offset;

    SD("+ (FileStorage)do_write()\n");

    if (scsi->write_protect) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_DATA_PROTECT, WRITE_PROTECTED);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	return 0;
    } else if (!scsi->unit_ready) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NOT_READY, MEDIUM_NOT_PRESENT);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	return 0;
    }

    /* set return values */
    *data_phase = PHASE_DEVICE_OUT;
    *data_len = xferlen * be32_to_cpu(scsi->capacity_data.blocklen_be32);
    bot->BOTXfer_pData = NULL;

    if (trans_dir_in != *data_phase) return 0;
    if (data_len_req == 0 || xferlen == 0) return 0;

    /* Register a timeout handler */
    init_timer(&tl);
    tl.expires = jiffies + 9 * HZ;
    tl.function = timeout_handler;
    tl.data = (unsigned long)current;
    add_timer(&tl);

    file_offset = (loff_t)lba * be32_to_cpu(scsi->capacity_data.blocklen_be32);
    amount_left = min(data_len_req, *data_len);

    while (amount_left) {
	unsigned int amount;
	loff_t file_offset_tmp;
	ssize_t nwrite;

	amount = min((int)amount_left, (int)BULK_EP_BUFLEN);

	/* Never write past the end of file */
	if (file_offset + amount > intf->file_length) {
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, LOGICAL_BLOCK_ADDRESS_OUT_OF_RANGE);
	    // FIXME: Set sense data info field to first invalid LBA
	    // curlun->sense_data_info = usb_offset >> 9;
	    goto bail;
	}

	bh = intf->next_buffhd_to_drain;
	if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) goto bail;
	bh->out_intended_length = bh->outreq->length = amount;
	start_transfer(intf->zero_dev, intf->Bout_ep, bh->outreq,
		&bh->outreq_busy, &bh->state);
	if (zero_wait_for_buf_full(intf->zero_dev, bh) != 0) goto bail;
	if (bh->outreq->actual != bh->outreq->length ||
		bh->outreq->status != 0) goto bail;
	intf->usb_amount_left -= amount;

	file_offset_tmp = file_offset;
	nwrite = intf->filp->f_op->write(intf->filp,
		(char *)bh->buf, amount, &file_offset_tmp);
	bh->state = BUF_STATE_EMPTY;

	if (nwrite == -EINTR) goto bail;

	/* If an error occurred, report it and its position */
	if (nwrite < (ssize_t)amount) {
	    SD("*** write error - set WRITE_ERROR - nwrite: %d, amount: %d\n", nwrite, amount);
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_MEDIUM_ERROR, WRITE_ERROR);
	    // FIXME: Set sense data info field to first invalid LBA
	    // curlun->sense_data_info = usb_offset >> 9;
	    goto bail;
	}
	intf->next_buffhd_to_drain = bh->next;
	file_offset += nwrite;
	amount_left -= nwrite;
	intf->residue -= nwrite;
    }

    if (ppstorage_sync_file(intf) == 0) {
	ret = 4;
    }

bail:
    del_timer(&tl);

    if (signal_pending(current)) {
	// SIGUSR2 means timeout, anything else is an exit condition
	for (;;) {
	    int sig;
	    siginfo_t info;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	    spin_lock_irq(&current->sigmask_lock);
	    sig = dequeue_signal(&current->blocked, &info);
	    spin_unlock_irq(&current->sigmask_lock);
#else
	    sig = dequeue_signal(current, &current->blocked, &info);
#endif
	    if (!sig) break;
	    if (sig != SIGUSR2) {
		return -EINTR;
	    }
	}
	SD("SCSI_Write(): Timeout\n");
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ABORTED_COMMAND, TIMEOUT_ON_LOGICAL_UNIT);
    }

    return ret;
}

static char SCSI_Read(bot_t *bot, struct scsi_s *scsi, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len, uint32_t xferlen, uint32_t lba)
{
    int ret = 0;
    struct FTC_zero_dev *fsg = bot->zero_dev;
    struct ppstorage_intf_t *intf = fsg->storage[bot->index];
    struct timer_list tl;
    u32 amount_left;
    loff_t file_offset;

    /* set return values */
    *data_phase = PHASE_DEVICE_IN;
    *data_len = xferlen * be32_to_cpu(scsi->capacity_data.blocklen_be32);
    bot->BOTXfer_pData = NULL;

    SD("+ (FileStorage)do_read()\n");

    if (!scsi->unit_ready) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NOT_READY, MEDIUM_NOT_PRESENT);
	return 0;
    }

    if (trans_dir_in != *data_phase) return 0;
    if (data_len_req == 0 || xferlen == 0) return 0;

    /* Register a timeout handler */
    init_timer(&tl);
    tl.expires = jiffies + 9 * HZ;
    tl.function = timeout_handler;
    tl.data = (unsigned long)current;
    add_timer(&tl);

    /* Carry out the file reads */
    file_offset = (loff_t)lba * be32_to_cpu(scsi->capacity_data.blocklen_be32);
    amount_left = min(data_len_req, *data_len);

    while (amount_left) {
	unsigned int amount;
	unsigned int partial_page;
	struct zero_buffhd *bh;
	loff_t file_offset_tmp;
	ssize_t nread;

	/* Figure out how much we need to read:
	 * Try to read the remaining amount.
	 * But don't read more than the buffer size.
	 * If we're not at a page boundary, don't read past the next page. */
	SD("amount_left: %d, BULK_EP_BUFLEN: %d, filelen: %llu, file-off: %llu\n", amount_left, BULK_EP_BUFLEN, intf->file_length, file_offset);
	amount = min((int)amount_left, (int)BULK_EP_BUFLEN);
	SD("amount: %d\n", amount);
	partial_page = file_offset & (PAGE_CACHE_SIZE - 1);
	if (partial_page > 0) {
	    amount = min(amount, (unsigned int)PAGE_CACHE_SIZE - partial_page);
	}

	/* Never read past the end of file */
	if (file_offset + amount > intf->file_length) {
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, LOGICAL_BLOCK_ADDRESS_OUT_OF_RANGE);
	    // FIXME: Set sense data info field to first invalid LBA
	    // curlun->sense_data_info = usb_offset >> 9;
	    goto bail;
	}

	/* Wait for the next buffer to become available */
	bh = intf->next_buffhd_to_fill;
	if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) goto bail;
	intf->next_buffhd_to_fill = bh->next;

	/* Perform the read */
	file_offset_tmp = file_offset;
	nread = intf->filp->f_op->read(intf->filp,
		(char *)bh->buf, amount, &file_offset_tmp);
	SD("file read %u @ %llu -> %d\n", amount,
		(unsigned long long) file_offset, (int) nread);
	if (signal_pending(current)) {
	    // SIGUSR2 means timeout, anything else is an exit condition
	    for (;;) {
		int sig;
		siginfo_t info;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		spin_lock_irq(&current->sigmask_lock);
		sig = dequeue_signal(&current->blocked, &info);
		spin_unlock_irq(&current->sigmask_lock);
#else
		sig = dequeue_signal(current, &current->blocked, &info);
#endif
		if (!sig) break;
		if (sig != SIGUSR2) {
		    ret = -EINTR;
		    goto bail;
		}
	    }
	    SD("SCSI_Read(): Timeout\n");
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ABORTED_COMMAND, TIMEOUT_ON_LOGICAL_UNIT);
	    goto bail;
	}

	/* If an error occurred, report it and its position */
	if (nread < (ssize_t)amount) {
	    SD("*** read error - set UNREC_READ_ERROR - nread: %d, amount: %d\n", nread, amount);
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_MEDIUM_ERROR, UNRECOVERED_READ_ERROR);
	    // FIXME: Set sense data info field to first invalid LBA
	    // curlun->sense_data_info = usb_offset >> 9;
	    goto bail;
	}

	/* Send this buffer, switch to next one */
	bh->state = BUF_STATE_FULL;
	bh->inreq->length = nread;
	bh->inreq->zero = 0;
	start_transfer(fsg, intf->Bin_ep, bh->inreq,
		&bh->inreq_busy, &bh->state);
	file_offset += nread;
	amount_left -= nread;
	intf->residue -= nread;
	intf->usb_amount_left -= nread;
    }
    SD("'default' reply -> EIO\n");
    SD("-------- resi after do_read: %d\n", intf->residue);
    SD("oooo nxt-to-fill 2: %p\n", intf->next_buffhd_to_fill);
    ret = 4;

bail:
    del_timer(&tl);
    return ret;
}


#else
static atomic_t next_scsi_id = ATOMIC_INIT(0);


static char SCSI_Read(bot_t *bot, struct scsi_s *scsi, uint32_t data_len_req, transfer_phase_t data_trans_in, transfer_phase_t *data_phase, uint32_t *data_len, uint32_t xferlen, uint32_t lba)
{
    SD("%s: Read: block addr: %08lx, len: %08x\n", __FUNCTION__, lba, xferlen);
    *data_phase = PHASE_DEVICE_IN;
    *data_len = xferlen * be32_to_cpu(scsi->capacity_data.blocklen_be32);

    g_ud.file_req[bot->index].id = (uint32_t)atomic_inc_return(&next_scsi_id);
    g_ud.file_req[bot->index].length = xferlen;
    g_ud.file_req[bot->index].block_address = lba;
    g_ud.file_req[bot->index].block_length = be32_to_cpu(scsi->capacity_data.blocklen_be32);
    g_ud.file_req[bot->index].cmd = PP_USB_READ;
	
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    
    return 4;
}

// generic SCSI Write function, xferlen and lba are in CPU byte order
static char SCSI_Write(bot_t *bot, struct scsi_s *scsi, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len, uint32_t xferlen, uint32_t lba)
{
    if (!scsi->write_protect) {
    	SD("%s: block addr: %08lx, len: %04x\n", __FUNCTION__, lba, xferlen);

    	g_ud.file_req[bot->index].id = (uint32_t)atomic_inc_return(&next_scsi_id);
    	g_ud.file_req[bot->index].length = xferlen;
    	g_ud.file_req[bot->index].block_address = lba;
    	g_ud.file_req[bot->index].block_length = scsi->capacity_data.blocklen_be32;
    	g_ud.file_req[bot->index].cmd = PP_USB_WRITE;
    
    	*data_phase = PHASE_DEVICE_OUT;
    	*data_len = g_ud.file_req[bot->index].block_length * g_ud.file_req[bot->index].length;
    
    	return 1;
    } else {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_DATA_PROTECT, WRITE_PROTECTED);
    	*data_len = 0;
    	*data_phase = PHASE_DEVICE_NONE;
    	
    	return 0;
    }
}
    
#endif


char RBC_ReadCapacity(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbReadCap cdb->RbcCdb_ReadCapacity
    uint8_t retStatus = 0;
    SD("ReadCapacity: OPC: %02x, slen: %08x\n",
       cdbReadCap.OperationCode,
       sizeof(scsi->capacity_data));

    if (!scsi->unit_ready) {
	bot->BOTXfer_pData = NULL;
	*data_len = 0;
	retStatus = 0; /* results in CMDFAILED */
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NOT_READY, MEDIUM_NOT_PRESENT);
    } else {
	SD("cap: bl: %x, lba: %xc\n", be32_to_cpu(scsi->capacity_data.lba_be32),
	   be32_to_cpu(scsi->capacity_data.blocklen_be32));
	*data_phase = PHASE_DEVICE_IN;
	*data_len = sizeof(scsi->capacity_data);
	bot->BOTXfer_pData = (uint8_t*)&scsi->capacity_data;
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
	retStatus = 1;
    }
    return retStatus;
#undef cdbReadCap
}

char SPC_ReportLuns(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbReportLuns cdb->SpcCdb_ReportLuns
    uint8_t retStatus = 0;
    uint32_t alloc_len = be32_to_cpu(cdbReportLuns.AllocationLength_be32);
    
    SD("ReportLuns: OPC: %02x, alloc len: %x\n",
       cdbReportLuns.OperationCode,
       alloc_len);

    /* theoretically we have to deal with unit attention stuff here -> see spec */

    if (alloc_len < 16) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);
	*data_phase = PHASE_DEVICE_NONE;
	*data_len = 0;
	retStatus = 0;
	return retStatus;
    }

    *data_phase = PHASE_DEVICE_IN;
    *data_len = min(sizeof(lunsData), (alloc_len / 8) * 8);
    
    bot->BOTXfer_pData = (uint8_t*)&lunsData;
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    retStatus = 1;

    return retStatus;
#undef cdbReadCap
}


char RBC_OnOffUnit(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbOnOffUnit cdb->RbcCdb_OnOffUnit
    char retStatus = 0;
    SD("Start Stop Unit: Power Cond: %x, LOEJ: %x, Start: %x, Immed: %x\n",
       cdbOnOffUnit.PowerConditions,
       cdbOnOffUnit.LoadEject,
       cdbOnOffUnit.Start,
       cdbOnOffUnit.Immediate);
    bot->BOTXfer_pData = NULL;
    *data_len = 0;
    retStatus = 1;
    
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    
    return retStatus;
#undef cdbOnOffUnit
}

char SPC_Inquiry(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbInquirySPC cdb->SpcCdb_Inquiry
    uint8_t length;
    char retStatus = 0;

    SD("Inquiry: OPC: %02x, VPD: %02x, CMDD: %02x, PC: %02x, AL: %02x, C: %02x\n", 
       cdbInquirySPC.OperationCode,
       cdbInquirySPC.EnableVPD,
       cdbInquirySPC.CmdSupportData,
       cdbInquirySPC.PageCode,
       cdbInquirySPC.AllocationLen,
       cdbInquirySPC.Control);

    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);

    if(cdbInquirySPC.EnableVPD) {
	switch(cdbInquirySPC.PageCode) {
	  case VPDPAGE_SUPPORTED:
	      length = sizeof(scsi->vpd_supported);
	      bot->BOTXfer_pData = (uint8_t*)&scsi->vpd_supported;
	      *data_len = length;
	      retStatus = 1;
	      *data_phase = PHASE_DEVICE_IN;
	      break;
	  case VPDPAGE_SERIAL_NUMBER:
	      length = sizeof(scsi->vpd_serial);
	      bot->BOTXfer_pData = (uint8_t*)&scsi->vpd_serial;
	      *data_len = length;
	      retStatus = 1;
	      *data_phase = PHASE_DEVICE_IN;
	      break;
	  case VPDPAGE_DEVICE_IDENTITY:
	      length = sizeof(scsi->vpd_device_id);
	      bot->BOTXfer_pData = (uint8_t*)&scsi->vpd_device_id;
	      *data_len = length;
	      retStatus = 1;
	      *data_phase = PHASE_DEVICE_IN;
	      break;
	  default:
	      retStatus = 0;
	      RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);
	      *data_phase = PHASE_DEVICE_NONE;
	}
    } else if(cdbInquirySPC.CmdSupportData) {
	retStatus = 0;
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);

	*data_phase = PHASE_DEVICE_NONE;
    } else {
	length = sizeof(scsi->inquiryData);
	SD("inquiry data in inq: %p\n", &scsi->inquiryData);
	bot->BOTXfer_pData = (uint8_t*)&scsi->inquiryData;
	*data_len = length;
	retStatus = 1;
	*data_phase = PHASE_DEVICE_IN;
    }

    *data_len = min(*data_len, (uint32_t)cdbInquirySPC.AllocationLen);

    return retStatus;
#undef cdbInquirySPC
}

char SPC_ModeSelect6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbModeSelectSPC    cdb->SpcCdb_ModeSelect
    char retStatus = 0;
    *data_phase = PHASE_DEVICE_NONE;
    bot->BOTXfer_pData = NULL;
    *data_len = 0;

    retStatus = 1;
#if 0
    bot->BOTXfer_pData = (uint8_t*)0;
    bot->BOTXfer_wResidue = cdbModeSelectSPC.ParameterLen;

    if(cdbModeSelectSPC.SavePage != 0x1) {
	if(en < BOTXfer_wResidue) {
	    BOTXfer_wResidue = CBW_wXferLen;
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);
	    BOTFSMstate = USBFSM4BOT_CSW;
	    return retStatus;
	} else if(CBW_wXferLen == BOTXfer_wResidue) {
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
	    retStatus = 1;
	} else {
	    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
	    retStatus = 1;
	}
	BOTFSMstate = USBFSM4BOT_DATAOUT;
    } else {
	BOTXfer_wResidue = CBW_wXferLen;
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);

	BOTFSMstate = USBFSM4BOT_CSW;
    }
#endif
    return retStatus;

#undef cdbModeSelectSPC
}

char SPC_ModeSelect10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    char retStatus = 0;
    *data_phase = PHASE_DEVICE_NONE;
    bot->BOTXfer_pData = NULL;
    *data_len = 0;

    retStatus = 1;
    return retStatus;
}


char SPC_ModeSense(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len,
		   char *start,
		   int hdr_len,
		   char *page_data,
		   int max_page_data,
		   int *page_data_len);


char SPC_ModeSense10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    CDB_RBC cdb2;
    int page_data_len;
    char retStatus;
#define c10 cdb->SpcCdb_ModeSense10
#define c6 cdb2.SpcCdb_ModeSense
    c6.OperationCode = c10.OperationCode;
    c6.DisableBlockDescriptor = c10.DisableBlockDescriptor;
    c6.PageControl = c10.PageControl;
    c6.PageCode = c10.PageCode;
    c6.ParameterLen = (uint8_t) (be16_to_cpu(c10.ParameterLen_be16) & 0xff); /* we send max 256 bytes */
    c6.Control = c10.Control;

    // build mode sense header
    memcpy(&mode_page_data.disk_page_10.head, &mode_parameter_head10_template, sizeof(mode_parameter_head10_template));
    mode_page_data.disk_page_10.head.DeviceParameter.WP = scsi->write_protect;
    
    retStatus = SPC_ModeSense(bot, scsi, &cdb2, data_len_req, data_phase, data_len,
			      (char*)&mode_page_data.disk_page_10, // start of complete struct
			      sizeof(mode_parameter_head10_template),
			      (char*)&mode_page_data.disk_page_10.page_data,
			      c6.ParameterLen-sizeof(mode_parameter_head10_template),
			      &page_data_len
			      );
    if (retStatus == 1) {
	// something will be sent, just adjust the data length in the header
	mode_page_data.disk_page_10.head.DataLen_be16 = cpu_to_be16(page_data_len + sizeof(mode_parameter_head10_template) - sizeof(mode_parameter_head10_template.DataLen_be16));

    }
    return retStatus;
}
    

char SPC_ModeSense6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbModeSenseSPC cdb->SpcCdb_ModeSense
    SD("------ %s: DBD: 0x%x, PC: 0x%x, PCcomplete: 0x%x, alloc-len: 0x%x\n",
       __FUNCTION__,
       cdbModeSenseSPC.DisableBlockDescriptor,
       cdbModeSenseSPC.PageControl,
       cdbModeSenseSPC.PageCode,
       cdbModeSenseSPC.ParameterLen);
    int retStatus;
    int page_data_len;
    
    // build mode sense header
    memcpy(&mode_page_data.disk_page_6.head, &mode_parameter_head6_template, sizeof(mode_parameter_head6_template));
    mode_page_data.disk_page_6.head.DeviceParameter.WP = scsi->write_protect;
    
    retStatus = SPC_ModeSense(bot, scsi, cdb, data_len_req, data_phase, data_len,
			      (char*)&mode_page_data.disk_page_6, // start of complete struct
			      sizeof(mode_parameter_head6_template),
			      (char*)&mode_page_data.disk_page_6.page_data,
			      cdbModeSenseSPC.ParameterLen-sizeof(mode_parameter_head6_template),
			      &page_data_len
			      );
    if (retStatus == 1) {
	// something will be sent, just adjust the data length in the header
	mode_page_data.disk_page_6.head.DataLen = page_data_len + sizeof(mode_parameter_head6_template) - sizeof(mode_parameter_head6_template.DataLen);
    }
    return retStatus;
}

char SPC_ModeSense(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len,
		   char *start,
		   int hdr_len,
		   char *page_data,
		   int max_page_data,
		   int *page_data_len)
{
    char retStatus = 1;

    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    
    if(cdbModeSenseSPC.PageCode == 0x3F || cdbModeSenseSPC.PageCode == 0x01)	{
	switch(cdbModeSenseSPC.PageControl) {
	  case PAGECTRL_CHANGEABLE:
	      bot->BOTXfer_pData = start;
	      /* thre:
		 commented the error recovery page out
		 reason: Windows ignored the WriteProtected flag of the
		 device when this page is sent. Without it, it works.
		 Don't know why. Maybe it misinterprets some bits in
		 the page, but I can't imagine which. */
	      
	      *data_len = hdr_len + 0;
	      *page_data_len = 0;
	      retStatus = 1;
	      *data_phase = PHASE_DEVICE_IN;
	      break;
	  case PAGECTRL_DEFAULT:
	  case PAGECTRL_CURRENT:
	      bot->BOTXfer_pData = start;
	      *data_len = hdr_len + 0;
	      *page_data_len = 0;
	      retStatus = 1;
	      *data_phase = PHASE_DEVICE_IN;
	      break;
	  case PAGECTRL_SAVED:
	  default:
	      RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, SAVING_PARAMETERS_NOT_SUPPORTED);
	      *data_len = 0;
	      *data_phase = PHASE_DEVICE_NONE;
	      return 1;
	}
    } else if (cdbModeSenseSPC.PageCode == 0x05 && scsi->device_type == PP_USB_IMAGE_TYPE_FLOPPY) {
	/* dt=0 and rm=1 means normally floppy */

	/* according to the USB Mass Storage Specification For Bootability _all_ devices (besides CDROM) should
           support the flexible disk page */
	int disk_page_len = min_t(int, sizeof(scsi->flexible_disk_page), max_page_data);
	
	memcpy(page_data, &scsi->flexible_disk_page, disk_page_len);
	bot->BOTXfer_pData = start;
	*data_len = hdr_len + disk_page_len;
	printk("hdr_len: %d, flex_disk_len: %d\n", hdr_len, disk_page_len);
	*page_data_len = disk_page_len;
	retStatus = 1;
	*data_phase = PHASE_DEVICE_IN;
    } else {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	retStatus = 0;
    }
    return retStatus;
#undef cdbModeSenseSPC
}


char SPC_LockMedia(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbLockSPC cdb->SpcCdb_Remove
    int retStatus;
    
    SD("%s: ----  Prevent: %x\n", __FUNCTION__, cdbLockSPC.Prevent);

    bot->BOTXfer_pData = NULL;
    *data_len = 0;
    retStatus = 1;
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);

    return 1;
#undef cdbLockSPC
}

char SPC_TestUnit(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbTestUnit cdb->SpcCdb_TestUnit
    uint8_t retStatus;

    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    
    SD("TestUnit: OPC: %02x, Control: %02x\n",
       cdbTestUnit.OperationCode,
       cdbTestUnit.Control);
    
    *data_phase = PHASE_DEVICE_NONE;
    *data_len = 0;

    if (scsi->unit_ready) {
	SD("unit ready\n");
	bot->BOTXfer_pData = NULL;
	*data_len = 0;
	retStatus = 1;
    } else {
	SD("unit _not_ ready\n");
	bot->BOTXfer_pData = NULL;
	*data_len = 0;
	retStatus = 0; /* results in CMDFAILED */
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NOT_READY, MEDIUM_NOT_PRESENT);
    }
    return retStatus;
#undef cdbTestUnit
}

char SPC_RequestSense(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbRequestSenseSPC cdb->SpcCdb_RequestSense
    //    PREQUEST_SENSE_DATA sd;
    SD("RequestSense: OPC: %02x, alloclen: %02x, C: %02x\n",
       cdbRequestSenseSPC.OperationCode,
       cdbRequestSenseSPC.AllocationLen,
       cdbRequestSenseSPC.Control);

    *data_phase = PHASE_DEVICE_IN;

    //sd = RBC_GetSenseData();
    
    //memcpy(&RBC_SenseData_tosend, sd, sizeof(RBC_SenseData_tosend));

    bot->BOTXfer_pData =(uint8_t*)&scsi->RBC_SenseData ;

    SD("sensedata: ResponseCode: %x (%x), SenseKey: %x (%x), ASQ: %x\n",
       scsi->RBC_SenseData.ResponseCode,
       ((char*)&scsi->RBC_SenseData)[0],
       scsi->RBC_SenseData.SenseKey,
       ((char*)&scsi->RBC_SenseData)[2],
       scsi->RBC_SenseData.ASC);
    // type case to uint8_t is ok, because cdbRequestSenseSPC.AllocationLen is 8bit and so the result of min is also within the 8bit range 
    *data_len = (uint8_t)min((uint32_t)cdbRequestSenseSPC.AllocationLen, sizeof(scsi->RBC_SenseData));
    SD("**** sensedatalen: %d\n", *data_len);
    //RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
	   
    return 1;
#undef cdbRequestSenseSPC
}

char RBC_Format(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_MEDIUM_ERROR, FORMAT_COMMAND_FAILED);
    // not supported
    *data_len = 0;
    *data_phase = PHASE_DEVICE_NONE;
    return 0;
}

/* these commands are only needed if we are in a multiple-initiator environment */
char SPC_Release(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
    *data_len = 0;
    *data_phase = PHASE_DEVICE_NONE;
    return 0;
}

char SPC_Reserve(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
    *data_len = 0;
    *data_phase = PHASE_DEVICE_NONE;
    return 0;
}

char SPC_SendDiagnostic(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    return SCSI_cmd_nop_no_error(bot, scsi, data_phase, data_len);
}

char UFI_ReZeroUnit(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    return SCSI_cmd_nop_no_error(bot, scsi, data_phase, data_len);
}

char UFI_Seek(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    return SCSI_cmd_nop_no_error(bot, scsi, data_phase, data_len);
}

char MMC_ReadTOC(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbReadToc cdb->MmcCdb_ReadToc
    SD("Read Toc: MSF: %x, Format: %x, Track Nr: %x, Alloc Len: %d\n",
       cdbReadToc.MSF,
       cdbReadToc.Format,
       cdbReadToc.TrackNumber,
       be16_to_cpu(cdbReadToc.AllocationLength_be16));

    if (cdbReadToc.Format == 0x00 && (cdbReadToc.TrackNumber == 0x00 || cdbReadToc.TrackNumber == 0x01)) {
	if (cdbReadToc.MSF) {
	    /* calculate the start address of the leadout in MSF format */
	    uint8_t m,s,f;
	    uint32_t l;
	    l = be32_to_cpu(scsi->capacity_data.lba_be32) + 1 + 150;
	    f = l % 75;
	    l = l / 75;
	    s = l % 60;
	    l = l / 60;
	    m = l % 255;
	    SD("l: %x m: %x, s: %x, f: %x\n", l, m, s, f);
	    scsi->tocdata_track1_and_lo.LeadOut.Adress.MSF.Reserved = 0x00;
	    scsi->tocdata_track1_and_lo.LeadOut.Adress.MSF.Minute = m;
	    scsi->tocdata_track1_and_lo.LeadOut.Adress.MSF.Second = s;
	    scsi->tocdata_track1_and_lo.LeadOut.Adress.MSF.Frame = f;
	    
	    // thre: don't know why, but this works with KDE auto mounter :-)
	    scsi->tocdata_track1_and_lo.Track1.Adress.MSF.Second = 2;
	    
	    bot->BOTXfer_pData = (uint8_t*) &scsi->tocdata_track1_and_lo;
	    *data_len = min((unsigned int)be16_to_cpu(cdbReadToc.AllocationLength_be16), (unsigned int)sizeof(scsi->tocdata_track1_and_lo));
	    *data_phase = PHASE_DEVICE_IN;
	} else {
	    bot->BOTXfer_pData = (uint8_t*) &scsi->tocdata_track1;
	    *data_len = min((unsigned int)be16_to_cpu(cdbReadToc.AllocationLength_be16), (unsigned int)sizeof(scsi->tocdata_track1));
	    *data_phase = PHASE_DEVICE_IN;
	}
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
        return 1;
    } else {
        SD("Illegal!!\n");
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_FIELD_IN_CDB);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
        return 0;
    }
}

char MMC_ReadFormatCapacities(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbReadFormatCapa cdb->MmcCdb_ReadFormatCapacities

    scsi->readformatcap.CapacityListHeader.CapacityListLength = sizeof(MAX_CAP_DESC);
    scsi->readformatcap.MaxCapacityDescriptor.NumberOfBlocks_be32 = cpu_to_be32(be32_to_cpu(scsi->capacity_data.lba_be32) + 1);
    scsi->readformatcap.MaxCapacityDescriptor.DescriptorType = (scsi->unit_ready)?2:3;
    scsi->readformatcap.MaxCapacityDescriptor.BlockLength_MSB = (be32_to_cpu(scsi->capacity_data.blocklen_be32) & 0x00ff0000) >> 16;
    scsi->readformatcap.MaxCapacityDescriptor.BlockLength_LSB_be16 = cpu_to_be16(be32_to_cpu(scsi->capacity_data.blocklen_be32) & 0x0000ffff);

    SD("red redformcap: caplist: %d, maxnob: %x, dt: %x, BL: %x\n",
       scsi->readformatcap.CapacityListHeader.CapacityListLength,
       scsi->readformatcap.MaxCapacityDescriptor.NumberOfBlocks_be32,
       scsi->readformatcap.MaxCapacityDescriptor.DescriptorType,
       ((uint32_t)scsi->readformatcap.MaxCapacityDescriptor.BlockLength_MSB << 16) |
       (uint32_t)(be16_to_cpu(scsi->readformatcap.MaxCapacityDescriptor.BlockLength_LSB_be16)));


    // check_command(need_medium = 0, ...)
    
    bot->BOTXfer_pData = (uint8_t*) &scsi->readformatcap;
    *data_len = sizeof(scsi->readformatcap);
    *data_phase = PHASE_DEVICE_IN;
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);

    return 1;
}

char MMC_GetConfiguration(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len) {
#define cdbCfg cdb->MmcCdb_GetConfiguration

    if (scsi->device_type == PP_USB_IMAGE_TYPE_ISO && be16_to_cpu(cdbCfg.StartingFeatureNumber_be16) == 0) {
        scsi->featureDescriptor.Header.CurrentProfile_be16 = scsi->unit_ready ? 
            scsi->featureDescriptor.Features.ProfileList.Profiles[0].ProfileNumber_be16 : cpu_to_be16(PROFILE_NUMBER_NONE);
        scsi->featureDescriptor.Features.ProfileList.Profiles[0].CurrentP = scsi->unit_ready ? 1 : 0;

        bot->BOTXfer_pData = (uint8_t*) &scsi->featureDescriptor;
        *data_len = min((unsigned int)be16_to_cpu(cdbCfg.AllocationLength_be16), (unsigned int)sizeof(scsi->featureDescriptor));
        *data_phase = PHASE_DEVICE_IN;
        RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
        return 1;
    } else {
        RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
        *data_len = 0;
        *data_phase = PHASE_DEVICE_NONE;
        return 0;
    }

#undef cdbCfg
}

char MMC_EventStatusNotification(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req,
                                 transfer_phase_t *data_phase, uint32_t *data_len) {
#define cdbEvtNotf cdb->MmcCdb_GetEventStatusNotification
    
    if (scsi->device_type == PP_USB_IMAGE_TYPE_ISO && cdbEvtNotf.IMMED) {
        if (cdbEvtNotf.NotificationClassRequest & (1 << EVENT_STATUS_MEDIA_STATUS)) {
            scsi->eventStatusNotificationMediaChange.Header.EventDataLength_be16 =
                __constant_cpu_to_be16(sizeof(scsi->eventStatusNotificationMediaChange) -
                                       sizeof(scsi->eventStatusNotificationMediaChange.Header.EventDataLength_be16));
            scsi->eventStatusNotificationMediaChange.Header.NotificationClass = EVENT_STATUS_MEDIA_STATUS;
            scsi->eventStatusNotificationMediaChange.Header.NEA = 0;                    /* no event available */
            scsi->eventStatusNotificationMediaChange.Header.SupportedEventClasses = (1 << EVENT_STATUS_MEDIA_STATUS);
            
            scsi->eventStatusNotificationMediaChange.DoorOpen = 0;
            scsi->eventStatusNotificationMediaChange.MediaPresent = scsi->unit_ready ? 1 : 0;

            switch (scsi->event_notf) {
                case SCSI_EVENT_INSERTED:
                    scsi->eventStatusNotificationMediaChange.MediaEvent = MEDIA_EVENT_NEW_MEDIA; break;
                case SCSI_EVENT_REMOVED:
                    scsi->eventStatusNotificationMediaChange.MediaEvent = MEDIA_EVENT_MEDIA_REMOVAL; break;
                case SCSI_EVENT_NONE:
                default:
                    scsi->eventStatusNotificationMediaChange.MediaEvent = MEDIA_EVENT_NO_EVENT; break;
            }
            scsi->event_notf = SCSI_EVENT_NONE;

            *data_len = sizeof(scsi->eventStatusNotificationMediaChange);
        } else {
            // we don't support any other events
            scsi->eventStatusNotificationMediaChange.Header.EventDataLength_be16 =
                __constant_cpu_to_be16(sizeof(scsi->eventStatusNotificationMediaChange.Header) -
                                       sizeof(scsi->eventStatusNotificationMediaChange.Header.EventDataLength_be16));
            scsi->eventStatusNotificationMediaChange.Header.NotificationClass = 0;      /* 0 means requested event not supported */
            scsi->eventStatusNotificationMediaChange.Header.NEA = 1;                    /* no event available */
            scsi->eventStatusNotificationMediaChange.Header.SupportedEventClasses = (1 << EVENT_STATUS_MEDIA_STATUS);

            *data_len = sizeof(scsi->eventStatusNotificationMediaChange.Header);
        }
        // Synchonous event notification, tell the requestor that we don't support any of them
        bot->BOTXfer_pData = (uint8_t*) &scsi->eventStatusNotificationMediaChange;
        *data_phase = PHASE_DEVICE_IN;
        RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
        return 1;
    } else {
        // Asynchronous events not supported!
        RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
        *data_len = 0;
        *data_phase = PHASE_DEVICE_NONE;
        return 0;
    }
#undef cdbEvtNotf
}

char MMC_DiscInformation(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len) {
#define cdbDisc cdb->MmcCdbDiscInformation

    if (scsi->device_type == PP_USB_IMAGE_TYPE_ISO) {
        bot->BOTXfer_pData = (uint8_t*) &scsi->discInformation;
        *data_len = min((unsigned int)be16_to_cpu(cdbDisc.AllocationLength_be16), (unsigned int)sizeof(scsi->discInformation));
        *data_phase = PHASE_DEVICE_IN;
        RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
        return 1;
    } else {
        RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
        *data_len = 0;
        *data_phase = PHASE_DEVICE_NONE;
        return 0;
    }

#undef cdbDisc
}

#ifdef __arm__

static int open_socket(struct scsi_s *scsi)
{
    int optval = 1;

    sock_create(AF_UNIX, SOCK_DGRAM, 0, &scsi->setup_proto_socket);

    /* Setting SO_PASSCRED also allows the socket to use the abstract socket
       namespace. This way we don't have to bind() to a name in file system (and
       take care about removing it later) for the userspace to send us response
       packets */
    return sock_setsockopt(scsi->setup_proto_socket,
	    SOL_SOCKET, SO_PASSCRED, (char *)&optval, sizeof(optval));
}

static void release_socket(struct scsi_s *scsi)
{
    if (scsi->setup_proto_socket) {
	sock_release(scsi->setup_proto_socket);
	scsi->setup_proto_socket = NULL;
    }
}

char SetupProtoReq(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    struct FTC_zero_dev *fsg = bot->zero_dev;
    struct ppstorage_intf_t *intf = fsg->storage[bot->index];
    struct zero_buffhd *bh;
    struct sockaddr_un sa;
    struct iovec iov;
    struct msghdr msg;
    pp_ipc_req_head_t *request;
    int r;

    if (!scsi_setup_proto_enabled) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	return 0;
    }

    /* Get request via USB */
    *data_phase = PHASE_DEVICE_OUT;
    *data_len = data_len_req;
    bot->BOTXfer_pData = NULL;

    bh = intf->next_buffhd_to_drain;
    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return 0;
    bh->out_intended_length = bh->outreq->length = data_len_req;
    start_transfer(intf->zero_dev, intf->Bout_ep, bh->outreq,
	    &bh->outreq_busy, &bh->state);
    if (zero_wait_for_buf_full(intf->zero_dev, bh) != 0) return 0;
    if (bh->outreq->actual != bh->outreq->length ||
	    bh->outreq->status != 0) return 0;
    intf->usb_amount_left -= data_len_req;
    intf->residue -= data_len_req;

    /* Pass request to userspace */
    release_socket(scsi);
    open_socket(scsi);
    request = kmalloc(data_len_req + sizeof(pp_ipc_req_head_t), GFP_KERNEL);
    request->request_type = PP_IPC_REQ_SETUP_PROTO;
    memcpy(request->data, bh->buf, data_len_req);

    sa.sun_family = AF_UNIX;
    strcpy(sa.sun_path, "/tmp/eric_control.socket");
    iov.iov_base = request;
    iov.iov_len = data_len_req + sizeof(pp_ipc_req_head_t);
    memset(&msg, 0, sizeof(msg));
    msg.msg_name = &sa;
    msg.msg_namelen = sizeof(sa);
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    r = sock_sendmsg(scsi->setup_proto_socket, &msg, iov.iov_len);
    kfree(request);

    bh->state = BUF_STATE_EMPTY;
    intf->next_buffhd_to_drain = bh->next;

    return (r >= 0);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
/**
 * msleep - sleep safely even with waitqueue interruptions
 * @msecs: Time in milliseconds to sleep for
 */
static void msleep(unsigned int msecs)
{
	unsigned long timeout = (msecs + 1000/HZ -1) / (1000/HZ) + 1;

	while (timeout) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		timeout = schedule_timeout(timeout);
	}
}
#endif

char SetupProtoRsp(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    struct FTC_zero_dev *fsg = bot->zero_dev;
    struct ppstorage_intf_t *intf = fsg->storage[bot->index];
    struct zero_buffhd *bh;

    struct iovec iov;
    struct msghdr msg;
    pp_ipc_rsp_head_t *rsp;
    int r;

    if (!scsi_setup_proto_enabled) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	return 0;
    }

    *data_phase = PHASE_DEVICE_IN;

    bh = intf->next_buffhd_to_fill;
    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return 0;

    rsp = kmalloc(PP_IPC_SOCKET_MTU + sizeof(pp_ipc_rsp_head_t), GFP_KERNEL);
    iov.iov_base = rsp;
    iov.iov_len = PP_IPC_SOCKET_MTU;
    memset(&msg, 0, sizeof(msg));
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;

//    r = sock_recvmsg(scsi->setup_proto_socket, &msg, 100, 0);
    r = sock_recvmsg(scsi->setup_proto_socket, &msg, PP_IPC_SOCKET_MTU, MSG_DONTWAIT);
    if (r == -EAGAIN) {
	// Send zero-length response as "not yet" signal for psetup
	*data_len = 0;
    } else if (r <= sizeof(pp_ipc_rsp_head_t)) {
	return 0;
    } else {
	*data_len = r - sizeof(pp_ipc_rsp_head_t);
	memcpy(bh->buf, rsp->data, *data_len);
	bh->inreq->zero = 0;
	bh->inreq->length = *data_len;
	start_transfer(fsg, intf->Bin_ep, bh->inreq,
		&bh->inreq_busy, &bh->state);
    }
    kfree(rsp);

    intf->residue -= *data_len;
    intf->usb_amount_left -= *data_len;

    if (intf->usb_amount_left > 0) {
	// make sure the whole packet has been written into the FIFO buffer
	// and transferred to the host before stalling. No idea how to do
	// this any better with Faraday's core.
	intf->usb_amount_left = 0;
	if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return 0;
	msleep(100);
	zero_set_halt(fsg, intf->Bin_ep);
	msleep(100);
    }
    intf->next_buffhd_to_fill = bh->next;

    return 1;
}

#else

char SetupProtoReq(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    if (!scsi_setup_proto_enabled) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	return 0;
    }
    
    UD("############################## get SetupProto request, requested: %d bytes\n", data_len_req);
    
    g_ud.file_req[bot->index].length = data_len_req;
    g_ud.file_req[bot->index].block_address = 0;
    g_ud.file_req[bot->index].block_length = 1;
    g_ud.file_req[bot->index].cmd = PP_USB_SETUP_PROTO_REQ;
    
    *data_phase = PHASE_DEVICE_OUT;
    *data_len = g_ud.file_req[bot->index].block_length * g_ud.file_req[bot->index].length;

    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    
    return 1;
}

char SetupProtoRsp(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    if (!scsi_setup_proto_enabled) {
	RBC_BuildSenseData(bot, scsi, SCSI_SENSE_ILLEGAL_REQUEST, INVALID_COMMAND_OPERATION_CODE);
	*data_len = 0;
	*data_phase = PHASE_DEVICE_NONE;
	return 0;
    }
	
    UD("**************************** get SetupProto response request, requested: %d bytes\n", data_len_req);

    *data_phase = PHASE_DEVICE_IN;
    *data_len = data_len_req;

    g_ud.file_req[bot->index].length = data_len_req;
    g_ud.file_req[bot->index].block_address = 0;
    g_ud.file_req[bot->index].block_length = 1;
    g_ud.file_req[bot->index].cmd = PP_USB_SETUP_PROTO_RSP;
    
    RBC_BuildSenseData(bot, scsi, SCSI_SENSE_NO_SENSE, NO_ADDITIONAL_SENSE_INFORMATION);
    return 4;
}

#endif

#ifdef PP_FEAT_IPMI_SERVER_SCSI_CHAN
char IpmiOverScsi(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
#define cdbIpmi cdb->Cdb_IpmiOverScsi
    struct FTC_zero_dev *fsg = bot->zero_dev;
    struct ppstorage_intf_t *intf = fsg->storage[bot->index];
    struct zero_buffhd *bh;

    uint16_t timeout;
    uint32_t session_id;
    int length;

    switch (cdbIpmi.Command) {
	case IPMI_OVER_SCSI_OPEN_SESSION:
	    timeout = be16_to_cpu(cdbIpmi.SessionTimeout);
	    session_id = scsi_ipmi_session_new(timeout);
	    scsi->ipmi_scsi_open_session_data = session_id;
	    *data_phase = PHASE_DEVICE_IN;
	    *data_len = 4;
	    bot->BOTXfer_pData = (char*)&scsi->ipmi_scsi_open_session_data;
	    return 1;

	case IPMI_OVER_SCSI_CLOSE_SESSION:
	    scsi_ipmi_session_close(cdbIpmi.SessionId);
	    *data_phase = PHASE_DEVICE_NONE;
	    *data_len = 0;
	    bot->BOTXfer_pData = NULL;
	    return 1;

	case IPMI_OVER_SCSI_SEND_REQUEST:
	    bh = intf->next_buffhd_to_drain;
	    intf->next_buffhd_to_drain = bh->next;
	    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return 0;
	    bh->out_intended_length = bh->outreq->length = data_len_req;
	    start_transfer(fsg, intf->Bout_ep, bh->outreq,
		    &bh->outreq_busy, &bh->state);
	    if (zero_wait_for_buf_full(intf->zero_dev, bh) != 0) return 0;
	    if (bh->outreq->actual != bh->outreq->length ||
		    bh->outreq->status != 0) return 0;
	    intf->usb_amount_left -= data_len_req;
	    intf->residue -= data_len_req;

	    length = scsi_ipmi_send_cmd(cdbIpmi.SessionId, data_len_req, bh->buf);
	    if (length < 0) {
		SD("scsi_ipmi_send_cmd() failed: %d\n", length);
		return 0;
	    }
	    bh->state = BUF_STATE_EMPTY;

	    *data_phase = PHASE_DEVICE_OUT;
	    *data_len = data_len_req;
	    bot->BOTXfer_pData = NULL;
	    return 1;

	case IPMI_OVER_SCSI_GET_RESPONSE:
	    bh = intf->next_buffhd_to_fill;
	    if (zero_wait_for_buf_free(intf->zero_dev, bh) != 0) return 0;
	    intf->next_buffhd_to_fill = bh->next;
	    /* Prepend a two-byte data length field before response; the Windows
	       SCSI-passthrough driver does not seem to regard the residue field. */
	    length = scsi_ipmi_get_response(cdbIpmi.SessionId, data_len_req - 2, bh->buf + 2);
	    *(unsigned short *)bh->buf = cpu_to_le16(length);
	    if (length < 0) {
		SD("scsi_ipmi_get_response() failed: %d\n", length);
		return 0;
	    }
	    bh->inreq->zero = 0;
	    bh->inreq->length = data_len_req;
	    bh->state = BUF_STATE_FULL;
	    start_transfer(fsg, intf->Bin_ep, bh->inreq,
		    &bh->inreq_busy, &bh->state);
	    intf->usb_amount_left -= data_len_req;
	    intf->residue -= (length + 2);

	    *data_phase = PHASE_DEVICE_IN;
	    *data_len = (length + 2);
	    bot->BOTXfer_pData = NULL;
	    return 1;
    }
    SD("Unknown IPMI-over-SCSI subcommand: %02x\n", cdbIpmi.Command);
    return 0;
#undef cdbIpmi
}
#endif

char GetPpId(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len)
{
    static char pp_id[] = PP_ID_VALUE;

    *data_phase = PHASE_DEVICE_IN;
    *data_len = PP_ID_LENGTH;
    bot->BOTXfer_pData = pp_id;
    return 1;
}

void RBC_BuildSenseData(bot_t *bot, struct scsi_s *scsi, int8_t SenseKey, uint16_t add_sense_data)
{
    scsi->RBC_SenseData.ResponseCode = SCSI_RESPONSECODE_CURRENT_ERROR;
    scsi->RBC_SenseData.Valid = 0;
    scsi->RBC_SenseData.SenseKey =  SenseKey;
    scsi->RBC_SenseData.AdditionalSenseLen = sizeof(scsi->RBC_SenseData) - 8;
    scsi->RBC_SenseData.ASC = ASC(add_sense_data);
    scsi->RBC_SenseData.ASCQ = ASCQ(add_sense_data);
}

