#ifndef __PP_SCSI_INTERN_H_
#define __PP_SCSI_INTERN_H_

#include <linux/types.h>

/* defines for endianess conversion, should be consolidated */
#include "../pp_byte_order.h"

/* a semi automatically generated list of all possible scan-codes extracted from the SPC */
#include "sense_codes.h"

/* debug */
#define noSCSI_DEBUG
#ifdef SCSI_DEBUG
#define SD(fmt, args...)	{ printk(fmt, ##args); }
#else
#define SD(fmt, args...)
#endif

/* SBC commands */

/* mandatory (SBC spec) */
#define SBC_CMD_FORMAT				        0x04
#define SBC_CMD_READ6                                   0x08
#define SBC_CMD_READ10					0x28
#define SBC_CMD_READCAPACITY				0x25

/* optional */
#define SBC_CMD_WRITE6                                  0x0A
#define SBC_CMD_WRITE10					0x2A
#define SBC_CMD_VERIFY10				0x2F
#define SBC_CMD_STARTSTOPUNIT				0x1B
#define SBC_CMD_SYNCCACHE				0x35

/* SPC-2 commands */

/* mandatory (SPC-2 spec) */
#define SPC_CMD_INQUIRY					0x12

/* mandatory (SBC spec) */
#define SPC_CMD_REQUESTSENSE				0x03
#define SPC_CMD_TESTUNITREADY				0x00
#define SPC_CMD_RELEASE                                 0x57
#define SPC_CMD_RESERVE                                 0x56
#define SPC_CMD_SEND_DIAGNOSTIC                         0x1D

/* optional */
#define SPC_CMD_MODESELECT6				0x15
#define SPC_CMD_MODESENSE6				0x1A
#define SPC_CMD_MODESELECT10		                0x55
#define SPC_CMD_MODESENSE10				0x5A
#define SPC_CMD_PERSISTANTRESERVIN			0x5E
#define SPC_CMD_PERSISTANTRESERVOUT			0x5F
#define SPC_CMD_PRVENTALLOWMEDIUMREMOVAL	        0x1E
#define SPC_CMD_RELEASE6				0x17
#define SPC_CMD_RESERVE6				0x16
#define SPC_CMD_WRITEBUFFER				0x3B
#define SPC_CMD_REPORTLUNS                              0xA0

/* SFF 8070 / UFI */
#define UFI_CMD_READ12					0xA8
#define UFI_CMD_REZEROUNIT				0x01
#define UFI_CMD_SEEK10					0x2B
#define UFI_CMD_WRITE12                                 0xAA
#define UFI_CMD_WRITEVERIFY				0x2E

/* vendor specific (http://www.t10.org/lists/op-num.htm) */
#define SCSI_VENDOR_CMD_SETUP_PROTO_REQ                 0xC0
#define SCSI_VENDOR_CMD_SETUP_PROTO_RSP                 0xC1
#define SCSI_VENDOR_CMD_IPMI                            0xC2
#define SCSI_VENDOR_CMD_GET_PP_ID                       0xC3

/* MMC-2 */
#define MMC_CMD_READ_FORMAT_CAPACITIES                  0x23
#define MMC_CMD_READ_TOC                                0x43
/* these two additional commands are (theoretically) mandatory for MMC-2 devices */
#define MMC_GET_EVENT_STATUS_NOTIFICATION               0x4A
#define MMC_GET_CONFIGURATION                           0x46
#define MMC_CMD_DISC_INFORMATION                        0x51

/* for START_STOP_UNIT: SBC, 6.1.14 */
#define PWR_NOCHANGE    0
#define PWR_ACTIVE      1
#define PWR_IDLE        2
#define PWR_STANDBY     3
#define PWR_SLEEP       5
#define PWR_DEVCTRL     7

/* DeviceType field for Inquiry Data: SPC, 7.3.2 */
#define DIRECT_ACCESS_DEVICE            0x00    /* disks */
#define SEQUENTIAL_ACCESS_DEVICE        0x01    /* tapes */
#define PRINTER_DEVICE                  0x02    /* printers */
#define PROCESSOR_DEVICE                0x03    /* scanners, printers, etc */
#define WRITE_ONCE_READ_MULTIPLE_DEVICE 0x04    /* worms */
#define READ_ONLY_DIRECT_ACCESS_DEVICE  0x05    /* cdroms */
#define SCANNER_DEVICE                  0x06    /* scanners */
#define OPTICAL_DEVICE                  0x07    /* optical disks */
#define MEDIUM_CHANGER                  0x08    /* jukebox */
#define COMMUNICATION_DEVICE            0x09    /* network */
#define ASC0T18_DEVICE                  0x0A    /* ASC IT8 */
#define ASC1T18_DEVICE                  0x0B    /* ASC IT8 */
#define SCC2_DEVICE                     0x0C    /* Storage Controller device */
#define SES_DEVICE                      0x0D    /* Enclousre Service device */
#define RBC_DEVICE                      0x0E    /* RBC device */
#define OCRW_DEVICE                     0x0F    /* optical Card Reader /writer Device */

/* Mode Sense/Select page constants. */
#define MODE_PAGE_RBC_DEVICE_PARAMETERS 0x06

#define PAGECTRL_CURRENT    0x0
#define PAGECTRL_CHANGEABLE 0x1
#define PAGECTRL_DEFAULT    0x2
#define PAGECTRL_SAVED      0x3

/* Prevent Allow Medium Removal: SPC, 7.12 */
#define MEDIUMREMOVAL_PROHIBITED_ALL    0x3
#define MEDIUMREMOVAL_ALLOWED_ALL       0x0
#define MEDIUMREMOVAL_ALLOWED_CHANGER   0x1
#define MEDIUMREMOVAL_ALLOWED_DATA      0x2


#define SCSI_RESPONSECODE_CURRENT_ERROR     0x70
#define SCSI_RESPONSECODE_PREVIOUS_ERROR    0x71


/* Unit Attention Conditions */
typedef enum {
    SCSI_UNIT_ATT_NONE,
    SCSI_UNIT_ATT_RESET,
    SCSI_UNIT_ATT_CHANGED,
    SCSI_UNIT_ATT_INQUIRY
} unit_attention_condition_t;

/* events */
typedef enum {
    SCSI_EVENT_INSERTED,
    SCSI_EVENT_REMOVED,
    SCSI_EVENT_NONE
} event_notification_t;

/* Sense codes: SPC, 7.20.6 */
#define SCSI_SENSE_NO_SENSE         0x00
#define SCSI_SENSE_RECOVERED_ERROR  0x01
#define SCSI_SENSE_NOT_READY        0x02
#define SCSI_SENSE_MEDIUM_ERROR     0x03
#define SCSI_SENSE_HARDWARE_ERROR   0x04
#define SCSI_SENSE_ILLEGAL_REQUEST  0x05
#define SCSI_SENSE_UNIT_ATTENTION   0x06
#define SCSI_SENSE_DATA_PROTECT     0x07
#define SCSI_SENSE_BLANK_CHECK      0x08
#define SCSI_SENSE_UNIQUE           0x09
#define SCSI_SENSE_COPY_ABORTED     0x0A
#define SCSI_SENSE_ABORTED_COMMAND  0x0B
#define SCSI_SENSE_EQUAL            0x0C
#define SCSI_SENSE_VOL_OVERFLOW     0x0D
#define SCSI_SENSE_MISCOMPARE       0x0E
#define SCSI_SENSE_RESERVED         0x0F

/* 16 bytes CDB (maximum CDB in Mass Storage Bulk Only Command Block Wrapper) */
typedef struct _GENERIC_CDB {
    uint8_t OperationCode;
    uint8_t Reserved[15];
} __attribute__((__packed__)) GENERIC_CDB,*PGENERIC_CDB;

typedef struct _GENERIC_RBC {
    uint8_t OperationCode;
    uint8_t Reserved[8];
    uint8_t Control;
} __attribute__((__packed__)) GENERIC_RBC,*PGENERIC_RBC;

/* FORMAT UNIT, SBC 6.1.1 */
typedef struct _FORMAT_CDB {
    uint8_t OperationCode;	/* 04H */

    BITFIELD4(uint8_t,
	      DefectListFormat: 3,
	      CmpLst: 1,
	      FmtData: 1,
	      Reserved0: 3
	      );
    
    uint8_t VendorSpecific;

    uint16_t Interleave_be16;
    uint8_t Control;
} __attribute__((__packed__)) FORMAT_RBC, *PFORMAT_RBC;

/* READ(10), SBC 6.1.5 */
typedef struct _READ10_RBC {
    uint8_t OperationCode;	/* 28H */

    BITFIELD5(uint8_t,
	      RelAdr: 1,
	      Reserved1: 2,
	      FUA: 1,
	      DPO: 1,
	      Reserved0: 3
	      );
    
    uint32_t LBA_W_be32;
    uint8_t Reserved;
    uint16_t XferLength_be16;
    uint8_t Control;
} __attribute__((__packed__)) READ10_RBC, *PREAD10_RBC;

/* READ(6), SBC 6.1.4 */
typedef struct _READ6_RBC {
    uint8_t OperationCode;	/* 08H */

    BITFIELD2(uint8_t,
	      LBA_MSB:5,
	      Reserved:3
	      );

    uint16_t LBA_W_be16;
    
    uint8_t XferLength;
    
    uint8_t Control;
} __attribute__((__packed__)) READ6_RBC, *PREAD6_RBC;

/* READ(12), UFI 4.8*/
typedef struct _READ12_UFI {
    uint8_t OperationCode;      /* A8H */

    BITFIELD5(uint8_t,
	      RelAdr: 1,
	      Reserved1: 2,
	      FUA: 1,
	      DPO: 1,
	      LogicalUnitNumber: 3
	      );
    
    uint32_t LBA_W_be32;
    uint32_t XferLength_be32; 
    
    uint8_t Reserved2;
    uint8_t Reserved3;
} __attribute__((__packed__)) READ12_UFI, *PREAD12_UFI;

/* REZEROUNIT, UFI 4.12 */
typedef struct _REZEROUNIT_UFI{
    uint8_t OperationCode;      /* 01H */
    BITFIELD2(uint8_t,
	      Reserved1: 5,
	      LogicalUnitNumber: 3
	      );
    
    uint8_t Reserved2[10];
} __attribute__((__packed__)) REZEROUNIT_UFI, *PREZEROUNIT_UFI;

/* SEEK, UFI 4.13 */
typedef struct _SEEK10_UFI{
    uint8_t OperationCode;      /* 2BH */

    BITFIELD2(uint8_t,
	      Reserved1: 5,
	      LogicalUnitNumber: 3
	      );

    uint32_t LBA_be32;
    uint8_t Reserved2[6];
} __attribute__((__packed__)) SEEK10_UFI, *PSEEK10_UFI;

/* READ CAPACITY, SBC 6.1.6 */
typedef struct _READ_CAPACITY_DATA {
    uint32_t lba_be32;
    uint32_t blocklen_be32;
} __attribute__((__packed__)) READ_CAPACITY_DATA, *PREAD_CAPACITY_DATA;

typedef struct _READ_CAPACITY_RBC {
    uint8_t  OperationCode;	/* 25H */

    BITFIELD2(uint8_t,
	      RelAdr: 1,
	      Reserved0: 7
	      );
    
    uint32_t LBA_W_be32;

    uint16_t Reserved1_be16;

    BITFIELD2(uint8_t,
	      PMI: 1,
	      Reserved2: 7
	      );
    
    uint8_t Control;
} __attribute__((__packed__)) READ_CAPACITY_RBC, *PREAD_CAPACITY_RBC;

/* START_STOP_UNIT, SBC 6.1.14 */
typedef struct _START_STOP_RBC {
    uint8_t OperationCode;    /*1BH*/

    BITFIELD2(uint8_t,
	      Immediate: 1,
	      Reserved1 : 7
	      );
    
    uint8_t Reserved2[2];

    BITFIELD4(uint8_t,
	      Start          : 1,
	      LoadEject      : 1,
	      Reserved3      : 2,
	      PowerConditions: 4
	      );
    
    uint8_t Control;
} __attribute__((__packed__)) START_STOP_RBC, *PSTART_STOP_RBC;


/* SEND DIAGNOSTIC, SPC 7.23 */
typedef struct _SEND_DIAGNOSTIC_RBC {
    uint8_t OperationCode;    /*1DH*/

    BITFIELD6(uint8_t,
	      UnitOffL: 1,
	      DevOffL:1,
	      SelfTest: 1,
	      Reserved0: 1,
	      PF: 1,
	      SelfTestCode: 3
	      );

    uint8_t Reserved1;

    uint16_t ParameterLen_be16;
    
    uint8_t Control;
} __attribute__((__packed__)) SEND_DIAGNOSTIC_RBC, *PSEND_DIAGNOSTIC_RBC;

/* Synchronize Cache, SBC 6.1.15 */
typedef struct _SYNCHRONIZE_CACHE_RBC {
    uint8_t OperationCode;    /* 0x35 */
    uint8_t Reserved[8];
    uint8_t Control;
} __attribute__((__packed__)) SYNCHRONIZE_CACHE_RBC, *PSYNCHRONIZE_CACHE_RBC;

/* vendor specific, setup proto request */
typedef struct _VENDOR_SETUP_PROTO_REQUEST {
    uint8_t OperationCode;    /* 0xC0 */
    uint8_t Reserved[4];
    uint8_t Control;
} __attribute__((__packed__)) VENDOR_SETUP_PROTO_REQUEST;

/* vendor specific, setup proto response request*/
typedef struct _VENDOR_SETUP_PROTO_RESPONSE {
    uint8_t OperationCode;    /* 0xC1 */
    uint8_t Reserved[4];
    uint8_t Control;
} __attribute__((__packed__)) VENDOR_SETUP_PROTO_RESPONSE;

/* WRITE(6), SBC 6.1.17 */
typedef struct _WRITE6_RBC {
    uint8_t OperationCode;	/* 0AH      */

    BITFIELD2(uint8_t,
	      LBA_MSB: 5,
	      Reserved0: 3
	      );
    
    uint16_t LBA_W_be16;

    uint8_t XferLength;

    uint8_t Control;
} __attribute__((__packed__)) WRITE6_RBC, *PWRITE6_RBC;

/* WRITE(10), SBC 6.1.18 */
typedef struct _WRITE10_RBC {
    uint8_t OperationCode;	/* 2AH      */

    BITFIELD5(uint8_t,
	      RelAdr: 1,
	      Reserved1: 2,
	      FUA: 1,
	      DPO: 1,
	      Reserved0: 3
	      );
    
    uint32_t LBA_W_be32;

    uint8_t Reserved2;

    uint16_t XferLength_be16;

    uint8_t Control;
} __attribute__((__packed__)) WRITE10_RBC, *PWRITE10_RBC;

/* REPORT LUNS, SPC 7.19 */
typedef struct _REPORT_LUNS_SPC {
    uint8_t OperationCode;	/* A0h */

    uint8_t Reserved1;
    uint8_t Reserved2;
    uint8_t Reserved3;
    uint8_t Reserved4;
    uint8_t Reserver5;

    uint32_t AllocationLength_be32;

    uint8_t Reserved6;

    uint8_t Control;
} __attribute__((__packed__)) REPORT_LUNS, *PREPORT_LUNS;

/* for now assume we have only one lun */
typedef struct _REPORT_LUNS_DATA {
    uint32_t LunListLength_be32;
    uint32_t Reserved1;
    uint64_t Lun0_be64;
} __attribute__((__packed__)) REPORT_LUNS_DATA, *PREPORT_LUNS_DATA;
    


/* WRITE(12), UFI 4.19 */
typedef struct _WRITE12_UFI {
    uint8_t OperationCode;	/* AAH      */

    BITFIELD5(uint8_t,
	      RelAdr: 1,
	      Reserved1: 2,
	      FUA: 1,
	      DPO: 1,
	      Reserved0: 3
	      );
    
    uint32_t LBA_W_be32;

    uint32_t XferLength_be32;

    uint8_t Reserve2;
    uint8_t Reserve3;
} __attribute__((__packed__)) WRITE12_UFI, *PWRITE12_UFI;

/* WRITEVERIFY, UFI 4.20 */
typedef struct _WRITEVERIFY_UFI {
    uint8_t OperationCode;	/* 2EH      */

    BITFIELD6(uint8_t,
	      RelAdr: 1,
	      ByteChk: 1,
	      Reserved2: 1,
	      Reserved1: 1,
	      DPO: 1,
	      Reserved0: 3
	      );
	      
    uint32_t LBA_W_be32;
    uint8_t Reserved3;
    uint16_t XferLength_be16;
    uint8_t Reserved4;
    uint8_t Reserved5;
    uint8_t Reserved6;
} __attribute__((__packed__)) WRITEVERIFY_UFI, *PWRITEVERIFY_UFI;

/* VERIFY, SBC 6.1.16 */
typedef struct _VERIFY_RBC {
    uint8_t OperationCode;	/* 2FH */
    
    uint8_t Reserved0;

    uint32_t LBA_W_be32;

    uint8_t Reserved1;
    uint16_t VerifyLength_be16;

    uint8_t Control;
} __attribute__((__packed__)) VERIFY_RBC, *PVERIFY_RBC;

/* INQUIRY, SPC 7.3.1 */
typedef struct _INQUIRY_SPC {
    uint8_t OperationCode;	/* 12H */

    BITFIELD3(uint8_t,
	      EnableVPD:1,
	      CmdSupportData:1,
	      Reserved0:6
	      );
	      
    uint8_t PageCode;
    uint8_t Reserved1;
    uint8_t AllocationLen;
    uint8_t Control;
} __attribute__((__packed__)) INQUIRY_SPC, *PINQUIRY_SPC;

/* SPC 7.3.2 */
typedef struct _STD_INQUIRYDATA {
    /* Byte 0 */
    BITFIELD2(uint8_t,
	      DeviceType : 5,
	      PeripheralQualifier: 3
	      );
    
    /* Byte 1 */
    BITFIELD2(uint8_t,
	      Reserved1 : 7,
	      RemovableMedia : 1
	      );
    
    /* Byte 2 */
    uint8_t Version;

    /* Byte 3 */
    BITFIELD5(uint8_t,
	      ResponseDataFormat: 4,
	      HiSup : 1,
	      NormACA : 1,
	      Obsolete0 : 1,
	      AERC : 1
	      );

    /* Byte 4 */
    uint8_t AdditionalLength;

    /* Byte 5-6 */
    uint8_t Reserved4[2];

    /* Byte 7 */
    BITFIELD8(uint8_t,
	      SoftReset : 1,
	      CommandQueue : 1,
	      Reserved5 : 1,
	      LinkedCommands : 1,
	      Synchronous : 1,
	      Wide16Bit : 1,
	      Wide32Bit : 1,
	      RelativeAddressing : 1
	      );
	      
    /* Byte 8-15 */
    uint8_t VendorId[8];

    /* Byte 16-31 */
    uint8_t ProductId[16];

    /* Byte 32-35 */
    uint8_t ProductRevisionLevel[4];

    /* Byte 35-55 */
    uint8_t VendorSpecific[20];

    /* Byte 56 */
    BITFIELD4(uint8_t,
	      InfoUnitSupport : 1,
	      QuickArbitSupport : 1,
	      Clocking : 2,
	      Reserved6 : 4
	      );

    /* Byte 57 */
    uint8_t  Reserved7 ;

    /* Byte 58-73 */
    int16_t VersionDescriptor_be16[8] ;

    /* Byte 74-95 */
    uint8_t Reserved8[22];
} __attribute__((__packed__)) STD_INQUIRYDATA, *PSTD_INQUIRYDATA;


/* SPC 8.4 */
#define VPDPAGE_SUPPORTED       0x00
#define VPDPAGE_SERIAL_NUMBER   0x80
#define VPDPAGE_DEVICE_IDENTITY 0x83

/* SPC 8.4.6 */
#define SERIALNUMBER_LENGTH 24
typedef struct _SERIALNUMBER_PAGE {
    BITFIELD2(uint8_t,
	      DeviceType : 5,
	      DeviceTypeQualifier : 3
	      );
    
    uint8_t PageCode ;
    uint8_t Reserved0 ;

    uint8_t PageLength ;
    uint8_t SerialNumber[SERIALNUMBER_LENGTH];
} __attribute__((__packed__)) VPD_SERIAL_PAGE,* PVPD_SERIAL_PAGE;

/* SPC 8.4.5 */
#define NUMBER_VPD_PAGES 3
typedef struct _SUPPORTED_PAGE {
    BITFIELD2(uint8_t,
	      DeviceType : 5,
	      DeviceTypeQualifier : 3
	      );
    
    uint8_t PageCode ;
    uint8_t Reserved0 ;

    uint8_t PageLength ;
    uint8_t SupportedPages[NUMBER_VPD_PAGES];
} __attribute__((__packed__)) VPD_SUPPORTED_PAGE, * PVPD_SUPPORTED_PAGE;

/* SPC 8.4.4. */
#define ASCII_ID_STRING 25
typedef struct _ID_DESCRIPTOR {
    BITFIELD2(uint8_t,
	      CodeSet : 4,
	      Reserved0 : 4
	      );

    BITFIELD3(uint8_t,
	      IDType : 4,
	      Association : 2,
	      Reserved1 : 2
	      );
    
    uint8_t   Reserved2;

    uint8_t   IDLength ;
    uint8_t   AsciiID[ASCII_ID_STRING];
} __attribute__((__packed__)) ASCII_ID_DESCRIPTOR,* PASCII_ID_DESCRIPTOR;

typedef struct _DEVICE_ID_PAGE {
    BITFIELD2(uint8_t,
	      DeviceType : 5,
	      DeviceTypeQualifier : 3
	      );
    
    uint8_t PageCode ;
    uint8_t Reserved0 ;
    
    uint8_t PageLength ;

    ASCII_ID_DESCRIPTOR   AsciiIdDescriptor;
} __attribute__((__packed__)) VPD_DEVICE_ID_PAGE, * PVPD_DEVICE_ID_PAGE;


typedef struct _CAP_LIST_HEADER {
    uint8_t Reserved0;
    uint8_t Reserved1;
    uint8_t Reserved2;
    uint8_t CapacityListLength;
} __attribute__((__packed__)) CAP_LIST_HEADER;

typedef struct _MAX_CAP_DESC {
    uint32_t NumberOfBlocks_be32;
    BITFIELD2(uint8_t,
	      DescriptorType:2,
	      Reserved0:6
	      );
    uint8_t BlockLength_MSB;
    uint16_t BlockLength_LSB_be16;
} __attribute__((__packed__)) MAX_CAP_DESC;

typedef struct _READ_FORMAT_CAP {
    CAP_LIST_HEADER CapacityListHeader;
    MAX_CAP_DESC MaxCapacityDescriptor;
} __attribute__((__packed__)) READ_FORMAT_CAP;

/* MODE SELECT, SPC 7.6 */
typedef struct _MODE_SELECT_SPC {
    uint8_t OperationCode;	/* 15H */

    BITFIELD4(uint8_t,
	      SavePage : 1,
	      Reseved0 : 3,
	      PageFormat : 1,
	      Reserved1 : 3
	      );

    uint8_t Reserved2[2];
    
    uint8_t ParameterLen;
    uint8_t Control;
} __attribute__((__packed__)) MODE_SELECT_SPC, * PMODE_SELECT_SPC;

/* MODE SENSE, SPC */
typedef struct _MODE_SENSE_SPC {
    uint8_t OperationCode;	/* 1AH */

    BITFIELD3(uint8_t,
	      Reserved1 : 3,	      
	      DisableBlockDescriptor : 1,
	      Reserved0 : 4
	      );

    BITFIELD2(uint8_t,
	      PageCode:6,
	      PageControl : 2
	      );

    uint8_t Reserved2;
    uint8_t ParameterLen;
    uint8_t Control;
} __attribute__((__packed__)) MODE_SENSE_SPC, * PMODE_SENSE_SPC;

/* MODE SENSE, SPC */
typedef struct _MODE_SENSE10_SPC {
    uint8_t OperationCode;	/* 1AH */

    BITFIELD3(uint8_t,
	      Reserved1 : 3,	      
	      DisableBlockDescriptor : 1,
	      Reserved0 : 4
	      );

    BITFIELD2(uint8_t,
	      PageCode:6,
	      PageControl : 2
	      );

    uint8_t Reserved2[4];
    uint16_t ParameterLen_be16;
    uint8_t Control;
} __attribute__((__packed__)) MODE_SENSE10_SPC, * PMODE_SENSE10_SPC;

typedef struct _FLEXIBLE_DISK_PAGE {
    BITFIELD3(uint8_t,
	      PageCode: 6,	      
	      Reserved: 1,
	      PageSavable: 1
	      );

    uint8_t PageLength;

    uint16_t TransferRate_be16;
    uint8_t NumberOfHeads;
    uint8_t SectorsPerTrack;
    uint16_t DataBytesPerSector_be16;
    uint16_t NumberOfCylinders_be16;
    uint16_t StartWritePreComp_be16;
    uint16_t StartWriteCurr_be16;
    uint16_t StepRate_be16;
    uint8_t PulseWidth;
    uint16_t HeadSettleDelay_be16;
    uint8_t MotorOnDelay;
    uint8_t MotorOffDelay;

    BITFIELD4(uint8_t,
	      Reserved2:5,
	      MO:1,
	      SSN:1,
	      TRDY:1
	      );

    BITFIELD2(uint8_t,
	      SPC: 4,
	      Reserved3:4
	      );
    
    uint8_t WriteCompensation;
    uint8_t HeadLoadDelay;
    uint8_t HeadUnloadDelay;

    BITFIELD2(uint8_t,
	      Pin2:4,
	      Pin34:4
	      );

    BITFIELD2(uint8_t,
	      Pin1:4,
	      Pin4:4
	      );

    uint16_t MediumRotationRate_be16;
    uint8_t Reserved4;
    uint8_t Reserved5;
} __attribute__((__packed__)) FLEXIBLE_DISK_PAGE; 
    
typedef struct _MODE_ERROR_RECOVERY_PAGE {
    BITFIELD3(uint8_t,
	      PageCode: 6,	      
	      Reserved: 1,
	      PageSavable: 1
	      );

    uint8_t PageLength;

    BITFIELD8(uint8_t,
	      DCR: 1,
	      DTe: 1,
	      PER: 1,
	      EER: 1,
	      RC: 1,
	      TB: 1,
	      ARRE: 1,
	      AWRE: 1
	      );

    uint8_t ReadRetryCount;
    uint8_t CorrectionSpan;
    uint8_t HeadOffsetCount;
    uint8_t DataStrobeOffsetCount;
    uint8_t Reserved0;
    uint8_t WriteRetryCount;
    uint8_t Reserved1;
    uint16_t RecoveryTimeLimit_be16;
} __attribute__((__packed__)) MODE_ERROR_RECOVERY_PAGE;
    
typedef struct _MODE_PARAMETER_HEAD6 {
    uint8_t DataLen;
    uint8_t MediumType;
    struct {
	BITFIELD4(uint8_t,
		  Reserved1: 4,
		  DPOFUA: 1,
		  Reserved0: 2,
		  WP: 1
		  );
    } __attribute__((__packed__)) DeviceParameter;
    unsigned char BlockDescriptorLen;
} __attribute__((__packed__)) MODE_PARAMETER_HEAD6, * PMODE_PARAMETER_HEAD6;

typedef struct _MODE_PARAMETER_HEAD10 {
    uint16_t DataLen_be16;
    uint8_t MediumType;
    struct {
	BITFIELD4(uint8_t,
		  Reserved1: 4,
		  DPOFUA: 1,
		  Reserved0: 2,
		  WP: 1
		  );
    } __attribute__((__packed__)) DeviceParameter;
    uint8_t Reserved0[2];
    uint16_t BlockDescriptorLen_be16;
} __attribute__((__packed__)) MODE_PARAMETER_HEAD10, * PMODE_PARAMETER_HEAD10;

typedef struct {
    uint8_t data[20];
} __attribute__((__packed__)) MODE_CD_CAP_PAGE;

#if 0
typedef struct {
    MODE_PARAMETER_HEAD10 head;
    MODE_CD_CAP_PAGE cd_cap_page;
} __attribute__((__packed__)) MODE_CD_CAPABILITIES_PAGE;
#endif

typedef struct {
    MODE_PARAMETER_HEAD10 head;
    char page_data[256];
} __attribute__((__packed__)) MODE_PAGE10;

typedef struct {
    MODE_PARAMETER_HEAD6 head;
    char page_data[256];
} __attribute__((__packed__)) MODE_PAGE6;


/* PREVENT ALLOW MEDIUM REMOVAL, SPC 7.12 */
typedef struct _MEDIA_REMOVAL_SPC {
    uint8_t OperationCode;    /* 1EH */

    uint8_t Reserved0[3];
    
    BITFIELD2(uint8_t,
	      Prevent:2,
	      Reserved1:6
	      );

    uint8_t Control;
} __attribute__((__packed__)) MEDIA_REMOVAL_SPC, *PMEDIA_REMOVAL_SPC;

/* READ TOC, MMC-3, 5.1.13 */
typedef struct _READ_TOC_MMC {
    uint8_t OperationCode;    /* 43H */

    BITFIELD4(uint8_t,
	      Reserved2:1,
	      MSF:1,
	      Reserved1:3,
	      Reserved0:3
	      );

    BITFIELD2(uint8_t,
	      Format:4,
	      Reserved3:4
	      );

    uint8_t Reserved4[3];
    uint8_t TrackNumber;
    uint16_t AllocationLength_be16;

    uint8_t Control;
} __attribute__((__packed__)) READ_TOC_MMC, *PREAD_TOC_MMC;


typedef struct _TOC_HEAD {
    uint16_t DataLength_be16;
    uint8_t FirstTrack;
    uint8_t LastTrack;
} __attribute__((__packed__)) TOC_HEAD;

/* mmc-3, 5.1.13.2 */
typedef struct _TOC_DATA_ENTRY {
    uint8_t Reserved0;
    BITFIELD2(uint8_t,
	      CONTROL: 4,
	      ADR: 4
	      );
    uint8_t TrackNumber;
    uint8_t Reserved1;
    union {
	uint32_t LBA_be32;
	struct {
	    uint8_t Reserved;
	    uint8_t Minute;
	    uint8_t Second;
	    uint8_t Frame;
	} __attribute__((__packed__)) MSF;
    } __attribute__((__packed__)) Adress;
} __attribute__((__packed__)) TOC_DATA_ENTRY;

typedef struct _TOC_DATA_TRACK1 {
    TOC_HEAD Head;
    TOC_DATA_ENTRY Track1;
} __attribute__((__packed__)) TOC_DATA_TRACK1;

typedef struct _TOC_DATA_TRACK1_AND_LO {
    TOC_HEAD Head;
    TOC_DATA_ENTRY Track1;
    TOC_DATA_ENTRY LeadOut;
} __attribute__((__packed__)) TOC_DATA_TRACK1_AND_LO;


/* READ FORMAT CAPACITIES MMC-2 6.1.20 */
typedef struct _READ_FORMAT_CAPACITIES_MMC {
    uint8_t OperationCode;    /* 23H */

    BITFIELD2(uint8_t,
	      Reserved1:5,
	      Reserved0:3
	      );
    
    uint8_t Reserved2[5];

    uint16_t AllocationLength_be16;
    uint8_t Control;
} __attribute__((__packed__)) READ_FORMAT_CAPACITIES_MMC;

/* GET CONFIGURATION MMC-2 6.1.4 */
typedef struct _GET_CONFIGURATION_MMC {
    uint8_t OperationCode;  /* 46h */
    
    BITFIELD3(uint8_t,
              RT:2,
              Reserved1:3,
              Reserved0:3
              );

    uint16_t StartingFeatureNumber_be16;
    uint8_t Reserved2[3];
    uint16_t AllocationLength_be16;
    uint8_t Control;
} __attribute__((__packed__)) GET_CONFIGURATION_MMC;

#define GET_CONFIGURATION_MMC_RT_RETURN_ALL         0x0
#define GET_CONFIGURATION_MMC_RT_RETURN_ALL_CURRENT 0x1
#define GET_CONFIGURATION_MMC_RT_RETURN_ONE         0x2

typedef struct _FEATURE_HEADER {
    uint32_t DataLength_be32;
    uint8_t Reserved[2];
    uint16_t CurrentProfile_be16;
} __attribute__((__packed__)) FEATURE_HEADER;

typedef struct _FEATURE_DESCRIPTOR_HEADER {
    uint16_t FeatureCode_be16;
    
    BITFIELD4(uint8_t,
              Current:1,
              Persistent:1,
              Version:4,
              Reserved:2
              );
    
    uint8_t AdditionalLength;
} __attribute__((__packed__)) FEATURE_DESCRIPTOR_HEADER;

#define PROFILE_NUMBER_NONE                 0x00
#define PROFILE_NUMBER_NON_REMOVABLE_DISK   0x01
#define PROFILE_NUMBER_REMOVABLE_DISK       0x02
#define PROFILE_NUMBER_MO_ERASABLE          0x03
#define PROFILE_NUMBER_MO_WRITE_ONCE        0x04
#define PROFILE_NUMBER_AS_MO                0x05
#define PROFILE_NUMBER_CDROM                0x08
#define PROFILE_NUMBER_CD_R                 0x09
#define PROFILE_NUMBER_CD_RW                0x0A
#define PROFILE_NUMBER_DVDROM               0x10
#define PROFILE_NUMBER_DVD_R                0x11
#define PROFILE_NUMBER_DVD_RAM              0x12

typedef struct _PROFILE_DESCRIPTOR {
    uint16_t ProfileNumber_be16;
    BITFIELD2(uint8_t,
              CurrentP:1,
              Reserved0:7
              );
    uint8_t Reserved1;
} __attribute__((__packed__)) PROFILE_DESCRIPTOR;

// this is the default feature descriptor for our devices
typedef struct _FEATURE_DESCRIPTOR {
    FEATURE_HEADER Header;
    struct {
        struct {
            FEATURE_DESCRIPTOR_HEADER Header;
            PROFILE_DESCRIPTOR Profiles[2];
        } __attribute__((__packed__)) ProfileList;
    } __attribute__((__packed__)) Features;
} __attribute__((__packed__)) FEATURE_DESCRIPTOR;

/* GET EVENT/STATUS NOTIFICATION MMC-2 6.1.5 */
#define EVENT_STATUS_RESERVED1              0
#define EVENT_STATUS_OPERATIONAL_CHANGE     1
#define EVENT_STATUS_POWER_MANAGEMENT       2
#define EVENT_STATUS_EXTERNAL_REQUEST       3
#define EVENT_STATUS_MEDIA_STATUS           4
#define EVENT_STATUS_MULTI_INITIATOR        5
#define EVENT_STATUS_DEVICE_BUSY            6
#define EVENT_STATUS_RESERVED2              7

typedef struct _GET_EVENT_STATUS_NOTIFICATION_MMC {
    uint8_t OperationCode;  /* 4Ah */
    
    BITFIELD3(uint8_t,
              IMMED:1,
              Reserved1:4,
              Reserved0:3
              );
    
    uint8_t Reserved2[2];
    uint8_t NotificationClassRequest;
    uint8_t Reserved3[2];
    uint16_t AllocationLength_be16;
    uint8_t Control;
} __attribute__((__packed__)) GET_EVENT_STATUS_NOTIFICATION_MMC;

typedef struct _EVENT_STATUS_NOTIFICATION_MMC_HEADER {
    uint16_t EventDataLength_be16;  /* length of data coming behind this header */
    
    BITFIELD3(uint8_t,
              NotificationClass:3,
              Reserved:4,
              NEA:1
              );
    
    uint8_t SupportedEventClasses;  /* bitmask of all supported events */
} __attribute__((__packed__)) EVENT_STATUS_NOTIFICATION_MMC_HEADER;

#define MEDIA_EVENT_NO_EVENT        0x00    // media state unchanged
#define MEDIA_EVENT_EJECT_REQUEST   0x01    // unit has received a request from the user to eject the slot/media
#define MEDIA_EVENT_NEW_MEDIA       0x02    // specified slot has received new media and is ready to access it
#define MEDIA_EVENT_MEDIA_REMOVAL   0x03    // media has been removed from the specified slot, unit is unable to access
#define MEDIA_EVENT_MEDIA_CHANGED   0x04    // user has requested that the media in the specified slot be loaded

typedef struct _EVENT_STATUS_NOTIFICATION_MMC_MEDIA_CHANGE {
    EVENT_STATUS_NOTIFICATION_MMC_HEADER Header;
    
    BITFIELD2(uint8_t,
              MediaEvent:4,
              Reserved0:4
              );
    BITFIELD3(uint8_t,
              DoorOpen:1,
              MediaPresent:1,
              Reserved1:6
              );
    uint8_t StartSlot;
    uint8_t EndSlot;
} __attribute__((__packed__)) EVENT_STATUS_NOTIFICATION_MMC_MEDIA_CHANGE;

/* DISC INFORMATION MMC-2 6.1.18 */
typedef struct _DISC_INFORMATION_CDB_MMC {
    uint8_t OperationCode;  /* 51h */
    uint8_t Reserved[6];
    uint16_t AllocationLength_be16;
    uint8_t Control;
} __attribute__((__packed__)) DISC_INFORMATION_CDB_MMC;

typedef struct _DISC_INFORMATION_MMC {
    uint16_t DiscInformationLength_be16;
    BITFIELD4(uint8_t,
              DiscStatus:2,
              StateOfLastSession:2,
              Erasable:1,
              Reserved0:3
              );
    uint8_t NumberOfFirstTrack;
    uint8_t NumberOfSessionsLSB;
    uint8_t FirstTrackInLastSessionLSB;
    uint8_t LastTrackInLastSessionLSB;
    BITFIELD4(uint8_t,
              Reserved1:5,
              URU:1,
              DBC_V:1,
              DID_V:1
              );
    uint8_t DiscType;
    uint8_t NumberOfSessionsMSB;
    uint8_t FirstTrackInLastSessionMSB;
    uint8_t LastTrackInLastSessionMSB;
    uint32_t DiscIdentification_be32;
    uint32_t LastSessionLeadInStartTime_be32;
    uint32_t LastPossibleStartTimeForLeadOut_be32;
    uint8_t DiscBarCode[8];
    uint8_t Reserved2;
    uint8_t NumberOfOPCTableEntries;
} __attribute__((__packed__)) DISC_INFORMATION_MMC;

/* REQUEST SENSE */
typedef struct _REQUEST_SENSE_SPC {
    uint8_t OperationCode;    /* 03H */
    uint8_t Reserved[3];
    uint8_t AllocationLen;
    uint8_t Control;
} __attribute__((__packed__)) REQUEST_SENSE_SPC, *PREQUEST_SENSE_SPC;

typedef struct _REQUEST_SENSE_DATA {
    /* order of the bit fields is architecture depended */
    BITFIELD2(uint8_t,
	      ResponseCode : 7,
	      Valid : 1
	      );
    
    uint8_t Obsolete0;

    BITFIELD5(uint8_t,
	      SenseKey : 4,
	      Reserved0 : 1,
	      WrongLenIndicator : 1,
	      EndofMedium : 1,
	      FileMark : 1
	      );
    
    uint8_t Info_0;
    uint8_t Info_1;
    uint8_t Info_2;
    uint8_t Info_3;

    uint8_t AdditionalSenseLen;

    uint8_t CommandSpecInfo_0;
    uint8_t CommandSpecInfo_1;
    uint8_t CommandSpecInfo_2;
    uint8_t CommandSpecInfo_3;

    uint8_t ASC;
    uint8_t ASCQ;
    uint8_t FieldReplacableUnitCode;

    BITFIELD2(uint8_t,
	      SenseKeySpec_0 : 7,
	      SenseKeySpecValid : 1
	      );

    uint8_t SenseKeySpec_1;

    uint8_t SenseKeySpec_2;

} __attribute__((__packed__)) REQUEST_SENSE_DATA, *PREQUEST_SENSE_DATA;

/* TEST UNIT READY, SPC 7.25 */
typedef struct _TEST_UNIT_SPC {
    uint8_t OperationCode;    /* 00H */
    uint8_t Reserved[4];
    uint8_t Control;
} __attribute__((__packed__)) TEST_UNIT_SPC, *PTEST_UNIT_SPC;

/* IPMI-over-SCSI, vendor-defined */
#define IPMI_OVER_SCSI_OPEN_SESSION  0x00
#define IPMI_OVER_SCSI_CLOSE_SESSION 0x01
#define IPMI_OVER_SCSI_SEND_REQUEST  0x02
#define IPMI_OVER_SCSI_GET_RESPONSE  0x03

typedef struct _CDB_IPMI_OVER_SCSI {
    uint8_t  OperationCode;    /* C2H */
    uint8_t  Command;
    uint32_t SessionId;
    uint16_t SessionTimeout;
} __attribute__((__packed__)) CDB_IPMI_OVER_SCSI;

/* get PP ID - this returns a special hard coded ID to be able
   to identify one of our devices from any client software */
#define PP_ID_LENGTH    32
#define PP_ID_VALUE     "PP_ID:1234567890ABCDEF1234567890"

typedef union _CDB_RBC {
    GENERIC_CDB             Cdb_Generic;
    GENERIC_RBC             RbcCdb_Generic;
    READ6_RBC               RbcCdb_Read6;
    READ10_RBC              RbcCdb_Read10;
    READ12_UFI              UfiCdb_Read12;
    SEEK10_UFI		    UfiCdb_Seek10;
    REZEROUNIT_UFI	    UfiCdb_ReZero;
    READ_CAPACITY_RBC       RbcCdb_ReadCapacity;
    START_STOP_RBC          RbcCdb_OnOffUnit;
    SYNCHRONIZE_CACHE_RBC   RbcCdb_SyncCache;
    VERIFY_RBC              RbcCdb_Verify;
    WRITE6_RBC              RbcCdb_Write6;
    WRITE10_RBC             RbcCdb_Write10;
    WRITE12_UFI             UfiCdb_Write12;
    REPORT_LUNS             SpcCdb_ReportLuns;
    WRITEVERIFY_UFI         UfiCdb_WriteVerify;
    INQUIRY_SPC             SpcCdb_Inquiry;
    MODE_SELECT_SPC         SpcCdb_ModeSelect;
    MODE_SENSE_SPC          SpcCdb_ModeSense;
    MODE_SENSE10_SPC        SpcCdb_ModeSense10;
    MEDIA_REMOVAL_SPC       SpcCdb_Remove;
    REQUEST_SENSE_SPC       SpcCdb_RequestSense;
    TEST_UNIT_SPC           SpcCdb_TestUnit;
    READ_FORMAT_CAPACITIES_MMC  MmcCdb_ReadFormatCapacities;
    GET_CONFIGURATION_MMC   MmcCdb_GetConfiguration;
    GET_EVENT_STATUS_NOTIFICATION_MMC   MmcCdb_GetEventStatusNotification;
    DISC_INFORMATION_CDB_MMC    MmcCdbDiscInformation;
    READ_TOC_MMC            MmcCdb_ReadToc;
    CDB_IPMI_OVER_SCSI      Cdb_IpmiOverScsi;
} __attribute__((__packed__)) CDB_RBC, *PCDB_RBC;

struct scsi_s {
    int index;

    /* unit_ready means "there is an image in the device" */
    volatile uint8_t unit_ready;

    /* to inform the host about device changes */
    volatile unit_attention_condition_t unit_att;
    volatile event_notification_t event_notf;

    /* the size of the image */
    volatile READ_CAPACITY_DATA capacity_data;

    /* write protection */
    volatile uint8_t write_protect;

    /* type of the device: cdrom or hdd */
    volatile usb_device_type_t device_type;

    TOC_DATA_TRACK1 tocdata_track1;
    TOC_DATA_TRACK1_AND_LO tocdata_track1_and_lo;
    READ_FORMAT_CAP readformatcap;
    FLEXIBLE_DISK_PAGE flexible_disk_page;
    VPD_SERIAL_PAGE vpd_serial;
    VPD_SUPPORTED_PAGE vpd_supported;
    VPD_DEVICE_ID_PAGE vpd_device_id;
    REQUEST_SENSE_DATA RBC_SenseData;
    STD_INQUIRYDATA inquiryData;
    uint32_t ipmi_scsi_open_session_data;
    EVENT_STATUS_NOTIFICATION_MMC_MEDIA_CHANGE eventStatusNotificationMediaChange;
    FEATURE_DESCRIPTOR featureDescriptor;
    DISC_INFORMATION_MMC discInformation;

#ifdef __arm__
    /* Socket for communication with setup protocol thread */
    struct socket *setup_proto_socket;
#endif
};

#include "scsi.h"

char RBC_Read6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len);
char RBC_Read10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len);

char RBC_ReadCapacity(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char RBC_OnOffUnit(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char RBC_Format(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);

char SPC_Inquiry(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_ModeSelect(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_ModeSense6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_ModeSense10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_LockMedia(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_TestUnit(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_RequestSense(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_Release(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_Reserve(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SPC_SendDiagnostic(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char MMC_ReadFormatCapacities(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char MMC_GetConfiguration(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char MMC_EventStatusNotification(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char MMC_DiscInformation(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SetupProtoReq(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char SetupProtoRsp(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char IpmiOverScsi(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char MMC_ReadTOC(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char GetPpId(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);

char RBC_Write6(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len);
char RBC_Write10(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req,transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len);
char UFI_Write12(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len);
char UFI_WriteVerify(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req,transfer_phase_t trans_dir_in,  transfer_phase_t *data_phase, uint32_t *data_len);

char UFI_Read12(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t trans_dir_in, transfer_phase_t *data_phase, uint32_t *data_len);
char UFI_ReZeroUnit(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);
char UFI_Seek(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);

char SPC_ReportLuns(bot_t *bot, struct scsi_s *scsi, CDB_RBC *cdb, uint32_t data_len_req, transfer_phase_t *data_phase, uint32_t *data_len);

void RBC_BuildSenseData(bot_t *bot, struct scsi_s *scsi, int8_t SenseKey, uint16_t add_sense_data);

#endif /* #ifndef __PP_SCSI_INTERN_H_ */
