#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <pp/base.h>
#include <liberic_pthread.h>
#include <liberic_cron.h>

typedef struct _cron_action_entry_t cron_action_entry_t;

struct _cron_action_entry_t {
    time_t		  rel_time;
    eric_cron_action_t	  action;
    cron_action_entry_t * next;
};

static int initialized = 0;

static pthread_t cron_thread;
static int cron_should_die = 0;
static pthread_mutex_t action_list_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
static cron_action_entry_t action_list;
static cron_action_entry_t * prev_action_entry;
static time_t prev_time;
static time_t prev_midnight_time;
static time_t prev_rel_time;

static time_t get_rel_time(time_t abs_time);
static void init_prev_action_entry(time_t rel_time);
static void * run_action(void * action);
static void * cron(void * arg);

int
eric_cron_init(void)
{
    action_list.rel_time = 24*3600;
    action_list.action = NULL;
    action_list.next = &action_list;
    prev_action_entry = &action_list;

    prev_time = time(NULL);
    prev_rel_time = get_rel_time(prev_time);
    prev_midnight_time = prev_time - prev_rel_time;

    if (eric_pthread_create(&cron_thread, 0, 65536, cron, NULL) != 0) {
	pp_log("%s(): Cannot create cron thread.\n", ___F);
	return -1;
    }

    initialized = 1;

    return 0;
}

void
eric_cron_cleanup(void)
{
    if (initialized) {
	cron_should_die = 1;
	pthread_join(cron_thread, NULL);
	prev_action_entry = &action_list;
    }
    initialized = 0;
}

int
eric_cron_insert_action(time_t rel_time, eric_cron_action_t action)
{
    cron_action_entry_t * new_ae = malloc(sizeof(cron_action_entry_t));
    cron_action_entry_t * ae;

    if (!new_ae || rel_time < 0 || rel_time >= (24*3600)) { return -1; }

    new_ae->rel_time = rel_time;
    new_ae->action = action;

    MUTEX_LOCK(&action_list_mtx);
    for (ae = &action_list;
	 ae->next->rel_time < new_ae->rel_time;
	 ae = ae->next);
    new_ae->next = ae->next;
    ae->next = new_ae;
    init_prev_action_entry(prev_rel_time);
    MUTEX_UNLOCK(&action_list_mtx);

    return 0;
}

int
eric_cron_delete_action(time_t rel_time, eric_cron_action_t action)
{
    cron_action_entry_t * ae, * old_ae;
    int need_init_prev_action_entry = 0;

    if (rel_time >= (24*3600)) { return -1; }

    MUTEX_LOCK(&action_list_mtx);
    ae = &action_list;
    if (rel_time < 0) {
	/* delete all actions that match */
	for (ae = &action_list; ae->next != &action_list;) {
	    if (ae->next->action == action) {
		old_ae = ae->next;
		ae->next = old_ae->next;
		free(old_ae);
		need_init_prev_action_entry = 1;
	    } else {
		ae = ae->next;
	    }
	}
    } else {
	/* delete the action that match exactly at a given time */
	for (ae = &action_list;
	     ae->next != &action_list &&
		 (ae->next->rel_time != rel_time ||
		  ae->next->action != action);
	     ae = ae->next);
	if (ae->next != &action_list) {
	    old_ae = ae->next;
	    ae->next = old_ae->next;
	    free(old_ae);
	    need_init_prev_action_entry = 1;
	}
    }
    if (need_init_prev_action_entry) {
	init_prev_action_entry(prev_rel_time);
    }
    MUTEX_UNLOCK(&action_list_mtx);

    return 0;
}

static time_t
get_rel_time(time_t abs_time)
{
    struct tm tm;

    localtime_r(&abs_time, &tm);
    return (tm.tm_hour * 3600 + tm.tm_min * 60 + tm.tm_sec);
}

static void
init_prev_action_entry(time_t rel_time)
{
    if (rel_time >= 0 && rel_time < (24*3600)) {

	cron_action_entry_t * ae = &action_list;

	MUTEX_LOCK(&action_list_mtx);
	for (ae = &action_list; ae->next->rel_time <= rel_time; ae = ae->next);
	prev_action_entry = ae;
	if (ae->next != &action_list) {
	    prev_action_entry = ae;
	} else {
	    prev_action_entry = ae->next;
	}
	MUTEX_UNLOCK(&action_list_mtx);
    }
}

static void *
run_action(void * action)
{
    if (action) {
	((eric_cron_action_t)action)();
    }

    pthread_exit(NULL);
}

static void *
cron(void * arg UNUSED)
{
    time_t current_time;
    struct timespec ts = { 1, 0 };

    while (!cron_should_die) {
	MUTEX_LOCK(&action_list_mtx);
	current_time = time(NULL);

	if (prev_time > (current_time + 30)) {	    
	    init_prev_action_entry(get_rel_time(current_time-1));
	}
	while (current_time >= (prev_midnight_time +
				prev_action_entry->next->rel_time)) {
	    if (prev_action_entry->next->action) {
		pthread_t action_thread;
		if (eric_pthread_create(&action_thread, 1, 65536, run_action,
					prev_action_entry->next->action)) {
		    pp_log("%s(): Cannot create action thread.\n", ___F);
		}
	    }
	    prev_action_entry = prev_action_entry->next;
	    if (prev_action_entry == &action_list) {
		prev_time = current_time;
		prev_rel_time = get_rel_time(current_time);
		prev_midnight_time = current_time - prev_rel_time;
	    }
	}
	prev_time = current_time;
	prev_rel_time = get_rel_time(current_time);
	prev_midnight_time = prev_time - prev_rel_time;
	MUTEX_UNLOCK(&action_list_mtx);
	nanosleep(&ts, NULL);
    }

    pthread_exit(NULL);
}
