#include <pp/base.h>
#include <liberic_pthread.h>
#include <sys/time.h>

/*
  gprof has the problem, that it doesn't work correctly with multithreaded apps
  This wrapper start function (http://sam.zoy.org/doc/programming/gprof.html)
  passes the timer data, which gprof uses, to all started threads
  define GPROF_COMPAT to use this workaround
*/

#ifdef GPROF_COMPAT
typedef struct gprof_wrapper_s
{
    void * (*start_routine)(void *);
    void * arg;

    pthread_mutex_t lock;
    pthread_cond_t  wait;

    struct itimerval itimer;

} gprof_wrapper_t;

static void * gprof_wrapper_routine(void *);
#endif

int
eric_pthread_create(pthread_t * thread, int detached, size_t stack_size,
		    void * (*start_routine)(void *), void * arg)
{
    pthread_attr_t attr;
    int r;
#ifdef GPROF_COMPAT   
    gprof_wrapper_t wrapper_data;

    /* Initialize the wrapper structure */
    wrapper_data.start_routine = start_routine;
    wrapper_data.arg = arg;
    getitimer(ITIMER_PROF, &wrapper_data.itimer);
    pthread_cond_init(&wrapper_data.wait, NULL);
    pthread_mutex_init(&wrapper_data.lock, NULL);
    pthread_mutex_lock(&wrapper_data.lock);
#endif
    
    /* create the thread */
    if ((r = pthread_attr_init(&attr)) != 0) {
	pp_log("Cannot initialize thread attributes (%s).\n",
		 strerror(r));
	goto finish;
    }

    if ((r = pthread_attr_setstacksize(&attr, stack_size)) != 0) {
	pp_log("Cannot set thread stack size (%s).\n", strerror(r));
	goto finish;
    }

    if (detached &&
	(r = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED))) {
	pp_log("Cannot set thread stack size (%s).\n", strerror(r));
	goto finish;

    }

#ifdef GPROF_COMPAT
    r = pthread_create(thread, &attr, &gprof_wrapper_routine, &wrapper_data);
    
    /* wait for the data to be released */
    if (r == 0) {
        pthread_cond_wait(&wrapper_data.wait, &wrapper_data.lock);
    }

    pthread_mutex_unlock(&wrapper_data.lock);
    pthread_mutex_destroy(&wrapper_data.lock);
    pthread_cond_destroy(&wrapper_data.wait);
#else
    r = pthread_create(thread, &attr, start_routine, arg);
#endif

 finish:
    return r;
}

#ifdef GPROF_COMPAT
static void * gprof_wrapper_routine(void * data)
{
    /* Put user data in thread-local variables */
    void * (*start_routine)(void *) = ((gprof_wrapper_t*)data)->start_routine;
    void * arg = ((gprof_wrapper_t*)data)->arg;

    /* Set the profile timer value */
    setitimer(ITIMER_PROF, &((gprof_wrapper_t*)data)->itimer, NULL);

    /* Tell the calling thread that we don't need its data anymore */
    pthread_mutex_lock(&((gprof_wrapper_t*)data)->lock);
    pthread_cond_signal(&((gprof_wrapper_t*)data)->wait);
    pthread_mutex_unlock(&((gprof_wrapper_t*)data)->lock);

    /* Call the real function */
    return start_routine(arg);
}
#endif
