/*****************************************************************************
 * Filename:    cmds_eric.c
 * Author:      Thomas Rendelmann
 *
 * DESCRIPTION: Implements miscalleanous  Terminal Commands.
 *
 *****************************************************************************

 Funktions implemented in this file and declared in commands.h

 long CmdReset(struct arg *arg);  void CmdResetHelp(const char *cmd);

*****************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h> 
#include <string.h>
#include <unistd.h>

#include <terminal.h>

#include "commands.h"

#include <sys/types.h>
#include <sys/ioctl.h>

#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <liberic_notify.h>
#include <pp/intl.h>
#include <pp/firmware.h>
#include <pp/powerswitch.h>
#include <pp/cfg.h>
#include <pp/um.h>
#if defined(PP_FEAT_TELNET_KVM_PSWITCH) && defined(PP_FEAT_POWERSWITCH)
#include <pp/kvm.h>
#endif
#if defined(PP_FEAT_PCI_ADC) || defined(PP_FEAT_AD_ATMEL)
#include <pp/sense.h>
#endif
#include <lara.h>

#include <term_internal.h>

#if defined(PP_FEAT_CLP)
	#include <pp/clp.h>
#endif

#if defined(PP_FEAT_POWERSWITCH)
/*---------------- CmdReset ----------------*/

#define RST_HOST	0
#define RST_CARD	1
#define RST_ERROR	2

long CmdReset(struct arg *arg, term_cl_t * clp) 
{
    int reset_host = RST_HOST; /* default is "host" */
    int confirm;
   
    if ((arg->type == t_string) && (arg->value.s != NULL)) {
	if (!strcasecmp(arg->value.s, "card")) {
	    reset_host = RST_CARD;
	} else if (strcasecmp(arg->value.s, "host")) {
	    reset_host = RST_ERROR;
	}
    }
   
    switch (reset_host) {
      case RST_HOST:
	  eric_term_printf(clp,
	      "Warning: This may result in data loss on the server!\r\ncontinue (y/n)? ");
	  if ((confirm = net_getchar(clp)) < 0) break;
	  if ((confirm == 'y') || (confirm == 'Y')) {
	      if (clp->session) {
		  const char * user = eric_session_get_user(clp->session);
		  if (PP_ERR == pp_um_user_has_permission(user, "pc",
						       pp_acl_raasip_yes_str)) {
		      eric_notify_security_violation(clp->session);
		      return -1;
		  }
	      }
	      if (PP_FAILED(pp_power_login(PP_POWER_PORT_ID_INTERNAL))
		  || PP_FAILED(pp_power_reset(0, 0, PP_POWER_PORT_ID_INTERNAL))) {
		  eric_term_printf(clp, "\r\nServer reset button press failed.\r\n");
	      } else {
		  eric_term_printf(clp, "\r\nServer reset button press performed.\r\n");
	      }
	      pp_power_logout(PP_POWER_PORT_ID_INTERNAL);
	  }
	  break;
      case RST_CARD:
	  eric_term_printf(clp,
		   "Warning: You'll be disconnected from the terminal\r\n         and a card reset will be performed.\r\ncontinue (y/n)? ");
	  if ((confirm = net_getchar(clp)) < 0) break;
	  if ((confirm == 'y') || (confirm == 'Y')) {
	      eric_term_printf(clp, "\r\nResetting card ...\r\n");
	      switch (pp_firmware_erla_reset_device(clp->session)) {
		case 0:
		    break;
		case -1:
		    eric_term_printf(clp,
		     "\r\nYou do not have the permission to reset the card.\r\n");
		    break;
		case -2:
		    eric_term_printf(clp,
		     "\r\nThere is a firmware upgrade running  currently. Try again later.\r\n");
		    break;
		default:
		    eric_term_printf(clp, "\r\nAn internal error occured.\r\n");
	      }
	  } else {
	      eric_term_printf(clp, "\r\nAborted.\r\n");
	  }
	  break;
      default:
	  CmdResetHelp("reset", clp);
    }
    return 0;
}

int CmdResetHelp(const char *cmd, term_cl_t * clp) 
{
    eric_term_printf(clp,
     "\r\nUsage: %s [host|card]\r\n\n  This resets the server system or the remote management card;\r\n  The default is 'host'.\r\n", cmd);
    return 0;
}

/*---------------- CmdPower ----------------*/

#define POWER_ON	0
#define POWER_SHORT	1
#define POWER_LONG	2
#define POWER_INFO	3
#define POWER_ERROR	4

static int
perform_powerbutton(term_cl_t * clp, int long_press) {
    if (clp->session) {
	const char * user = eric_session_get_user(clp->session);
	if (PP_ERR == pp_um_user_has_permission(user, "pc",
                                                pp_acl_raasip_yes_str)) {
	    eric_notify_security_violation(clp->session);
	    return -1;
	}
    }
    if (PP_FAILED(pp_power_login(PP_POWER_PORT_ID_INTERNAL))
	|| PP_FAILED(pp_power_switch(0, 0, !long_press, 0, PP_POWER_PORT_ID_INTERNAL))) {
	eric_term_printf(clp, "\r\nServer power button press failed.\r\n");
    } else {
	eric_term_printf(clp, "\r\nServer power button press performed.\r\n");
    }
    pp_power_logout(PP_POWER_PORT_ID_INTERNAL);
    return 0;
}

long CmdPower(struct arg *arg, term_cl_t * clp) 
{
    int power_req = POWER_INFO; /* default is info */
    int host_has_power;
    int confirm;
   
    if ((arg->type == t_string) && (arg->value.s != NULL)) {
	pp_log("arg: %s\n", arg->value.s);
	if (!strcasecmp(arg->value.s, "on")) {
	    power_req = POWER_ON;
	} else if (!strcasecmp(arg->value.s, "off")) {
	    power_req = POWER_SHORT;
	} else if (!strcasecmp(arg->value.s, "off-short")) {
	    power_req = POWER_SHORT;
	} else if (!strcasecmp(arg->value.s, "off-long")) {
	    power_req = POWER_LONG;
	} else if (strcasecmp(arg->value.s, "info")) {
	    power_req = POWER_ERROR;
	}
    }
   
    if (eric_misc_get_host_power(&host_has_power) == -1) {
	eric_term_printf(clp, "Error: Getting power status of server failed.\r\n");
	return 0;
    }
   
    switch (power_req) {
      case POWER_ON:
	  if (!host_has_power) {
	      perform_powerbutton(clp, 0);
	  } else {
	      eric_term_printf(clp, "The server is powered on already!\r\n");
	  }
	  break;
      case POWER_LONG:
      case POWER_SHORT:
	  if (host_has_power) {
	      eric_term_printf(clp,
  	         "Warning: This may result in data loss on the server!\r\ncontinue (y/n)? ");
	      if ((confirm = net_getchar(clp)) < 0) break;
	      if ((confirm == 'y') || (confirm == 'Y')) {
		  perform_powerbutton(clp, power_req == POWER_SHORT);
	      } else {
		  eric_term_printf(clp, "\r\nAborted.\r\n");
	      }
	  } else {
	      eric_term_printf(clp, "The server is powered off already!\r\n");
	  }
	  break;
      case POWER_INFO:
	  eric_term_printf(clp, "The server is powered %s.\r\n",
		     host_has_power ? "on" : "off");
	  break;
      default:
	  CmdPowerHelp("power", clp);
    }
    return 0;
}

int CmdPowerHelp(const char *cmd, term_cl_t * clp) 
{
    eric_term_printf(clp,
	       "\r\nUsage: %s [on|off|off-short|off-long|info]\r\n\n",
		     "  This powers the host server system on or off\r\n  'off-long' sends a long power-off and\r\n    unconditionally powers the system off\r\n  'info' shows whether the server is powered on or off\r\n  The default is info.\r\n", cmd);
    return 0;
}
#endif

#if defined(PP_FEAT_TELNET_KVM_PSWITCH) && defined(PP_FEAT_POWERSWITCH)
/*---------------- KVM Power Switch ----------------*/

#define KVM_POWER_OFF	 0
#define KVM_POWER_ON	 1
#define KVM_POWER_CYCLE	 2

#define POWER_CYCLE_TIME 5

long CmdKVMPSwitch(struct arg *arg, term_cl_t * clp)
{
    int ret = 0, cycle_time = POWER_CYCLE_TIME;
// FIXME: - kvm unit support
    int i = 0, power_req = 0, kvm_unit = 0, kvm_port = 0;
    struct arg *param = arg;
    
    while ( param->type != t_endoflist ) {
	i++;
	
	if ( i == 1 && param->type == t_number ) {	
	    kvm_port = (*(param->value.n)) - 1;

	    if ( kvm_port < 0 ||
		 kvm_port > pp_kvm_get_unit_port_count(kvm_unit) ) {
		eric_term_printf(clp, "Error: Argument 1 must be a number from 1 to %d\r\n",
			    pp_kvm_get_unit_port_count(kvm_unit));
		goto finish;
	    }	    
	    
	} else if ( i == 2 && param->type == t_string && (param->value.s != NULL)) {	    
	    if (!strcasecmp(param->value.s, "on")) {
		power_req = KVM_POWER_ON;
	    } else if (!strcasecmp(param->value.s, "off")) {
		power_req = KVM_POWER_OFF;
	    } else if (!strcasecmp(param->value.s, "cycle")) {
		power_req = KVM_POWER_CYCLE;
	    } else {
		eric_term_printf(clp, "Error: Argument 2 must be one of the following strings: (on | off | cycle)\r\n");
		goto finish;
	    }

	} else if ( i == 3 ) {
	    if ( param->type == t_number ) {
		cycle_time = *(param->value.n);
	    }
	} else {
	    eric_term_printf(clp, "Error: Argument %d\r\n", i);
	    CmdKVMPSwitchHelp("powerswitch", clp);
	    goto finish;
	}
	param++;
    }

    if ( i == 0 ) {
	CmdKVMPSwitchHelp("powerswitch", clp);
	goto finish;
    }    

    if ( (ret = pp_power_login(PP_POWER_PORT_ID_SERIAL_1)) ) {
	eric_term_printf(clp, "Error: Open powerswitch failed: %d\r\n", ret);
	goto finish;
    }

    switch ( power_req ) {
      case KVM_POWER_ON:
	  ret = pp_power_switch_kvm_ports(kvm_unit, kvm_port, 1);
	  break;
      case KVM_POWER_OFF:
	  ret = pp_power_switch_kvm_ports(kvm_unit, kvm_port, 0);
	  break;
      case KVM_POWER_CYCLE:
	  if ( (ret = pp_power_switch_kvm_ports(kvm_unit, kvm_port, 0)) ) {
	      break;
	  }
	  
	  sleep(cycle_time);
	  
	  ret = pp_power_switch_kvm_ports(kvm_unit, kvm_port, 1);
	  break;
      default: break;	  
    }

    if ( !ret ) {
	eric_term_printf(clp, "KVM powerswitch succeeded\r\n");
    } else {
	eric_term_printf(clp, "KVM powerswitch failed: %d\r\n", ret);
    }

    
 finish:
    pp_power_logout(PP_POWER_PORT_ID_SERIAL_1);
    return 0;
}

int CmdKVMPSwitchHelp(const char *cmd, term_cl_t * clp)
{
    eric_term_printf(clp, "\r\nUsage: %s [<kvm_port> <action> <time>]\r\n\n  All assigned power ports of the given KVM port will be switched.\r\n  <port>   is the KVM port to switch.\r\n  <action> is one of ( on | off | cycle ).\r\n  <time>   is optional power cycle time in seconds. if not present,\r\n           default of %d seconds will be used.\r\n\n", cmd, POWER_CYCLE_TIME );
    return 0;
}
#endif /* PP_FEAT_TELNET_KVM_PSWITCH && PP_FEAT_POWERSWITCH */

/*---------------- CmdTerminal ----------------*/

#if !defined(PRODUCT_ASMIDC)

long CmdTerminal(struct arg *arg, term_cl_t * clp) 
{    
    if ((arg->type == t_number) && (arg->value.n != NULL)) {
	clp->serial_id = *arg->value.n;
	
#ifdef PRODUCT_ICPMMD
	/* HACK: hide the two processor serial */
	clp->serial_id += 2;
#endif
    } else {
	clp->serial_id = 1;
    }

    /* we are starting with 1 counting the user-visible serial ports*/
    if (clp->serial_id == 0) {
	eric_term_printf(clp, "Serial port 0 does not exist, start counting with 1.\r\n");
	goto bail;
    }

    clp->serial_id--;
    
    if ((clp->serial_fd = serial_attach_client(clp)) >= 0) {
	switch_to_sub_state(clp, SUB_STATE_TERMINAL, "");
    } else {
	eric_term_printf(clp, "The selected serial port is not available.\r\n");
    }

 bail:
    return 0;
}

int CmdTerminalHelp(const char *cmd, term_cl_t * clp) 
{
    eric_term_printf(clp,
     "\r\nUsage: %s [<nr>]\r\n\n  Opens a connection to the serial port.\r\n  <nr> the serial port nr (1 or 2), default is 1.\r\n You may leave this mode by entering the following key sequence: ESC e x i t\r\n\n", cmd);
  
    return 0;
}

#endif

/*---------------- CmdVersion ----------------*/

long CmdVersion(struct arg *arg UNUSED, term_cl_t * clp) 
{
    eric_term_printf(clp,
 	       "\r\nFirmware Version: \t%c%c.%c%c.%c%c\r\nFirmware Build Number: \t%s\r\nFirmware Description: \t%s\r\n",
	       pp_firmware_erla_version[0], pp_firmware_erla_version[1],
	       pp_firmware_erla_version[2], pp_firmware_erla_version[3],
	       pp_firmware_erla_version[4], pp_firmware_erla_version[5],
	       pp_firmware_erla_build_nr_str, pp_firmware_erla_tag);
    return 0;
} 

int CmdVersionHelp(const char *cmd, term_cl_t * clp) 
{
    eric_term_printf(clp, "\r\nUsage: %s\r\n\n  Prints version info.\r\n", cmd);
  
    return 0;
}

#if defined(PP_FEAT_CLP)

/*---------------- CLP commands ----------------*/


long CmdClp(struct arg *arg UNUSED, term_cl_t * clp) {

    if (clp->clp_enabled) {
	clp->clp_session = pp_clp_open(clp);
	switch_to_sub_state(clp, SUB_STATE_CLP, clp->clp_session->prompt);
    } else {
	eric_term_printf(clp, "CLP access has been disabled.\r\n");
    }

    return 0;
}

int CmdClpHelp(const char *cmd, term_cl_t * clp) {

    eric_term_printf(clp, "\r\nUsage: %s \r\n\n", cmd);
  
    return 0;
}

#endif /* PP_FEAT_CLP */

#if defined(PP_FEAT_PCI_ADC) || defined(PP_FEAT_AD_ATMEL)
long CmdVoltages(struct arg *arg UNUSED, term_cl_t * clp) 
{
    float raw_value;

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_12V) ) {
	eric_term_printf(clp, "PCI Voltage      (+12V): %.2f V\r\n", raw_value);
    } else {
	eric_term_printf(clp, "PCI Voltage      (+12V):failed\r\n");
    }

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_5V) ) {
	eric_term_printf(clp, "PCI Voltage       (+5V):  %.2f V\r\n", raw_value);
    } else {
        eric_term_printf(clp, "PCI Voltage       (+5V):failed\r\n");
    }

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_VIO) ) {
	eric_term_printf(clp, "PCI Voltage       (VIO):  %.2f V\r\n", raw_value);
    } else {
	  eric_term_printf(clp, "PCI Voltage       (VIO):failed\r\n");
    }

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_PCI) ) {
	eric_term_printf(clp, "PCI Voltage (PCI +3.3V):  %.2f V\r\n", raw_value);
    } else {
	  eric_term_printf(clp, "PCI Voltage (PCI +3.3V):failed\r\n");
    }
	
    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_AUX) ) {
	eric_term_printf(clp, "PCI Voltage (AUX +3.3V):  %.2f V\r\n", raw_value);
    } else {
	  eric_term_printf(clp, "PCI Voltage (AUX +3.3V):failed\r\n");
    }

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_3V_PLANE) ) {
	eric_term_printf(clp, "Plane Voltage   (+3.3V):  %.2f V\r\n", raw_value);
    } else {
	  eric_term_printf(clp, "Plane Voltage   (+3.3V):failed\r\n");
    }

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_12V_NEG) ) {
	eric_term_printf(clp, "PCI Voltage      (-12V):%.2f V\r\n", raw_value);
    } else {
	  eric_term_printf(clp, "PCI Voltage      (-12V):failed\r\n");
    }

    if ( !pp_sense_read_adc_value_raw(&raw_value, PCI_VOLTAGE_5V_EXTERNAL_PWR_SUPPLY) ) {
	eric_term_printf(clp, "EXT_PWR_SUPPLY   (0V||+5V):  %.2f V\r\n", raw_value);
    } else {
	eric_term_printf(clp, "EXT_PWR_SUPPLY   (0V||+5V):failed\r\n");
    }
	
   return 0;
}

int CmdVoltagesHelp(const char *cmd, term_cl_t * clp) 
{
    eric_term_printf(clp,
           "\r\nUsage: %s \r\n\n Shows the current PCI voltages.\r\n", cmd);
    return 0;
}
#endif /* defined(PP_FEAT_PCI_ADC) || defined(PP_FEAT_AD_ATMEL) */
