#include <stdlib.h>
#include <liberic_pthread.h>
#include <pp/base.h>
#include "ringbuf.h"

ring_buf_t *
ring_buf_init(ring_buf_t * ring_buf, u_int buf_size)
{
    if (ring_buf == NULL) {
	ring_buf = malloc(sizeof(ring_buf_t));
	ring_buf->on_heap = 1;
    } else {
	ring_buf->on_heap = 0;
    }
    ring_buf->buf = malloc(buf_size);
    ring_buf->buf_size = buf_size;
    ring_buf->head_offs = ring_buf->tail_offs = 0;
    pthread_mutex_init(&ring_buf->lock, NULL);
    pthread_cond_init(&ring_buf->cond, NULL);
    return ring_buf;
}

void
ring_buf_cleanup(ring_buf_t * ring_buf)
{
    if (ring_buf) {
	pthread_mutex_destroy(&ring_buf->lock);
	pthread_cond_destroy(&ring_buf->cond);
	free(ring_buf->buf);
	if (ring_buf->on_heap) free(ring_buf);
    }
}

size_t
ring_buf_put(ring_buf_t * ring_buf, const char * data, size_t size)
{
    size_t real_size = min(size, ring_buf_get_free_size(ring_buf));
    if (real_size > 0) {
	u_int tail_offs = ring_buf->tail_offs;
	u_int new_tail_offs = (tail_offs + real_size) % ring_buf->buf_size;
	if (new_tail_offs > tail_offs || new_tail_offs == 0) {
	    memcpy(&ring_buf->buf[tail_offs], data, real_size);
	} else {
	    u_int first_seg_len = ring_buf->buf_size - tail_offs;
	    memcpy(&ring_buf->buf[tail_offs], data, first_seg_len);
	    memcpy(ring_buf->buf, &data[first_seg_len], real_size - first_seg_len);
	}
	ring_buf->tail_offs = new_tail_offs;
    }

    return real_size;
}

size_t
ring_buf_get(ring_buf_t * ring_buf, char * data, size_t size)
{
    size_t real_size = min(size, ring_buf_get_used_size(ring_buf));
    if (real_size > 0) {
	u_int head_offs = ring_buf->head_offs;
	u_int new_head_offs = (head_offs + real_size) % ring_buf->buf_size;
	if (new_head_offs > head_offs || new_head_offs == 0) {
	    memcpy(data, &ring_buf->buf[head_offs], real_size);
	} else {
	    u_int first_seg_len = ring_buf->buf_size - head_offs;
	    memcpy(data, &ring_buf->buf[head_offs], first_seg_len);
	    memcpy(&data[first_seg_len], ring_buf->buf, real_size - first_seg_len);
	}
	MUTEX_LOCK(&ring_buf->lock);
	ring_buf->head_offs = new_head_offs;
    }

    return real_size;
}

