#include <fcntl.h>
#include <unistd.h>
#include <sys/poll.h>

#include <liberic_pthread.h>
#include <pp/base.h>
#include <pp/serial.h>
#include <pp/termios.h>
#include <pp/cfg.h>

#include <term_internal.h>

typedef struct {
    int id;
    int	fd;
    int socket_pair[2];
    pthread_t handler_thread;
    int handler_thread_started;
    volatile int terminate_thread;
    term_cl_t * clp;
} serial_port_data_t;

static pp_hash_i_t * serial_port_data_hash = NULL;

static void free_serial_port(void * _spd);
static int init_serial_port(u_int id, u_int fd);
static void uninit_serial_port(u_int id);
static int _serial_write(term_cl_t* clp, const void * buf, size_t len);
static void * serial_handler_thread_func(void * arg);
static int serial_term_init(const char * device_path, u_int id, u_int fd, char ** errstr_p);
static int serial_term_uninit(const char * device_path, u_int id, u_int fd UNUSED, char ** errstr_p);

int
serial_init(void)
{
    if ((serial_port_data_hash = pp_hash_create_i(17)) == NULL) {
	pp_log("%s(): pp_hash_create_i() failed\n", ___F);
	return PP_ERR;
    }

    pp_serial_register_port_callbacks(PP_SERIAL_USAGE_PASSTHROUGH, serial_term_init, serial_term_uninit);

    return PP_SUC;
}

void
serial_cleanup(void)
{
    pp_serial_unregister_port_callbacks(PP_SERIAL_USAGE_PASSTHROUGH);
    if (serial_port_data_hash) {
	pp_hash_delete(serial_port_data_hash);
	serial_port_data_hash = NULL;
    }
}

int
serial_attach_client(term_cl_t * clp)
{
    serial_port_data_t * spd = pp_hash_get_entry_i(serial_port_data_hash, clp->serial_id);
    int fd = -1;

    if (spd != NULL && spd->clp == NULL) {
	spd->clp = clp;
	fd = spd->socket_pair[1];
    }

    return fd;
}

int
serial_detach_client(term_cl_t * clp)
{
    serial_port_data_t * spd = pp_hash_get_entry_i(serial_port_data_hash, clp->serial_id);

    if (spd != NULL && spd->clp == clp) {
	spd->clp = NULL;
	clp->serial_fd = -1;
	return PP_SUC;
    }

    return PP_ERR;
}

int
serial_write_string(term_cl_t* clp, const char* c)
{
    return _serial_write(clp, c, strlen(c));
}

int
serial_write_byte(term_cl_t* clp, char c)
{
    return _serial_write(clp, &c, 1);
}

int
serial_write_break(term_cl_t* clp, int duration)
{
    return tcsendbreak(clp->serial_fd, duration) == 0 ? PP_SUC : PP_ERR;
}

static void
free_serial_port(void * _spd)
{
    serial_port_data_t * spd = (serial_port_data_t *)_spd;
    if (spd->socket_pair[0] >= 0) close(spd->socket_pair[0]);
    /* FIXME: close other side too? */
    if (spd->handler_thread_started) {
	spd->terminate_thread = 1;
	pthread_join(spd->handler_thread, NULL);
    }
    free(spd);
}


static int
init_serial_port(u_int id, u_int fd)
{
    serial_port_data_t * spd;
    const char *parity;
    char *parity_str, *handshake_str;
    u_int speed, data_bits, stop_bits, stop2, hwf, swf;
    int n;

    pp_cfg_get_uint(&speed, "serialport[%u].passthrough.speed", id);
    pp_cfg_get_uint(&data_bits, "serialport[%u].passthrough.data", id);
    pp_cfg_get(&parity_str, "serialport[%u].passthrough.parity", id);
    pp_cfg_get_uint(&stop_bits, "serialport[%u].passthrough.stop", id);
    pp_cfg_get(&handshake_str, "serialport[%u].passthrough.handshake", id);

    if (!strcmp(handshake_str, "sw")) {
	hwf = 0; swf = 1;
    } else if (!strcmp(handshake_str, "hw")) {
	hwf = 1; swf = 0;
    } else {
	hwf = swf = 0;
    }
    free(handshake_str);
    
    if (!strcmp(parity_str, "odd")) {
	parity = "O";
    } else if (!strcmp(parity_str, "even")) {
	parity = "E";
    } else {
	parity = "";
    }
    free(parity_str);
    
    stop2 = stop_bits == 2 ? 1 : 0;

    pp_base_set_tty_params(fd, speed, parity, data_bits, stop2, hwf, swf);

    spd = calloc(1, sizeof(serial_port_data_t));
    spd->id = id;
    spd->fd = fd;
    spd->clp = NULL;
    spd->socket_pair[0] = spd->socket_pair[1] = -1;
    if (socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, spd->socket_pair) != 0) {
	pp_log_err("%s(): socketpair()", ___F);
    } else if ((n = fcntl(spd->socket_pair[0], F_GETFL, 0)) < 0) {
	pp_log_err("%s(): fcntl(F_GETFL)", ___F);
    } else if (fcntl(spd->socket_pair[0], F_SETFL, n | O_NDELAY) < 0) {
	pp_log_err("%s(): fcntl(F_SETFL)", ___F);
    } else if ((n = fcntl(fd, F_GETFL, 0)) < 0) {
	pp_log_err("%s(): fcntl(F_GETFL)", ___F);
    } else if (fcntl(fd, F_SETFL, n | O_NDELAY) < 0) {
	pp_log_err("%s(): fcntl(F_SETFL)", ___F);
    } else {
	spd->handler_thread_started = 1;
	spd->terminate_thread = 0;
	if (eric_pthread_create(&spd->handler_thread, 0, 128 * 1024, serial_handler_thread_func, spd) == 0) {
	    pp_hash_set_entry_i(serial_port_data_hash, id, spd, free_serial_port);
	} else {
	    pp_log("%s(): Cannot create serial handler thread for port id %d\n", ___F, id);
	    spd->handler_thread_started = 0;
	}
    }
    if (!spd->handler_thread_started) {
	free_serial_port(spd);
	return PP_ERR;
    }
    return PP_SUC;
}

static void
uninit_serial_port(u_int id)
{
    pp_hash_delete_entry_i(serial_port_data_hash, id);
}

static int
_serial_write(term_cl_t* clp, const void * buf, size_t len)
{
    size_t written = 0;

    while (written != len) {
	ssize_t n = write(clp->serial_fd, buf + written, len - written);
	if (n > 0) {
	    written += n;
	} else if (n == 0) {
	    break;
	} else if (n < 0) {
	    if (errno != EAGAIN && errno != EINTR) {
		pp_log_err("%s(): write() to inter link socket", ___F);
		break;
	    }
	}
    }
    return written == len ? PP_SUC : PP_ERR;
}

static void *
serial_handler_thread_func(void * arg)
{
    serial_port_data_t * spd = (serial_port_data_t *)arg;
    char rbuf[512];
    size_t rbuf_pos = 0;
    ssize_t rbuf_len = 0;
    char wbuf[512];
    size_t wbuf_pos = 0;
    ssize_t wbuf_len = 0;
    int n;
    struct pollfd fds[2] = {
	{
	    .fd = spd->fd,
	    .events = POLLIN
	},
	{
	    .fd = spd->socket_pair[0],
	    .events = POLLIN
	}
    };
    u_int pollfd_cnt = sizeof(fds) / sizeof(struct pollfd);

    while (!spd->terminate_thread) {
	if (poll(fds, pollfd_cnt, 500) < 0) {
	    if (errno == EINTR) continue;
	    pp_log_err("%s(): poll()", ___F);
	    break;
	}

	/* handle input data from serial port */
	if (fds[0].revents & POLLIN) {
	    errno = 0;
	    n = read(fds[0].fd, rbuf, sizeof(rbuf));
	    if (n > 0) {
		rbuf_len = n;
	    } else if (n == 0) {
		pp_log("read 0 bytes from serial - strange\n");
		break;
	    } else if (n < 0) {
		if (errno != EAGAIN && errno != EINTR) {
		    pp_log_err("%s(): read() from serial", ___F);
		    break;
		}
		rbuf_len = 0;
	    }
	    rbuf_pos = 0;
	}

	/* handle input data from inter link socket */
	if (fds[1].revents & POLLIN) {
	    n = read(fds[1].fd, wbuf, sizeof(wbuf));
	    if (n > 0) {
		wbuf_len = n;
	    } else if (n == 0) {
		pp_log("read 0 bytes from inter link socket - strange\n");
		break;
	    } else if (n < 0) {
		if (errno != EAGAIN && errno != EINTR) {
		    pp_log_err("%s(): read() from inter link socket", ___F);
		    break;
		}
		wbuf_len = 0;
	    }
	    wbuf_pos = 0;
	}

	/* handle output data to serial port */
	if (fds[0].revents & POLLOUT || (wbuf_len > 0 && !(fds[0].events & POLLOUT))) {
	    n = write(fds[0].fd, &wbuf[wbuf_pos], wbuf_len);
	    if (n == 0) {
		break;
	    } else if (n != wbuf_len) {
		if (n > 0 || errno == EAGAIN || errno == EINTR) {
		    fds[0].events |= POLLOUT;
		    fds[1].events &= ~POLLIN;
		} else {
		    pp_log_err("%s(): write() to serial", ___F);
		    break;
		}
	    }
	    if (n > 0) wbuf_pos += n;
	    wbuf_len -= n;
	    if (wbuf_len == 0) {
		fds[0].events &= ~POLLOUT;
		fds[1].events |= POLLIN;
		wbuf_pos = 0;
	    }
	}

	/* handle output data to inter link socketnetwork */
	if (fds[1].revents & POLLOUT || (rbuf_len > 0 && !(fds[1].events & POLLOUT))) {
	    n = write(fds[1].fd, &rbuf[rbuf_pos], rbuf_len);
	    if (n == 0) {
		break;
	    } else if (n != rbuf_len) {
		if (n > 0 || errno == EAGAIN || errno == EINTR) {
		    fds[1].events |= POLLOUT;
		    fds[0].events &= ~POLLIN;
		} else {
		    pp_log_err("%s(): write() to inter link socket", ___F);
		    break;
		}
	    }
	    if (n > 0) rbuf_pos += n;
	    rbuf_len -= n;
	    if (rbuf_len == 0) {
		fds[1].events &= ~POLLOUT;
		fds[0].events |= POLLIN;
		rbuf_pos = 0;
	    }
	}
    }

    pthread_exit(NULL);
}

static int
serial_term_init(const char * device_path UNUSED, u_int id, u_int fd, char ** errstr_p UNUSED)
{
    return init_serial_port(id, fd);
}

static int
serial_term_uninit(const char * device_path UNUSED, u_int id, u_int fd UNUSED, char ** errstr_p UNUSED)
{
    uninit_serial_port(id);
    return PP_SUC;
}
