#include <liberic_net.h>
#include <pp/um.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"

#if defined(PP_FEAT_AUTH_RADIUS)

FV_SPEC = {
    {
	id:		FV_ID_AUTH_RAD_RADIUS_SERVER,
	cfgkey:		"radius[%I]",
	elemkey:	"server"
    },
    {
	id:		FV_ID_AUTH_RAD_AUTH_PORT,
	cfgkey:		"radius[%I]",
	elemkey:	"auth_port"
    },
    {
	id:		FV_ID_AUTH_RAD_ACCT_PORT,
	cfgkey:		"radius[%I]",
	elemkey:	"acct_port"
    },
    {
	id:		FV_ID_AUTH_RAD_SECRET,
	cfgkey:		"radius[%I]",
	elemkey:	"secret"
    },
    {
	id:		FV_ID_AUTH_RAD_TIMEOUT,
	cfgkey:		"radius[%I]",
	elemkey:	"timeout"
    },
    {
	id:		FV_ID_AUTH_RAD_RETRIES,
	cfgkey:		"radius[%I]",
	elemkey:	"retries"
    },
    {
	id:		FV_ID_AUTH_RAD_AUTH_TYPE,
	cfgkey:		"radius_chap_enabled",
	elemkey:	"auth_type"
    },
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);
static int is_entry_empty(form_var_t * form_vars, u_int idx);

int
auth_radius_tmpl_init(void)
{
    form_handler_t * fh;

    fh = CREATE_FH_INSTANCE(TEMPLATE_AUTH_RADIUS, ACL_OBJ_AUTHOPT);

    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh)
{
#define NUM_IDS 6
    int ids[NUM_IDS] = {
	FV_ID_AUTH_RAD_RADIUS_SERVER,
	FV_ID_AUTH_RAD_AUTH_PORT,
	FV_ID_AUTH_RAD_ACCT_PORT,
	FV_ID_AUTH_RAD_SECRET,
	FV_ID_AUTH_RAD_TIMEOUT,
	FV_ID_AUTH_RAD_RETRIES
    };
    char *mode = get_form_var_value_unsaved(wp, TEMPLATE_AUTH, FV_ID_AUTH_MODE);
    int radius_selected = !pp_strcmp_safe(mode, "radius");
    free(mode);

    if (!radius_selected) {
	fh_disable_validate_and_save(fh, -1);
    }
    
    /* ------------------------------------------------------------------------
     * Check if the user wants more entries. Give him more if maximum
     * is not reached yet.
     */
    if (form_button_clicked(wp, "action_more_radius_servers")) {
	form_var_t * fv = &fh->fv[FV_ID_AUTH_RAD_RADIUS_SERVER];
	fv->val_cnt = form_var_vec_increase(wp, fv->fvname, fv->val_cnt,
					    4, fv->val_cnt_max, "");
	
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
	return 0;
    } else if (form_button_clicked(wp, "action_apply")) {
	/* ------------------------------------------------------------------
	 * remove empty entries
	 */
	if (radius_selected) {
	    if (remove_entries(wp, fh->fv,
			       fh->fv[FV_ID_AUTH_RAD_RADIUS_SERVER].val_cnt,
			       ids, NUM_IDS, is_entry_empty) == -1) {
		return -1;
	    }
	}
    }

    return 0;
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    char *mode = get_form_var_value_unsaved(wp, TEMPLATE_AUTH, FV_ID_AUTH_MODE);
    int radius_selected = !pp_strcmp_safe(mode, "radius");
    free(mode);

    if (radius_selected) {
	form_var_t * fv = &fh->fv[FV_ID_AUTH_RAD_RADIUS_SERVER];
	int i;

	if (fv->val_cnt <= 0) {
	    set_response(wp, ERIC_RESPONSE_ERROR,
			 _("At least one RADIUS server must be defined."));
	    return -1;
	} else {
	    for (i = 0; i < fv->val_cnt; ++i) {
		if (fv->val.m[i][0] == '\0') {
		    set_response(wp, ERIC_RESPONSE_WARNING,
				 _("Server %d requires a server name."), i + 1);
		}
	    }
	}
    }
    return 0;
}

static int
is_entry_empty(form_var_t * form_vars, u_int idx)
{
    int r;
    r = form_vars[FV_ID_AUTH_RAD_RADIUS_SERVER].val.m[idx][0] == '\0';
    return r;
}

#else /* !PP_FEAT_AUTH_RADIUS */

int auth_radius_tmpl_init(void) { return 0; }

#endif /* !PP_FEAT_AUTH_RADIUS */
