#include <pp/intl.h>
#include <pp/um.h>
#include <pp/vsc.h>
#include <pp/grab.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"

#define VSC_FILE		"/flashdisk/vsc.rbf"
#define VSC_MAX_SIZE		(512 * 1024)
#define GPIO_2_4_8_MASK		(PP_IBM_GPIO_MASK(2) | PP_IBM_GPIO_MASK(4) | PP_IBM_GPIO_MASK(8))

FV_SPEC = {
    {
	id:		FV_ID_DEBUG_VSC_FILE_NAME,
	cfgkey:		"video.vsc.file"
    }
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int vsc_upload_cb(webs_t wp, char * name, char * filename, void * data, size_t data_offset, size_t data_len, int more_data);

int
debug_vsc_tmpl_init(void)
{
    form_handler_t * fh;
   
    websAddPostDataCallback("vsc_upload_file", vsc_upload_cb);
    
    fh = CREATE_FH_INSTANCE(TEMPLATE_DEBUG_VSC, ACL_OBJ_SUPER);

    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED)
{
    const char* action;
    int ret = -1;
    int vscfile = -1;
    
    if (wp->upload_failed) {
	/* do nothing here, errors already reported */
	goto bail;
    }
   
    if (wp->sd->vsc_data) {
	if ((vscfile = open(VSC_FILE, O_CREAT | O_RDWR | O_TRUNC | O_SYNC, 0644)) == -1) {
	    pp_log("Cannot open vsc file.\n");
	    goto bail;
	}

	if (write(vscfile, wp->sd->vsc_data, wp->sd->vsc_data_size) == -1) {
	    pp_log("Cannot write vsc file.\n");
	    goto bail;
	}

	pp_vsc_reset(0, VSC_RESET_PROGRAM_FPGA);
    }

    action = websGetVar(wp, "vsc_debug_action", NULL);

    if (!pp_strcmp_safe(action, "delete")) {
	if (unlink(VSC_FILE) == 0) {
	    pp_vsc_reset(0, VSC_RESET_PROGRAM_FPGA);
	} else {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Could not delete VSC binary."));
	    goto bail;
	}
    } else if (!pp_strcmp_safe(action, "jtagon")) {
	/* set GPIOs 2, 4, 8 to output mode (0,0,1)*/
	pp_gpio_set(PP_GPIO_DEV_IBM, PP_IBM_GPIO_MASK(8), 
		    PP_IBM_GPIO_MASK(2) | PP_IBM_GPIO_MASK(4),  0);
    } else if (!pp_strcmp_safe(action, "jtagoff")) {
	/* set GPIOs 2, 4, 8 to input mode */
	pp_gpio_set(PP_GPIO_DEV_IBM, 0, 0, GPIO_2_4_8_MASK);
    } else if (!pp_strcmp_safe(action, "resetprogram")) {
	pp_vsc_reset(0, VSC_RESET_PROGRAM_FPGA);
    } else if (!pp_strcmp_safe(action, "resetplain")) {
	pp_vsc_reset(0, VSC_RESET_SOFTWARE_ONLY);
    } else if (!pp_strcmp_safe(action, "diffmap_zero")) {
	pp_grab_debug_use_diffmap(0);
    } else if (!pp_strcmp_safe(action, "diffmap_vsc")) {
	pp_grab_debug_use_diffmap(1);
    } else if (!pp_strcmp_safe(action, "diffmap_one")) {
	pp_grab_debug_use_diffmap(2);
    } else if (!pp_strcmp_safe(action, "testpic_on")) {
	pp_vsc_testpic(0, 1);
    } else if (!pp_strcmp_safe(action, "testpic_off")) {
	pp_vsc_testpic(0, 0);
    } else if (!pp_strcmp_safe(action, "ire_toggle")) {
	pp_vsc_debug_ire_toggle(0);
    }
    
    ret = 0;
        
 bail:
    if (vscfile >= 0) close(vscfile);
    if (wp->sd->vsc_data) {
	free(wp->sd->vsc_data);
	wp->sd->vsc_data = NULL;
	wp->sd->vsc_data_size = 0;
    }    
    return ret;
}

static int
vsc_upload_cb(webs_t wp, char * name UNUSED, char * filename UNUSED,
	      void * data, size_t data_offset UNUSED, size_t data_len,
	      int more_data)
{
    printf("VSC upload callback\n");
    eric_session_touch(wp->session);

    if (PP_ERR == pp_um_user_has_permission(wp->user, ACL_OBJ_SUPER,
                                            pp_acl_raasip_yes_str)) {
	set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	goto fail;
    }

    if (wp->sd->vsc_data_size + data_len > VSC_MAX_SIZE) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Uploaded file exceeds maximum size of %d KB."),
		     (VSC_MAX_SIZE / 1024));
	goto fail;
    }
    
    if ((wp->sd->vsc_data = realloc(wp->sd->vsc_data, wp->sd->vsc_data_size + data_len)) == NULL) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Not enough free memory to upload file."));
	goto fail;
    }
    
    if (data_len > 0) {
	memcpy(wp->sd->vsc_data + wp->sd->vsc_data_size, data, data_len);
	wp->sd->vsc_data_size += data_len;
    }
    
    if (data == NULL) {
	free(wp->sd->vsc_data);
	wp->sd->vsc_data = NULL;
	wp->sd->vsc_data_size = 0;
    } else if (!more_data) {
    	if (wp->sd->vsc_data_size >= 2) {
	    wp->sd->vsc_data_size -= 2;
	}
    }
    
    wp->upload_failed = 0;

    return 0;

 fail:
    free(wp->sd->vsc_data);
    wp->sd->vsc_data = NULL;
    wp->sd->vsc_data_size = 0;
    wp->upload_failed = 1;
    
    return -1;
}
