#include <pp/firmware.h>
#include <pp/intl.h>
#include <liberic_misc.h>
#include <pp/um.h>
#if defined(PRODUCT_FLASHX4)
# include <pp/flashprog.h>
#endif
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "wsIntrn.h"
#include "tmpl_firmware_common.h"
#if defined(PP_FEAT_RPCCFG)
# include <pp/rpc_ripc.h>
#endif


FV_SPEC = {
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int fw_upload_cb(webs_t wp, char * name, char * filename, void * data, size_t data_offset, size_t data_len, int more_data);

int
firmware_upload_tmpl_init(void)
{
    form_handler_t * fh;

    if (firmware_common_tmpl_init() != 0) return -1;

    /* register POST callbacks */
    websAddPostDataCallback("firmware_file", fw_upload_cb);

    fh = CREATE_FH_INSTANCE(TEMPLATE_FIRMWARE_UPLOAD, ACL_OBJ_FIRMWARE);

    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp,  form_handler_t * fh UNUSED)
{
#if defined(PRODUCT_FLASHX4)
    int  file_type = 0;
    char *fp_upload;
#endif    
    char logmsg[PP_NOTIFY_MAX_MSG_LEN+1];
    const char * fCrossOEM = websGetVar(wp, "crossoem", "");
    const char * fCrossHWID = websGetVar(wp, "crosshwid", "");
    const char * error_msg = NULL;
    char * ver = NULL;
    char * min_req_ver = NULL;
    char * min_dwngrd_ver = NULL;
    int bnr;
    int ret = -1;

    if (wp->upload_failed) {
	return -1;
    } else if (form_button_clicked(wp, "action_firmware_upload")) {
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;

#if defined(PRODUCT_FLASHX4)
	fp_upload = websGetVar(wp, "upload_file", "");

	if ( !strcmp(fp_upload, "8mb") ) {
	    file_type = FILE_UBOOT_8MB;
	} else if ( !strcmp(fp_upload, "16mb") ) {
	    file_type = FILE_UBOOT_16MB;
	} else {
#endif /* PRODUCT_FLASHX4 */

	    /* get firmware info */
	    if (PP_FAILED(pp_firmware_erla_get_info(wp->sd->firmware_ctx, &ver, &bnr,
			  NULL, &min_req_ver, &min_dwngrd_ver))) {
		set_response(wp, ERIC_RESPONSE_ERROR, _("Getting firmware information failed."));
		goto bail;
	    }

	    if (pp_firmware_erla_validate(wp->sd->firmware_ctx, !strcmp(fCrossOEM, "yes"),
					  !strcmp(fCrossHWID, "yes")) != 0) {
		char * custom_err_msg = NULL;
		if (errno == PP_FIRMWARE_ERR_NEW_VER_TOO_OLD) {
		    asprintf(&custom_err_msg,
			     _("You may not downgrade to the uploaded firmware. "
			       "The minimum version you may downgrade to is %c%c.%c%c.%c%c."),
			     min_dwngrd_ver[0], min_dwngrd_ver[1], min_dwngrd_ver[2],
			     min_dwngrd_ver[3], min_dwngrd_ver[4], min_dwngrd_ver[5]);
		} else if (errno == PP_FIRMWARE_ERR_CUR_VER_TOO_OLD) {
		    asprintf(&custom_err_msg,
			     _("The current firmware is too old. "
			       "You need to update to version %c%c.%c%c.%c%c first."),
			     min_req_ver[0], min_req_ver[1], min_req_ver[2],
			     min_req_ver[3], min_req_ver[4], min_req_ver[5]);
		} else {
		    error_msg = pp_error_string(errno);
		}
		set_response(wp, ERIC_RESPONSE_ERROR, custom_err_msg ? custom_err_msg : error_msg);
		free(custom_err_msg);
		eric_notify_post_event(_("Firmware validation failed."), "device", PP_NOTIFY_EVENT_GENERIC);
	    } else {
		snprintf(logmsg, sizeof(logmsg),
			 _("Firmware file uploaded by user '%s'. %c%c.%c%c.%c%c (Build %u)."),
			 eric_session_get_user(wp->session),
			 ver[0], ver[1], ver[2], ver[3], ver[4], ver[5], bnr);
		eric_notify_post_event(logmsg, "device", PP_NOTIFY_EVENT_GENERIC);

#ifdef PP_FEAT_RPCCFG
		/*
		 * Send fw via RPC to the slave KIM. This is only executed by the master KIM
		 * because no web-access to slave KIMs.
		 */
		if (PP_FAILED(pp_rpc_fw_upload(wp->sd->firmware_ctx))) {
		    set_response(wp, ERIC_RESPONSE_ERROR,
				 _("A internal error occurred during sharing the firmware internally."));
		    goto bail;
		}
#endif /* PP_FEAT_RPCCFG */
		ret = 0;
	    }
#if defined(PRODUCT_FLASHX4)
	}
	if (file_type) {
	    int ret;
	    size_t size = pp_firmware_erla_data_get_len(wp->sd->firmware_ctx);
	    if (size == 0x20000) {
		void * flash_data = malloc(size);
		if ((ret = pp_firmware_erla_data_get(wp->sd->firmware_ctx, flash_data, 0, size)) != 0
		    || (ret = pp_flashprog_set_data(flash_data, size, file_type)) != 0) {
		    set_response(wp, ERIC_RESPONSE_ERROR, _("Error while uploading flash prog file."));
		} else {
		    set_response(wp, ERIC_RESPONSE_OK, _("Flash file successfully uploaded"));
		}
		free(flash_data);
	    } else {
		set_response(wp, ERIC_RESPONSE_ERROR, _("Flash file has wrong size: 0x%x. It should have 0x20000 bytes."), size);
	    }
	    pp_firmware_erla_ctx_free(wp->sd->firmware_ctx);
	    wp->sd->firmware_ctx = 0;
	    return ret;
	}
#endif /* PRODUCT_FLASHX4 */

    } else {
	ret = 0;
    }

 bail:
    free(ver);
    free(min_req_ver);
    free(min_dwngrd_ver);
    if (ret != 0) {
	pp_firmware_erla_ctx_free(wp->sd->firmware_ctx);
	wp->sd->firmware_ctx = 0;
    }
    return ret;
}

static int
fw_upload_cb(webs_t wp, char * name UNUSED, char * filename UNUSED,
	     void * data, size_t data_offset, size_t data_len, int more_data)
{
    eric_session_touch(wp->session);

    if (PP_ERR == pp_um_user_has_permission(wp->user, ACL_OBJ_FIRMWARE,
                                            pp_acl_raasip_yes_str)) {
	set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	goto fail;
    }

    if (data_offset == 0 && data_len > 0) {
	if ((wp->sd->firmware_ctx = pp_firmware_erla_ctx_new(PP_FIRMWARE_ERLA_MAX_SIZE+2, 1)) == 0) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("A firmware upload/update is already running by another user."));
	    goto fail;
	}
    }

    if (pp_firmware_erla_data_set(wp->sd->firmware_ctx, data, data_offset, data_len) != 0) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Firmware too big (max. %u MB)."), PP_FIRMWARE_ERLA_MAX_SIZE / (1024 * 1024));
	goto fail;
    }

    if (data == NULL) {
	pp_firmware_erla_ctx_free(wp->sd->firmware_ctx);
	wp->sd->firmware_ctx = 0;
    } else if (!more_data) {
	if ((data_offset + data_len) >= 2) pp_firmware_erla_data_set_len(wp->sd->firmware_ctx, data_offset + data_len - 2);
    }
  
    return 0;

 fail:
    pp_firmware_erla_ctx_free(wp->sd->firmware_ctx);
    wp->sd->firmware_ctx = 0;
    wp->upload_failed = 1;
    eric_notify_post_event(_("Firmware upload failed."), "device", PP_NOTIFY_EVENT_GENERIC);

    return -1;
}
