#include <stdio.h>

#include <pp/base.h>
#include <pp/ipmi.h>
#include <pp/intl.h>
#include <liberic_notify.h>

#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"

#ifdef PP_FEAT_ESB2_FNI

#define ESB2_I2C_DELAY  250000 // usecs

FV_SPEC = {
    {
        id:             FV_ID_IPMI_FNI_ENABLED,
        cfgkey:         "ipmi.fni.enabled"
    },
};

static int fni_debug = 0;

static int post_validate_hook(webs_t wp, form_handler_t * fh);
static int ipmi_fni_get_vars_asp(int eid, webs_t wp, int argc, char **argv);
static int get_fni_debug(void);
static int esb2_fni_debug_trigger(pp_cfg_chg_ctx_t * ctx);

int
ipmi_fni_tmpl_init(void)
{
    form_handler_t * fh;
    
    fni_debug = get_fni_debug();
    pp_cfg_add_post_tx_change_listener(esb2_fni_debug_trigger, "ipmi.fni.debug");

    websAspDefine("ipmiFniGetVars", ipmi_fni_get_vars_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_IPMI_FNI, ACL_OBJ_IPMI);
    fh->post_validate_hook = post_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static void fni_log(int no_log, const char *format, ...) {
    va_list ap;
    char *txt;
    va_start(ap, format);
    
    vasprintf(&txt, format, ap);
    if (!no_log) pp_log(txt);
    if (fni_debug) eric_notify_post_event(txt, "asmi", PP_NOTIFY_EVENT_GENERIC);
    
    free(txt);
    va_end(ap);
}

static int get_fni_debug(void) {
    int debug;
    pp_cfg_is_enabled(&debug, "ipmi.fni.debug");
    return debug;
}

static int esb2_fni_debug_trigger(pp_cfg_chg_ctx_t * ctx UNUSED) {
    fni_debug = get_fni_debug();
    return 0;
}

static int
post_validate_hook(webs_t wp,  form_handler_t * fh UNUSED)
{
    pp_ipmi_parameter_t param;
    pp_ipmi_return_t ret;
    int err;
    int fni_chan;
    int sol_enabled;
    int anon_enabled;
    int anon_sol_enabled;
    const char *webs_var;
    int retval = PP_ERR;
    
    // if FNI is not enabled, we can skip this whole section
    if (strcmp(fh->fv[FV_ID_IPMI_FNI_ENABLED].val.s, "yes")) {
        return 0;
    }

    fni_log(0, "ASMI: Enabling FNI\n");

    // query whether FNI is enabled and query channel number
    if (PP_SUCCED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_FNI_STATUS, NULL, &ret, &err, NULL))) {
        fni_chan = ret.data.oem_intel_fni_status.chan;
    } else {
	fni_log(0, "ASMI: FNI failed to query FNI channel number (%s)\n", pp_ipmi_get_error_string(err));
	set_response(wp, ERIC_RESPONSE_ERROR, _("Failed to enable IPMI Channel 3 Forwarding."));
	pp_ipmi_cleanup_ret(&ret);
	goto bail;
    }
    pp_ipmi_cleanup_ret(&ret);
    
    fni_log(0, "ASMI: FNI Enabled, channel=%d\n", fni_chan);

    // do the SOL config
    webs_var = websGetVar(wp, "__ipmi_fni_sol_enabled_check", "");
    sol_enabled = !strcmp(webs_var, "yes");
    err = 0;
    bzero(&param, sizeof(param));
    param.data.sol_set_enable.chan = fni_chan;
    param.data.sol_set_enable.enabled = sol_enabled;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_SOL, PP_IPMI_SOL_SUBCMD_SET_ENABLE, &param, &ret, &err, NULL))) {
	fni_log(0, "ASMI: FNI failed to set SOL enabled to %d (%s)\n", sol_enabled, pp_ipmi_get_error_string(err));
	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("Failed to configure IPMI Channel 3 Forwarding."));
	pp_ipmi_cleanup_ret(&ret);
	goto bail;
    } else {
	fni_log(0, "ASMI: FNI set SOL enabled to %d\n", sol_enabled);
    }
    pp_ipmi_cleanup_ret(&ret);
    
    // do the anonymous user configuration
    webs_var = websGetVar(wp, "__ipmi_fni_anon_enabled_check", "");
    anon_enabled = !strcmp(webs_var, "yes");
    err = 0;
    bzero(&param, sizeof(param));
    param.data.channel_setaccess.channel = fni_chan;
    param.data.channel_setaccess.user_id = 1;
    param.data.channel_setaccess.access.privilege = anon_enabled ? PP_IPMI_PRIV_ADMIN : PP_IPMI_PRIV_NO_ACCESS;
    if (anon_enabled) {
        param.data.channel_setaccess.access.callin = -1;
        param.data.channel_setaccess.access.link = 1;
        param.data.channel_setaccess.access.ipmi = 1;
    }
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_CHANNEL, PP_IPMI_CHANNEL_SUBCMD_SET_ACCESS, &param, &ret, &err, NULL))) {
	fni_log(0, "ASMI: FNI failed to set user access to %d (%s)\n", param.data.channel_setaccess.access.privilege, pp_ipmi_get_error_string(err));
	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("Failed to configure IPMI Channel 3 Forwarding."));
	pp_ipmi_cleanup_ret(&ret);
	goto bail;
    } else {
	fni_log(0, "ASMI: FNI set anonymous user access to %d\n", param.data.channel_setaccess.access.privilege);
    }
    pp_ipmi_cleanup_ret(&ret);
    
    // do the SOL config
    webs_var = websGetVar(wp, "__ipmi_fni_anon_sol_enabled_check", "");
    anon_sol_enabled = !strcmp(webs_var, "yes");
    err = 0;
    bzero(&param, sizeof(param));
    param.data.sol_set_user_enable.chan = fni_chan;
    param.data.sol_set_user_enable.uid = 1;
    param.data.sol_set_user_enable.enabled = anon_sol_enabled;
    if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_SOL, PP_IPMI_SOL_SUBCMD_SET_USER_ENABLE, &param, &ret, &err, NULL))) {
	fni_log(0, "ASMI: FNI failed to set SOL for anonymous user enabled to %d (%s)\n", anon_sol_enabled, pp_ipmi_get_error_string(err));
	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("Failed to configure IPMI Channel 3 Forwarding."));
	pp_ipmi_cleanup_ret(&ret);
	goto bail;
    } else {
	fni_log(0, "ASMI: FNI set SOL for anonymous user enabled to %d\n", anon_sol_enabled);
    }
    pp_ipmi_cleanup_ret(&ret);
    
    // do the channel auth configuration
    bzero(&param, sizeof(param));
    param.data.lanp_set_auth_types.chan = fni_chan;

    webs_var = websGetVar(wp, "__ipmi_fni_admin_none_auth_check", "");
    param.data.lanp_set_auth_types.admin_auth.none = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_admin_password_auth_check", "");
    param.data.lanp_set_auth_types.admin_auth.password = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_admin_md2_auth_check", "");
    param.data.lanp_set_auth_types.admin_auth.md2 = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_admin_md5_auth_check", "");
    param.data.lanp_set_auth_types.admin_auth.md5 = !strcmp(webs_var, "yes");

    webs_var = websGetVar(wp, "__ipmi_fni_operator_none_auth_check", "");
    param.data.lanp_set_auth_types.operator_auth.none = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_operator_password_auth_check", "");
    param.data.lanp_set_auth_types.operator_auth.password = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_operator_md2_auth_check", "");
    param.data.lanp_set_auth_types.operator_auth.md2 = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_operator_md5_auth_check", "");
    param.data.lanp_set_auth_types.operator_auth.md5 = !strcmp(webs_var, "yes");

    webs_var = websGetVar(wp, "__ipmi_fni_user_none_auth_check", "");
    param.data.lanp_set_auth_types.user_auth.none = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_user_password_auth_check", "");
    param.data.lanp_set_auth_types.user_auth.password = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_user_md2_auth_check", "");
    param.data.lanp_set_auth_types.user_auth.md2 = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_user_md5_auth_check", "");
    param.data.lanp_set_auth_types.user_auth.md5 = !strcmp(webs_var, "yes");

    webs_var = websGetVar(wp, "__ipmi_fni_callback_none_auth_check", "");
    param.data.lanp_set_auth_types.callback_auth.none = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_callback_password_auth_check", "");
    param.data.lanp_set_auth_types.callback_auth.password = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_callback_md2_auth_check", "");
    param.data.lanp_set_auth_types.callback_auth.md2 = !strcmp(webs_var, "yes");
    webs_var = websGetVar(wp, "__ipmi_fni_callback_md5_auth_check", "");
    param.data.lanp_set_auth_types.callback_auth.md5 = !strcmp(webs_var, "yes");

    err = 0;
    if(PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, PP_IPMI_LANP_SUBCMD_SET_AUTH_TYPES, &param, &ret, &err, NULL))) {
	fni_log(0, "ASMI: FNI failed to set channel auth types (%s)\n", pp_ipmi_get_error_string(err));
	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("Failed to configure IPMI Channel 3 Forwarding."));
	pp_ipmi_cleanup_ret(&ret);
	goto bail;
    } else {
	fni_log(0, "ASMI: FNI set channel auth types\n");
    }
    pp_ipmi_cleanup_ret(&ret);

    retval = 0;

bail:
    return retval;
}

static int ipmi_fni_get_vars_asp(int eid UNUSED, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    pp_ipmi_parameter_t param;
    pp_ipmi_return_t ret;
    int err;
    int fni_chan = 0;
    int fni_enabled;

    bzero(&param, sizeof(param));
    bzero(&ret, sizeof(ret));

    // query whether FNI is enabled and query channel number
    if (PP_SUCCED(pp_ipmi_send_command(PP_IPMI_CMD_OEM_INTEL_TPT, PP_IPMI_OEM_INTEL_FNI_STATUS, NULL, &ret, &err, NULL))) {
        fni_enabled = ret.data.oem_intel_fni_status.status;
        fni_chan = ret.data.oem_intel_fni_status.chan;
    } else {
	fni_log(0, "ASMI: FNI failed to query current FNI state (%s)\n", pp_ipmi_get_error_string(err));
	websSetVar(wp, "__ipmi_fni_error", "1");
	pp_ipmi_cleanup_ret(&ret);
	return 0;
    }
    pp_ipmi_cleanup_ret(&ret);
    
    fni_log(0, "ASMI: FNI %sabled, channel=%d\n", fni_enabled ? "en" : "dis", fni_chan);

    // query whether SOL is enabled
    bzero(&param, sizeof(param));
    param.data.sol_get_enable.chan = fni_chan;
    if (PP_SUCCED(pp_ipmi_send_command(PP_IPMI_CMD_SOL, PP_IPMI_SOL_SUBCMD_GET_ENABLE, &param, &ret, &err, NULL))) {
        websSetVar(wp, "__ipmi_fni_sol_enabled", ret.data.sol_get_enable.enabled ? "1" : "0");
    } else {
	fni_log(0, "ASMI: FNI failed to query SOL enabled state (%s)\n", pp_ipmi_get_error_string(err));
    }
    pp_ipmi_cleanup_ret(&ret);

    // query whether the anonymous user is enabled
    bzero(&param, sizeof(param));
    param.data.channel_getaccess.channel = fni_chan;
    param.data.channel_getaccess.user_id = 1;
    if (PP_SUCCED(pp_ipmi_send_command(PP_IPMI_CMD_CHANNEL, PP_IPMI_CHANNEL_SUBCMD_GET_ACCESS, &param, &ret, &err, NULL))) {
        if (ret.data.channel_getaccess.privilege >= PP_IPMI_PRIV_USER) {
            websSetVar(wp, "__ipmi_fni_anon_enabled", "1");
        } else {
            websSetVar(wp, "__ipmi_fni_anon_enabled", "0");
        } 
    } else {
	fni_log(0, "ASMI: FNI failed to query anonymous user state (%s)\n", pp_ipmi_get_error_string(err));
    }
    pp_ipmi_cleanup_ret(&ret);
    
    // query whether SOL for anonymous user is enabled
    bzero(&param, sizeof(param));
    param.data.sol_get_user_enable.chan = fni_chan;
    param.data.sol_get_user_enable.uid = 1;
    if (PP_SUCCED(pp_ipmi_send_command(PP_IPMI_CMD_SOL, PP_IPMI_SOL_SUBCMD_GET_USER_ENABLE, &param, &ret, &err, NULL))) {
        websSetVar(wp, "__ipmi_fni_anon_sol_enabled", ret.data.sol_get_user_enable.enabled ? "1" : "0");
    } else {
	fni_log(0, "ASMI: FNI failed to query SOL enabled for anonymous user state (%s)\n", pp_ipmi_get_error_string(err));
    }
    pp_ipmi_cleanup_ret(&ret);

    // query the channel auth settings
    bzero(&param, sizeof(param));
    param.data.lanp_get.chan = fni_chan;
    param.data.lanp_get.need_auth_types = 1;
    if (PP_SUCCED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, PP_IPMI_LANP_SUBCMD_GET, &param, &ret, &err, NULL))) {
        websSetVar(wp, "__ipmi_fni_admin_none_auth", ret.data.lanp_get.auth_type_enables.admin.none ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_admin_md2_auth", ret.data.lanp_get.auth_type_enables.admin.md2 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_admin_md5_auth", ret.data.lanp_get.auth_type_enables.admin.md5 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_admin_password_auth", ret.data.lanp_get.auth_type_enables.admin.password ? "1" : "0");

        websSetVar(wp, "__ipmi_fni_operator_none_auth", ret.data.lanp_get.auth_type_enables._operator.none ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_operator_md2_auth", ret.data.lanp_get.auth_type_enables._operator.md2 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_operator_md5_auth", ret.data.lanp_get.auth_type_enables._operator.md5 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_operator_password_auth", ret.data.lanp_get.auth_type_enables._operator.password ? "1" : "0");

        websSetVar(wp, "__ipmi_fni_user_none_auth", ret.data.lanp_get.auth_type_enables.user.none ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_user_md2_auth", ret.data.lanp_get.auth_type_enables.user.md2 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_user_md5_auth", ret.data.lanp_get.auth_type_enables.user.md5 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_user_password_auth", ret.data.lanp_get.auth_type_enables.user.password ? "1" : "0");

        websSetVar(wp, "__ipmi_fni_callback_none_auth", ret.data.lanp_get.auth_type_enables.callback.none ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_callback_md2_auth", ret.data.lanp_get.auth_type_enables.callback.md2 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_callback_md5_auth", ret.data.lanp_get.auth_type_enables.callback.md5 ? "1" : "0");
        websSetVar(wp, "__ipmi_fni_callback_password_auth", ret.data.lanp_get.auth_type_enables.callback.password ? "1" : "0");
    } else {
	fni_log(0, "ASMI: FNI failed to query LAN parameters for channel %d (%s)\n", fni_chan, pp_ipmi_get_error_string(err));
    }
    pp_ipmi_cleanup_ret(&ret);
    
    return 0;
}

#endif /* PP_FEAT_ESB2_FNI */
