#include <liberic_config.h>
#include <pp/intl.h>
#include <pp/ipmi.h>
#include "ej.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "eric_validate.h"
#include "templates.h"

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND)

#define NUMBERS_OF_ENTRIES_PER_PAGE 15

extern pp_ipmi_return_t sdr_list;
extern pthread_mutex_t sdr_list_mutex;

#endif /* PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND */

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND)

static pthread_t sel_cache_thread;
static pthread_t sel_cache_cleanup_thread;

static void * sel_cache_thread_func(void *arg);
static void * sel_cache_cleanup_thread_func(void *arg);

static void ipmi_sel_cache_invalidate(int force_availability);
static int ipmi_sel_cache_update(void);

static volatile int sel_list_uptodate = 0;
static volatile int sel_available = 1;
static vector_t * sel_list;
static pthread_mutex_t sel_list_mutex = PTHREAD_MUTEX_INITIALIZER;

static volatile unsigned int interval;
static unsigned int get_interval(void);
static int interval_ch(pp_cfg_chg_ctx_t *ctx);

#endif /* PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND */

FV_SPEC = {
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int ipmi_sel_build_table_asp(int eid, webs_t wp, int argc, char **argv);

int ipmi_sel_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("ipmiGetSelTable", ipmi_sel_build_table_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_IPMI_SEL, ACL_OBJ_IPMI_STATUS);
    fh->pre_validate_hook = pre_validate_hook;

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND)
    interval = get_interval();
    pp_cfg_add_change_listener(interval_ch, "ipmi.sel_poll_interval");
#endif /* PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND */

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

void eric_webs_ipmi_sel_list_start(void)
{
#ifdef PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND
    pthread_create(&sel_cache_thread, NULL, sel_cache_thread_func, NULL);
    pthread_create(&sel_cache_cleanup_thread, NULL, sel_cache_cleanup_thread_func, NULL);
#endif
}

/* the formular handler (parses the buttons) */
static int pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED)
{
    if (form_button_clicked(wp, "action_clear_log")) {
        if (ipmi_sel_clear_log(wp)) {
            set_response(wp, ERIC_RESPONSE_ERROR, _("Could not clear System Event Log!"));
        } else {
#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND)
            ipmi_sel_cache_invalidate(0);
#endif
            set_response(wp, ERIC_RESPONSE_OK, _("Successfully cleared System Event Log."));
        }
#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND)
    } else if (form_button_clicked(wp, "action_invalidate")) {
        ipmi_sel_cache_invalidate(1);
        set_response(wp, ERIC_RESPONSE_OK, _("System Event Log Cache invalidated."));
    } else if (form_button_clicked(wp, "action_switch_page")) {
	int page = 0;
	const char *page_txt = websGetVar(wp, "_page", "0");
	if (sscanf(page_txt, "%d", &page) < 1) page = 0;

        wp->sd->sel_list_page = page;

	set_response(wp, ERIC_RESPONSE_OK, "");
#endif /* PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND */
    } else if (form_button_clicked(wp, "action_refresh")) {
        /* dont show "Operation Completed" */
        set_response(wp, ERIC_RESPONSE_OK, "");
    }

    return 0;
}

/* backend for the buttons */
int ipmi_sel_clear_log(webs_t wp) {
    pp_ipmi_return_t ipmi_ret;
    int ret;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    pp_log("Clearing IPMI Event Log.\n");
    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_CLEAR, NULL, &ipmi_ret, NULL, wp->user);

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return ret;
}

/* ASP calls */
static char* get_string(char *s) {
    static char nbsp[] = "&nbsp;";
    if (s) pp_trim_string(s);
    return (s && s[0]) ? s : nbsp;
}

static void add_sel_table_entry(pp_strstream_t *sel_table, int idx, int rec_id, char *type, char *date,
                                char *mytime, char *source, char *description, char* direction) {
    (void)rec_id;
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(sel_table,
            idx % 2 ? "<tr class=\"loggingRowOdd\">\n" : "<tr class=\"loggingRowEven\">\n");

    pp_strappend(sel_table, "  <td>");
#if 0
    pp_strappendf(sel_table, "%d - %s", rec_id, get_string(type));
#else
    pp_strappend(sel_table, get_string(type));
#endif
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(date));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(mytime));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(source));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(description));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(direction));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "</tr>\n");
#else /* PP_FEAT_RARITAN_DESIGN */
    (void)idx;
    pp_strappend(sel_table, "<script>table_entry('");
    pp_strappend(sel_table, get_string(type));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(date));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(mytime));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(source));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(description));
    
    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(direction));
    pp_strappend(sel_table, "');</script>\n");
#endif /* PP_FEAT_RARITAN_DESIGN */
}

static void add_empty_table_entry(pp_strstream_t *sel_table, const char *str) {
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(sel_table, "<tr><td colspan=\"6\" class=\"normal\">");
    pp_strappend(sel_table, str);
    pp_strappend(sel_table, "</td></tr>\n");
#else
    pp_strappend(sel_table, "<script>table_entry('");
    pp_strappend(sel_table, str);
    pp_strappend(sel_table, "','','','','','');</script>\n");
#endif    
}

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND)

static int interval_ch(pp_cfg_chg_ctx_t *ctx UNUSED) {
    interval = get_interval();
    return PP_SUC;
}

static unsigned int get_interval(void) {
    unsigned int _interval = 0;
    pp_cfg_get_uint(&_interval, "ipmi.sel_poll_interval");
    return _interval;
}

/* invalidates the SEL cache - clears all IDs */
static void ipmi_sel_cache_invalidate(int force_availability) {
    MUTEX_LOCK(&sel_list_mutex);
    vector_delete(sel_list);
    sel_list = vector_new(NULL, 0, NULL);
    sel_list_uptodate = 0;
    if (force_availability) {
        sel_available = 1;
    }
    MUTEX_UNLOCK(&sel_list_mutex);
}

/* update the SEL cache - appends SEL entry IDs to the list of known IDs */
static int ipmi_sel_cache_update(void) {
    int start_from, count = 20;
    size_t size;
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    int ret, retried = 0;
    
    MUTEX_LOCK(&sel_list_mutex);
    if ((size = vector_size(sel_list)) == 0) {
        start_from = 0;
    } else {
        start_from = vector_get_u_int(sel_list, size - 1);
    }
    MUTEX_UNLOCK(&sel_list_mutex);
    
    while (1) {
        int finished = 0;

	// skip further SEL updates if the cache has been disabled
        if (interval == 0) {
            //pp_log("SEL cache currently disabled, break polling the SEL\n");
            return 0;
        }
        
 retry:
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        memset(&ipmi_param, 0, sizeof(ipmi_param));
        
        ipmi_param.data.sel_list_ids.start_from = start_from;
        ipmi_param.data.sel_list_ids.count = count;
        
        if ((ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_LIST_IDS, &ipmi_param, &ipmi_ret, NULL, NULL))) {
            pp_ipmi_cleanup_ret(&ipmi_ret);
            if (!retried) {
                retried = 1;
                pp_log("Could not query SEL entries, retrying.\n");
                goto retry;
            } else {
                pp_log("Could not query SEL entries! Restarting SEL cache enumeration!\n");
                ipmi_sel_cache_invalidate(0);
                return -1;
            }
        }
        
        MUTEX_LOCK(&sel_list_mutex);

        // if we didn't get any more new entries from IPMI, we're done
        if ((size = vector_size(ipmi_ret.data.sel_list_ids)) == 0) {
            finished = 1;
        } else {
            // copy only entries which we don't already have
            // (the first one is always a duplicate, others may be)
            unsigned int i, j;
            unsigned int added = 0;
            size_t size2 = vector_size(sel_list);

            for (i = 0; i < size; i++) {
                unsigned int id = vector_get_u_int(ipmi_ret.data.sel_list_ids, i);
                int got_it = 0;
                for (j = 0; j < (size2 = vector_size(sel_list)); j++) {
                    if (vector_get_u_int(sel_list, j) == id) {
                        got_it = 1;
                        break;
                    }
                }
                if (!got_it) {
                    vector_add_u_int(sel_list, id);
                    // prepare next round
                    start_from = id;
                    added++;
                }
            }
            
            // if we didn't copy any new ones, we're done, too
            if (added == 0) {
                finished = 1;
            }
        }
        
        if (finished) {
            sel_list_uptodate = 1;
        }

        MUTEX_UNLOCK(&sel_list_mutex);

        pp_ipmi_cleanup_ret(&ipmi_ret);
        
        if (finished) {
            return 0;
        }
        
        sleep(1);
    }
}

static void * sel_cache_thread_func(void *arg UNUSED) {
    // if cache is disabled (this is used to reduce IPMB traffic), don't start reading
    while (interval == 0) {
        //pp_log("SEL cache currently disabled, won't start polling the SEL\n");
        sleep(1);
    }

    // first, we query the complete sel IDs
    sel_list = vector_new(NULL, 0, NULL);
    if (ipmi_sel_cache_update()) {
        sleep(30);

        // if cache is disabled (this is used to reduce IPMB traffic), don't start reading
        while (interval == 0) {
            pp_log("SEL cache currently disabled, won't start polling the SEL\n");
            sleep(1);
        }

        if (ipmi_sel_cache_update()) {
            pp_log("SEL not available.\n");
            sel_available = 0;
        }
    }
    
    // once we got them, we query for updates each <interval> seconds
    while (1) {
        // if cache is disabled (this is used to reduce IPMB traffic), don't continue reading
        if (interval == 0) {
            //pp_log("SEL cache currently disabled, won't continue polling the SEL\n");
            sleep(1);
            continue;
        }

        sleep(interval);
        if (sel_available) ipmi_sel_cache_update();
    }
    
    return NULL;
}

/* checks whether one SEL entry exists */
static int check_sel_id(unsigned int id) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    int ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    
    ipmi_param.data.sel_get_list.id_list = vector_new(NULL, 0, NULL);
    vector_add_u_int(ipmi_param.data.sel_get_list.id_list, id);

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND)
    MUTEX_LOCK(&sdr_list_mutex);
    ipmi_param.data.sel_get_list.sdr_raw_list = pp_ipmi_clone_sdr_list_raw(&sdr_list);
    MUTEX_UNLOCK(&sdr_list_mutex);
#endif
    
    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_GET, &ipmi_param, &ipmi_ret, NULL, NULL);
    
    pp_ipmi_cleanup_ret(&ipmi_ret);
    if (ipmi_param.data.sel_get_list.id_list) vector_delete(ipmi_param.data.sel_get_list.id_list);
    if (ipmi_param.data.sel_get_list.sdr_raw_list) vector_delete(ipmi_param.data.sel_get_list.sdr_raw_list);

    return ret;
}

/* checks the whole SEL cache whether the entries are valid and removes invalid entries */
static void * sel_cache_cleanup_thread_func(void *arg UNUSED) {
    unsigned int checked_id_index = 0;
    unsigned int checked_id = 0;
    
    while (1) {
        // if cache is disabled (this is used to reduce IPMB traffic), don't start reading
        while (interval == 0) {
            pp_log("SEL cache currently disabled, won't cleanup the SEL\n");
            sleep(1);
        }

        MUTEX_LOCK(&sel_list_mutex);
        if (!sel_list_uptodate || !sel_available) {
            MUTEX_UNLOCK(&sel_list_mutex);
            sleep(10);
            continue;
        }
        
        size_t size = vector_size(sel_list);
        if (size == 0) {
            MUTEX_UNLOCK(&sel_list_mutex);
            sleep(10);
            continue;
        }

        // if we are through the list, start over again
        if (checked_id_index >= size) checked_id_index = 0;
        checked_id = vector_get_u_int(sel_list, checked_id_index);
        
        MUTEX_UNLOCK(&sel_list_mutex);
        
        // check whether an SEL element with this ID exists - we better do this twice ;-)
        if (check_sel_id(checked_id) && check_sel_id(checked_id)) {
            unsigned int i;

            MUTEX_UNLOCK(&sel_list_mutex);
            size = vector_size(sel_list);
            for (i = 0; i < size; i++) {
                if (vector_get_u_int(sel_list, i) == checked_id) {
                    vector_remove(sel_list, i);
                    break;
                }
            }
            MUTEX_UNLOCK(&sel_list_mutex);
        }

        checked_id_index++;
        sleep(1);
    }
    return NULL;
}

static void set_page_count_vars(webs_t wp, int page_count, int curr_page) {
    char page_count_val[32];
    snprintf(page_count_val, sizeof(page_count_val), "%d", page_count);
    websSetVar(wp, "sel_page_count", page_count_val);

    char page_val[32];
    snprintf(page_val, sizeof(page_val), "%d", curr_page);
    websSetVar(wp, "sel_current_page", page_val);
}

static int ipmi_sel_build_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    char empty_str[1] = { 0 };
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    
    int idx;
    unsigned int i;
    size_t sel_list_size;

    if (!form_was_submitted(wp)) {
	wp->sd->sel_list_page = 0;
    }
    
    pp_strstream_t sel_table = PP_STRSTREAM_INITIALIZER;

    if (interval == 0) {
        add_empty_table_entry(&sel_table,
            _("The SEL Polling is currently disabled. Please enable it in the <a href=\"ipmi.asp\">IPMI Settings."));
        set_page_count_vars(wp, 1, 0);
        goto bail;
    }

    MUTEX_LOCK(&sel_list_mutex);
    if (!sel_list_uptodate || !sel_available) {
        add_empty_table_entry(&sel_table,
            !sel_available ?
            _("System Event Log not available.") :
            _("Event Log Cache not filled yet. Please try again later."));
        MUTEX_UNLOCK(&sel_list_mutex);
        set_page_count_vars(wp, 1, 0);
        goto bail;
    }
    
    idx = (sel_list_size = vector_size(sel_list)) - NUMBERS_OF_ENTRIES_PER_PAGE * wp->sd->sel_list_page - 1;
    if (idx < 0) idx = 0;
   
    pp_strstream_init(&sel_table);
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));

    ipmi_param.data.sel_get_list.id_list = vector_new(NULL, 0, NULL);
    for (i = 0; i < NUMBERS_OF_ENTRIES_PER_PAGE && idx >= 0; i++, idx--) {
        vector_add_u_int(ipmi_param.data.sel_get_list.id_list, vector_get_u_int(sel_list, idx));
    }
    MUTEX_UNLOCK(&sel_list_mutex);

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND)
    MUTEX_LOCK(&sdr_list_mutex);
    ipmi_param.data.sel_get_list.sdr_raw_list = pp_ipmi_clone_sdr_list_raw(&sdr_list);
    MUTEX_UNLOCK(&sdr_list_mutex);
#endif

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_GET, &ipmi_param, &ipmi_ret, NULL, wp->user)) {
        unsigned int sel_count = vector_size(ipmi_ret.data.sel_list);

        pp_log("found %d SEL entries.\n", sel_count);

        if (sel_count > 0) {
            char * s_preinit = strdup(_("Pre-Init"));
            char * s_unknown = strdup(_("Unknown"));
            for (i = 0; i < sel_count; i++) {
                pp_ipmi_sel_list_entry_t *entry = vector_get(ipmi_ret.data.sel_list, i);

                if (!entry) {
                    pp_log("Error: could not read entry %d\n", i);
                    continue;
                }

                add_sel_table_entry(&sel_table, sel_count - 1 - i,
                    entry->record_id, entry->record_type_string.buf,
                    entry->timestamp.is_valid ? (entry->timestamp.is_pre_init ? s_preinit : entry->timestamp.string_date.buf) : empty_str,
                    entry->timestamp.is_valid ? entry->timestamp.string_time.buf : empty_str,
                    strcmp(entry->sensor_name.buf, s_unknown) ? entry->sensor_name.buf : entry->sensor_type.buf,
                    entry->description.buf,
                    entry->event_direction.buf);
            }
            if (s_preinit) free(s_preinit);
            if (s_unknown) free(s_unknown);
        } else {
            add_empty_table_entry(&sel_table, _("Event log has no entries."));
        }
    } else {
        add_empty_table_entry(&sel_table, _("Error: Could not query event log."));
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    if (ipmi_param.data.sel_get_list.id_list) {
        vector_delete(ipmi_param.data.sel_get_list.id_list);
    }
    if (ipmi_param.data.sel_get_list.sdr_raw_list) {
        vector_delete(ipmi_param.data.sel_get_list.sdr_raw_list);
    }

    set_page_count_vars(wp, (sel_list_size / NUMBERS_OF_ENTRIES_PER_PAGE) + 1, wp->sd->sel_list_page);

bail:
    ejSetResult(eid, pp_strstream_buf_and_free(&sel_table));

    return 0;
}

#else /* PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND */

static int ipmi_sel_build_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    char empty_str[1] = { 0 };
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;

    pp_strstream_t sel_table = PP_STRSTREAM_INITIALIZER;

    pp_strstream_init(&sel_table);
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));

    ipmi_param.data.sel_list.range = PP_IPMI_SEL_LIST_RANGE_ALL;

#if defined(PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND)
    MUTEX_LOCK(&sdr_list_mutex);
    ipmi_param.data.sel_list.sdr_raw_list = pp_ipmi_clone_sdr_list_raw(&sdr_list);
    MUTEX_UNLOCK(&sdr_list_mutex);
#endif

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_LIST, &ipmi_param, &ipmi_ret, NULL, wp->user)) {
        int i;
        int sel_count = vector_size(ipmi_ret.data.sel_list);

        pp_log("found %d SEL entries.\n", sel_count);

        if (sel_count > 0) {
            char * s_preinit = strdup(_("Pre-Init"));
            char * s_unknown = strdup(_("Unknown"));
            for (i = sel_count - 1; i >= 0; i--) {
                pp_ipmi_sel_list_entry_t *entry = vector_get(ipmi_ret.data.sel_list, i);

                if (!entry) {
                    pp_log("Error: could not read entry %d\n", i);
                    continue;
                }

                add_sel_table_entry(&sel_table, sel_count - 1 - i,
                    entry->record_id, entry->record_type_string.buf,
                    entry->timestamp.is_valid ? (entry->timestamp.is_pre_init ? s_preinit : entry->timestamp.string_date.buf) : empty_str,
                    entry->timestamp.is_valid ? entry->timestamp.string_time.buf : empty_str,
                    strcmp(entry->sensor_name.buf, s_unknown) ? entry->sensor_name.buf : entry->sensor_type.buf,
                    entry->description.buf,
                    entry->event_direction.buf);
            }
            if (s_preinit) free(s_preinit);
            if (s_unknown) free(s_unknown);
        } else {
            add_empty_table_entry(&sel_table, _("Event log has no entries."));
        }
    } else {
        add_empty_table_entry(&sel_table, _("Error: Could not query event log."));
    }

    ejSetResult(eid, pp_strstream_buf_and_free(&sel_table));

    pp_ipmi_cleanup_ret(&ipmi_ret);
    if (ipmi_param.data.sel_list.sdr_raw_list) {
        vector_delete(ipmi_param.data.sel_list.sdr_raw_list);
    }
    return 0;
}

#endif /* PP_FEAT_IPMI_CLIENT_QUERY_SEL_IN_BACKGROUND */
