#include <liberic_config.h>
#include <pp/intl.h>
#include <pp/ipmi.h>
#include "ej.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "eric_validate.h"

FV_SPEC = {
    /* we have no form vars (yet) */
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int ipmi_sensors_build_table_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED);

#ifdef PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND

static int ipmi_sdr_list_update(void);
static void * ipmi_sdr_list_thread(void *arg);

pp_ipmi_return_t sdr_list;
pthread_mutex_t sdr_list_mutex = PTHREAD_MUTEX_INITIALIZER;

static pthread_t sensor_thread;

static volatile unsigned int interval;
static unsigned int get_interval(void);
static int interval_ch(pp_cfg_chg_ctx_t *ctx);

#endif

int ipmi_sensors_tmpl_init(void)
{
    form_handler_t * fh;

#ifdef PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND
    memset(&sdr_list, 0, sizeof(sdr_list));
    interval = get_interval();
    pp_cfg_add_change_listener(interval_ch, "ipmi.sensor_poll_interval");
#endif

    /* register ASPs */
    websAspDefine("ipmiGetSensorTable", ipmi_sensors_build_table_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_IPMI_SENSORS, ACL_OBJ_IPMI_STATUS);
    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

void eric_webs_ipmi_sdr_list_start(void)
{
#ifdef PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND
    pthread_create(&sensor_thread, NULL, ipmi_sdr_list_thread, NULL);
#endif
}

/* the formular handler (parses the buttons) */
static int pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED)
{
    if (form_button_clicked(wp, "action_refresh")) {
            /* dont show "Operation Completed" */
            set_response(wp, ERIC_RESPONSE_OK, "");
    }

    return 0;
}

/* ASP calls */
static const char* fixup_string(char *s) {
    pp_trim_string(s);
    if (!s || !s[0]) {
        return "&nbsp;";
    }
    return s;
}

#define TABLE_BG_WHITE  0
#define TABLE_BG_RED    1
#define TABLE_BG_GREEN  2
#define TABLE_BG_YELLOW 3

static void add_sensor_table_entry(pp_strstream_t *sensor_table, int idx, int color,
                                   char *type, char *name, char *status, char *reading) {
#if !defined(PP_FEAT_RARITAN_DESIGN)
    const char *colors[] = { "#FFFFFF", "#FF0000", "#00FF00", "#FFFF00" };
    pp_strappend(sensor_table,
            idx % 2 ? "<tr class=\"loggingRowOdd\">\n" : "<tr class=\"loggingRowEven\">\n");

    if (color != TABLE_BG_WHITE) {
        pp_strappend(sensor_table, "  <td style=\"background-color:");
        pp_strappend(sensor_table, colors[color]);
        pp_strappend(sensor_table, "\">&nbsp;&nbsp;&nbsp;</td>\n");
    } else {
        pp_strappend(sensor_table, "<td>&nbsp;&nbsp;&nbsp;</td>\n");
    }

    pp_strappend(sensor_table, "  <td>");
    pp_strappend(sensor_table, fixup_string(type));
    pp_strappend(sensor_table, "</td>\n");

    pp_strappend(sensor_table, "  <td>");
    pp_strappend(sensor_table, fixup_string(name));
    pp_strappend(sensor_table, "</td>\n");

    pp_strappend(sensor_table, "  <td>");
    pp_strappend(sensor_table, fixup_string(status));
    pp_strappend(sensor_table, "</td>\n");

    pp_strappend(sensor_table, "  <td>");
    pp_strappend(sensor_table, fixup_string(reading));
    pp_strappend(sensor_table, "</td>\n");

    pp_strappend(sensor_table, "</tr>\n");
#else /* PP_FEAT_RARITAN_DESIGN */
    (void)idx;
    (void)color;
    pp_strappend(sensor_table, "<script>table_entry('");
    pp_strappend(sensor_table, fixup_string(type));

    pp_strappend(sensor_table, "','");
    pp_strappend(sensor_table, fixup_string(name));

    pp_strappend(sensor_table, "','");
    pp_strappend(sensor_table, fixup_string(status));

    pp_strappend(sensor_table, "','");
    pp_strappend(sensor_table, fixup_string(reading));
    pp_strappend(sensor_table, "');</script>\n");
#endif /* PP_FEAT_RARITAN_DESIGN */
}

static void add_empty_table_entry(pp_strstream_t *sensor_table, const char *str) {
    pp_strappend(sensor_table, "\n");
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(sensor_table, "<tr><td colspan=\"5\" class=\"normal\">");
    pp_strappend(sensor_table, str);
    pp_strappend(sensor_table, "</td></tr>\n");
#else /* PP_FEAT_RARITAN_DESIGN */
    pp_strappend(sensor_table, "<script>table_entry('");
    pp_strappend(sensor_table, str);
    pp_strappend(sensor_table, "','','','');</script>\n");
#endif /* PP_FEAT_RARITAN_DESIGN */
}

static int get_color_from_status(pp_ipmi_sdr_list_entry_t *entry) {
    switch (entry->data.full.status) {
        case PP_IPMI_SENSOR_STATUS_NO_READING:
            return TABLE_BG_WHITE;
        case PP_IPMI_SENSOR_STATUS_OK:
            return TABLE_BG_GREEN;
        case PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL:
        case PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL:
            return TABLE_BG_YELLOW;
        default:
            return TABLE_BG_RED;
    }
}

#ifdef PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND

static int interval_ch(pp_cfg_chg_ctx_t *ctx UNUSED) {
    interval = get_interval();
    return PP_SUC;
}

static unsigned int get_interval(void) {
    unsigned int _interval = 0;
    pp_cfg_get_uint(&_interval, "ipmi.sensor_poll_interval");
    return _interval;
}

static int ipmi_sensors_build_table_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED) {
    pp_strstream_t sensor_table = PP_STRSTREAM_INITIALIZER;
    u_int i, j, sensor_cnt = 0;

    pp_strstream_init(&sensor_table);

    MUTEX_LOCK(&sdr_list_mutex);

    if (interval == 0) {
	add_empty_table_entry(&sensor_table, _("The Sensor Polling is currently disabled. Please enable it in the <a href=\"ipmi.asp\">IPMI Settings."));
	goto bail;
    }

    if (!sdr_list.data.sdr_list) {
	add_empty_table_entry(&sensor_table, _("No sensor data available yet."));
	goto bail;
    }

    sensor_cnt = vector_size(sdr_list.data.sdr_list);

    if(sensor_cnt == 0) {
	add_empty_table_entry(&sensor_table, _("No sensors found."));
	goto bail;
    }

    for (i = 0, j = 0; i < sensor_cnt; i++) {
	pp_ipmi_sdr_list_entry_t *entry = vector_get(sdr_list.data.sdr_list, i);
	int tolerance_valid = entry->data.full.tolerance.valid;
	float tolerance = entry->data.full.tolerance.value;
	float reading = entry->data.full.reading.value;
	int tolerance_is_0, tolerance_prec, reading_prec;
	int color = TABLE_BG_WHITE;
	char reading_str[50] = "";
	char *disp_reading_str = reading_str;
	char *status = NULL;

	if (!entry) {
	    pp_log("%s(): Error: could not read entry %u\n", ___F, i);
	    continue;
	}

	if (entry->type == PP_IPMI_SENSOR_TYPE_FULL) {
	    if (entry->data.full.reading_present == PP_IPMI_SENSOR_READING_OK) {
		tolerance_is_0 = (int)(tolerance * 1000 + 0.5) == 0;
		tolerance_prec = ((int)(tolerance * 1000 + 0.5) % 1000) ? 3 : 0;
		reading_prec = ((int)(reading * 1000 + 0.5) % 1000) ? 3 : 0; 
		color = get_color_from_status(entry);
		status = strdup(pp_strstream_buf(&entry->data.full.status_string));
		if (entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_ANALOG) {
		    if (!tolerance_valid || tolerance_is_0) {
			snprintf(reading_str, sizeof(reading_str), "%.*f %s\n",
			         reading_prec, reading,
				 entry->data.full.unit.buf ? : "");
		}    else {
			snprintf(reading_str, sizeof(reading_str), "%.*f (+/- %.*f) %s\n",
			         reading_prec, reading, tolerance_prec, tolerance,
				 entry->data.full.unit.buf ? : "");
		    }
		} // else: discrete sensors don't have a reading, just a status
	    } else if (entry->data.full.reading_present == PP_IPMI_SENSOR_READING_SENSOR_NOT_OWNED_BY_BMC) {
	        continue;
	    } else {		
		status = strdup(_("No reading"));
	    }
	} else if (entry->type == PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR) {
	    if (entry->data.compact.status == PP_IPMI_SENSOR_STATUS_NO_READING) {
		status = strdup(_("No reading"));
	    } else if (entry->data.compact.status == PP_IPMI_SENSOR_READING_SENSOR_NOT_OWNED_BY_BMC) {
	        continue;
	    } else if (!entry->data.compact.scanning_disabled) {
		status = pp_strdup_safe(entry->data.compact.description.buf);
	    }
	} else {
	    continue;
	}
	add_sensor_table_entry(&sensor_table, j++, color, entry->sdr_type_sub.buf,
			       entry->sdr_name.buf, status, disp_reading_str);
	free(status);
    }

 bail:
    ejSetResult(eid, sensor_table.buf);

    pp_strstream_free(&sensor_table);
    MUTEX_UNLOCK(&sdr_list_mutex);
    return 0;
}

static int ipmi_sdr_list_update()
{
    pp_ipmi_return_t ipmi_ret;
    int ret;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    ret=pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_LIST, NULL, &ipmi_ret, NULL, NULL);

    MUTEX_LOCK(&sdr_list_mutex);
    pp_ipmi_cleanup_ret(&sdr_list);
    sdr_list=ipmi_ret;
    MUTEX_UNLOCK(&sdr_list_mutex);

    return ret;
}

static void * ipmi_sdr_list_thread(void *arg UNUSED)
{
    size_t sdr_count;

    // if cache is disabled (this is used to reduce IPMB traffic), don't start reading
    while (interval == 0) {
        //pp_log("SDR cache currently disabled, won't start polling the SDR\n");
        sleep(1);
    }

    ipmi_sdr_list_update();

    sdr_count=vector_size(sdr_list.data.sdr_list);

    while (1) {
	unsigned int i;

        // if cache is disabled (this is used to reduce IPMB traffic), don't continue reading
        if (interval == 0) {
            //pp_log("SDR cache currently disabled, won't continue polling the SDR\n");
            sleep(1);
            continue;
        }

	sleep(interval);

	for(i = 0; i < sdr_count; i++) {
	    pp_ipmi_parameter_t param;
	    pp_ipmi_return_t ipmi_ret;
	    int ret,error;

	    memset(&ipmi_ret,0,sizeof(ipmi_ret));
	    memset(&param,0,sizeof(param));

	    MUTEX_LOCK(&sdr_list_mutex);
	    pp_ipmi_clone_sdr_list(&ipmi_ret,&sdr_list,i);
	    MUTEX_UNLOCK(&sdr_list_mutex);

	    ret=pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_UPDATE, &param, &ipmi_ret, &error, NULL);

	    if(ret)
		continue;

	    if(vector_size(ipmi_ret.data.sdr_list)>1)
	    {
		MUTEX_LOCK(&sdr_list_mutex);
		vector_set(sdr_list.data.sdr_list,i,vector_pop(ipmi_ret.data.sdr_list));
		MUTEX_UNLOCK(&sdr_list_mutex);
	    }

	    pp_ipmi_cleanup_ret(&ipmi_ret);
	    usleep(10000);
	    
	    // skip further sensor updates if the cache has been disabled
	    if (interval == 0) {
	        //pp_log("SDR cache currently disabled, break polling the SDR\n");
	        break;
	    }
	}
    }
}

#else /* PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND */

static int ipmi_sensors_build_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    pp_ipmi_return_t ipmi_ret;
    pp_strstream_t sensor_table = PP_STRSTREAM_INITIALIZER;
    u_int i, sensor_cnt = 0;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    pp_strstream_init(&sensor_table);

    if (pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_LIST,
			     NULL, &ipmi_ret, NULL, wp->user) != 0) {
        add_empty_table_entry(&sensor_table, _("Error: Could not query Sensor repository."));
	goto bail;
    }

    sensor_cnt = vector_size(ipmi_ret.data.sdr_list);
    //pp_log("found %d SDR entries.\n", sensor_cnt);

    if (sensor_cnt == 0) {
	add_empty_table_entry(&sensor_table, _("No sensors found."));
	goto bail;
    }

    for (i = 0; i < sensor_cnt; i++) {
	pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret.data.sdr_list, i);
	int tolerance_valid = entry->data.full.tolerance.valid;
	float tolerance = entry->data.full.tolerance.value;
	float reading = entry->data.full.reading.value;
	int tolerance_is_0, tolerance_prec, reading_prec;
	int color = TABLE_BG_WHITE;
	char reading_str[50] = "";
	char *disp_reading_str = reading_str;
	char *status = NULL;

	if (!entry) {
	    pp_log("%s(): Error: could not read entry %u\n", ___F, i);
	    continue;
	}

	if (entry->type == PP_IPMI_SENSOR_TYPE_FULL) {
	    if (entry->data.full.reading_present == PP_IPMI_SENSOR_READING_OK) {
		tolerance_is_0 = (int)(tolerance * 1000 + 0.5) == 0;
		tolerance_prec = ((int)(tolerance * 1000 + 0.5) % 1000) ? 3 : 0;
		reading_prec = ((int)(reading * 1000 + 0.5) % 1000) ? 3 : 0; 
		color = get_color_from_status(entry);
		status = strdup(pp_strstream_buf(&entry->data.full.status_string));
		if (entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_ANALOG) {
		    if (!tolerance_valid || tolerance_is_0) {
			snprintf(reading_str, sizeof(reading_str), "%.*f %s\n",
			         reading_prec, reading,
				 entry->data.full.unit.buf ? : "");
		}    else {
			snprintf(reading_str, sizeof(reading_str), "%.*f (+/- %.*f) %s\n",
			         reading_prec, reading, tolerance_prec, tolerance,
				 entry->data.full.unit.buf ? : "");
		    }
		} // else: discrete sensors don't have a reading, just a status
	    } else {		
		status = strdup(_("No reading"));
	    }
	} else if (entry->type == PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR) {
	    if (entry->data.compact.status == PP_IPMI_SENSOR_STATUS_NO_READING) {
		status = strdup(_("No reading"));
	    } else if (!entry->data.compact.scanning_disabled) {
		status = pp_strdup_safe(entry->data.compact.description.buf);
	    }
	} else {
	    continue;
	}
	add_sensor_table_entry(&sensor_table, i, color, entry->sdr_type_sub.buf,
			       entry->sdr_name.buf, status, disp_reading_str);
	free(status);
    }

 bail:
    ejSetResult(eid, sensor_table.buf);

    pp_strstream_free(&sensor_table);
    pp_ipmi_cleanup_ret(&ipmi_ret);
    return 0;
}

#endif /* PP_FEAT_IPMI_CLIENT_QUERY_SENSORS_IN_BACKGROUND */

