#include <pp/base.h>
#include <pp/powerswitch.h>
#include <liberic_config.h>
#include <liberic_misc.h>
#include <pp/kvm.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"

FV_SPEC = {
    {
	id:		FV_ID_KVM_KEY_PAUSE_DURATION,
	cfgkey:		"kvm.key_pause_duration"
    },
/* we have port index %I not target port %P here! */
    {
	id:		FV_ID_KVM_PORTNAME,
	cfgkey:		"unit[%N].port[%I]",
	elemkey:	"name"
    },
    {
	id:		FV_ID_KVM_HOTKEY,
	cfgkey:		"unit[%N].port[%I]",
	elemkey:	"hotkey.key"
    },
    {
	id:		FV_ID_KVM_HOTKEYCODE,
	cfgkey:		"unit[%N].port[%I]",
	elemkey:	"hotkey.code"
    },
    {
	id:		FV_ID_KVM_SHOW_IN_RC,
	cfgkey:		"unit[%N].port[%I]",
	elemkey:	"show_in_rc"
    },
    {
        /* this has to be the last entry to store correct number of ports! */
	id:		FV_ID_KVM_UNIT_PORT_COUNT,
	cfgkey:		"unit[%N].port._s_"
    },
};

static void kvm_update_port_count(webs_t wp, form_var_t *fv, u_int cnt);
static int kvm_port_backswitch(webs_t wp, u_int channel, u_int new_cnt);
static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);
static int kvm_init_form_data_asp(int eid, webs_t wp, int argc, char ** argv);

int
kvm_tmpl_init(void)
{
    form_handler_t * fh;

    if (kvm_common_tmpl_init() != 0) return -1;

    /* register ASPs */
    websAspDefine("kvmInitFormData", kvm_init_form_data_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_KVM, ACL_OBJ_KVM_S);

    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh)
{
    form_var_t * fv;
    u_int i;

    /* ------------------------------------------------------------------------
     * user updated the number of available ports
     */
    if (form_button_clicked(wp, "action_update_kvm_port")) {
	u_int new_cnt;

	/* don't save (but only on pressing "apply") */
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
	
	fv = &fh->fv[FV_ID_KVM_UNIT_PORT_COUNT];
	new_cnt = pp_strtoul_10(fv->val.s, 0, NULL);

	kvm_update_port_count(wp, &fh->fv[FV_ID_KVM_PORTNAME], new_cnt);
	kvm_update_port_count(wp, &fh->fv[FV_ID_KVM_HOTKEY], new_cnt);
	kvm_update_port_count(wp, &fh->fv[FV_ID_KVM_HOTKEYCODE], new_cnt);
	kvm_update_port_count(wp, &fh->fv[FV_ID_KVM_SHOW_IN_RC], new_cnt);

	return 0;
    }

#if !defined(PRODUCT_XX01IP_ANY)
    /* ------------------------------------------------------------------------
     * fill in default values for the hotkeys and names
     */
    if (form_button_clicked(wp, "action_set_defaults")) {
        char **default_keys, **default_names;
        int8_t *default_showvals;
        char var_name[MAX_OPT_KEY_LEN + 1];
        char var_value[MAX_OPT_VALUE_LEN + 1];
	u_int kvm_default_idx;

	/* don't save (but only on pressing "apply") */
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;

	kvm_default_idx = pp_strtoul_10(websGetVar(wp, "_kvm_default_idx", "0"), 0, NULL);
	default_keys = pp_kvm_get_defaultkeys(kvm_default_idx);
	fv = &fh->fv[FV_ID_KVM_HOTKEY];
	for (i = 0; default_keys && default_keys[i]; i++) {
            form_var_vec_name(fv->fvname, i, var_name, sizeof(var_name));
	    websSetVar(wp, var_name, default_keys[i]);
	}
	default_names = pp_kvm_get_defaultnames(kvm_default_idx);
	fv = &fh->fv[FV_ID_KVM_PORTNAME];
	for (i = 0; default_names && default_names[i]; i++) {
            form_var_vec_name(fv->fvname, i, var_name, sizeof(var_name));
	    websSetVar(wp, var_name, default_names[i]);
	}
	default_showvals = pp_kvm_get_default_show_in_rc_vals(kvm_default_idx);
	fv = &fh->fv[FV_ID_KVM_SHOW_IN_RC];
	for (i = 0; default_showvals && default_showvals[i] >= 0; i++) {
            form_var_vec_name(fv->fvname, i, var_name, sizeof(var_name));
	    websSetVar(wp, var_name, default_showvals[i] ? "yes": "no");
	}
	fv = &fh->fv[FV_ID_KVM_KEY_PAUSE_DURATION];
	snprintf(var_value, sizeof(var_value), "%u", pp_kvm_get_defaultduration(kvm_default_idx));
	websSetVar(wp, fv->fvname, var_value);

	return 0;	
    }
#else /* !PRODUCT_XX01IP_ANY */
    /* FIXME: is there a feature for both PRODUCT_XX01IP_ANY and KX2? */
    {
        /* fixed for our KVM switches */
        int fixed_handler[] = { FV_ID_KVM_KEY_PAUSE_DURATION,
                                FV_ID_KVM_HOTKEY,
                                FV_ID_KVM_HOTKEYCODE,
                                FV_ID_KVM_UNIT_PORT_COUNT };
        
        for(i = 0; i < sizeof(fixed_handler) / sizeof(int); ++i) {
            fh->fv[fixed_handler[i]].flags |= FV_FLAG_SKIP_VALIDATE |
                                              FV_FLAG_DONT_SAVE;
        }
    }
#endif /* PRODUCT_XX01IP_ANY */

    return 0;
}

static int post_validate_hook(webs_t wp, form_handler_t * fh)
{
    u_int i, cnt;
    char var_value[MAX_OPT_KEY_LEN + 1];
    
    /* ------------------------------------------------------------------------
     * parse the hotkey sequences, set them and the nr of ports, maybe
     * switch the port if the nr decreased
     */
    if (form_button_clicked(wp, "action_apply")) {
	form_var_t *fv_cnt, *fv_hk, *fv_hkc;

	fv_cnt = &fh->fv[FV_ID_KVM_UNIT_PORT_COUNT];
	cnt = pp_strtoul_10(fv_cnt->val.s, 0, NULL);

	/* hotkeys */
	fv_hk = &fh->fv[FV_ID_KVM_HOTKEY];
	fv_hkc = &fh->fv[FV_ID_KVM_HOTKEYCODE];
	if (fv_hk->val_cnt != fv_hkc->val_cnt) {
            set_response(wp, ERIC_RESPONSE_ERROR, _("Internal error."));
            return -1;
        }

	if (fv_hk->val_cnt > 0 && cnt > fv_hk->val_cnt) {
	    cnt = fv_hk->val_cnt;
	    snprintf(var_value, sizeof(var_value), "%u", cnt);
	    websSetVar(wp, fv_cnt->fvname, var_value);
	    fh->fv[FV_ID_KVM_UNIT_PORT_COUNT].val.s = websGetVar(wp, fv_cnt->fvname, "0");
	}

	if (cnt > 1) {
	    for (i = 0; i < cnt; ++i) {
		char label[MAX_LABEL_LEN];
		char hkc_var_name[FV_KEY_MAX_LEN+1];
		char * hkc_var_val;
		snprintf(label, sizeof(label), fv_hk->label, i + 1);
		if (translate_hotkey(wp, fv_hk->val.m[i], &hkc_var_val,
				     NULL, label, KEYPARSER_WITH_DELIS) < 0) {
		    return -1;
		}
		form_var_vec_name(fv_hkc->fvname, i, hkc_var_name, sizeof(hkc_var_name));
		websSetVar(wp, hkc_var_name, hkc_var_val);
		fv_hkc->val.m[i] = websGetVar(wp, hkc_var_name, "");
		free(hkc_var_val);
	    }
	}

#if !defined(PRODUCT_XX01IP_ANY)
	/* make "show in console" require non-empty "hotkey" */
	{
	    form_var_t *fv_hkey = &fh->fv[FV_ID_KVM_HOTKEY];
	    form_var_t *fv_show = &fh->fv[FV_ID_KVM_SHOW_IN_RC];
	    char port[4];
	    
	    for (i = 0; i < fv_show->val_cnt; i++) {
		if (!pp_strcmp_safe(fv_show->val.m[i], "yes")) {
		    if (fv_hkey->val.m[i] == NULL || !strcmp(fv_hkey->val.m[i], "")) {
			kvm_get_portname_from_index(i, port, sizeof(port));
			set_response(wp, ERIC_RESPONSE_ERROR,
				_("Setting '%s' for port %s<br>"
				  "requires corresponding '%s' to be non-empty!\n"),
				fv_show->label, port, fv_hkey->label);
			return -1;
		    }
		}
	    }
	}
#endif /* !PRODUCT_XX01IP_ANY */

	/* all validation tests passed -> make settings effective */
	for (i = 0; i < PP_KVM_CHANNEL_COUNT; ++i) {
	    if (kvm_port_backswitch(wp, i, cnt) < 0) return -1;
	}
	/* notify kvm lib of number change */
	pp_kvm_set_unit_port_count(wp->target_unit, cnt);
    } else {
	/* don't save unless "action_apply" (useful e.g. for action_switch) */
	wp->fh_flags |= FH_FLAG_ABORT_AT_SAVE;
    }
    return 0;	
}


static void
kvm_update_port_count(webs_t wp, form_var_t *fv, u_int cnt)
{
    char cnt_opt_name[MAX_OPT_KEY_LEN];
    char cnt_opt_val[11];

    form_var_sz_name(fv->fvname, cnt_opt_name, sizeof(cnt_opt_name));
    snprintf(cnt_opt_val, sizeof(cnt_opt_val), "%u", cnt);
    websSetVar(wp, cnt_opt_name, cnt_opt_val);
}

/**
 * switch kvm port, if currently active port > the new max number 
 * of ports the settings get changed to
 */
static int
kvm_port_backswitch(webs_t wp, u_int channel, u_int new_cnt)
{
    u_char unit;
    u_short port;
    int ret = -1;

    if (pp_kvm_get_unit_port_for_video_link(channel, &unit, &port) != 0) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Cannot determine current KVM unit/port."));
	goto bail;
    }

    if ((u_int)port + 1 > new_cnt) {
	/* TODO: at some point we may have to deal with kvm unit != 0 */
	/* FIXME: fix pp_kvm_switch_port */
	u_char dl, vl;
	if (pp_kvm_switch_port(NULL, 0 /*unit*/, new_cnt-1, &vl, &dl)) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Backswitching the KVM port failed."));
	    goto bail;
	}
    }

    ret = 0;
 bail:
    return ret;
}

static int
kvm_init_form_data_asp(int eid UNUSED, webs_t wp, int argc UNUSED, char ** argv UNUSED)
{
    char var_name[MAX_OPT_KEY_LEN + 1];
    char var_value[MAX_OPT_VALUE_LEN + 1];
    int  i, cnt;      
    
    /* ------------------------------------------------------------------------
     * fill the kvm default-values list
     */
    cnt = pp_kvm_get_defaultcount();
    snprintf(var_value, sizeof(var_value), "%u", cnt);
    websSetVar(wp, "kvm_default_entry_cnt", var_value);
    for (i = 0; i < cnt; i++) {
        form_var_vec_name("kvm_default_entry", i, var_name, sizeof(var_name));
	snprintf(var_value, sizeof(var_value), "%s", pp_kvm_get_defaultname(i));
	websSetVar(wp, var_name, var_value);
    }
    return 0;
}
