/* system includes */
/* firmware includes */
#include <pp/ipmi.h>
#include <pp/hal_rpc.h>

/* local includes */
#include "webs.h"
#include "templates.h"
#include "eric_forms.h"
#include "tmpl_rpc_common.h"

FV_SPEC = {
};

/* ASPs */
static int rpc_init_outlet_details_asp(int eid, webs_t wp, 
                                       int argc, char **argv);

int rpc_outlet_detail_tmpl_init(void)
{
    form_handler_t * fh;

    rpc_common_tmpl_init();
    
    /* register ASPs */
    websAspDefine("rpcInitOutletDetails", rpc_init_outlet_details_asp);
    
    fh = CREATE_FH_INSTANCE(TEMPLATE_RPC_OUTLET_DETAIL, ACL_OBJ_POWER_SWITCH);

    fh->pre_validate_hook = rpc_outlet_switch_hook;
    
    REGISTER_FH_INSTANCE_AND_RETURN(fh);
    
    return PP_SUC;
}

static void set_outlet_detail_unavailable(webs_t wp, unsigned int sensor_num) {
    char var_buf[20];

    form_var_vec_name("sensor_value", sensor_num, var_buf, sizeof(var_buf));
    websSetVar(wp, var_buf, "");
    form_var_vec_name("sensor_status", sensor_num, var_buf, sizeof(var_buf));
    websSetVar(wp, var_buf, _("currently unavailable"));
}

static void set_outlet_details_unavailable(webs_t wp, unsigned int sensors_sz) {
    unsigned int u;
    
    for(u = 0; u < sensors_sz; ++u) {
        set_outlet_detail_unavailable(wp, u);
    }
}

static int rpc_init_outlet_details_asp(int eid UNUSED, webs_t wp, 
                                       int argc, char **argv) {
    int ret = -1;
    int outlet, num_outlets = pp_hal_rpc_get_no_outlets();
    
    /* check if argv is valid outlet number */
    if(argc == 1 &&
       (outlet = pp_strtol_10(argv[0], -1, NULL) - 1) >= 0 &&
       outlet >= 0 && outlet < num_outlets) {
        int power_branch = rpc_get_power_branch_for_outlet(outlet);
        int sensors[] = { RPC_RECEPTACLE_SENSOR_RMS_CURRENT(outlet),
                          RPC_RECEPTACLE_SENSOR_MAX_RMS_CURRENT(outlet),
                          RPC_POWER_BRANCH_SENSOR_RMS_VOLTAGE(power_branch),
                          RPC_RECEPTACLE_SENSOR_TRUE_RMS_CURRENT(outlet),
                          RPC_RECEPTACLE_SENSOR_APPARENT_RMS_POWER(outlet),
                          RPC_RECEPTACLE_SENSOR_AVERAGE_REAL_RMS_POWER(outlet)
                        };
#if !defined(USE_SDR_SENSOR_NAMES)
        const char *sensor_strs[] = { N_("RMS Current"),
                                      N_("RMS max Current"),
                                      N_("RMS Voltage"),
                                      N_("True RMS Current"),
                                      N_("RMS Power"),
                                      N_("Real RMS Power") };
#endif /* !USE_SDR_SENSOR_NAMES */
        unsigned int sensors_sz = sizeof(sensors) / sizeof(int);
        unsigned int u;
        char var_buf[20], val_buf[20];
        pp_ipmi_return_t ipmi_ret;
        pp_ipmi_parameter_t params;
        vector_t *sdr_entries = vector_new(NULL, sensors_sz, NULL);
        vector_t *get_list;
        int state = rpc_get_outlet_state(outlet);
        const char *outlet_state_str[6] = { "failure",
                                            "on",
                                            "on (power critical)",
                                            "on (circuit breaker off)",
                                            "off",
                                            "off (circuit breaker off)" };
        
        /* init */
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        memset(&params, 0, sizeof(params));
        
        params.data.sensor_get_by_id_list = 
            vector_new2(NULL, sensors_sz, sizeof(unsigned int), NULL);
        get_list = params.data.sensor_get_by_id_list;
        
        snprintf(val_buf, sizeof(val_buf), "%u", sensors_sz);
        websSetVar(wp, "sensors_sz", val_buf);
        
        websSetVar(wp, "outlet_state", _(outlet_state_str[state]));
            
        /* build vector for sensor reading */
        for(u = 0; u < sensors_sz; ++u) {
            unsigned int sensornum = sensors[u];
            pp_ipmi_sdr_list_entry_t *entry;
            char sensor_name[32], sensor_unit[32],
                 sensor_value[32], sensor_status[32];
            
            form_var_vec_name("sensor_name", u,
                              sensor_name, sizeof(sensor_name));
            form_var_vec_name("sensor_unit", u,
                              sensor_unit, sizeof(sensor_unit));
            form_var_vec_name("sensor_value", u,
                              sensor_value, sizeof(sensor_value));
            form_var_vec_name("sensor_status", u,
                              sensor_status, sizeof(sensor_status));
            
            if(NULL != (entry = get_sdr_entry_for_sensor(sensornum))) {
                /* valid sdr entry */

#if defined(USE_SDR_SENSOR_NAMES)
                websSetVar(wp, sensor_name, 
                           entry->sdr_name.buf ? : _("unknown"));
#else /* !USE_SDR_SENSOR_NAMES */
                websSetVar(wp, sensor_name, _(sensor_strs[u]));
#endif /* !USE_SDR_SENSOR_NAMES */
                if(state == STATE_RELAIS_OFF_POWER_ON ||
                   state == STATE_RELAIS_OFF_POWER_OFF || 
                   state == STATE_RELAIS_ON_POWER_OFF) {
                    websSetVar(wp, sensor_value, "");
                    websSetVar(wp, sensor_unit, "");
                    websSetVar(wp, sensor_status, _("off"));
                } else {
                    vector_add2(get_list, &sensornum);
                    vector_add(sdr_entries, entry);
                    websSetVar(wp, sensor_unit, 
                               entry->data.full.unit.buf ? : "");
                }
            } else {
                /* invalid sensor, no sdr entry */
                vector_add(sdr_entries, NULL);

                websSetVar(wp, sensor_name, _("not initialized"));
                websSetVar(wp, sensor_unit, "");
                websSetVar(wp, sensor_value, "");
                websSetVar(wp, sensor_status, "");
            }
        }

        /* do not disturb bmc if we have nothing to query ;-) */
        if(vector_size(get_list)) {
            /* get raw sensor reading */
            
            if (pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
                                     PP_IPMI_SENSOR_SUBCMD_GET_RAW_BY_ID,
                                     &params, &ipmi_ret,
                                     NULL, NULL) != 0) {
                RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_SENSOR,  PP_IPMI_SENSOR_SUBCMD_GET_RAW_BY_ID failed");
                
                set_outlet_details_unavailable(wp, sensors_sz);
            } else {
                vector_t *raw_readings = ipmi_ret.data.sensor_get_raw_list;
                
                if(!raw_readings || !vector_size(raw_readings)) {
                    RPC_DEBUG_IPMI("sensor get raw returned no reading");
                    set_outlet_details_unavailable(wp, sensors_sz);
                } else {
                    /* we got raw readings */
                    
                    unsigned int val_count = 0;
                    
                    for(u = 0; u < sensors_sz; ++u) {
                        /* check if we requested reading for sensor */
                        pp_ipmi_sdr_list_entry_t *entry =
                            vector_get(sdr_entries, u);
                        
                        if(entry) {
                            /* now get reading from sensor */
                            pp_ipmi_raw_reading_list_entry_t *reading;
                            double value;
                            
                            assert(val_count < vector_size(raw_readings));
                            reading = vector_get(raw_readings, val_count++);
                            
                            /* paranoia */
                            assert(reading);
                            assert(reading->id == entry->sdr_id);
    
                            // TODO: discrete sensors?
                            value = pp_ipmi_sdr_convert_sensor_reading_core(
                                                entry->data.full.m,
                                                entry->data.full.b,
                                                entry->data.full.b_exp,
                                                entry->data.full.r_exp,
                                                entry->data.full.analog,
                                                entry->data.full.linearization,
                                                reading->raw_value);
                                                    
                            /* TODO: format string! */
                            snprintf(val_buf, sizeof(val_buf), "%.1f", value);
    
                            form_var_vec_name("sensor_value", u, 
                                              var_buf, sizeof(var_buf));
                            websSetVar(wp, var_buf, val_buf);
                            form_var_vec_name("sensor_status", u,
                                              var_buf, sizeof(var_buf));
                            if(reading->threshold.bits.below_critical) {
                                websSetVar(wp, var_buf, _("below critical"));
                            } else
                            if(reading->threshold.bits.below_non_critical) {
                                websSetVar(wp, var_buf, 
                                           _("below non-critical"));
                            } else if(reading->threshold.bits.above_critical) {
                                websSetVar(wp, var_buf, _("above critical"));
                            } else
                            if(reading->threshold.bits.above_non_critical) {
                                websSetVar(wp, var_buf, 
                                           _("above non-critical"));
                            } else {
                                websSetVar(wp, var_buf, _("ok"));
                            }
                        } // else already set unavailable!
                    }
                }
            }
        }
        
        ret = 0; // all vars set

        /* clean up */
        vector_delete(sdr_entries);
        vector_delete(get_list);
        pp_ipmi_cleanup_ret(&ipmi_ret);
    } else {
        /* invalid outlet number */
        if(argc != 1) {
            pp_log("invalid number of arguments!\n");
        } else {
            pp_log("outlet number '%s' invalid!\n", argv[0]);
        }
        abort();
    }
    
    return ret;
}
