#include <liberic_notify.h>
#include <pp/um.h>
#include <liberic_net.h>
#include <pp/ldap_prof.h>
#include <openssl/err.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"
#include <liberic_config.h>
#include <pp/cfgx.h>

/* Documentation 
   Summary:	Common user settings
   Action: 	none
*/

static int um_init_blocking_info_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_am_i_root_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_ldap_type_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_ldap_info_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_ldap_moname_asp(int eid, webs_t wp, int argc, char** argv);
static char* um_ldap_format_srv_type(char* srv_type);
static int um_ldap_moinfo_asp(int eid, webs_t wp, int argc, char** argv);
static int um_profile_info_asp(int eid, webs_t wp, int argc, char** argv);
static int um_target_user_is_root_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_group_available_for_user(int eid, webs_t wp, int argc, char ** argv);
static int um_group_is_individual(int eid, webs_t wp, int argc, char ** argv);

int
um_common_tmpl_init(void)
{
    static int initialized = 0;

    if (!initialized) {
	/* register ASPs */
	websAspDefine("umInitBlockingInfo", um_init_blocking_info_asp);
	websAspDefine("umAmIroot", um_am_i_root_asp);
	websAspDefine("umLdapType", um_ldap_type_asp);
	websAspDefine("umLdapInfo", um_ldap_info_asp);
	websAspDefine("umLdapMoName", um_ldap_moname_asp);
	websAspDefine("umLdapMoInfo", um_ldap_moinfo_asp);
	websAspDefine("umProfileInfo", um_profile_info_asp);
	websAspDefine("umTUIsRoot", um_target_user_is_root_asp);
	websAspDefine("umGroupAvailableForUser", um_group_available_for_user);
        websAspDefine("umGroupIsIndividual", um_group_is_individual);
	
	initialized = 1;
    }

    return 0;
}

static int
um_init_blocking_info_asp(int eid UNUSED, webs_t wp, int argc UNUSED,
			  char **argv UNUSED)
{
    char opt_val[32];
    int mins;
    websSetVar(wp, "_current_user", wp->user);
    if (wp->target_user
	&& strcmp(wp->target_user, "--- select ---")
	&& pp_um_user_is_blocked(wp->target_user, &mins))
    {
	websSetVar(wp, "_user_is_blocked", "1");
	snprintf(opt_val, sizeof(opt_val), "%d", mins);
	websSetVar(wp, "_user_blocked_mins", opt_val);
    } else {
	websSetVar(wp, "_user_is_blocked", "0");
    }

    return 0;
}

static int
um_am_i_root_asp(int eid, webs_t wp UNUSED, int argc UNUSED,
		 char **argv UNUSED)
{
    int root = 0;

    pp_um_user_is_root(&root, wp->user);
    ejSetResult(eid, root ? "1" : "0");

    return 0;
}

static int
um_ldap_type_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char ** argv UNUSED)
{
    if (pp_ldap_is_active(NULL, NULL)) {
	  ejSetResult(eid, "pemx");
    } else if (pp_ldap_mini_is_active(NULL)) {
	  ejSetResult(eid, "mini");
    } else {
	  ejSetResult(eid, "none");
    }
    return 0;
}

static int
um_ldap_info_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char ** argv UNUSED)
{
    char *ld_profile_srv = NULL, *ld_user_srv = NULL;
    char *ld_usrsrv_type = NULL, *buf = NULL;
    int buf_len = 0;

    if (pp_ldap_is_active(&ld_profile_srv, &ld_user_srv)) {
	if(ld_usrsrv_type == NULL) ld_usrsrv_type = strdup("Peppercon-EMX");
	ld_usrsrv_type = um_ldap_format_srv_type(ld_usrsrv_type);
	if(ld_profile_srv && ld_user_srv &&
	   !strcmp(ld_profile_srv, ld_user_srv)) {
	    //sprintf(buf, "%s (%s)", ld_usrsrv_type, ld_user_srv);
	    // if ld_profile_srv equals ld_user_srv than we have EMX control
	    pp_dsprintf(&buf, &buf_len, 0, "Peppercon-EMX (%s)",
			ld_profile_srv);
	} else {
	    pp_dsprintf(&buf, &buf_len, 0,
			"%s (%s), Profiles: Peppercon-EMX (%s)",
			ld_usrsrv_type, ld_user_srv, ld_profile_srv);
	}
    } else if (pp_ldap_mini_is_active(&ld_user_srv)) {
        if (PP_SUCCED(pp_cfg_get(&ld_usrsrv_type, "ldap.server_type"))) {
	    ld_usrsrv_type = um_ldap_format_srv_type(ld_usrsrv_type);
	} else {
	    assert(0);
	    ld_usrsrv_type = strdup("unknown");
	}
	pp_dsprintf(&buf, &buf_len, 0, "%s (%s), Profiles: self contained",
		    ld_usrsrv_type, ld_user_srv);
    } else {
	buf = strdup("self contained");
    }
    
    ejSetResult(eid, buf);
    free(buf);
    free(ld_usrsrv_type);
    free(ld_profile_srv);
    free(ld_user_srv);
    return 0;
}

static char *
um_ldap_format_srv_type(char* srv_type)
{
    char* ret;
    if(!srv_type || !strcmp(srv_type, "generic")) {
	free(srv_type);
	ret = strdup("LDAP");
    } else {
	char* c;
	for(c = srv_type; *c != '\0'; ++c) *c = toupper(*c);
	ret = srv_type;
    }
    return ret;
}

static int
um_ldap_moname_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
    char* name;
    if (NULL == (name = pp_ldap_get_nbx_moname())) {
	ejSetResult(eid, "");
    } else {
	ejSetResult(eid, name);
	free(name);
    }
    return 0;
}

static int
um_ldap_moinfo_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char** argv UNUSED)
{
    char* name;
    pp_strstream_t os = PP_STRSTREAM_INITIALIZER;
    if (NULL != (name = pp_ldap_get_nbx_moname())) {
	pp_strappend(&os, name);
	free(name);
    }

    if (pp_cfg_ldap_prof_is_active(-1)) {
	if (os.pos > 0) pp_strappend(&os, " : ");
	pp_strappend(&os, "PEMX Profile active.");
    }
    
    ejSetResult(eid, os.buf);
    pp_strstream_free(&os);
    return 0;
}

static int
um_profile_info_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char** argv)
{
    //char* principal;
    int uid;
    char* fielddeli;
    pp_cfg_profile_info_t* pi;
    pp_strstream_t os = PP_STRSTREAM_INITIALIZER;
    
    assert(argc == 2);
    //principal = argv[0];
    uid=pp_um_user_get_uid(argv[0]);
    fielddeli = argv[1];
    
    if (PP_ERR == pp_cfg_get_profile_info(&pi, uid)) {
	ejSetResult(eid, pp_error_string(errno));
    } else {
	pp_strappend(&os, "Local active tag: ");
	pp_strappend(&os, pi->local_tag ? pi->local_tag : "none");
	if (pp_ldap_is_active(NULL, NULL)) {
	    pp_strappend(&os, fielddeli);
 	    pp_strappend(&os, "Pemx device profile: ");
	    pp_strappend(&os, pi->ldap_prof_is_active ?
			 pi->ldap_profile_name : "none");
	    if ((pi->ldap_active_tag) != NULL) {
		pp_strappend(&os, fielddeli);
		pp_strappend(&os, "Pemx active tag: ");
		pp_strappend(&os, pi->ldap_active_tag);
	    }
	    if (pi->ldap_requested_tag != NULL &&
		(!pi->ldap_prof_is_active ||
		strcmp(pi->ldap_active_tag, pi->ldap_requested_tag))) {
		pp_strappend(&os, fielddeli);
		pp_strappend(&os, "Pemx request tag: ");
		pp_strappend(&os, pi->ldap_requested_tag);
	    }
	    if (pi->ldap_error != PP_LDAP_ERR_NO_ERROR
		&& ((pi->ldap_error != PP_LDAP_ERR_NO_SUCH_TAG) ||
		    (!pi->ldap_prof_is_active
		     && pi->ldap_requested_tag != NULL))) {
		pp_strappend(&os, fielddeli);
		pp_strappend(&os, "Last error: ");
		pp_strappend(&os, pp_error_string(pi->ldap_error));
	    }
	}
	ejSetResult(eid, os.buf);
	pp_strstream_free(&os);
	pp_cfg_profile_info_destroy(pi);
    }    
    return 0;
}

static int
um_target_user_is_root_asp(int eid, webs_t wp, int argc UNUSED, char ** argv UNUSED)
{
    ejSetResult(eid, pp_um_user_get_uid(wp->target_user) ? "0" : "1");
    
    return 0;
}

/**
 * umGroupAvailableForUser(groupname, username)
 * a group is available for a user, if
 * - its not the builtin group "<None>"
 * - the group is not individual
 * - the group is already taken by the user
 */
static int 
um_group_available_for_user(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    int available = 0;
    pp_um_group_state_t groupstate;
    char *group = NULL;
    
    if(argc == 2 && argv[0] && strcmp(argv[0], "<None>") && argv[1] && 
       PP_ERR != pp_um_group_is_individual_for_user(argv[0], argv[1],
                                                    &groupstate) &&
       (groupstate == PP_UM_GROUP_GENERIC ||
        (PP_ERR != pp_um_user_get_group(argv[1], &group) &&
         !strcmp(argv[0], group)))) {
        available = 1;
    }
    
    free(group);
    ejSetResult(eid, available ? "1" : "0");
        
    return 0;
}

static int 
um_group_is_individual(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    pp_um_group_state_t groupstate;
    int individual = 0;
    
    if(argc == 1 && argv[0] &&
       PP_ERR != pp_um_group_is_individual_for_user(argv[0], NULL, &groupstate)
       && groupstate != PP_UM_GROUP_GENERIC) {
        individual = 1;
    }
    
    ejSetResult(eid, individual ? "1" : "0");
        
    return 0;
}

