/**	
 *	@file	CSC.cpp
 *	@brief	Implementation of CCSC
 *  This is the Port 5000 multiplexer
 *  It listens for TCP connections and hands them to the installed CCSCProtocol handlers
 *  It listens for UDP messages and hands the to udp message handles or to CCSCInfo for discovery
 *
 */

#include "pp/CSC.h"
#include "pp/SXDB.h"
#include "pp/CSCAccept.h"
#include "pp/NetConn.h"
#include "pp/Session.h"
#include "pp/CSCProtocol.h"
#include "pp/CC.h"

#include "assert.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	static data
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	CCSC Class
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/

CCSC::CCSC( CSessionManager *pNewSessionManager, int tcpPort, int udpPort ) :
	CTCPListener( tcpPort ),
	CUDPListener( udpPort ),
	pSessionManager(pNewSessionManager)
{
	pFirst = NULL;
}

/*  --------------------------------------------------------------------*/

CCSC::~CCSC()
{
}

/*  --------------------------------------------------------------------*/

int CCSC::Initialize( )
{
	int result;

	// Initialize the SSL contexts

	result = CNetConn_SSL::SSL_Library_Init(SSL2_TXT_RC4_128_WITH_MD5);
	assert( result == 0 );
	result = CreatePrivateKeyFile("PrivateKey.pem");
	assert( result != 0 );
	result = CreateCertConfigFile("CertConfig.pem");
	assert( result != 0 );
	result = CreateCertificateFile("PrivateKey.pem","CertConfig.pem","Cert.pem");
	assert( result != 0 );

	pSSLCtx = new CNetConn_SSL_CTX("PrivateKey.pem","Cert.pem");
	assert(pSSLCtx != NULL);

	DestroyFile("PrivateKey.pem");
	DestroyFile("CertConfig.pem");
	DestroyFile("Cert.pem");

	if (pSSLCtx == NULL) {
		printf("ERROR: CCSC::Initialize failed (pSSLCtx == NULL)\n");
		return -1;
	}

	result = CTCPListener::Initialize();
	if (result < 0) {
		printf("ERROR: CCSC::Initialize failed (CTCPListener::Initialize failed)\n");
		return result;
	}

	result = CUDPListener::Initialize();
	if (result < 0) {
		printf("ERROR: pCCSC::Initialize failed (CUDPListener::Initialize failed)\n");
		return result;
	}

	AddUDPHandler((CCSCInfo *) this);

	return 0;
}

/*  --------------------------------------------------------------------*/

void CCSC::AcceptClientSocket( SOCKET s, int ipAddress, int tcpPort )
{
	CNetConn_Socket *	pNetConn_Socket = new CNetConn_Socket( s );
	CNetConn_SSL	*	pNetConn_SSL = NULL;
	CNetConn 		*	pNetConn = (CNetConn *) pNetConn_Socket;
	CCSCAccept			csc(pSessionManager,(CCSCInfo *) this);
	CSession			*pSession = NULL;
	int					result;
	int					bytesSent;

	if (pNetConn_Socket == NULL)
		return;

	// Do the CSC negotications

	csc.SetSocket( pNetConn_Socket );
	csc.SetClientIP( ipAddress ); 
	result = csc.NegotiateProtocol( &bytesSent );

	if (result < 0)
	{
		// CSC negotiations failed...
		// Send out fake, old style serverID packet with big version
		// so that old RRC's will know that they are too old
		// to work with this server.

		/*
		BYTE * p = (BYTE *) &localServerID;
		p += bytesSent;
		bOk = WriteSocket( pNetConn, p, sizeof(localServerID) - bytesSent );
		*/
		goto EXIT;
	}

	// Do SSL if needed

	if (strcmp(csc.GetEncryptionName(),"SSL") == 0)
	{
		// Switch to SSL CNetConn
		pNetConn_SSL = new CNetConn_SSL( pNetConn_Socket );
		pNetConn = (CNetConn *) pNetConn_SSL;
		csc.SetSocket( (CNetConn_Socket *) pNetConn_SSL );
		delete pNetConn_Socket;
		pNetConn_Socket = NULL;

		// Do the SSL Accept
		result = pNetConn_SSL->SSL_Accept(pSSLCtx);
		if (result < 0)
			goto EXIT;
	}

	// Authenticate the user

	result = csc.Authenticate();
	if (result < 0)
		goto EXIT;

	// Connect to the Session object

	if (csc.GetSessionID() == NULL )
	{
		// Create a new session

		pSession = pSessionManager->NewSession();

		if (pSession == NULL)
			goto EXIT;

		// Set the user object

		pSession->SetUserObject( csc.GetUserObjectID() );
	}
	else
	{
		// Attach to existing session

		pSession = pSessionManager->GetSession( csc.GetSessionID() );
	}

	if (pSession == NULL)
		goto EXIT;

	// Hand off the socket to the protocol handler

	if (csc.GetProtocol() != NULL)
	{
		csc.GetProtocol()->AcceptClientSocket(pSession,pNetConn,ipAddress,tcpPort);
		pNetConn_Socket = NULL;
		pNetConn_SSL = NULL;
	}

EXIT:
	delete pNetConn_Socket;
	delete pNetConn_SSL;
}

