/**	
 *	@file	CSCInfo.h
 *	@brief	This class manages the CSCInfo data and acts as the UDP handler
 * 			for CSC Discovery.
 * 			The CSCInfo data is a description of the device and a list of
 * 			supported protocols.
 *
 */

#ifndef _CSCInfo_h_
#define _CSCInfo_h_

#include "pp/UDPListener.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

class CCSCProtocol;

/*----------------------------------------
 *	CNetConn Class
 *--------------------------------------*/

class CCSCInfo : public CUDPHandler
{
public:

	CCSCInfo();
	virtual ~CCSCInfo() {}

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Tells CSCInfo the device description data.
	 *
	 * 	@param	pID				///< Ptr to the type name
	 *  @param	pEncrypt		///< Ptr to the model name
	 *  @param	pVersion		///< Ptr to the product name
	 *  @param	pOldestVersion	///< Ptr to the version number
	 *  @param	pCompression	///< Ptr to the device name
	 *  @param	pAuth			///< Ptr to the CC cluster name (Can be NULL)
	 *	@return					0 = no error
	 */

	void SetDeviceDescription(const char *pType, const char *pModel, const char *pProductName, const char *pVersion, const char *pName, const char *pCluster);

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Returns the protocol XML string
	 *
	 *  @see CCSCProtocol
	 * 
	 *  @param	pProtcol		Ptr to the CCSCProtcol object to add 
	 *	@return					Ptr to XML Protocol string to be used in CSC_Info
	 */

	void AddProtocol( CCSCProtocol * pProtocol );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Enumerates the CCSCProtocol objects
	 * 
	 *  @see CCSCProtocol
	 * 
	 *  @param	index			The index into the list of protocols
	 *  @return					Ptr to the CCSCProtocol object or NULL if index is too big
	 */

	CCSCProtocol * EnumProtocols( int index );


	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Builds a CSCInfo block
	 *
	 *  @param	msgFormat		0 = text only version of CSC
	 * 							1 = CSC_Info UDP message format (with 4-byte length header)
	 *  @param	*pLength		Ptr to an int to recieve the length of the CSC_Info...can be null
	 *	@return					Returns a NEW string with CSC_Info inside
	 * 							Caller must delete[] the string when done.
	 */

	char * MakeCSCInfo( int msgFormat = 0, int * pLength = (int *) 0 );


	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Virtual call back functin to handle UDP message.
	 *  Derived from CUDPHandler::ReceiveUDPMessage
	 *  Determines if the UDP message is a CSC_Discover request and responds.
	 * 
	 *  @see CUDPHandler
	 * 
	 *  @param	ipAddress		The client IP address
	 *  @param	pData			The data received
	 * 	@param	count			# of bytes
	 *	@return					true = message was handled, false = call next handler
	 */

	virtual bool ReceiveUDPMessage( CUDPListener * pUDP, int ipAddress, char * pData, int count );


protected:
	CCSCProtocol *	pFirst;			///< List of protocols
	char *			pDevInfo;		///< The device Info
};

#endif // _CSCInfo_h_




