/**	
 *	@file	CSCProtocol.h
 *	@brief	This class defines a protocol that can be negotiated by
 * 			CSC. Adding a CCSCProtocol class to the CCSC object causes
 * 			A new protocol descriptor to be added the the CSC_Info struct.
 * 			This handler is also called when a new TCP socket requests
 * 			This protocol
 *
 */

#ifndef _CSCProtocol_h_
#define _CSCProtocol_h_

#include "pp/OS_Port.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

class CCSCInfo;
class CSession;
class CNetConn;

/*----------------------------------------
 *	CNetConn Class
 *--------------------------------------*/

class CCSCProtocol
{
public:

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Creates a CSC Protocol description
	 *
	 * 	@param	pID				///< Ptr to Protocol name ("RDM", "RDMEvent", "RFP", "RFB")
	 *  @param	pEncrypt		///< Encruptions type ("SSL", "None")
	 *  @param	pVersion		///< Protocol current version
	 *  @param	pOldestVersion	///< Oldest supported protocol version
	 *  @param	pCompression	///< Compression = "None"
	 *  @param	pAuth			///< Ptr to the authenticaton mode ("CSC","Natice")
	 *	@return					0 = no error
	 */

	CCSCProtocol(const char *pID, const char *pEncrypt, const char *pVersion, const char *pOldestVersion, const char *pCompression, const char *pAuth);
	virtual ~CCSCProtocol();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Returns the protocol XML string
	 *
	 *	@return					Ptr to XML Protocol string to be used in CSC_Info
	 */

	const char * GetProtocolXML( ) { return pProtocolString; }

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Call back function. Called when a new tcp connection has
	 *  been accepted that matches this protocol.
	 * 
	 *  @see CTCPListener
	 * 
	 *  This function must be over ridden to accept TCP connections.
	 *  If the protocol requires SSL, then SSL will have already been started
	 *  on this socket.
	 *  If the protocol requires CSC authentication, then the user has
	 *  already been authenticated.
	 * 
	 *  @param	pSession		The user session for this connection
	 * 	@param	s				The socket
	 *  @param	ipAddress		The client IP address
	 *  @param	tcpPort			The client TCP Port
	 */

	virtual void AcceptClientSocket( CSession *pSession, CNetConn *pNetConn, int ipAddress, int tcpPort ) = 0;



protected:
	friend class CCSCInfo;

	char *	pProtocolString;			///< The formatted protocol string
	CCSCProtocol * pNext;				///< Linked list
};

#endif // _CSCProtocol_h_



