/**	
 *	@file	UDPListener.h
 *	@brief	Simple class for listening for UDP messages
 *
 */

#ifndef _UDPListener_h_
#define _UDPListener_h_

#include	"pp/OS_Port.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

 class CUDPHandler;

/*----------------------------------------
 *	CUDPListener Class
 *--------------------------------------*/

class CUDPListener
{
public:

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Constructor
	 * 
	 * 	@param	port			The TCP port # to listen to
	 */

	CUDPListener( int port, int maxUDPMsgSize=256 );
	virtual ~CUDPListener();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Initialize the TCP socket and start the listen thread
	 * 
	 *	@return					0 = no error
	 */

	int Initialize( );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Adds UDP handler  to this listern.
	 * 
	 *  The handler is called when a UDP message is received.
	 *  @see CUDPListened::RecieveMessage
	 *  @see CUDPHandler
	 * 
	 *	@return					0 = no error
	 */

	void AddUDPHandler( CUDPHandler * pUDPHandler );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Stops the listening thread
	 * 
	 *	@return					0 = no error
	 */

	void Shutdown( );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Call back function. Called when a new udp message has
	 *  been received
	 * 
	 *  The default implementation calls the install CUDPHandlers.
	 *  This function can be over ridden to accept UDP data directly
	 * 
	 *  @param	ipAddress		The client IP address
	 *  @param	pData			The data received
	* 	@param	count			# of bytes
	 */

	virtual void ReceiveMessage( int ipAddress, char * pData, int count );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Call back function. Called when a new tcp connection has
	 *  been accepted
	 * 
	 *  This function must be over ridden to accept TCP connections.
	 * 
	 *  @param	pData			The data to send
	 * 	@param	count			# of bytes to send
	 *  @return					errno
	 */

	int ReturnMessage( char * pData, int count );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Closes the socket
	 * 
	 */

	void CloseSocket( SOCKET s );


	void ListenThread( );

protected:
	int			initialized;			// !0 = we are ready to read and write
	SOCKET		listenSocket;			// The socket used to listen for connections
	int			port;					// The TCP port we are listening to
	int			maxUDPMsgSize;			// Max # of bytes in one UDP message
	OS_THREAD	thread;					// Our listen thread
	void	*	pLastClientAddress;		// Used by ReturnMessage to send back to the last client
	CUDPHandler *pFirst;				// First handler

};

/*----------------------------------------
 *	CUDP_Handler Class
 *--------------------------------------*/

class CUDPHandler
{
public:

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Constructor
	 * 
	 */

	CUDPHandler( ) { pNext = NULL; }

	virtual ~CUDPHandler() {}

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Virtual call back functin to handle UDP message.
	 * 
	 *  @param	ipAddress		The client IP address
	 *  @param	pData			The data received
	 * 	@param	count			# of bytes
	 *	@return					true = message was handled, false = call next handler
	 */

	virtual bool ReceiveUDPMessage( CUDPListener * pUDP, int ipAddress, char * pData, int count ) = 0;


	CUDPHandler * pNext;		///< ptr to next handler
};

#endif // _UDPListener_h_
