/*--------------------------------------------------------------------------------
	OS_Posix.cpp
	Copyright (c) 2001, Raritan Computer, Inc.
	ClientLib OS porting layer
	Posix implementation

	JL	OCT.23, 2001		Add RedHat 7.2 Linux POSIX codes
--------------------------------------------------------------------------------*/

#define __USE_UNIX98

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <semaphore.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <pp/base.h>
#include <pp/OS_Port.h>


#ifdef OS_MFC
#ifdef _DEBUG
#define _CRTDBG_MAP_ALLOC
#include <stdlib.h>
#include <crtdbg.h>
#endif //_DEBUG
#else //OS_MFC
#ifdef DMALLOC
#include "dmalloc.h"
#endif //DMALLOC
#endif //OS_MFC



//static DWORD	instanceCount = 0;		// # of times we have been initialized

//static void Print_Msg();
/***********************************************************/
/*static void Print_Msg(void *argv)
{
	printf("Thread T=%x cancel\n", (int) pthread_self());
}*/

/***********************************************************/
void OS_Startup ()
{
}

/***********************************************************/
void OS_Shutdown ()
{
}

/***********************************************************/
OS_THREAD OS_CreateThread (void *pFunction, DWORD dwAttributes, void *pParam)
{
	static int flag = 0;
	pthread_t thread;
	int ret;

	ret = pthread_create(&thread,
		(pthread_attr_t *)NULL,
		(void *(*)(void*))pFunction,
		pParam);
    
    if ( !ret && (dwAttributes == OS_THREAD_DETACHED) )
        pthread_detach(thread);

	if ( ret ) {
		fprintf(stderr, "OS_CreateThread() pthread_create failed -- %s\n", strerror(errno));
		return (OS_THREAD) NULL;
	}
#ifdef	OS_POSIX_DEBUG
	printf("CreateThread : T=%x\n", thread);
#endif
	if ( !flag ) {
		pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, 0);
		// FIXME: this sets the cancel type of the current thread, but
		//        that of the new thread should be set, as that one is
		//        to be cancelled later on by OS_DeleteThread
		// FIXME: cancelling asynchronously is almost always unsafe, as
		//        it may happen outside of cancellation points and
		//        resources, like mutexes, may not be released properly
		pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, 0);
		flag = 1;
	}
	return (OS_THREAD)thread;
}


/***********************************************************/
DWORD OS_DeleteThread (OS_THREAD hThread)
{
	int ret;
	ret = pthread_cancel((pthread_t)hThread);
	// FIXME: return of pthread_cancel and actual cancellation happen
	//        asynchronously -> pthread_join should be added
#ifdef	OS_POSIX_DEBUG
	printf("DeleteThread : T=%x ret=%d\n", hThread, ret);
#endif
	return !ret;
}

/***********************************************************/
DWORD OS_WaitForThread (OS_THREAD hThread )
{
	int ret = 0;
	int result;
	
	result = pthread_join((pthread_t)hThread, (void **)ret); // sjc
#ifdef	OS_POSIX_DEBUG
	printf("WaitForThread : T=%x ret=%d\n", hThread, ret);
#endif
	return ret; // sjc
}

/***********************************************************/
OS_THREAD OS_GetCurrentThread()
{
	pthread_t thd = pthread_self();
	return (OS_THREAD)thd;
}

/***********************************************************/

typedef struct
{
	pthread_mutex_t	mutex_t;
	pthread_cond_t	cond_t;
	int				signalled;
} MY_EVENT;

OS_EVENT OS_CreateEvent ( UNUSED DWORD dwAttributes)
{
	MY_EVENT	*myEvent;
	int	ret;

	myEvent = (MY_EVENT *)malloc(sizeof(MY_EVENT));
	
	assert( myEvent != NULL );
	
	if ( !myEvent )
		return 0;
	
	myEvent->signalled = 0;
	ret = pthread_mutex_init(&myEvent->mutex_t, (pthread_mutexattr_t *) NULL);
	ret = pthread_cond_init(&myEvent->cond_t, (pthread_condattr_t *) NULL);

	return (OS_EVENT) myEvent;
}

/***********************************************************/
void OS_DeleteEvent ( OS_EVENT	hOSEvent)
{
	int ret;
	MY_EVENT *myEvent = (MY_EVENT *) hOSEvent;
	
	ret = pthread_mutex_destroy( &myEvent->mutex_t );
	ret = pthread_cond_destroy( &myEvent->cond_t );
	free(hOSEvent);
}

/***********************************************************/
void OS_SetEvent ( OS_EVENT hOSEvent)
{
	int ret;
	MY_EVENT *myEvent = (MY_EVENT *) hOSEvent;
	
	ret = pthread_mutex_lock(&myEvent->mutex_t);
	myEvent->signalled = 1;
	ret = pthread_cond_broadcast( &myEvent->cond_t );
	ret = pthread_mutex_unlock(&myEvent->mutex_t );
}

/***********************************************************/
void OS_ResetEvent ( OS_EVENT hOSEvent)
{
	int ret;
	MY_EVENT *myEvent = (MY_EVENT *) hOSEvent;
	
	ret = pthread_mutex_lock(&myEvent->mutex_t);
	myEvent->signalled = 0;
	ret = pthread_mutex_unlock(&myEvent->mutex_t );
}

/***********************************************************/
DWORD OS_WaitForEvent ( OS_THREAD hOSEvent, DWORD	timeout)
{
	int ret;
	MY_EVENT *myEvent = (MY_EVENT *) hOSEvent;
	struct timespec	ts;
	struct timeval tv;
	int		x;
	
	ret = pthread_mutex_lock(&myEvent->mutex_t);

	if ( myEvent->signalled )
	{
		// Already signalled...just reset and go
		
		myEvent->signalled = 0;
		ret = pthread_mutex_unlock(&myEvent->mutex_t );
		return 0;
	}
	else
	{
		// Wait for signal
		
		if (timeout == INFINITE)
			ret = pthread_cond_wait( &myEvent->cond_t, &myEvent->mutex_t );
		else
		{
			gettimeofday(&tv, NULL);
			ts.tv_sec = tv.tv_sec + timeout / 1000;
			x = tv.tv_usec + ((timeout % 1000) * 1000);
			while (x > 1000 * 1000)
			{
				ts.tv_sec++;
				x -= 1000*1000;
			}
			ts.tv_nsec = x * 1000;
			
			ret = pthread_cond_timedwait( &myEvent->cond_t, &myEvent->mutex_t, &ts );
		}

		if (ret == ETIMEDOUT)
		{
			// Timeout
			
			ret = pthread_mutex_unlock(&myEvent->mutex_t );
			
			return OS_ERROR_TIMEOUT;
		}
		else
		{
			// Signalled
			
			if (myEvent->signalled != 0)
			{
				// Reset the signal
				
				myEvent->signalled = 0;
			}
			
			ret = pthread_mutex_unlock(&myEvent->mutex_t );
			
			return 0;
		}
	}
}

/***********************************************************/
void OS_Sleep ( DWORD ms)
{
	usleep(ms * 1000);
}

/***********************************************************/

typedef struct
{
	pthread_mutex_t	mutex_t;
	pthread_t		handle;
	int				count;
	int			flg_locked;	
} MY_MUTEX;

OS_CRITICAL_SECTION OS_CreateCriticalSection ( UNUSED DWORD dwAttributes)
{
	MY_MUTEX	*myMutex;
	int ret;

	myMutex = (MY_MUTEX *)malloc(sizeof(MY_MUTEX));
	if ( !myMutex ) {
		fprintf(stderr, "OS_CreateCriticalSection failed\n");
		return 0;
	}
	
	myMutex->handle = (pthread_t) 0;
	myMutex->count = 0;
	myMutex->flg_locked = 0;
	
	ret = pthread_mutex_init(&myMutex->mutex_t, (pthread_mutexattr_t *) NULL);
	
	return (OS_CRITICAL_SECTION)myMutex;
}

/***********************************************************/
void OS_DeleteCriticalSection ( OS_CRITICAL_SECTION hCS )
{	
	int ret;
	MY_MUTEX *myMutex = (MY_MUTEX *) hCS;
	
	#ifdef _DEBUG
		assert( myMutex->count == 0 );
		assert( myMutex->handle == (pthread_t) 0 );
		myMutex->handle = (pthread_t) 0;
		myMutex->count = 0;
		myMutex->flg_locked = 0;
	#endif

	ret = pthread_mutex_destroy( &myMutex->mutex_t );
	free(hCS);
#ifdef	OS_POSIX_DEBUG
	if ( ret )
		printf("T=%x pthread_mutex_destory failed %x ret=%d\n", (int) pthread_self(), (int) hCS, ret);
#endif
}

/***********************************************************/
void OS_EnterCriticalSection ( OS_CRITICAL_SECTION hCS)
{
	int ret;
	MY_MUTEX *myMutex = (MY_MUTEX *) hCS;
	
	if (pthread_equal( pthread_self(), myMutex->handle ) )
	{
		// Recursive call to EnterCriticalSection, increase count
		
		myMutex->count++;
	}
	else
	{
		ret = pthread_mutex_lock(&myMutex->mutex_t);
		myMutex->handle = pthread_self();
		myMutex->flg_locked = 1;
	}
}

/***********************************************************/
void OS_LeaveCriticalSection ( OS_CRITICAL_SECTION hCS)
{
	int ret;
	MY_MUTEX *myMutex = (MY_MUTEX *) hCS;

	#ifdef _DEBUG
	{
		pthread_t self = pthread_self();
		assert( pthread_equal( self, myMutex->handle ) );
	}
	#endif
	
	if ( myMutex->count != 0 )
	{
		// Decrement the recursive count
		
		myMutex->count--;
	}
	else
	{
		myMutex->handle = (pthread_t) 0;
		ret = pthread_mutex_unlock(&myMutex->mutex_t);
		myMutex->flg_locked = 0;
	}
}

/***********************************************************/
OS_THREAD OS_CheckCriticalSection ( OS_CRITICAL_SECTION hCS)
{
    MY_MUTEX *myMutex = (MY_MUTEX *) hCS;

    if ( myMutex->flg_locked == 0 )
    {
        // cs is not locked
        return NULL;
    }
    else
    {
        return (OS_THREAD) myMutex->handle;
	}
}

/***********************************************************/
DWORD OS_GetTickCount ()
{
	struct timeval tv;

	gettimeofday(&tv, NULL);
	return tv.tv_sec * 1000 + tv.tv_usec / 1000;
}

/*********************************************************/
int OS_GetTimeStamp(char *buf, const int limit_size)
{
    // generate a finer( millisec ) timestamp



    struct timeval  tv;

    struct tm*      ptm;



    if (!buf)

        return 0;



    gettimeofday(&tv, NULL);

    ptm = localtime(&tv.tv_sec);

    strftime(buf, limit_size, "[%b %d %H:%M:%S", ptm);

    sprintf(buf+strlen(buf), ".%05d]", (int)tv.tv_usec/10000);

    

    return strlen(buf);
}

//**********************************************************/
// Read/Write Lock implementation
//**********************************************************/

//Structure for read/write locks
typedef struct rwlock_tag 
{
    pthread_mutex_t     mutex;          // mutex to protect this structure
    pthread_cond_t      read;           // wait for read 
    pthread_cond_t      write;          // wait for write 
    int                 valid;          // set when valid
    int                 r_active;       // readers active
    int                 w_active;       // writer  active
    int                 r_wait;         // readers waiting
    int                 w_wait;         // writers waiting
} rwlock_t;
#define RWLOCK_VALID    0xfacade

//------------------------------------------------------------------------
//
    OS_RWLOCK   
    OS_CreateRWLock
    ( 
        UNUSED DWORD dwAttributes
    )
//
// Initialize a read/write lock
//
{
    int status;

    rwlock_t * rwl = (rwlock_t *)malloc( sizeof( rwlock_t ) );
    rwl->r_active = 0;
    rwl->r_wait = rwl->w_wait = 0;
    rwl->w_active = 0;

    status = pthread_mutex_init( &rwl->mutex, NULL );
    if (status != 0)
        return NULL;

    status = pthread_cond_init( &rwl->read, NULL );
    if(status != 0)
    {
        // If unable to create read CV, destroy mutex 
        pthread_mutex_destroy( &rwl->mutex );
        return NULL;
    }
    
    status = pthread_cond_init( &rwl->write, NULL );
    if (status != 0)
    {
        // If unable to create read CV, destroy read CV and mutex 
        pthread_cond_destroy( &rwl->read );
        pthread_mutex_destroy( &rwl->mutex );
        return NULL; 
    }
    rwl->valid = RWLOCK_VALID;
    return ((OS_RWLOCK)rwl);
}


//------------------------------------------------------------------------
//
    int         
    OS_DeleteRWLock
    ( 
        OS_RWLOCK  rwlock 
    )
//
// Destroy a read/write lock
//
{
    int status, status1, status2;
    rwlock_t * rwl = (rwlock_t *)rwlock;
    
    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;
    status = pthread_mutex_lock( &rwl->mutex );
    if (status != 0)
        return status;
    //
    // Check whether any threads own the lock; report "EBUSY" if so
    //
    if (rwl->r_active > 0 || rwl->w_active)
    {
        pthread_mutex_unlock( &rwl->mutex );
        return EBUSY;
    }
    
    //
    // Check whether any threads are known to be waiting; report EBUSY if so
    //
    if (rwl->r_wait > 0 || rwl->w_wait > 0)
    {
        pthread_mutex_unlock( &rwl->mutex );
        return EBUSY;
    }

    rwl->valid = 0;
    status = pthread_mutex_unlock( &rwl->mutex );
    if (status != 0)
        return status;

    status = pthread_mutex_destroy( &rwl->mutex );
    status1 = pthread_cond_destroy( &rwl->read );
    status2 = pthread_cond_destroy( &rwl->write );

    return ( status != 0 ? status : (status1 != 0 ? status1: status2));
}

//------------------------------------------------------------------------
//
    static void 
    OS_ReadLockCleanup
    ( 
        OS_RWLOCK rwlock
    )
//
// Hamdle cleanup when the read lock condition variable wait is canceled.
//
// Simply record that the thread is no longer waiting, and unlock the mutex.
//
{
    rwlock_t * rwl = (rwlock_t *)rwlock;
    
    rwl->r_wait--;
    pthread_mutex_unlock( &rwl->mutex );
}

//------------------------------------------------------------------------
//
    static void
    OS_WriteLockCleanup
    ( 
        OS_RWLOCK  rwlock 
    )
//
// Handle cleanup when the write lock condition variable
// wait is canceled.
//
// Simply record that the thread is no longer waiting, 
// and unlock the mutex.
{
    rwlock_t * rwl = (rwlock_t *)rwlock;
    
    rwl->w_wait--;
    pthread_mutex_unlock( &rwl->mutex );
}

//------------------------------------------------------------------------
//
    int         
    OS_ReadLock
    ( 
        OS_RWLOCK  rwlock 
    )
//
// Lock a read/write lock for read access.
//
{
    int status;
    
    rwlock_t * rwl = (rwlock_t *)rwlock;

    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;
    status = pthread_mutex_lock( &rwl->mutex );
    if (status != 0)
        return status;
    
    if (rwl->w_active)
    {
        rwl->r_wait++;
        pthread_cleanup_push( OS_ReadLockCleanup, (OS_RWLOCK)rwl );
        while( rwl->w_active )
        {
            status = pthread_cond_wait( &rwl->read, &rwl->mutex );
            if (status != 0)
                break;
        }
        pthread_cleanup_pop( 0 );
        rwl->r_wait--;
    }
    
    if (status == 0)
        rwl->r_active++;
    
    pthread_mutex_unlock( &rwl->mutex );
    return status;
}

//------------------------------------------------------------------------
//
    int         
    OS_TryReadLock
    ( 
        OS_RWLOCK rwlock 
    )
//
// Attempt to lock a read/write lock for read access (don't
// block if unavailable)
//
{
    int status, status2;

    rwlock_t * rwl = (rwlock_t *)rwlock;

    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;

    status = pthread_mutex_lock( &rwl->mutex );
    if (status != 0)
        return status;
    if (rwl->w_active)
        status = EBUSY;
    else
        rwl->r_active++;

    status2 = pthread_mutex_unlock( &rwl->mutex );
    return (status2 != 0 ? status2 : status);
}

//------------------------------------------------------------------------
//
    int         
    OS_ReadUnlock
    ( 
        OS_RWLOCK rwlock 
    )
//
// Unlock a read/write lock from read access
//
{
    int status, status2;

    rwlock_t * rwl = (rwlock_t *)rwlock;

    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;

    status = pthread_mutex_lock( &rwl->mutex );

    if (status != 0)
        return status;

    rwl->r_active--;

    if (rwl->r_active == 0 && rwl->w_wait > 0)
        status = pthread_cond_signal( &rwl->write );

    status2 = pthread_mutex_unlock( &rwl->mutex );

    return (status2 == 0 ? status : status2);
}

//------------------------------------------------------------------------
//
    int         
    OS_WriteLock
    ( 
        OS_RWLOCK rwlock 
    )
//
// Lock a read/write lock for write access
//
{
    int status;

    rwlock_t * rwl = (rwlock_t *)rwlock;

    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;
    
    status = pthread_mutex_lock( &rwl->mutex );
    if (status != 0)
        return status;
    if (rwl->w_active || rwl->r_active > 0)
    {
        rwl->w_wait++;
        pthread_cleanup_push( OS_WriteLockCleanup, (void *)rwl );
        while( rwl->w_active || rwl->r_active > 0)
        {
            status = pthread_cond_wait( &rwl->write, &rwl->mutex );
            if (status != 0)
                break;
        }
        pthread_cleanup_pop( 0 );
        rwl->w_wait--;
    }
    if (status == 0)
        rwl->w_active = 1;
    pthread_mutex_unlock( &rwl->mutex );
    return status;
}

//------------------------------------------------------------------------
//
    int         
    OS_TryWriteLock
    ( 
      OS_RWLOCK rwlock 
    )
//
// Attempt to lock a read/write lock for write access. Don't block
// if unavailable.
//
{
    int status, status2;
    
    rwlock_t * rwl = (rwlock_t *)rwlock;

    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;

    status = pthread_mutex_lock( &rwl->mutex );
    if (status != 0)
        return status;
    if (rwl->w_active || rwl->r_active > 0)
        status = EBUSY;
    else
        rwl->w_active = 1;
    status2 = pthread_mutex_unlock( &rwl->mutex );
    return( status != 0 ? status : status2 );
}

//------------------------------------------------------------------------
//
    int         
    OS_WriteUnlock
    ( 
        OS_RWLOCK rwlock 
    )
//
// Unlock a read/write lock from write access.
//
{
    int status;
    
    rwlock_t * rwl = (rwlock_t *)rwlock;
    
    if (rwl->valid != RWLOCK_VALID)
        return EINVAL;
    status = pthread_mutex_lock( &rwl->mutex );
    if (status != 0)
        return status;
    rwl->w_active = 0;
    if (rwl->r_wait > 0)
    {
        status = pthread_cond_broadcast( &rwl->read );
        if (status != 0)
        {
            pthread_mutex_unlock( &rwl->mutex );
            return status;
        }
    }
    else if (rwl->w_wait > 0)
    {
        status = pthread_cond_signal( &rwl->write );
        if(status != 0)
        {
            pthread_mutex_unlock( &rwl->mutex );
            return status;
        }    
    }
    status = pthread_mutex_unlock( &rwl->mutex );
    return status;
}

int WSAGetLastError()
{ 
	return errno;
}

int OS_GetLastSocketError()
{
	return WSAGetLastError();
}
