/*-------------------------------------------------------------------------------

	RDM_DDA_Utility.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Utility functions useful to DDA modules.


--------------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <pp/syms.h>
#include <pp/RDM_DDA_Utility.h>
#include <pp/RDM.h>
#include <pp/SXML_Out.h>
#include <pp/SXDB.h>
#include <pp/SXDB_Parse_Table.h>

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

typedef struct 
{
    const char    *pComponent;
    const char    *pVersionType;
    int     compIndex;
    const char    *pVersionInfo;
} VERSION_INFO;

#define PT_STRUCT   VERSION_INFO
PT_BEGIN    ( "Version",        verInfoTable,    PT_NO_UNKNOWN )
PT_ATT      ( "Component",      pComponent,        0,  PT_STRING_PTR )
PT_ATT      ( "VersionType",    pVersionType,      0,  PT_STRING_PTR )
PT_ATT      ( "CompIndex",       compIndex,         0,  PT_INT )
PT_ROOT_DATA(                   pVersionInfo,      0,  PT_STRING_PTR )
PT_END
#undef  PT_STRUCT

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//
	int									// <0 = error (port was not found)
	DDA_UpdatePortStatus
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// ID of the port
		int		status					// Port status (0,1,2)
	)
//
//	Updates the port status and sends out the notification
//
//------------------------------------------------------------------------------//
{
	char		string[RDM_MAX_NAME + RDM_MAX_PORT_NAME + RDM_MAX_ID + 110];
	CSXDB_Node	*pNode;
	const char		*pPortName;
	char		xmlPortName[RDM_MAX_PORT_NAME];
	const char		*pDeviceName;
	char		xmlDeviceName[RDM_MAX_NAME];
	int			result = 0;
	char		statStr[2];

	pRDM->db->BeginAccess( 1 );

	// Find the port

	pNode = pRDM->db->GetNodeFromID( pID );

	if (pNode == NULL)
	{
		result = -1;
		goto EXIT;
	}

	// Get the name ptrs

	pDeviceName = pRDM->db->GetData("../Name",NULL,pNode);
	pPortName = pRDM->db->GetData("Name", NULL, pNode);
	
	if (pPortName == NULL) pPortName = "";
	if (pDeviceName == NULL) pDeviceName = "";

	SXMLOut(xmlPortName,pPortName,RDM_MAX_PORT_NAME );
	SXMLOut(xmlDeviceName,pDeviceName,RDM_MAX_NAME );
	
	// Update the status

	statStr[0] = (char) (status + '0');
	statStr[1] = 0;

	pRDM->db->PutData("","Status",SXDB_TYPE_ATTRIBUTE,statStr,NULL,pNode);

	// Make the event string

	sprintf(string,"<Device><Name>%s</Name><Port Status=\"%d\"><Name>%s</Name></Port></Device>",
					xmlDeviceName,
					status,
					xmlPortName );

	// Send the event
	pRDM->nm.Notify(pID, RDM_EC_Port_Status_Changed, string );

EXIT:
	pRDM->db->EndAccess( );

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// <0 = error (port was not found)
	DDA_UpdatePathStatus
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// ID of the path
		int		status					// Port status (0,1,2)
	)
//
//	Updates the path status and sends out the notification
//
//------------------------------------------------------------------------------//
{
	char		string[RDM_MAX_NAME + RDM_MAX_PORT_NAME + RDM_MAX_ID + 110];
	CSXDB_Node	*pNode;
	const char		*pPortName;
	char		xmlPortName[RDM_MAX_PORT_NAME];
	const char		*pDeviceName;
	char		xmlDeviceName[RDM_MAX_NAME];
    const char        *pPortID;
	int			result = 0;
	char		statStr[2];

	pRDM->db->BeginAccess( 1 );

	// Find the port

	pNode = pRDM->db->GetNodeFromID( pID );

	if (pNode == NULL)
	{
		result = -1;
		goto EXIT;
	}

	// Get the name ptrs

	pDeviceName = pRDM->db->GetData("../../Name",NULL,pNode);
	pPortID = pRDM->db->GetData("../@id",NULL,pNode);
	pPortName = pRDM->db->GetData("../Name",NULL,pNode);
	
	if (pPortID == NULL) pPortID = "";
	if (pPortName == NULL) pPortName = "";
	if (pDeviceName == NULL) pDeviceName = "";

	SXMLOut(xmlPortName,pPortName,RDM_MAX_PORT_NAME );
	SXMLOut(xmlDeviceName,pDeviceName,RDM_MAX_NAME );
	
    // Update the status

	statStr[0] = (char) (status + '0');
	statStr[1] = 0;

	pRDM->db->PutData("","Status",SXDB_TYPE_ATTRIBUTE,statStr,NULL,pNode);

	// Make the event string

    sprintf(string,
            "<Device><Name>%s</Name><Port id=\"%s\"><Name>%s</Name><Path id=\"%s\" Status=\"%d\"></Path></Port></Device>",
            xmlDeviceName,
            pPortID,
            xmlPortName,
            pID,
            status
           );

	// Send the event

	pRDM->nm.Notify(pID, RDM_EC_Path_Status_Changed, string );

EXIT:
	pRDM->db->EndAccess( );

	return result;
}


//--------------------------------------------------------------------------------
//
	void
	DDA_NotifyWithName
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pElement,				// Root Element name (<Device> <User> <Group>)
		const char	*pName,					// Name of the device, user or group
		const char	*pID,					// The ID to send to
		int		eventID					// The event ID
	)
//
//	Many of the notifications contain data in the form of:
//		<Device><Name>xxx</Name></Device>
//		<User><Name>xxx</Name></User>
//		<Group><Name>xxx</Name></Group>
//	This function is just a handy short cut to create the data
//	Sends a notification one data element  <%pElement><Name>%pData</Name></%pElement>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<%s><Name>%s</Name></%s>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) - 15 );

	sprintf(data,pFormat,pElement,xmlName,pElement);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_UserName
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*pName					// Name of the user
	)
//
//	Sends a notification with <User><Name>%s</Name></User>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<User><Name>%s</Name></User>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,xmlName);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_GroupName
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*pName					// Name of the group
	)
//
//	Sends a notification with <Group><Name>%s</Name></Group>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Group><Name>%s</Name></Group>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,xmlName);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_System
	(
		CRDM	*pRDM,					// Ptr to RDM
		int		eventID					// The event ID
	)
//
//	Sends an event with the "System" id an no data
//
//------------------------------------------------------------------------------//
{
	pRDM->nm.Notify( "System", eventID, NULL );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_System
	(
		CRDM	*pRDM,					// Ptr to RDM
		int		eventID,				// The event ID
		const char	*mode					// The mode value
	)
//
//	Sends an event with the "System" id and  data
//
//------------------------------------------------------------------------------//
{
	pRDM->nm.Notify( "System", eventID, mode );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_Text
	(
		CRDM	*pRDM,					// Ptr to RDM
		int		eventID,				// The event ID
		const char	*pText					// Name of the group
	)
//
//	Sends a notification with <Group><Name>%s</Name></Group>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Text>%s</Text>";
	char	xmlText[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlText,pText,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,pText);

	pRDM->nm.Notify( "System", eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_Net
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pSessionID,			// The session ID
		int		eventID,				// The event ID
		const char	*pName,					// Name of the group
		DWORD	ipAddress				// The IP Address
	)
//
//	Sends a notification with <User><Name>%s</Name></User><IPAddress>%s</IPAddress>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<User><Name>%s</Name></User><IPAddress>%d.%d.%d.%d</IPAddress>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) - 4 );

	sprintf(	data,
				pFormat,
				xmlName,
				ipAddress >> 24,
				(ipAddress >> 16) & 0xFF,
				(ipAddress >> 8) & 0xFF,
				(ipAddress & 0xFF)
		   );

	pRDM->nm.Notify( pSessionID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_ID
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID					// The event ID
	)
//
//	Sends a notification with <Group><Name>%s</Name></Group>
//
//------------------------------------------------------------------------------//
{
	pRDM->nm.Notify( pID, eventID, NULL );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_Error
	(
		CRDM	*pRDM,					// Ptr to RDM
		int		eventID,				// The event ID
		DWORD	errorCode,				// The error code
		const char	*pFileName,				// File name the error occurred in
		DWORD	lineNo					// Line number in the file
	)
//
//	Sends a notification with <Error>%d</Error><File>%s</File><Line>%d</Line>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Error>%08X</Error><File>%s</File><Line>%d</Line>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pFileName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,errorCode,xmlName,lineNo);

	pRDM->nm.Notify( "System", eventID, NULL );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_Access_Port
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pSessionID,			// The session ID
		int		eventID,				// The event ID
		const char	*pUserName,				// The User name
		const char	*pPortID,				// The port ID
		const char	*pPortName				// The Port name
	)
//
//	Sends a notification with <Error>%d</Error><File>%s</File><Line>%d</Line>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<User><Name>%s</Name></User><Port id=\"%s\"><Name>%s</Name></Port>";
	char	xmlPortName[RDM_MAX_NOTIFICATION_DATA];
	char	xmlUserName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlUserName,pUserName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) - 16 );
	SXMLOut(xmlPortName,pPortName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) - strlen(xmlUserName) );

	sprintf(data,pFormat,xmlUserName,pPortID,xmlPortName);

	pRDM->nm.Notify( pSessionID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_Port_Change
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pPortID,				// The session ID
		const char	*pDeviceName,			// The device name
		const char	*pPortName,				// The Port name
		const char	*pStatus				// The Port Status
	)
//
//	Sends a notification with Port_Changed format
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Device><Name>%s</Name><Port Status=\"%s\"><Name>%s</Name></Port></Device>";
	char	xmlDevice[RDM_MAX_NOTIFICATION_DATA];
	char	xmlPort[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlDevice,pDeviceName,(RDM_MAX_NOTIFICATION_DATA - strlen(pFormat)) / 2 );
	SXMLOut(xmlPort,  pPortName,  (RDM_MAX_NOTIFICATION_DATA - strlen(pFormat)) / 2 );

	sprintf(data,pFormat,xmlDevice,pStatus,xmlPort);

	pRDM->nm.Notify( pPortID, RDM_EC_Port_Changed, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_Path_Change
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pPathID				// The Path ID
	)
//
//	Sends a notification with Path_Changed format.
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA]; //Data to be sent in notification
	char	query[128];
    const char *  Format = "<Device>"
                        "<Name>%s</Name>"
                        "<Port Status=\"%s\">"
                            "<Name>%s</Name>"
                            "<Path id=\"%s\" Status=\"%s\">"
                            "</Path>"
                        "</Port>"
                      "</Device>";
    CSXPath p;

    CSXDB_Node * DeviceNode = NULL;
    CSXDB_Node * PortNode   = NULL;
    CSXDB_Node * PathNode   = NULL;

    CSXDB_Node * DeviceNameNode = NULL;
    CSXDB_Node * PortNameNode = NULL;
    CSXDB_Node * PortStatusNode = NULL;
    CSXDB_Node * PathStatusNode = NULL;

    const char * devName      = NULL; //Device Name
    const char * portName     = NULL; //Port Name
    const char * portStatus   = NULL; //Port Status Name
    const char * pathStatus   = NULL; //Path Status Name

    //Retrive each of the above data from RDM

    //Retrive PathNode 
    sprintf( query, "/System/Device/Port/Path[@id=\"%s\"]", pPathID );
    
    int count = p.Parse( query, pRDM->db, NULL, 1, NULL );
    if (count >= 0)
        PathNode = p.Enum( 0 );

    //Get the status from Path Node
    if (PathNode != NULL)
    {
        PathStatusNode = PathNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "Status", NULL );
        if (PathStatusNode != NULL)
            pathStatus = PathStatusNode->GetData();
        else
            pathStatus = "0";
    }

    //Retrive Port Name and Status
    if (PathNode != NULL)
        PortNode = PathNode->Parent();
    
    if (PortNode != NULL)
    {
        PortNameNode = PathNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "Name", NULL );
        if (PortNameNode != NULL)
            portName = PortNameNode->GetData();
        else
            portName = "";

        PortStatusNode = PathNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "Status", NULL );
        if (PortStatusNode != NULL)
            portStatus = PortStatusNode->GetData();
        else
            portStatus = "0";
    }

    //Retrive Device Name 
    if (PortNode != NULL)
        DeviceNode = PortNode->Parent();

    if (DeviceNode != NULL)
    {
        DeviceNameNode = PathNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "Name", NULL );
        if (DeviceNameNode != NULL)
            devName = DeviceNameNode->GetData();
        else
            devName = "";
    }

    sprintf( data, Format, devName, portStatus, portName, pPathID, pathStatus );
    pRDM->nm.Notify( pPathID, RDM_EC_Path_Changed, data );
}

//--------------------------------------------------------------------------------
//
int
DDA_FindDownloadPath
(
    CRDM *pRDM
)
//
//	Find the PMA Download path
//
//------------------------------------------------------------------------------//
{
    char query[128];
    int count = 0;
    int pathCount = 0;
    CSXPath p;
    CSXDB_Node * PathNode = NULL;
    CSXDB_Node * PathLANode = NULL;

    strcpy( query, "/System/Device/Port/Path" );

    pathCount = p.Parse( query, pRDM->db, NULL, 0, NULL ); 
    if (pathCount == 0)
        return -1;

    for (int index = 0; index < pathCount; index++)
    {
        sprintf( query, "/System/Device/Port/Path[%d]",index + 1 );

       //Retrive the path Node 
        count =  p.Parse( query, pRDM->db, NULL, 1, NULL );
        if (count >= 0)
            PathNode = p.Enum( 0 );

        //Retrive the Local Access Element
        if (PathNode != NULL)
        {
            PathLANode =  PathNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, "LocalAccess", NULL );

            if (PathLANode != NULL)
            {
                const char * type = PathLANode->GetData();

                if (type != NULL)
                {
                    if (!strcmp( type, "0" ))
                    {
                        return index;
                    }
                }
            }
        }
    }

    return 0;
}

//--------------------------------------------------------------------------------
//
    char *
    DDA_ConvertVerInfo
    (
        const char * dottedVer,   //The dotted format version information
        char * charCodeVer // Out value giving you the character code version
    )

//
//	Converts a version information from its dotted representation to character code
//  version
//
//  ex: 0.10.1.37 will be converted to 0A1b
//------------------------------------------------------------------------------//
{
    const char * seps = ".";
    char * token = NULL;
    char retVer[5];
    memset( retVer, 0, 5 );

    if (dottedVer == NULL)
        return NULL;

    char * actualVer = new char[strlen( dottedVer ) + 1];

    if (actualVer != NULL)
        strcpy( actualVer, dottedVer );
    else
        return NULL;

    char * delPtr = actualVer;
    token = strtok( actualVer, seps );
    while ( token != NULL )
    {
        int n = atoi( token );

        if ( n >= 0 && n < 10)
        {
            strcat( retVer, token );
        }
        else if (n >= 10 && n < 36)
        {
            char temp[2];
            sprintf( temp, "%c", (n + '0') + ('A' - '9') - 1);
            strcat( retVer, temp );
        }
        else if (n >= 36 && n < 62)
        {
            char temp[2];
            sprintf( temp, "%c", (n + 'A') - ( 'e' - 'a' ) );
            strcat( retVer, temp );
        }
        else
            strcat( retVer, "*" );

        token = strtok( NULL, seps );
    }

    strcpy( charCodeVer, retVer );

    delete delPtr;

    return charCodeVer;
}

//--------------------------------------------------------------------------------
//
    int
    DDA_AddVersionInfo
    (
        CRDM    *pRDM,                  // Ptr to RDM
        const char    *pDeviceID,             // Ptr to Device ID
        const char    *pComponent,            // Which component, e.g. CLIPR, UMT, etc
        const char    *pVersionType,          // Type of version, e.g. SW, FW, HW, FPGA, etc
        const char    *pVersionInfo,          // String representing the version
        int     componentIdx            // Component index attribute, -1 => Will not add
    )
//
//	Adds the version information to the given device. If the component index is -1
//  then we do not add the attribute component index. If the version node exist
//  this function updates the values
//
//------------------------------------------------------------------------------//
{
    int result;
    char xpath[128];
    char query[128];
    VERSION_INFO verInfo;
    CSXDB_Node * pNode = NULL;

    sprintf( xpath, "/System/Device[@id=\"%s\"]", pDeviceID );
    verInfo.pComponent         = pComponent;
    verInfo.pVersionType       = pVersionType;
    verInfo.compIndex          = componentIdx;
    verInfo.pVersionInfo       = pVersionInfo;

    sprintf( query, "/System/Device[@id=\"%s\"]/Version[@VersionType=\"%s\"]" ,
                pDeviceID, pVersionType );
    pNode = pRDM->db->GetNodeFromXPath( query );

    if (pNode != NULL)
    {
        //Check if the component and the component index match too
        CSXDB_Node * pNodeComp = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "Component", NULL);
        if (pNodeComp != NULL)
        {
            if ( strcmp(pNodeComp->GetData(), pComponent) != 0)
               pNode = NULL; 
            else
            {
                CSXDB_Node * pNodeIdx = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE,"CompIndex", NULL);
                if (pNodeIdx != NULL)
                {
                    char temp[16];
                    sprintf( temp, "%d",componentIdx );
                    if ( strcmp(pNodeIdx->GetData(), temp) != 0 )
                        pNode = NULL;
                }
            }
        }
    }

    if ( pNode == NULL )
    {
        //None of the attributes match, so append
        result = SXDB_PT_Put(pRDM->db, xpath, PT_CHILD_APPEND , verInfoTable, &verInfo, 0 );
    }
    else
    {
        result = SXDB_PT_Put(pRDM->db, query, PT_NODE_UPDATE, verInfoTable, &verInfo, 0 );
    }
    
    return result;
}

//--------------------------------------------------------------------------------
//
	int									// <0 = error (port was not found)
	DDA_UpdateResourceStatus
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The session ID
		int		status					// Port or Path status (0,1,2)
	)
//
//	Updates the port status and sends out the notification
//
//------------------------------------------------------------------------------//
{
	CSXDB_Node	*pNode;
	int			result = 0;
    const char *pNodeName;

	pRDM->db->BeginAccess( 1 );

	// Find the port

	pNode = pRDM->db->GetNodeFromID( pID );
    
	if (pNode == NULL)
	{
		result = -1;
		goto EXIT;
	}
    
    pNodeName = pNode->GetName();

    if (strcmp (pNodeName, "Path") == 0)
    {
        result = DDA_UpdatePathStatus (pRDM, pID, status);
    }
    else if (strcmp (pNodeName, "Port") == 0)
    {
        result = DDA_UpdatePathStatus (pRDM, pID, status);
    }

EXIT:
	pRDM->db->EndAccess( );

	return result;
}

//--------------------------------------------------------------------------------
//
    int
    DDA_ReleaseResources
	(
		CRDM	*pRDM,				// Ptr to RDM
		const char	*pID,			// The session ID
		int		NOTUSED(status)		// Port or Path status (0,1,2)
	)
// 
// Release the resources held by the session with id=pID
//--------------------------------------------------------------------------------
{
    int result = 0;
    const char *resourceid;
    char xpath[128];

    CSXDB_Node	*pNode;
	pRDM->db->BeginAccess( 1 );

    sprintf (xpath, "/System/Sessions/Session[@id=\"%s\"]/ResourceUsed", pID);
    
    pNode = pRDM->db->GetNodeFromXPath (xpath);

    while (pNode != NULL)
    {
        // Get resource id from session and update the path status

		if ( strcmp(pNode->GetName(), "ResourceUsed")==0 )
		{
        resourceid = pNode->GetData();

        result = DDA_UpdateResourceStatus (pRDM, resourceid, 1);
		}

        pNode = pNode->Next();
    }
	pRDM->db->EndAccess( );

    return result;
}

//--------------------------------------------------------------------------------
//
	void
	DDA_NotifyWithNameChanged
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pElement,				// Root Element name (<Device> <User> <Group>)
		const char	*pName,					// Name of the device, user or group
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*pAdminUser				// Admin user who changed the pElement
	)
//
//	Many of the notifications contain data in the form of:
//		<Device><Name>xxx</Name></Device>
//		<User><Name>xxx</Name></User>
//		<Group><Name>xxx</Name></Group>
//	This function is just a handy short cut to create the data
//	Sends a notification one data element  <%pElement><Name>%pData</Name></%pElement>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<LoginUser><Name>%s</Name></LoginUser><%s><Name>%s</Name></%s>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) - 15 );

	sprintf(data, pFormat,pAdminUser, pElement, xmlName, pElement);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_UserChanged
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*pName,					// Name of the user
		const char	*pAdminUser				// Admin user who changed the user	    
	)
//
//	Sends a notification with <User><Name>%s</Name></User>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<LoginUser><Name>%s</Name></LoginUser><User><Name>%s</Name></User>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,pAdminUser, xmlName);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_GroupChanged
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*pName,					// Name of the group
		const char	*pAdminUser				// Admin user who changed the group
	)
//
//	Sends a notification with <Group><Name>%s</Name></Group>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<LoginUser><Name>%s</Name></LoginUser><Group><Name>%s</Name></Group>";
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,pAdminUser, xmlName);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_CCManagement
	(
		CRDM	*pRDM,					// Ptr to RDM
		int		eventID,				// The event ID
		const char	*pServerName,			// Name of this server
		const char	*pName,					// CC user name
		DWORD	address					// CC ipAddress
	)
//
//	Sends a notification with <Group><Name>%s</Name></Group>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
    const char *  pFormat = "<User><Name>%s</Name><IPAddress>%d.%d.%d.%d</IPAddress></User>";

	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf ( data, pFormat, pName, 
		(address >> 24),((address >> 16) & 0xFF),((address >> 8) & 0xFF),(address & 0xFF) );

	pRDM->nm.Notify( pServerName, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_System_FactoryReset
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pName,					// Name of the user
		DWORD	address					// IPAddress if reset if performed by remote user
	)
//
//	Sends a notification with <Group><Name>%s</Name></Group>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
    const char *  pFormat = "<User><Name>%s</Name><IPAddress>%d.%d.%d.%d</IPAddress></User>";

	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pName,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf ( data, pFormat, xmlName,
		(address >> 24),((address >> 16) & 0xFF),((address >> 8) & 0xFF),(address & 0xFF) );
	
	pRDM->nm.Notify( "System", RDM_EC_System_Factory_Reset, data );
}


//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_PasswordChanged
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*pTargetUser,			// User whose password is changed
		const char	*pUser,					// User who changed the password
		DWORD	address					// IPAddress if reset if performed by remote user
	)
//
//	Sends a notification with <User><Name>%s</Name></User>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<PasswdChg><LoginUser><Name>%s</Name><IPAddress>%d.%d.%d.%d</IPAddress></LoginUser><User><Name>%s</Name></User></PasswdChg>";

	char	xmlName[RDM_MAX_NOTIFICATION_DATA];
	SXMLOut(xmlName,pUser,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data,pFormat,xmlName, 
		(address >> 24),((address >> 16) & 0xFF),((address >> 8) & 0xFF),(address & 0xFF), pTargetUser );

	pRDM->nm.Notify( pID, eventID, data );
}


//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_DeviceUpdateStarted
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		const char	*fwVersion,				// Firmware version being upgraded
		const char	*pUser,					// User who changed the password
		DWORD	address					// IPAddress if reset if performed by remote user
	)
//
//	Sends a notification with <User><Name>%s</Name></User>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Device><User><Name>%s</Name><IPAddress>%d.%d.%d.%d</IPAddress></User><Firmware><Version>%s</Version></Firmware></Device>";

	char	xmlName[RDM_MAX_NOTIFICATION_DATA];
	SXMLOut(xmlName,pUser,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );

	sprintf(data, pFormat, xmlName, 
		(address >> 24),((address >> 16) & 0xFF),((address >> 8) & 0xFF),(address & 0xFF), fwVersion);

	pRDM->nm.Notify( pID, RDM_EC_Device_Update_Started, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_DeviceUpdateCompleted
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		const char	*fwVersion,				// Firmware version being upgraded
		const char	*pUser,					// User who changed the password
		DWORD	address,				// IPAddress if reset if performed by remote user
		DWORD	status
	)
//
//	Sends a notification with <User><Name>%s</Name></User>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Device><User><Name>%s</Name><IPAddress>%d.%d.%d.%d</IPAddress></User><Firmware><Version>%s</Version><Status>%d</Status></Firmware></Device>";

	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlName,pUser,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );
	
	sprintf(data, pFormat, xmlName, 
		(address >> 24),((address >> 16) & 0xFF),((address >> 8) & 0xFF),(address & 0xFF), fwVersion, status);

	pRDM->nm.Notify( pID, RDM_EC_Device_Update_Completed, data );
}

//--------------------------------------------------------------------------------
//
	void
	DDA_Notify_DeviceConfig_Backup_Restore
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// The ID to send to
		int		eventID,				// The event ID
		const char	*fwVersion,				// Firmware version being upgraded
		const char	*type,					// Type of Config file (Device or User)
		const char	*pUser,					// User who changed the password
		DWORD	address,				// IPAddress if reset if performed by remote user
		DWORD	status
	)
//
//	Sends a notification with <User><Name>%s</Name></User>
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Device><User><Name>%s</Name><IP>%d.%d.%d.%d</IP></User><Config><Ver>%s</Ver><Type>%s</Type><Sts>%d</Sts></Config></Device>";

	char	xmlName[RDM_MAX_NOTIFICATION_DATA];
	SXMLOut(xmlName,pUser,RDM_MAX_NOTIFICATION_DATA - strlen(pFormat) );
	
	sprintf(data, pFormat, xmlName, 
		(address >> 24),((address >> 16) & 0xFF),((address >> 8) & 0xFF),(address & 0xFF), fwVersion, type, status);

	pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
    void
    DDA_Notify_DeviceReset
    (
        CRDM    *pRDM,                  // Ptr to RDM
        const char    *pID,                   // The ID to send to
        int     eventID,                // The event ID
        const char    *pName,					// Device name
        const char    *pResetMode             // Device reset mode
    )
//
//  Sends a notification with <Device><Name>%s</Name><DeviceResetMode>%s</DeviceResetMode></Device>
//
//------------------------------------------------------------------------------//
{
    char    data[RDM_MAX_NOTIFICATION_DATA];
    const char    *pFormat = "<Device><Name>%s</Name></Device>%s";

    sprintf(data, pFormat, pName, pResetMode);

    pRDM->nm.Notify( pID, eventID, data );
}

//--------------------------------------------------------------------------------
//
	void									// <0 = error (port was not found)
	DDA_UpdatePowerSupplyStatus
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// ID of the power supply node
		const char	*pDeviceName,			// The device name
		const char	*pPortName,				// The power supply node name
		int		status,					// status (0 or 1)
		int		index
	)
//
//	Updates the power supply status and sends out the notification
//
//------------------------------------------------------------------------------//
{
	char	data[RDM_MAX_NOTIFICATION_DATA];
	const char	*pFormat = "<Name>%s</Name><PowerSupply Status=\"%d\" id=\"%s\" index=\"%d\"><Name>%s</Name></PowerSupply>";

	char	xmlDevice[RDM_MAX_NOTIFICATION_DATA];
	char	xmlName[RDM_MAX_NOTIFICATION_DATA];

	SXMLOut(xmlDevice,pDeviceName,(RDM_MAX_NOTIFICATION_DATA - strlen(pFormat)) / 2 );
	SXMLOut(xmlName,  pPortName,  (RDM_MAX_NOTIFICATION_DATA - strlen(pFormat)) / 2 );

	sprintf ( data, pFormat, xmlDevice, status, pID, index, xmlName );

	pRDM->nm.Notify( pID, RDM_EC_Device_PowerSupply_Status_Changed, data );
}
