/*-------------------------------------------------------------------------------

	RDM_Notify.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Raritan Device Manager Protocol Notify service handler class (CRDM_Notify)

	This module provides two interfaces:
		1) It is the RDM_Service for the Notify messages and
		2) It provides the Notify function for sending notifications.

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<stdlib.h>				// for atoi()
#include	"pp/base.h"
#include	"pp/RDM.h"
#include	"pp/Hash32.h"
#include	"pp/RDM_Notify.h"
#include	"pp/CmdLineParse.h"
#include	"pp/SXDB_Parse_Table.h"
#include	"pp/RDM_Remote_Notify.h"
#include	"pp/RDM_EventCode.h"
#include	"pp/SXDB_Parse.h"
#include	"pp/SXML_Out.h"
//#include	"TimeDate.h"
//----------------------------------------
//				Equates
//----------------------------------------

	// Function IDs

enum
{
	CMD_Version,
	CMD_Subscribe,
	CMD_Unsubscribe,
	CMD_Notify,
	CMD_GetEvents
};

//----------------------------------------
//				Data Types
//----------------------------------------

typedef struct 
{
	DWORD		eventCode;
	const char	*	pName;
} EVENTNAME;

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

static int nextSubscriptionID = 1;		// Subscription ID counter

	// Function List

static TOKEN	functions[] = 
{
	{ "Version",	CMD_Version },
	{ "Subscribe",	CMD_Subscribe },
	{ "Cancel",		CMD_Unsubscribe },
	{ "Notify",		CMD_Notify },
	{ "GetEvents",	CMD_GetEvents },
	{ NULL,			0 } // End of list
};

//----------------------------------------
//		Subscribe Function Parse Table
//----------------------------------------

	// Subscribe Function Structure

typedef	struct
{
	const char		*pXPath;
	const char		*pEvents;
	const char		*pIDs;		
	CSXDB_Node	*time;
	CSXDB_Node	*serialNo;
	CSXDB_Node	*nodeID;
	CSXDB_Node	*parentNodeID;
	CSXDB_Node	*subID;
	CSXDB_Node	*sendData;
	CSXDB_Node	*deviceID;
} SUB_FUNC_DATA;

	// Subscribe Function Parse Table

#define	PT_STRUCT	SUB_FUNC_DATA
PT_BEGIN	( "Subscribe",		subTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Select",			pXPath,			0,	PT_STRING_PTR )
PT_ELEM		( "ID",				pIDs,			0,	PT_STRING_PTR )
PT_ELEM		( "Events",			pEvents,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Time",			time,			0,	PT_NODE )
PT_ELEM		( "NodeID",			nodeID,			0,	PT_NODE )
PT_ELEM		( "SerialNo",		serialNo,		0,	PT_NODE )
PT_ELEM		( "ParentNodeID",	parentNodeID,	0,	PT_NODE )
PT_ELEM		( "SubscriptionID",	subID,			0,	PT_NODE )
PT_ELEM		( "SendData",		sendData,		0,	PT_NODE )
PT_ELEM		( "DeviceID",		deviceID,		0,	PT_NODE )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Unsubscribe Function Parse Table
//----------------------------------------

	// Unsubscribe Function Structure

typedef	struct
{
	int		subID;
} UNSUB_FUNC_DATA;

	// Subscribe Function Parse Table

#define	PT_STRUCT	UNSUB_FUNC_DATA
PT_BEGIN	( "Cancel",			unsubTable,		PT_NO_UNKNOWN )
PT_ELEM		( "SubscriptionID",	subID,			0,	PT_INT )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Notify Function Parse Table
//----------------------------------------

	// Unsubscribe Function Structure

typedef	struct
{
	const char	*pEvent;
	const char	*pID;
	const char	*pData;
} NOTIFY_FUNC_DATA;

	// Subscribe Function Parse Table

#define	PT_STRUCT	NOTIFY_FUNC_DATA
PT_BEGIN	( "Notify",			notifyTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Event",			pEvent,			0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "ID",				pID,			0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Data",			pData,			0,	PT_STRING_PTR | PT_REQUIRED )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		GetEvents Function Parse Table
//----------------------------------------

	// Subscribe Function Structure

typedef	struct
{
	int			count;
	const char	*	pStart;
	const char	*	pDir;
	const char	*	pFormat;		
} GETEVENTS_FUNC_DATA;

	// Subscribe Function Parse Table

#define	PT_STRUCT	GETEVENTS_FUNC_DATA
PT_BEGIN	( "GetEvents",		getEventsTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Count",			count,			0,	PT_INT )
PT_ELEM		( "Start",			pStart,			0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Direction",		pDir,			0,	PT_STRING_PTR )
PT_ELEM		( "Format",			pFormat,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT


	// Event Format strings
	// These strings are closely linked with ConvertEventToText()

EVENTNAME	eventName[]=
{
	{	RDM_EC_Invalid,						"Unknown event %d"	},
	{	RDM_EC_Device_Added,				"Device %s added"	},
	{	RDM_EC_Device_Changed,				"Device %s changed"	},
	{	RDM_EC_Device_Deleted,				"Device %s removed"	},
	{	RDM_EC_Device_Change_Request,		"%s changed a device"	},
	{	RDM_EC_Device_Setting_Change,		"Device config changed"	},
	{	RDM_EC_Device_Setting_Change_Request,"%s changed device config"	},
	{	RDM_EC_Device_Status_Changed,		"Device %s status -- %s"},
	{	RDM_EC_Device_Key_Event,             "Special Key detected on %s"},
	{	RDM_EC_Device_Update_Started,		"%s updating device with firmware ver %s"	},
	{	RDM_EC_Device_Update_Completed,		"Device update completed (status = %s)"	},
	{	RDM_EC_Device_Config_Backup,		"%s backed up %s version %s"	},
	{	RDM_EC_Device_Config_Restore,		"%s restored %s	version %s"	},
	{	RDM_EC_Device_PowerSupply_Status_Changed,"PowerSupply %s is %s"	},
    {   RDM_EC_Device_Reset,				"Device %s reset, reset mode is %s" },

	{	RDM_EC_Port_Added,					"Port %s added"	},
	{	RDM_EC_Port_Changed,				"Port %s changed"	},
	{	RDM_EC_Port_Deleted,				"Port removed"	}, // Hack - No name ?
	{	RDM_EC_Port_Change_Request,			"%s changed a port"	},
	{	RDM_EC_Port_Status_Changed,			"Port %s is %s"	},
	{	RDM_EC_User_Added,					"User %s added"	},
	{	RDM_EC_User_Changed,				"User %s changed"	},
	{	RDM_EC_User_Deleted,				"User %s deleted"	},
	{	RDM_EC_User_Change_Request,			"%s changed a user"	},
	{	RDM_EC_User_Password_Changed,		"%s changed %s's password"	},
	{	RDM_EC_Group_Added,					"Group %s added"	},
	{	RDM_EC_Group_Changed,				"Group %s changed"	},
	{	RDM_EC_Group_Deleted,				"Group %s deleted"	},
	{	RDM_EC_Group_Change_Request,		"%s changed a Group"	},
	{	RDM_EC_System_Startup,				"System startup"	},
	{	RDM_EC_System_Shutdown,				"System shutdown"	},
	{	RDM_EC_System_Init_Error,			"Startup Error %s"	},
	{	RDM_EC_System_Fatal_Error,			"Fatal Error %s"	},
	{	RDM_EC_System_Run_Error,			"Runtime Error %s"	},
	{	RDM_EC_System_General,				"%s"	},
	{	RDM_EC_System_Reset,				"System Reset option is %s"	},
	{	RDM_EC_System_Start_Management,		"Start CC Management (user=%s, ip=%s)"	},
	{	RDM_EC_System_Stop_Management,		"Stop CC Management"	},
	{   RDM_EC_System_Progress_Begin,		"System Progress Begin "    },
	{   RDM_EC_System_Progress_End,			"System Progress End "    },
	{	RDM_EC_System_Factory_Reset,		"%s (%s) factory reset device" },
	{	RDM_EC_Modem_Connect,				"Modem connected"	},
	{	RDM_EC_Modem_Disconnect,			"Modem disconnected"	},
	{	RDM_EC_Access_Login,				"%s logged in (%s)"	},
	{	RDM_EC_Access_Logout,				"%s logged out (%s)"	},
	{	RDM_EC_Access_Login_Failed,			"%s login failed (%s)"	},
	{	RDM_EC_Access_Connection_Lost,		"%s disconnected (%s)"	},
	{	RDM_EC_Access_Connection_Timeout,	"%s timeout (%s)"	},
	{	RDM_EC_Access_Connection_Denied,	"Connection denied, system busy"	},
	{	RDM_EC_Access_Console_Login,		"%s logged into console"	},
	{	RDM_EC_Access_Console_Logout,		"%s logged out from console"	},
	{	RDM_EC_Access_Wrong_IP,				"%s connected from restricted ip"	},
	{	RDM_EC_Access_Port_Connect,			"%s enters port %s"	},
	{	RDM_EC_Access_Port_Disconnect,		"%s leaves port %s"	},
    {   RDM_EC_Path_Added,                  "Path %s added" },
    {   RDM_EC_Path_Changed,                "Path %s changed" },
    {   RDM_EC_Path_Deleted,                "Path %s deleted" },
    {   RDM_EC_Path_Change_Request,         "%s changed a path" },
    {   RDM_EC_Path_Status_Changed,         "Port %s is %s" },
	{	0,									NULL }
};

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CRDM_Notify
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CRDM_Notify::CRDM_Notify
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	pName				= "Notify";
	hashCode			= Hash32( 0, pName );
	pSubscriptionList	= NULL;
	this->pRDM			= pRDM;
	pLogFile			= NULL;
	serialNo			= 1;

	cs = OS_CreateCriticalSection( OS_CRITICAL_SECTION_NORMAL );	
}

//--------------------------------------------------------------------------------
//
	CRDM_Notify::~CRDM_Notify
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	// Remove all sub scriptions

	OS_EnterCriticalSection( cs );

	while (this->pSubscriptionList != NULL)
		RemoveSubscription( this->pSubscriptionList );

	OS_LeaveCriticalSection( cs );

	OS_DeleteCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	void
	CRDM_Notify::LogToFile
	(
		CCircularFile *pNewLogFile		// The new log file
	)
//
//	Sets the file that the events will be logged to.
//
//	Calling this method with pNewLogFile == NULL disables logging
//
//---------------------------------------------------------------------------
{
	int		tempSerialNo;

	OS_EnterCriticalSection( cs );

	// Flish the old log file, if there is one

	if (pLogFile != NULL)
		pLogFile->Flush();

	// Assign the new file

	pLogFile = pNewLogFile;

	// Get the serial No from the file

	if (pNewLogFile != NULL)
	{
		tempSerialNo = pNewLogFile->GetUserData();
	
		if (tempSerialNo > serialNo)
			serialNo = tempSerialNo;
	}

	OS_LeaveCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	int							// 0 or error code if request failed
	CRDM_Notify::Command
	(
		CSession	*	pUserSession,		// The user session or NULL for super user
		CSXDB_Node	*	pRootNode,		// Ptr to the root node of our functions
		CSIO		*	pResponse		// The response -
								// Contains the return value on exit
	)
//
//	Processes an RDMP request for the notification service and returns the response.
//
//------------------------------------------------------------------------------//
{
	const char		*pFunctionName;
	int			cmdID;
	int			result;
	CSXDB_Node	*pNode;

	// Get ptr to first function node

	pNode = this->GetFirstFunction( pRootNode );

	while (pNode != NULL)
	{
		// Function Name

		pFunctionName = pNode->GetName();

		// Find function

		cmdID = GetTokenValue( pFunctionName, functions );

		// Exexute the function

		pResponse->Printf("<%s>",pFunctionName);

		switch (cmdID)
		{
			case CMD_Version:
				pResponse->Puts("1");
				break;

			case CMD_Subscribe:
				this->Subscribe( pUserSession, pNode, pResponse );
				break;

			case CMD_Unsubscribe:
				this->Unsubscribe( pUserSession, pNode, pResponse );
				break;

			case CMD_Notify:
				this->NotifyFunction( pUserSession, pNode, pResponse );
				break;

			case CMD_GetEvents:
				this->GetEventsFunction( pUserSession, pNode, pResponse );
				break;

			default:
				this->PrintError( pResponse, RDM_ERROR_FUNCTION_NOT_FOUND );
				break;

		}

		result = pResponse->Printf("</%s>",pFunctionName);

		pNode = this->GetNextFunction(pNode);

	}

	return 0;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_Notify::Subscribe
	(
		CSession	*pUserSession,		// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Subscribe function
//
//---------------------------------------------------------------------------
{
	SUB_FUNC_DATA		sub;
	int					result = 0;;
	CRDM_Remote_Notify	*pSub;
	int					x;

	// Default values

	memset(&sub,0, sizeof(sub));

	// Parse the function

	result = SXDB_PT_Get(pNode, subTable, &sub, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	if ( (sub.pXPath != NULL && sub.pIDs != NULL) || (sub.pXPath == NULL && sub.pIDs == NULL) )
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	if (pUserSession == NULL)
		return PrintError( pResponse, RDM_ERROR_INVALID_SESSION );

	// Do it

	pSub = new CRDM_Remote_Notify;

	if (pSub == NULL)
		result = RDM_ERROR_MEMORY_ERROR;
	else
	{
		pSub->subscriptionID= nextSubscriptionID++;
		pSub->pUserSession	= pUserSession;
		pSub->time			= sub.time == NULL ? 0 : 1;
		pSub->serialNo		= sub.serialNo == NULL ? 0 : 1;
		pSub->nodeID		= sub.nodeID == NULL ? 0 : 1;
		pSub->parentNodeID	= sub.parentNodeID == NULL ? 0 : 1;
		pSub->subID			= sub.subID == NULL ? 0 : 1;
		pSub->sendData		= sub.sendData == NULL ? 0 : 1;
		pSub->deviceID		= sub.deviceID == NULL ? 0 : 1;

		if (sub.pIDs != NULL)
		{
			// Subscription has a list of IDs

			result = pSub->SetSubscription( sub.pIDs, sub.pEvents );

			result = result ? 0 : RDM_ERROR_BAD_PARAMETER;
		}
		else
		{
			// Subscription from an xPath

			CSXPath	p;

			result = p.Parse( sub.pXPath, pRDM->db, NULL, 0, pUserSession );

			if (result > 0)
			{
				// Add the IDs for all found nodes

				if (pSub->Allocate( result ) == 0)
				{

					for (x=0;x<result;x++)
					{
						// Get the ID from the found nodes

						pNode = p.Enum(x);
						pNode = pNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "id", NULL);
						if (pNode)
							pSub->SetID( x, pNode->GetData() );
					}

					// Add the events to the subscription

					pSub->AddEvents( sub.pEvents );
				}
				else
					result = RDM_ERROR_BAD_PARAMETER;
			}
			else if(result == 0)
				result = RDM_ERROR_DEVSET_NODE_NOT_FOUND;
		}
	}

	// Show results

	if (result < 0)
	{
		// Error...delete the subscription and return error

		if (pSub)
			delete pSub;

		return PrintError( pResponse, result );
	}
	else
	{
		// Everything is good, add the subscription and return sub ID

		this->AddSubscription( pSub );
		pUserSession->AddSessionObject( pSub );

		return pResponse->Printf("<SubscriptionID>%d</SubscriptionID>", pSub->subscriptionID);
	}
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_Notify::Unsubscribe
	(
		CSession	*pUserSession,		// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Unsubscribe function
//
//---------------------------------------------------------------------------
{
	UNSUB_FUNC_DATA		unsub;
	int					result = 0;;
	CRDM_Subscription	*pSub;

	OS_EnterCriticalSection( cs );
	
	// Default values

	memset(&unsub,0, sizeof(unsub));

	// Parse the function

	result = SXDB_PT_Get(pNode, unsubTable, &unsub, 0);

	if (result != 0)
	{
		result = PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Find the subscription

	pSub = this->pSubscriptionList;

	while (pSub)
	{
		if (pSub->GetSubID() == unsub.subID)
			break;

		pSub = pSub->pNext;
	}

	// Error if no matching sub found

	if (pSub == NULL)
	{
		result =  PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Make sure the sub belongs to this user

	if ( ((CRDM_Remote_Notify *) pSub)->pUserSession != pUserSession )
		pSub = NULL;

	// Error if no matching sub found

	if (pSub == NULL)
	{
		result =  PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Delete the subscription

	RemoveSubscription( pSub );

	if (((CRDM_Remote_Notify *)pSub)->pUserSession != NULL)
	{
		((CRDM_Remote_Notify *) pSub)->pUserSession->RemoveSessionObject( (CRDM_Remote_Notify *) pSub );
		delete pSub;
	}

EXIT:
	OS_LeaveCriticalSection( cs );
	return result;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_Notify::NotifyFunction
	(
		CSession	* NOTUSED(pUserSession),// User session
		CSXDB_Node	* pNode,		// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the Notify function
//
//---------------------------------------------------------------------------
{
	NOTIFY_FUNC_DATA	notify;
	int					result = 0;
	int					event;

	OS_EnterCriticalSection( cs ); // HACK - optimize
	
	// Default values

	memset(&notify,0, sizeof(notify));

	// Parse the function

	result = SXDB_PT_Get(pNode, notifyTable, &notify, 0);

	if (result != 0)
	{
		result = PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Convert the Event name to an event ID

	event = CRDM_EventCode::GetIDByName( notify.pEvent );

	if (event == RDM_EC_Invalid)
	{
		result = PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Send the event
	
	Notify( notify.pID, event, notify.pData );
	
	// Done
EXIT:
	OS_LeaveCriticalSection( cs );
	
	return result;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_Notify::GetEventsFunction
	(
		CSession	* NOTUSED(pUserSession),// User session
		CSXDB_Node	* pNode,		// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the GetEvents function
//
//---------------------------------------------------------------------------
{
	GETEVENTS_FUNC_DATA	getEvents;
	int					result = 0;
	int					start = RDM_NOTIFY_OLDEST;
	int					direction = 1;
	int					xmlFormat = 1;
	RDM_EVENT		*	events = NULL;
	char				msg[RDM_MAX_NOTIFICATION];
	int					x;

	events = new RDM_EVENT[32];

	if (events == NULL)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Default values

	memset(&getEvents,0, sizeof(getEvents));

	getEvents.count = 1;

	// Parse the function

	result = SXDB_PT_Get(pNode, getEventsTable, &getEvents, 0);

	if (result != 0)
	{
		result = PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Get the parameters

	if (strcmp(getEvents.pStart, "Oldest") == 0)
		start = RDM_NOTIFY_OLDEST;
	else if (strcmp(getEvents.pStart, "Newest") == 0)
		start = RDM_NOTIFY_NEWEST;
	else
		start = atoi(getEvents.pStart);

	if (getEvents.count > 32)
		getEvents.count = 32;

	if (getEvents.pDir != NULL && *getEvents.pDir == '-')
		direction = -1;

	if (getEvents.pFormat != NULL && strcmp(getEvents.pFormat,"Text") == 0)
		xmlFormat = 0;

	// Get the events
	
	result = GetEvents( &events[0], start, getEvents.count * direction );

	if (result < 0)
	{
		result = PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
		goto EXIT;
	}

	// Output the events

	for (x=0;x<result;x++)
	{
		if (xmlFormat)
		{
			RDM_ConvertEventToXML( &events[x], msg );
			pResponse->Puts( msg );
		}
		else
		{
			RDM_ConvertEventToText( &events[x], msg );
			pResponse->Printf("<Text serialNo=\"%d\">", events[x].serialNo);
			SXMLOut( pResponse, msg );
			pResponse->Printf("</Text>\n");
		}
	}

	// Done
EXIT:

	if (events != NULL)
		/* FIXME: use delete[] ? */
		delete events;
	
	return result;
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Notify::AddSubscription
	(
		CRDM_Subscription	*pSub		// Ptr to the subscription to add
	)
//
//	Adds a subscription to the notification manager
//
//------------------------------------------------------------------------------//
{
	OS_EnterCriticalSection( cs );

	pSub->pNext = pSubscriptionList;
	pSub->pNotifyObject = this;
	pSubscriptionList = pSub;

	OS_LeaveCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Notify::RemoveSubscription
	(
		CRDM_Subscription	*pSub		// Ptr to the subscription to remove
	)
//
//	Removes a subscription from the notification manager
//
//------------------------------------------------------------------------------//
{
	CRDM_Subscription *pTemp;

	if (pSub == NULL || pSub->pNotifyObject == NULL)
		return;
		
	OS_EnterCriticalSection( cs );

	if (pSub->pNotifyObject->pSubscriptionList == pSub)
		pSub->pNotifyObject->pSubscriptionList = pSub->pNext;
	else
	{
		for (pTemp = pSub->pNotifyObject->pSubscriptionList; pTemp != NULL && pTemp->pNext != pSub; pTemp = pTemp->pNext)
			;

		if (pTemp != NULL)
			pTemp->pNext = pSub->pNext;
	}

	pSub->pNext = NULL;
	pSub->pNotifyObject = NULL;

	OS_LeaveCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Notify::Notify
	(
		const char	*	pID,				// Ptr to the Unique ID
		int			event,				// The event code
		const char	*	pData				// Ptr to the data 
	)
//
//	Sends a notification
//
//------------------------------------------------------------------------------//
{
	RDM_EVENT			eventRecord;
	CRDM_Subscription	*pSub;
	int					length;

	if (pID == NULL)
		pID = "";

	int		hash = Hash32( 0, pID );

	OS_EnterCriticalSection( cs );

	// Assemble the event record

	if (event == RDM_EC_Heart_Beat || event == RDM_EC_Progress_Update)
		eventRecord.serialNo = 0;
	else
		eventRecord.serialNo = serialNo++;
	strncpy( eventRecord.id, pID, RDM_MAX_ID-1 );
    eventRecord.id[RDM_MAX_ID-1] = '\0';
	RDM_GetSystemTime( &eventRecord.time );
	eventRecord.event = (BYTE) event;
	if (pData != NULL)
	{
		strncpy( eventRecord.data, pData, RDM_MAX_NOTIFICATION_DATA-1 );
		eventRecord.data[RDM_MAX_NOTIFICATION_DATA-1] = 0;
		length = 1 + strlen(pData) + sizeof(RDM_EVENT) - RDM_MAX_NOTIFICATION_DATA;
	}
	else
	{
		length = 1 + sizeof(RDM_EVENT) - RDM_MAX_NOTIFICATION_DATA;
		eventRecord.data[0] = 0;
	}

	// Save the event in the log

	if (pLogFile != NULL && eventRecord.serialNo != 0)
		pLogFile->Write( (char *) &eventRecord, length );

	// Find subscriptions that match this event

	pSub = pSubscriptionList;

	while (pSub != NULL)
	{
		if (pSub->Test( pID, hash, event ))
		{
			// Send the event to the subscription

			pSub->Notify( &eventRecord );
		}

		pSub = pSub->pNext;
	}

	OS_LeaveCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	int									// The serialNo of the found event
										// 0 = no events in file
	CRDM_Notify::FindEvent
	(
		int			_serialNo			// The event serial number
	)
//
//	Sets the file position to the event with the specified serialNo.
//	If serialNo > than the newest, then position is at the end of file
//		and RDM_NOTIFY_NEWEST is returned
//	If serialNo < than the oldest, then position is at begining of file
//		and RDM_NOTIFY_OLDEST is returned
//	NOTE: This function assumes the caller has the critical section
//
//------------------------------------------------------------------------------//
{
	int		result;
	int		foundSerialNo;

	if (pLogFile == NULL)
		return -1;

	pLogFile->Rewind();

	result = pLogFile->Read( (char *) &foundSerialNo, sizeof(foundSerialNo), 0 );

	if (result < (int)sizeof(foundSerialNo) )
		return 0;

	if (foundSerialNo > _serialNo)
		return RDM_NOTIFY_OLDEST;

	if (foundSerialNo == _serialNo)
		return foundSerialNo;

	result = pLogFile->Seek( _serialNo - foundSerialNo );

	if (result < 0)
		return 0;

	if (result == _serialNo - foundSerialNo)
		return _serialNo;

	return RDM_NOTIFY_NEWEST;
}

//--------------------------------------------------------------------------------
//
	int									// # of events returned
	CRDM_Notify::GetEvents
	(
		RDM_EVENT *	pEvent,				// Where to put the event data
		int		_serialNo,			// The starting serialNo
		int		count				// # of events to get 
										// (+ = get newer events, - = get older events)
	)
//
//	Returns events starting from specified context position. The sign of inCount
//	determines if newer or older events will be returned.
//
//------------------------------------------------------------------------------//
{
	int		found = 0;
	int		result;
	int		direction = count >=0 ? 1 : -1;

	OS_EnterCriticalSection( cs );

	result = FindEvent( _serialNo );

	if (result == 0)
		goto EXIT;

	while (count != 0)
	{
		memset(pEvent,0,sizeof(*pEvent));

		result = pLogFile->Read( (char *) pEvent, 0, direction );

		if (result < 0)
			break;

		count -= direction;
		found ++;
		pEvent++;
	}

EXIT:

	OS_LeaveCriticalSection( cs );

	return found;
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CRDM_Notify::Sync
	(
		CSIO	*pDestFile			// File to write to
	)
//
//	Syncronizes the log file SIO (usually memory) with another file (usually
//	on disk).
//
//--------------------------------------------------------------------------//
{
	int		result = 0;

	if (pLogFile != NULL)
	{
		OS_EnterCriticalSection( cs );
		pLogFile->SetUserData( serialNo );
		result = pLogFile->Sync( pDestFile );
		OS_LeaveCriticalSection( cs );
	}

	return result;
}

//--------------------------------------------------------------------------------
//
	int
	CRDM_Notify::GetLastSerialNo
	(
	)

//

//------------------------------------------------------------------------------//	

{

	return (this->serialNo - 1);

}





//--------------------------------------------------------------------------------
//
	void
	RDM_ConvertEventToXML
	(
		RDM_EVENT *	pEvent,				// Where to put the event data
		char	  * pString,			// Where to put the text
										// Must be <RDM_MAX_NOTIFICATION> bytes long
		int			useTime,			// !0 = include time
		int			useID,				// !0 = include Node ID
		int			useSerialNo,		// !0 = include serial number
		int			useData,			// !0 = include data
		int			subID,				// >=0 = include this subscription ID
		int			useDeviceID			// !0 = include deviceID data
	)
//
//	Converts an event record into an XML event string.
//
//	<e ec="11"><t>1234-12-12 15:45:02</t><s>12341</s><i>1234567890</i><n>1234567890123456</n><d><User>Admin</User></d></e>
//
//------------------------------------------------------------------------------//
{
	char	*p = pString;
	const char 	*deviceID = RDM_GetDeviceID();

	if (!useDeviceID || deviceID == NULL)
		p += sprintf(p,"<e ec=\"%d\">",pEvent->event);
	else
		p += sprintf(p,"<e ec=\"%d\" id=\"%s\">",pEvent->event,deviceID);

	if (useTime)
		p += sprintf(p,"<t>%04d-%02d-%02d %02d:%02d:%02d</t>",
							pEvent->time.year,
							pEvent->time.month,
							pEvent->time.day,
							pEvent->time.hour,
							pEvent->time.minute,
							pEvent->time.second
					);

	if (useID)
		p += sprintf(p,"<n>%s</n>",pEvent->id);

	if (useSerialNo)
		p += sprintf(p,"<s>%d</s>",pEvent->serialNo);

	if (subID >= 0)
		p += sprintf(p,"<i>%d</i>",subID);

	if (useData)
		p += sprintf(p,"<d>%s</d>",pEvent->data);

	strcpy(p,"</e>");
}

//--------------------------------------------------------------------------------
//
	void
	RDM_ConvertEventToText
	(
		RDM_EVENT *	pEvent,				// Where to put the event data
		char	  * pString,			// Where to put the text
										// Must be <RDM_MAX_TEXT_EVENT> bytes long
		BOOL		bUTCFormat
	)
//
//	Converts an event record into an text event string.
//
//------------------------------------------------------------------------------//
{
	CSXDB_Parse	db;
	char		*p = pString;
	int		result;
	const char	*pNull = "";
	const char	*pFormat = pNull;
	const char	*pInvalidFormat = pNull;
	const char	*data;
	char		*pData1 = NULL;
	char		*pData2 = NULL;
	char		*pData3 = NULL;
	unsigned int	availableSpace;
	int		x;
	char		xml[RDM_MAX_NOTIFICATION];
	RDM_TIME	mytime;
	char		time_str[32];
	size_t		time_str_len = 0;

	// Get the format string
	for (x=0; eventName[x].pName != NULL; x++)
	{
		if (eventName[x].eventCode == pEvent->event)
		{
			pFormat = eventName[x].pName;
			break;
		}
		else if (eventName[x].eventCode == RDM_EC_Invalid)
			pFormat = pInvalidFormat = eventName[x].pName;
	}		

	// Create the time string
	if ( bUTCFormat ) {
		mytime = pEvent->time;
	} else {
		RDM_ConvertUTCToLocalTime ( &pEvent->time, &mytime );
	}

	snprintf(time_str, sizeof(time_str), "%04u-%02u-%02u %02u:%02u:%02u %s",
		 mytime.year, mytime.month, mytime.day, mytime.hour,
		 mytime.minute, mytime.second, bUTCFormat ? "GMT " : "");
	time_str_len = strlen(time_str);

	// Figure out how much space is left in the string
	availableSpace = RDM_MAX_TEXT_EVENT - 1 - strlen(pFormat) - time_str_len;

	// Parse the XML data
	snprintf(xml, sizeof(xml), "<d>%s</d>", pEvent->data);
	result = db.Parse( xml, strlen( xml ) );

	if (result >= 0)
	{
		// Get the data strings needed for the event

		switch (pEvent->event)
		{
			// These events need /Device/Name
			case RDM_EC_Device_Added:
			case RDM_EC_Device_Changed:
			case RDM_EC_Device_Deleted:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Name"));
				break;

			case RDM_EC_Device_Status_Changed:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Name"));

				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/@Status"));
				break;

			case RDM_EC_Device_Update_Started:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/User/Name"));
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Firmware/Version"));
				break;

			case RDM_EC_Device_Update_Completed:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Firmware/Status"));
				break;

			case RDM_EC_Device_Config_Backup:
			case RDM_EC_Device_Config_Restore:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/User/Name"));
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Config/Type"));
				pData3 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Config/Ver"));
				break;

			case RDM_EC_Device_PowerSupply_Status_Changed :
				data = db.GetDataBase()->GetData("/d/PowerSupply/@Status");
				if (data) {
					switch (atoi(data)) {
					  case 0:
						pData2 = strdup("unavailable");
						break;
					  default:
					  case 1:
						pData2 = strdup("available");
						break;
					}
				}
				if (pData2) availableSpace -= strlen(pData2);
				pData1 = pp_strndup_safe(db.GetDataBase()->GetData("/d/PowerSupply/Name"), availableSpace);
				break;

			case RDM_EC_Device_Reset:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Name"));
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/DeviceResetMode"));
				break;

			// These events need /User/Name
			case RDM_EC_Device_Change_Request:
			case RDM_EC_Device_Setting_Change_Request:
			case RDM_EC_Device_Key_Event:

			case RDM_EC_Port_Change_Request:
			case RDM_EC_User_Added:
			case RDM_EC_User_Changed:
			case RDM_EC_User_Deleted:
			case RDM_EC_User_Change_Request:
			case RDM_EC_Group_Change_Request:
			case RDM_EC_Access_Console_Login:
			case RDM_EC_Access_Console_Logout:
			case RDM_EC_Access_Wrong_IP:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/User/Name"));
				break;

			case RDM_EC_User_Password_Changed:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/PasswdChg/LoginUser/Name"));
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/PasswdChg/User/Name"));		// Target user being modified
				break;

			case RDM_EC_Path_Added:
			case RDM_EC_Path_Changed:
			case RDM_EC_Path_Deleted:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Port/Path/@id"));
				break;

			case RDM_EC_Path_Change_Request:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/User/Name"));
				break;

			case RDM_EC_Path_Status_Changed:
				data = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Port/Path/@Status"));
				if (data) {
					switch (atoi(data)) {
					  case 0:
						pData2 = strdup("unavailable");
						break;
					  default:
					  case 1:
						pData2 = strdup("available");
						break;
					  case 2:
						pData2 = strdup("in use");
						break;
					}
				}
				if (pData2) availableSpace -= strlen(pData2);
				pData1 = pp_strndup_safe(db.GetDataBase()->GetData("/d/Device/Port/Path/@id"), availableSpace);
				break;

			// These events need /Device/Port/Name
			case RDM_EC_Port_Added:
			case RDM_EC_Port_Changed:
			case RDM_EC_Port_Deleted:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Port/Name"));
				break;

			// This event needs /Device/Port/Name & /Device/Port@Status
			case RDM_EC_Port_Status_Changed:
				data = pp_strdup_safe(db.GetDataBase()->GetData("/d/Device/Port/@Status"));
				if (data) {
					switch (atoi(data)) {
					  case 0:
						pData2 = strdup("unavailable");
						break;
					  default:
					  case 1:
						pData2 = strdup("available");
						break;
					  case 2:
						pData2 = strdup("in use");
						break;
					}
				}
				if (pData2) availableSpace -= strlen(pData2);
				pData1 = pp_strndup_safe(db.GetDataBase()->GetData("/d/Device/Port/Name"), availableSpace);
				break;

			// These events need /Group/Name
			case RDM_EC_Group_Added:
			case RDM_EC_Group_Changed:
			case RDM_EC_Group_Deleted:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Group/Name"));
				break;

			// These events need /Error
			case RDM_EC_System_Init_Error:
			case RDM_EC_System_Fatal_Error:
			case RDM_EC_System_Run_Error:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Error"));
				break;

			// These events need /Text
			case RDM_EC_System_General:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Text"));
				break;

			case RDM_EC_System_Start_Management:
			case RDM_EC_System_Factory_Reset:
				pData1 = pp_strdup_safe(db.GetDataBase()->GetData("/d/User/Name"));
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/User/IPAddress"));
				break;	

			case RDM_EC_System_Stop_Management:
			case RDM_EC_System_Progress_Begin:
			case RDM_EC_System_Progress_End:
				break;

			// These events need /User/Name & /IPAddress
			case RDM_EC_Access_Login:
			case RDM_EC_Access_Logout:
			case RDM_EC_Access_Login_Failed:
			case RDM_EC_Access_Connection_Lost:
			case RDM_EC_Access_Connection_Timeout:
			case RDM_EC_Access_Connection_Denied:
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/IPAddress"));
				if (pData2) availableSpace -= strlen(pData2);
				pData1 = pp_strndup_safe(db.GetDataBase()->GetData("/d/User/Name"), availableSpace);
				break;
		
			// These events need /User/Name & /Port/Name
			case RDM_EC_Access_Port_Connect:
			case RDM_EC_Access_Port_Disconnect:
				pData2 = pp_strdup_safe(db.GetDataBase()->GetData("/d/Port/Name"));
				if (pData2) availableSpace -= strlen(pData2);
				pData1 = pp_strndup_safe(db.GetDataBase()->GetData("/d/User/Name"), availableSpace);
				break;

			default:
				pData1 = pp_strdup_safe(pEvent->data);
//				pFormat = pInvalidFormat;
				break;

		}
	}

	// Create the message
	snprintf(p, RDM_MAX_TEXT_EVENT, "%s", time_str);
	snprintf(p + time_str_len,
		 RDM_MAX_TEXT_EVENT - time_str_len,
		 pFormat,
		 pData1 ? pData1 : "",
		 pData2 ? pData2 : "",
		 pData3 ? pData3 : "");

	// Free allocated data
	free(pData1);
	free(pData2);
	free(pData3);
}
